/* ckuuid.c, Test program for Kristoffer Eriksson's uid-managing code. */
/* fdc, 5-26-90 */
/* Modified by Dean Long, 5-28-90 */
/* Modified ske, fdc, 6-10-90 */

/*

INSTRUCTIONS FOR TESTING THIS CODE

Compile and load the program in one of the following ways:

   $ cc -DANYBSD -DSAVEDUID -o ckuuid ckuuid.c     (1)
   $ cc -DANYBSD -o ckuuid ckuuid.c                (2)
   $ cc -DANYBSD -DNOSETREU -o ckuuid ckuuid.c     (3)
   $ cc -o ckuuid ckuuid.c                         (4)

(1) is for Berkeley-based systems that have setregid() and setreuid() and that
    have the saved-original-effective-uid feature, similar to AT&T System V.

(2) is for Berkeley-based systems that have setregid and setreuid, but do not
    have the saved-original-effective-uid feature.

(3) is for Berkeley-based systems that don't have setregid and setreuid.

(4) is for all others, including all AT&T-based versions, Xenix, etc.

If you don't know which one applies to your system, try all three and compare
the results.  If you try (1) or (2), but get complaints about _setreuid and
_setregid not found, let me know which version of UNIX you are running, and
try (3).

After building the program, run it to make sure that the uid's don't change
(they shouldn't if the program is not setuid'd).

Now make the program setuid and setgid to someone else:

   $ su
   Password: xxxxx
   su% chown <somebody-else> ckuuid
   su% chmod ug+s ckuuid
   su% exit

and then run it, recording the results.

   $ script
   $ who am i
   $ ls -lg ./ckuuid
   $ ./ckuuid
   $ exit

See if the uid's and gid's ever change after the first time, and report the
results (mail the typescript file) back to fdc@watsun.cc.columbia.edu, letting
me know exactly what kind of machine you have, and which version of UNIX.
Thanks!
*/ 

/* Includes */

#include <stdio.h>

/* UID-holding variables */

int ttpuid, ttpgid;
int uida, uidb;
int gida, gidb;

/* Function to check and print the current real and effective uid and gid */

chuid() {
    uida = getuid();
    uidb = geteuid();
    printf(" getuid = %d, geteuid = %d\n",uida,uidb);
    gida = getgid();
    gidb = getegid();
    printf(" getgid = %d, getegid = %d\n",gida,gidb);
}

/* Main program */

main() {
    int x;

/* Announce which options we were compiled with. */

#ifdef ANYBSD
#ifndef SAVEDUID
      printf("BSD, with SAVEDUID not defined\n");
#else
      printf("BSD, with SAVEDUID defined\n");
#endif
#ifdef NOSETREU
      printf("No setre[ug]id, using set[ug]id\n");
#else
      printf("Using setre[ug]id\n");
#endif
#else
      printf("Not BSD\n");
#endif

/* Print uids and gids before, during and after switching back & forth. */

    printf("ids at startup...\n");
    chuid();

    /* Initialize uid package, change to real uid. */
    x = priv_ini();
    printf("priv_ini returns %d\n",x);
    chuid();

    /* Try to change back to effective uid */
    x = priv_on();
    printf("priv_on returns %d\n",x);
    chuid();

    /* Change back to real uid */
    x = priv_off();
    printf("priv_off returns %d\n",x);
    chuid();

    /* Change back to real uid for a fork*/
    x = priv_can();
    printf("priv_can returns %d\n",x);
    chuid();

    /* Try to change back to effective uid */
    x = priv_on();
    printf("priv_on returns %d\n",x);
    chuid();

    exit(0);
}
  
/* 
  Starting here is Kristoffer's code, as modified by fdc, dlong, et al.
*/

/*
  setuid package, by Kristoffer Eriksson, with contributions from Dean
  Long and fdc.
*/

#ifndef AIX370
extern int getuid(), getgid(), geteuid(), getegid(), getreuid(), getregid();
#endif

/*
Subject: Set-user-id
To: fdc@watsun.cc.columbia.edu (Frank da Cruz)
Date: Sat, 21 Apr 90 4:48:25 MES
From: Kristoffer Eriksson <ske@pkmab.se>

This is a set of functions to be used in programs that may be run set-user-id
and/or set-group-id. They handle both the case where the program is not run
with such privileges (nothing special happens then), and the case where one
or both of these set-id modes are used.  The program is made to run with the
user's real user and group ids most of the time, except for when more
privileges are needed.  Don't set-user-id to "root".

This works on System V.  On BSD, it depends on the "saved-set-user-id" feature.
*/

#define UID_ROOT 0
#define GID_ROOT 0

/* User and group ID:s. */

static int realuid = -1, privuid = -1;

static int realgid = -1, privgid = -1;

/* Called as early as possible in a set-uid or set-gid program to store the
 * set-to uid and/or gid and step down to the users real uid and gid. The
 * stored id:s can be temporarily restored (allowed in System V) during
 * operations that require the privilege. Most of the time, the program
 * should execute in unpriviliged state, to not impose any security threat.
 *
 * Note: Don't forget that access() always uses the real id:s to determine
 * file access, even with privileges restored.
 *
 * Returns an error mask, with error values or:ed together:
 * 1 if setuid() fails,
 * 2 if setgid() fails, and
 * 4 if the program is set-user-id to "root", which can't be handled.
 *
 * Only the return value 0 indicates real success. In case of failure,
 * those privileges that could be reduced have been, at least, but the
 * program should be aborted none-the-less.
 *
 * Also note that these functions do not expect the uid or gid to change
 * without their knowing. It may work if it is only done temporarily, but
 * you're on your own.
 */
priv_ini() {
    int err = 0;

    /* Save real ID:s. */
    realuid = getuid();
    realgid = getgid();

    /* Save current effective ID:s, those set to at program exec. */
    ttpuid = privuid = geteuid();
    ttpgid = privgid = getegid();

    /* If running set-uid, go down to real uid, otherwise remember that
     * no privileged uid is available.
     *
     * Exceptions:
     *
     * 1) If the real uid is already "root" and the set-uid uid (the
     * initial effective uid) is not "root", then we would have trouble
     * if we went "down" to "root" here, and then temporarily back to the
     * set-uid uid (not "root") and then again tried to become "root". I
     * think the "saved set-uid" is lost when changing uid from effective
     * uid "root", which changes all uid, not only the effective uid. But
     * in this situation, we can simply go to "root" and stay there all
     * the time. That should give sufficient privilege (understatement!),
     * and give the right uids for subprocesses.
     *
     * 2) If the set-uid (the initial effective uid) is "root", and we
     * change uid to the real uid, we can't change it back to "root" when
     * we need the privilege, for the same reason as in 1). Thus, we can't
     * handle programs that are set-user-id to "root" at all. The program
     * should be aborted. Use some other uid. "root" is probably to
     * privileged for such things, anyway. (The uid is reverted to the
     * real uid until abortion.)
     *
     * These two exceptions have the effect that the "root" uid will never
     * be one of the two uids that are being switched between, which also
     * means we don't have to check for such cases in the switching
     * functions.
     *
     * Note that exception 1) is handled by these routines (by constantly
     * running with uid "root", while exception 2) is a serious error, and
     * is not provided for at all in the switching functions.
     */
    if (realuid == privuid)
	privuid = -1;			/* Not running set-user-id. */

    /* If running set-gid, go down to real gid, otherwise remember that
     * no privileged gid is available.
     *
     * There are no exception like there is for the user id, since there
     * is no group id that is privileged in the manner of uid "root".
     * There could be equivalent problems for group changing if the
     * program sometimes ran with uid "root" and sometimes not, but
     * that is already avoided as explained above.
     *
     * Thus we can expect always to be able to switch to the "saved set-
     * gid" when we want, and back to the real gid again. You may also
     * draw the conclusion that set-gid provides for fewer hassles than
     * set-uid.
     */

    if (realgid == privgid)		/* If not running set-user-id, */
      privgid = -1;			/*  remember it this way. */

    err = priv_off();			/* Turn off setuid privilege. */

    if (privuid == UID_ROOT)		/* If setuid to root, */
      err |= 4;				/* return this error. */

    if (realuid == UID_ROOT)		/* If real id is root, */
      privuid = -1;			/* stay root at all times. */

    return(err);
}


/* Macros for hiding the differences in UID/GID setting between various Unix
 * systems. These macros should always be called with both the privileged ID
 * and the non-privileged ID. The one in the second argument, will become the
 * effective ID. The one in the first argument will be retained for later
 * retrieval.
 */
 
#ifdef ANYBSD
#ifdef SAVEDUID
/* On BSD systems with the save-UID feature, we just juggle the effective
 * UID back and forth, and leave the real UID to it's true value. The kernel
 * allows swithing to both the current real UID, the effective UID, and the
 * UID which the program is set-UID to. The saved set-UID always holds the
 * privileged UID for us, and the real UID will always be the non-privileged,
 * and we can freely choose one of them for the effective UID at any time.
 */
#define switchuid(hidden,active)	setreuid(-1,active)
#define switchgid(hidden,active)	setregid(-1,active)
#else   /* ANYBSD,NOSETREU,!SAVEDUID */

#ifdef NOSETREU
#define switchuid(hidden,active)	setuid(active)
#define switchgid(hidden,active)	setgid(active)

#else	/* ANYBSD,!SAVEDUID,!NOSETREU */
/* On other BSD systems, we swap the real and effective UIDs each time. It's
 * the effective UID that we are interrested in, but we have to retain the
 * unused UID somewhere to enable us to restore it later, and that we do in
 * the real UID. The kernel only allows switching to either the current real
 * or the effective UID, unless you're "root".
 */
#define switchuid(hidden,active)	setreuid(hidden,active)
#define switchgid(hidden,active)	setregid(hidden,active)

#endif
#endif

#else	/* !ANYBSD */
/* On System V, the only thing we can change is the effective UID (unless
 * the current effective UID is "root", but initsuid() avoids that for us).
 * The kernel allows switching to the current real UID or to the saved
 * set-UID. These are always set to the non-privileged UID and the privileged
 * UID, respectively, and we only change the effective UID. This breaks if
 * the current effective UID is "root", though, because for "root" setuid/gid
 * becomes more powerful, which is why initsuid() treats "root" specially.
 * Note: That special treatment maybe could be ignored for BSD?
 * Note: For systems that don't fit any of these three cases, we simply can't
 * support set-UID.
 */
#define switchuid(hidden,active)	setuid(active)
#define switchgid(hidden,active)	setgid(active)
#endif
  
/* Go to the privileged uid (gid) that the program is set-user-id
 * (set-group-id) to, unless the program is running unprivileged.
 * If setuid() fails, return value will be 1. If getuid() fails it
 * will be 2.  Return immediately after first failure, and the function
 * tries to restore any partial work done.  Returns 0 on success.
 * Group id is changed first, since it is less serious than user id.
 */
priv_on() {
    if (privgid != -1)
      if (switchgid(realgid,privgid))
        return(2);

    if (privuid != -1)
      if (switchuid(realuid,privuid)) {
	  if (privgid != -1)
	    switchgid(privgid,realgid);
	  return(1);
      }
    return(0);
}

/* Return to the unprivileged uid (gid) after an temporary visit to
 * privileged status, unless the program is running without set-user-id
 * (set-group-id). Returns 1 for failure in setuid() and 2 for failure
 * in setgid() or:ed together. The functions tries to return both uid
 * and gid to unprivileged state, regardless of errors. Returns 0 on
 * success.
 */
priv_off() {
    int err = 0;

    if (privuid != -1)
       if (switchuid(privuid,realuid))
	  err |= 1;
/*
  This added by fdc.  If the result of the above operation is that the
  real uid is the privileged uid (because of swapping), then it is possible
  for the user to start a fork, e.g. with Kermit's "!" or PUSH command,
  and swap them back in order to regain the privileges.  To prevent this,
  we must cancel all privileges permanently.
*/
    if (getuid() == privuid)
      priv_can();			/* Cancel all privileges. */

/* Now the same deal for the group id. */

    if (privgid != -1)
       if (switchgid(privgid,realgid))
	err |= 2;

/* Same safety measure as before. */

    if (getgid() == privgid)
      priv_can();			/* Cancel all privileges. */

    return(err);
}

/* Turn off privilege permanently.  No going back.  This is necessary before
 * a fork() on BSD43 machines that don't save the setUID or setGID, because
 * we swap the real and effective ids, and we don't want to let the forked
 * process swap them again and get the privilege back. It will work on other
 * machines too, such that you can rely on its effect always being the same,
 * for instance, even when you're in priv_on() state when this is called.
 * (Well, that part about "permanent" is on System V only true if you follow
 * this with a call to exec(), but that's what we want it for anyway.)
 * Added by Dean Long -- dlong@midgard.ucsc.edu
 */

priv_can() {
    int err = 0;

#ifdef ANYBSD
    if (privuid != -1)
       if (setreuid(realuid,realuid))
	  err |= 1;

    if (privgid != -1)
        if (setregid(realgid,realgid))
 	  err |= 2;

    return(err);

#else
    /* Easy way of using setuid()/setgid() instead of setreuid()/setregid().*/
    return(priv_off());

#endif /* ANYBSD */
}
