/*
 * 5799-WZQ (C) COPYRIGHT IBM CORPORATION 1986,1988
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */
/* $Header:aedloc.c 12.0$ */
/* $ACIS:aedloc.c 12.0$ */
/* $Source: /ibm/acis/usr/sys/cacons/RCS/aedloc.c,v $ */

#if !defined(lint) && !defined(NO_RCS_HDRS)
static char *rcsid = "$Header:aedloc.c 12.0$";
#endif

#include "param.h"
#include "../machinecons/screen_conf.h"
#include "../machinecons/apaaed.h"
#include "../machine/io.h"

/*
 * This module is the stuff needed to implement the "AED" display,
 * a horrible but very useful kludge that treats a wired down hunk of
 * memory as the refresh bitmap for the AED display, echoing changed
 * rectangles to the real display. In addition all screen to screen
 * copies and line drawing are done by the AED microcode.
 *
 * This version actually accepts a limited set of commands via a command
 * queue in shared memory (between the user level server and the kernel).
 *
 * Commands supported in this version are:
 *	echo rectangle from simulated bitmap to the screen (AED_ECHO_RECT)
 *	draw a line on the screen (AED_DRAW_LINE)
 *	blt from screen to screen (AED_SCREEN_COPY).
 */

/*
 * AED macros
 */

#define WAIT()  {do DELAY(1) while (GET_PC2B(aed[0]));}
#define START() {PUT_PC2B(aed[0] , 1);  DELAY(1) }

#ifdef ATR
/*
 * since the AED is at a fixed address we can optimize the aedrd
 * macros (once the 128k window is set we just use the offset)
 */
#define AED_MMAP	0x0a0000
#define AED_BASE ((char *) (AED_MMAP & MASK_128K))
#define AED_ENTER(xxx)	int window = get_128_window(); \
	xxx set_128_window(AED_MMAP);
#define AED_EXIT()	set_128_window(window)
#define MASK_128K	0x1ffff
/* general loop - not unrolled */
#define LOOP(count,s) {							\
    register int cnt;							\
    for (cnt=(count); cnt>0; --cnt) { s ; } }

#else

#define AED_MMAP	0xf40a0000
#define AED_BASE ((char *) AED_MMAP)
#define AED_ENTER(xxx)
#define AED_EXIT()
#define aed_rd(X,Y,Z) bcopy(X, Y, Z+Z)	/* use bcopy */
#define aed_wr(X,Y,Z) bcopy(X, Y, Z+Z)	/* use bcopy */

/*
 * A general unrolled loop
 */

#define LOOP(count,s) {							\
    register int cnt;							\
    for (cnt=(count); cnt>=16; cnt-=16) {s;s;s;s;s;s;s;s;s;s;s;s;s;s;s;s;}\
    switch (cnt) {							\
    case 15: s; case 14: s; case 13: s; case 12: s;			\
    case 11: s; case 10: s; case  9: s; case  8: s;			\
    case  7: s; case  6: s; case  5: s; case  4: s;			\
    case  3: s; case  2: s; case  1: s; case  0: ; }			\
}
#endif ATR

/*
 * Global defines for others to see.
 */

static int old_cursor_x, old_cursor_y;
static short *display_base;
int aed_kernel_assist = 0;
int aed_blaster();

/*
 * This is really the initialization entrance to this module.
 */

aed_apa_init()
{
    display_base = (short *)AED;

    if (aed_reset())
	return;

    AED_CMDQ_SEMA = 0;
    AED_CMDQ_INDEX = 0;
    AED_COLOR = 0;
    AED_CURSOR_LOAD = 0;
    AED_CURSOR_X = 0;
    AED_CURSOR_Y = 0;
    aed_echo_rect(0, 0, 1023, 799, 15);	/* whiten display */
    old_cursor_x = -1;
    old_cursor_y = -1;
    aed_kernel_assist = 1;
    untimeout(aed_blaster, 0); 		/* stop previous polling, if any */
    timeout(aed_blaster, 0, 0);		/* start polling chain */
}


/*
 * This is really the termination entrance to this module!
 */

aed_apa_close()
{
    untimeout(aed_blaster, 0); 		/* stop polling */
}

/*
 * Show AED cursor
 */

aed_show_loc()
{
	/*
	 * just in case X10 does call aed_show_loc
	 */
	aed_kernel_assist = 1;
   	untimeout(aed_blaster, 0); 		/* stop previous polling, if any */
    	timeout(aed_blaster, 0, 0);		/* start polling chain */
}

/*
 * Hide AED cursor
 */

aed_hide_loc()
{
	/*
	 * this routine should never be called by X10, which uses
	 * the aed's hardware cursor. If some does call us, it must
	 * mean that they want the kernel to keep it's hands of the
	 * aed display, so let aed_blaster know..
	 */
	aed_kernel_assist = 0;
}

/*
 * Position cursor entrance. Remember coordinates, then move the
 * cursor there at update time.
 */

aed_pos_loc(x, y)
{
 	AED_CURSOR_X = x;
	AED_CURSOR_Y = y;
}

/*
 * Move AED cursor to new x,y position
 */

aed_move_cursor()
{
	register short *aed = (short *)display_base;
	register x = AED_CURSOR_X, y = AED_CURSOR_Y;
	AED_ENTER();

	if (x == old_cursor_x && y == old_cursor_y)
		return;
	old_cursor_x = x;
	old_cursor_y = y;

	WAIT();

	PUT_PC2B(aed[1] , 10);		/* position cursor */
	PUT_PC2B(aed[2] , x);		/* x position */
	PUT_PC2B(aed[3] , y);		/* y position */

	START();
	AED_EXIT();
}

/*
 * Load a new cursor image. Save image in shared memory, it's sent to
 * the AED at poll time.
 */

aed_load_loc(cursor)
register Locator *cursor;
{
	*AED_CURSOR = *cursor;
	AED_CURSOR_LOAD = 1;
}

/*
 * Send cursor to AED
 */

aed_set_cursor()
{
    register short *aed = (short *)display_base;
    register Locator *cursor = AED_CURSOR;
    int flipit = AED_COLOR ? 0 : -1;
    AED_ENTER();

    WAIT();

    if (cursor == (Locator *)0)
	PUT_PC2B(aed[1] , 9);			/* disable cursor */
    else {
	register short *bbits;
	register short *wbits;
	int col;
	int cursor_width, cursor_height;
	short *black_bits, *white_bits;
	
	PUT_PC2B(aed[1] , 8);			/* enable cursor */
	START();
	WAIT();

	cursor_width = 16;
	cursor_height = 16;
	black_bits = aed + 4;
	white_bits = aed + 260;

#define OFFSET 16	/* How much our little cursor is offset in hardware */

	PUT_PC2B(aed[1] , 7);				/* define cursor */
	PUT_PC2B(aed[2] , cursor->hotSpot.h + OFFSET);	/* x offset */
	PUT_PC2B(aed[3] , cursor->hotSpot.v + OFFSET);	/* y offset */

	/*
	 * black
	 */

	bbits = (short *) &cursor->data[0];
	wbits = (short *) &cursor->mask[0];

	for (col=0; col<64; col += 16) {
	    register short *orders = black_bits + col / 16;

	    if (col < OFFSET) {
		LOOP(64, (PUT_PC2B(*orders , 0), orders += 4));
	    } else if (col < cursor_height + OFFSET) {
		LOOP(OFFSET, (PUT_PC2B(*orders , 0), orders += 4));
		LOOP(cursor_width,
			(PUT_PC2B(*orders , (*bbits++) ^ (flipit ? *wbits++ : 0)),
							orders += 4));
		LOOP(64 - cursor_width - OFFSET, (PUT_PC2B(*orders , 0), orders += 4));
	    } else {
		LOOP(64, (PUT_PC2B(*orders , 0), orders += 4));
	    }
	}

	/*
	 * white
	 */

	bbits = (short *) &cursor->data[0];
	wbits = (short *) &cursor->mask[0];

	for (col=0; col<64; col += 16) {
	    register short *orders = white_bits + col / 16;

	    if (col < OFFSET) {
		LOOP(64, (PUT_PC2B(*orders , 0), orders += 4));
	    } else if (col < cursor_height + OFFSET) {
		LOOP(OFFSET, (PUT_PC2B(*orders , 0), orders += 4));
		LOOP(cursor_width,
		 (PUT_PC2B(*orders , *wbits & ~((*bbits++) ^ (flipit ? *wbits : 0))),
					wbits++, orders += 4));
		LOOP(64 - cursor_width - OFFSET, (PUT_PC2B(*orders , 0), orders += 4));
	    } else {
		LOOP(64, (PUT_PC2B(*orders , 0), orders += 4));
	    }
	}
    }

    START();
    AED_EXIT();
}

/*
 * Routine run at hardclock speed to tell the real display to
 * do things requested via a command queue from the server.
 */

aed_blaster()
{
	register i;
	AED_ENTER();

	if (aed_kernel_assist == 0) {
		return;
	}

	if (AED_CURSOR_LOAD) {
		aed_set_cursor();
		AED_CURSOR_LOAD = 0;
	}
	aed_move_cursor();
		
	if (AED_CMDQ_SEMA == 0) {
		for (i = 0; i < AED_CMDQ_INDEX; ++i) {
			register struct aed_cmd *ptr = &AED_CMDQ[i];

			switch (ptr->cmd) {

			case AED_ECHO_RECT:
				aed_echo_rect(ptr->ECHO_RECT.origin_x,
					  ptr->ECHO_RECT.origin_y,
					  ptr->ECHO_RECT.corner_x -
						ptr->ECHO_RECT.origin_x,
				          ptr->ECHO_RECT.corner_y -
						ptr->ECHO_RECT.origin_y,
					  12);
				break;
				
			case AED_SCREEN_COPY:
				aed_screen_copy(ptr->SCREEN_COPY.from_x,
					    ptr->SCREEN_COPY.from_y,
					    ptr->SCREEN_COPY.to_x,
					    ptr->SCREEN_COPY.to_y,
					    ptr->SCREEN_COPY.width,
					    ptr->SCREEN_COPY.height,
					    ptr->SCREEN_COPY.rule);
				break;

			case AED_DRAW_LINE:
				aed_draw_line(ptr->DRAW_LINE.from_x,
					     ptr->DRAW_LINE.from_y,
					     ptr->DRAW_LINE.to_x,
					     ptr->DRAW_LINE.to_y,
					     ptr->DRAW_LINE.rule,
					     ptr->DRAW_LINE.width,
					     ptr->DRAW_LINE.color,
					     ptr->DRAW_LINE.pat,
					     ptr->DRAW_LINE.patlen,
					     ptr->DRAW_LINE.top,
					     ptr->DRAW_LINE.left,
					     ptr->DRAW_LINE.bottom,
					     ptr->DRAW_LINE.right);
				break;
			}
		}
		AED_CMDQ_INDEX = 0;
	}
	timeout(aed_blaster, 0, 0);
	AED_EXIT();
}

/*
 * Memory-to-screen rasterop. Echos updated areas of simulated
 * display bitmap to the AED.
 */

aed_echo_rect (x, y, width, height, rule)
short x, y, width, height, rule;
{
	register short *aed = (short *) display_base;
	register short *orders;
	int rows_left, words_per_row, rows_per_chunk, cur_y;
	short *first_word;
	register short *word_ptr;
	AED_ENTER();

	if (width <=0 || height <= 0)
		return;

	width += x % 16;
	width = ((width + 15) / 16) * 16;
	x = (x / 16) * 16;
	rows_left = height;
	words_per_row = width / 16;

	rows_per_chunk = 2000 / words_per_row;
	if (rows_per_chunk <= 0)
		return;

	cur_y = y;
	first_word = (short *)AED_BM_ADDR + (y * 1024 / 16) + x / 16;

	while (rows_left > 0) {

		short cur_height = rows_left < rows_per_chunk ?
					rows_left : rows_per_chunk;

		WAIT();

		PUT_PC2B(aed[1] , 1);		/* orders */
		PUT_PC2B(aed[2] , rule);		/* combination rule */
		PUT_PC2B(aed[7] , x);		/* x position */
		PUT_PC2B(aed[8] , cur_y);		/* y position */
		PUT_PC2B(aed[9] , 0);		/* window x origin */
		PUT_PC2B(aed[10] , 0);		/* window y origin */
		PUT_PC2B(aed[11] , 0);		/* clip lo x */
		PUT_PC2B(aed[12] , 0);		/* clip lo y */
		PUT_PC2B(aed[13] , 1023); 	/* clip hi x */
		PUT_PC2B(aed[14] , 799);		/* clip hi y */
		PUT_PC2B(aed[15] , (short) AED_COLOR);	/* color */

		orders = aed + 17;
		PUT_PC2B(orders[0] , 9);		/* draw image */
		PUT_PC2B(orders[1] , width);		/* width */
		PUT_PC2B(orders[2] , cur_height);	/* height */
		orders += 3;

		while (cur_height-- > 0) {
			word_ptr = first_word;
			LOOP(words_per_row, PUT_PC2B(*orders++ , *word_ptr++));
			first_word += 1024 / 16;
		}
		rows_left -= rows_per_chunk;
		cur_y += rows_per_chunk;

		PUT_PC2B(aed[16] , orders - (aed + 17));

		START();
	}
	AED_EXIT();
}

/*
 * Screen-to-screen rasterop. All screen to screen copies 
 * are done by this routine and are not echoed.
 */

aed_screen_copy (xs, ys, xd, yd, width, height, rule) 
short xs, ys, xd, yd, width, height, rule;
{
	register short *aed = (short *) display_base;

	if (width <= 0 || height <= 0)
		return;

	WAIT();

	PUT_PC2B(aed[1] , 3);		/* copy */
	PUT_PC2B(aed[2] , xs);		/* source x */
	PUT_PC2B(aed[3] , ys);		/* source y */
	PUT_PC2B(aed[4] , xd);		/* dest x */
	PUT_PC2B(aed[5] , yd);		/* dest y */
	PUT_PC2B(aed[6] , width);		/* area width */
	PUT_PC2B(aed[7] , height);	/* area height */
	PUT_PC2B(aed[8] , rule);		/* combination rule */
	PUT_PC2B(aed[9] , (short) AED_COLOR);	/* source color */
	PUT_PC2B(aed[10] , 0);		/* dest color */

	START();
}

/*
 * Draw a line. All lines not more than 16 bits wide who have a
 * a square brush are done by this routine. All other lines are
 * done by a user level blt to the simulated display bitmap and
 * then echoed to the AED.
 */

aed_draw_line (x0, y0, x1, y1, rule, width, color, pat, patlen,
	       top, left, bottom, right)
short x0, y0, x1, y1, rule;
short width, color, pat, patlen;
short top, left, bottom, right;
{
	register short *aed = (short *) display_base;

	if (width <= 0)
		return;

	WAIT();			/* wait for previous op */

	PUT_PC2B(aed[1] , 1);		/* orders */
	PUT_PC2B(aed[2] , rule);		/* combination rule */
	PUT_PC2B(aed[4] , width);		/* line width */
	PUT_PC2B(aed[5] , pat);		/* dash pattern */
	PUT_PC2B(aed[6] , patlen);	/* pattern length */
	PUT_PC2B(aed[7] , x0);		/* x position */
	PUT_PC2B(aed[8] , y0);		/* y position */
	PUT_PC2B(aed[9] , 0);		/* window x origin */
	PUT_PC2B(aed[10] , 0);		/* window y origin */
	PUT_PC2B(aed[11] , left);		/* clip lo x */
	PUT_PC2B(aed[12] , top);		/* clip lo y */
	PUT_PC2B(aed[13] , right);	/* clip hi x */
	PUT_PC2B(aed[14] , bottom);	/* clip hi y */
	PUT_PC2B(aed[15] , AED_COLOR ? color : (~color) & 1);	/* color */

	PUT_PC2B(aed[16] , 3);		/* length */
	PUT_PC2B(aed[17] , 7);		/* line absolute */
	PUT_PC2B(aed[18] , x1);		/* end x */
	PUT_PC2B(aed[19] , y1);		/* end y */

	START();
}

/*
 * load AED microcode, start processor and wait for
 * initialization to complete
 */

aed_reset()
{
	short s;
	register short *aed = (short *) display_base;
	register unsigned short *ucode = (unsigned short *) UCODE;
	extern unsigned short aed_ucode[];
	AED_ENTER();

	s = GET_PC2B(ucode[0]);			/* stop processor */
#ifdef ATR
	aed_wr(aed_ucode, ucode, 0x4000>>1);
#else
	bcopy(aed_ucode, ucode, 0x4000);
#endif ATR

	if (GET_PC2B(ucode[0]) != aed_ucode[0]) {
		printf("aed_reset: can't find AED\n");
		return 1;
	}

	/*
	 * START processor
	 */

	PUT_PC2B(aed[0] , 1);			/* start processor */

	/*
	 * WAIT for reset to complete
	 */

	s = 1000;
	do {
		if (--s < 0) {
			printf("aed_reset: can't kick AED\n");
			return 1;
		}
		DELAY(1);
	} 
	while (GET_PC2B(aed[0]));

	return 0;
}

/*
 * The WHIM microcode for the AED.
 * Derived from whim.vik via od on 6 June 1985
 */

unsigned short aed_ucode[0x2000] = {
    0xd188, 0x001e, 0xce00, 0xf8e0, 0xee12, 0x0000, 0xee73, 0xf900,
    0xce00, 0xf8e0, 0xee12, 0x0000, 0xee40, 0x7140, 0xee03, 0xf8c1,
    0xd381, 0x0004, 0xce00, 0xf8e0, 0xee52, 0x0001, 0xee03, 0xf8c1,
    0xce00, 0xe480, 0xee1f, 0x000f, 0xd308, 0x7140, 0xd388, 0x0055,
    0xd388, 0x0057, 0xd388, 0x0079, 0xd388, 0x049b, 0xd388, 0x04df,
    0xd388, 0x0508, 0xd388, 0x0510, 0xd388, 0x058d, 0xd388, 0x05b7,
    0xd388, 0x05bc, 0xd388, 0x05c1, 0xce00, 0xf8e0, 0xee12, 0x0000,
    0xee73, 0xf900, 0xd388, 0x0004, 0xd188, 0x0041, 0xaef1, 0xf900,
    0xaef0, 0xf900, 0xce00, 0xf8e0, 0xaefd, 0x0200, 0xce00, 0xf8e0,
    0xaefc, 0x0190, 0xce00, 0xf8e0, 0xaefb, 0x0020, 0xce00, 0xf8e0,
    0xaefa, 0x0020, 0xd188, 0x05cf, 0xaeed, 0xf900, 0xaeec, 0xf900,
    0xce00, 0xf8e0, 0xaeeb, 0x03ff, 0xce00, 0xf8e0, 0xaeea, 0x031f,
    0xaee3, 0xf900, 0xaeff, 0xf900, 0xaec0, 0xf900, 0xce00, 0xf8e0,
    0xaebb, 0x036e, 0xaebc, 0xf900, 0xce00, 0xf8e0, 0xee12, 0x0001,
    0xce00, 0xf8e1, 0xce00, 0x0400, 0xee73, 0xf880, 0xce00, 0xf8e0,
    0xee12, 0x0002, 0xce00, 0xf8e1, 0xce00, 0x0320, 0xee73, 0xf880,
    0xca00, 0x7140, 0xee18, 0xf900, 0xee19, 0xf900, 0xee1a, 0xf900,
    0xee1b, 0xf900, 0xce00, 0xd8fc, 0xee15, 0x0320, 0xce00, 0xf8e0,
    0xaeef, 0xff20, 0xce00, 0xf8e0, 0xaeee, 0xfff0, 0xce00, 0xd91d,
    0xeeb4, 0xd85d, 0xce00, 0x9c9d, 0xce00, 0x0040, 0xbeee, 0xfcc0,
    0xd383, 0x004c, 0xee15, 0xdd7c, 0xbeef, 0xfcc0, 0xd383, 0x0049,
    0xca00, 0x7140, 0xd188, 0x001e, 0xd388, 0x001a, 0xce00, 0xf8e0,
    0xee52, 0x0010, 0xee03, 0xf8c1, 0xaeda, 0xf880, 0xce00, 0xd8f8,
    0xee52, 0x0011, 0x9eda, 0xf8c1, 0xaeef, 0xfe80, 0xd381, 0x00d9,
    0xce00, 0xd8fc, 0xee15, 0x032c, 0xee14, 0xd91d, 0xee03, 0xf8c1,
    0xee18, 0xf880, 0xee52, 0xdd78, 0xee03, 0xf8c1, 0xee19, 0xf880,
    0xee52, 0xdd78, 0xee03, 0xf8c1, 0xee1a, 0xf880, 0xee52, 0xdd78,
    0xee03, 0xf8c1, 0xeebb, 0xf880, 0xee52, 0xdd78, 0xce00, 0x9c9d,
    0xee14, 0x0040, 0xce00, 0x845d, 0xce00, 0x0400, 0xd383, 0x0076,
    0xce00, 0xd91d, 0xee15, 0xdd7c, 0xce00, 0xea80, 0xbeef, 0x0004,
    0xd383, 0x0063, 0xce00, 0xf8e0, 0xee52, 0x0002, 0xee03, 0xf8c1,
    0xaeff, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x0003, 0xee03, 0xf8c1,
    0xaee3, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x0004, 0xee03, 0xf8c1,
    0xaee2, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x0005, 0xee03, 0xf8c1,
    0xaee1, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x0006, 0xee03, 0xf8c1,
    0xaee0, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x0007, 0xee03, 0xf8c1,
    0xaedf, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x0008, 0xee03, 0xf8c1,
    0xaede, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x0009, 0xee03, 0xf8c1,
    0xaedd, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x000a, 0xee03, 0xf8c1,
    0xaedc, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x000b, 0xee03, 0xf8c1,
    0xaeed, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x000c, 0xee03, 0xf8c1,
    0xaeec, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x000d, 0xee03, 0xf8c1,
    0xaeeb, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x000e, 0xee03, 0xf8c1,
    0xaeea, 0xf880, 0xce00, 0xf8e0, 0xee52, 0x000f, 0xee03, 0xf8c1,
    0xaedb, 0xf880, 0x9eed, 0xf8c1, 0x9eeb, 0xe241, 0xd383, 0x00d9,
    0x9eec, 0xf8c1, 0x9eea, 0xe241, 0xd383, 0x00d9, 0xd188, 0x052b,
    0x9edb, 0xf8c1, 0xd189, 0x0493, 0xd188, 0x0b77, 0xce00, 0xf8e0,
    0xaed7, 0x032c, 0xaed8, 0xf900, 0x9eda, 0xf8c1, 0xaed9, 0xfe80,
    0xd381, 0x00d9, 0xaec2, 0xf900, 0xaeb6, 0xf900, 0x9ed9, 0xf8c1,
    0xd38b, 0x00d9, 0xd188, 0x0481, 0xee0c, 0xf8c1, 0xce00, 0xe480,
    0xee1f, 0x00ca, 0xd308, 0x7140, 0xd388, 0x00c3, 0xd388, 0x0102,
    0xd388, 0x0109, 0xd388, 0x010e, 0xd388, 0x0116, 0xd388, 0x011a,
    0xd388, 0x0121, 0xd388, 0x0128, 0xd388, 0x013a, 0xd388, 0x014e,
    0xd388, 0x0188, 0xd388, 0x02db, 0xd388, 0x03d6, 0xd388, 0x03db,
    0xd388, 0x0417, 0x9eb6, 0xf8c1, 0xd381, 0x00dd, 0xd188, 0x0301,
    0xd388, 0x00c3, 0x9edb, 0xf8c1, 0xd189, 0x0493, 0xee13, 0xf900,
    0xce00, 0xf8e0, 0xee72, 0x0001, 0x9eff, 0xf8c1, 0xee13, 0xf880,
    0xce00, 0xf8e0, 0xee72, 0x0002, 0x9ee3, 0xf8c1, 0xee13, 0xf880,
    0xce00, 0xf8e0, 0xee72, 0x0003, 0x9ee2, 0xf8c1, 0xee13, 0xf880,
    0xce00, 0xf8e0, 0xee72, 0x0004, 0x9ee1, 0xf8c1, 0xee13, 0xf880,
    0xce00, 0xf8e0, 0xee72, 0x0005, 0x9ee0, 0xf8c1, 0xee13, 0xf880,
    0xce00, 0xf8e0, 0xee72, 0x0006, 0x9edf, 0xf8c1, 0xee13, 0xf880,
    0xce00, 0xf8e0, 0xee72, 0x0007, 0x9ede, 0xf8c1, 0xee13, 0xf880,
    0xce00, 0xf8e0, 0xee72, 0x0008, 0x9ef0, 0xf8c1, 0xd189, 0x060f,
    0xaef0, 0xf900, 0xd388, 0x001a, 0xd188, 0x0481, 0xee0c, 0xf8c1,
    0xaeff, 0xf880, 0x9edb, 0xf8c1, 0xd189, 0x0493, 0xd188, 0x0b77,
    0xd388, 0x00c3, 0xd188, 0x0481, 0xee0c, 0xf8c1, 0xaee3, 0xf880,
    0xd188, 0x052b, 0xd388, 0x00c3, 0xd188, 0x0481, 0xee0c, 0xf8c1,
    0xaee1, 0xf880, 0xd188, 0x0481, 0xee0c, 0xf8c1, 0xaee0, 0xf880,
    0xaec3, 0xfe80, 0xd388, 0x00c3, 0xd188, 0x0481, 0xee0c, 0xf8c1,
    0xaee2, 0xf880, 0xd388, 0x00c3, 0xd188, 0x0481, 0xee0c, 0xf8c1,
    0xaedf, 0xf880, 0xd188, 0x0481, 0xee0c, 0xf8c1, 0xaede, 0xf880,
    0xd388, 0x00c3, 0xd188, 0x0481, 0xee0c, 0xf8c1, 0xbedf, 0xe280,
    0xd188, 0x0481, 0xee0c, 0xf8c1, 0xbede, 0xe280, 0xd388, 0x00c3,
    0x9edf, 0xf8c1, 0x9edd, 0xe281, 0xaed2, 0xf880, 0xd188, 0x0481,
    0xee0c, 0xf8c1, 0xaedf, 0xf880, 0x9edd, 0xe281, 0xaed0, 0xf880,
    0x9ede, 0xf8c1, 0x9edc, 0xe281, 0xaed1, 0xf880, 0xd188, 0x0481,
    0xee0c, 0xf8c1, 0xaede, 0xf880, 0x9edc, 0xe281, 0xaecf, 0xf880,
    0xd188, 0x07cb, 0xd388, 0x00c3, 0x9edf, 0xf8c1, 0x9edd, 0xe281,
    0xaed2, 0xf880, 0xd188, 0x0481, 0xee0c, 0xf8c1, 0x9edf, 0xe281,
    0xaedf, 0xf880, 0x9edd, 0xe281, 0xaed0, 0xf880, 0x9ede, 0xf8c1,
    0x9edc, 0xe281, 0xaed1, 0xf880, 0xd188, 0x0481, 0xee0c, 0xf8c1,
    0x9ede, 0xe281, 0xaede, 0xf880, 0x9edc, 0xe281, 0xaecf, 0xf880,
    0xd188, 0x07cb, 0xd388, 0x00c3, 0xd188, 0x0481, 0xee0c, 0xf8c1,
    0xce00, 0xd880, 0xd188, 0x0481, 0xee0c, 0xf8c1, 0xce00, 0xd881,
    0xce00, 0xd801, 0xaeef, 0xfe80, 0xae07, 0xf900, 0xce00, 0x8480,
    0xce00, 0x000f, 0xce00, 0xe4c1, 0xce00, 0xfff0, 0xbe07, 0xe280,
    0xbeef, 0xfcc0, 0xd383, 0x015b, 0x9edf, 0xf8c1, 0x9edd, 0xe281,
    0xaed6, 0xf880, 0xce00, 0x8080, 0xce00, 0xe440, 0xaed4, 0x0001,
    0x9ede, 0xf8c1, 0x9edc, 0xe281, 0xaed5, 0xf880, 0xce00, 0x8081,
    0xce00, 0xe440, 0xaed3, 0x0001, 0xd188, 0x063e, 0x9ed6, 0xf8c1,
    0xaee9, 0xf880, 0x9ed5, 0xf8c1, 0xaee8, 0xf880, 0x9ed4, 0xf8c1,
    0xaee7, 0xf880, 0x9ed3, 0xf8c1, 0xaee6, 0xf880, 0x9ed8, 0xf8c1,
    0xaee5, 0xf880, 0x9ed7, 0xf8c1, 0xaee4, 0xf880, 0xd188, 0x0656,
    0x9e07, 0xf8c1, 0xbed8, 0xe280, 0xce00, 0xea41, 0x9ed8, 0x0400,
    0xd383, 0x0185, 0xaed8, 0xf880, 0xbed7, 0xfcc0, 0xce00, 0xea41,
    0x9ed7, 0x0400, 0xd383, 0x017a, 0xce00, 0xf8e0, 0xaed7, 0x036e,
    0xd388, 0x017a, 0x9e07, 0xf999, 0xbed9, 0xe280, 0xd388, 0x00c3,
    0x9edf, 0xf8c1, 0x9edd, 0xe281, 0xae0b, 0xf880, 0xd188, 0x0481,
    0x9e0b, 0xf8c1, 0xee0c, 0xe281, 0xce00, 0xe440, 0xae0d, 0x0001,
    0x9ede, 0xf8c1, 0x9edc, 0xe281, 0xae0c, 0xf880, 0xd188, 0x0481,
    0x9e0c, 0xf8c1, 0xee0c, 0xe281, 0xce00, 0xe440, 0xae0e, 0x0001,
    0xd188, 0x0481, 0xee0c, 0xf8c1, 0xae0f, 0xf880, 0xd188, 0x0481,
    0xee0c, 0xf8c1, 0xae10, 0xf880, 0x9e10, 0xf8c1, 0xaeef, 0xfe80,
    0xae12, 0xf900, 0xce00, 0xea81, 0x9e0f, 0x000f, 0xce00, 0xe4c1,
    0xce00, 0xfff0, 0xbe12, 0xe280, 0xbeef, 0xfcc0, 0xd383, 0x01a5,
    0x9eed, 0xf8c1, 0xae07, 0xf880, 0x9eec, 0xf8c1, 0xae08, 0xf880,
    0x9eeb, 0xf8c1, 0xae09, 0xf880, 0x9eea, 0xf8c1, 0xae0a, 0xf880,
    0x9e0b, 0xf8c1, 0x9eed, 0xc209, 0xd383, 0x01b4, 0xaeed, 0xf880,
    0x9e0c, 0xf8c1, 0x9eec, 0xc209, 0xd383, 0x01b8, 0xaeec, 0xf880,
    0x9e0d, 0xf8c1, 0x9eeb, 0xc249, 0xd383, 0x01bc, 0xaeeb, 0xf880,
    0x9e0e, 0xf8c1, 0x9eea, 0xc249, 0xd383, 0x01c0, 0xaeea, 0xf880,
    0x9eeb, 0xf8c1, 0x9eed, 0xe201, 0xd383, 0x02c3, 0x9eea, 0xf8c1,
    0x9eec, 0xe201, 0xd383, 0x02c3, 0x9eed, 0xf8c1, 0xaed6, 0xf880,
    0x9eec, 0xf8c1, 0xaed5, 0xf880, 0x9eeb, 0xf8c1, 0xaed4, 0xf880,
    0x9eea, 0xf8c1, 0xaed3, 0xf880, 0xd188, 0x063e, 0xce00, 0xea41,
    0x9e10, 0x0001, 0xd389, 0x029b, 0xce00, 0xea41, 0x9e0f, 0x0001,
    0xd389, 0x029b, 0xd188, 0x0481, 0xce00, 0xeac1, 0xee0c, 0x8000,
    0xd381, 0x01e3, 0xce00, 0xeac1, 0x9eff, 0x000c, 0xd381, 0x0226,
    0xce00, 0xe441, 0xce00, 0x0004, 0xd381, 0x0261, 0xce00, 0xe441,
    0xce00, 0x0004, 0xd381, 0x02d2, 0xd388, 0x01ed, 0xce00, 0xeac1,
    0x9eff, 0x0003, 0xd381, 0x0226, 0xce00, 0xe441, 0xce00, 0x0001,
    0xd381, 0x0261, 0xce00, 0xe441, 0xce00, 0x0001, 0xd381, 0x02d2,
    0xd388, 0x01ed, 0x9eeb, 0xf8c1, 0xce00, 0xe4c0, 0xae13, 0xfff0,
    0xce00, 0xeac1, 0x9eed, 0xfff0, 0xbe13, 0xe240, 0xbe13, 0xf998,
    0xbe13, 0xfcc0, 0x9eed, 0xf8c1, 0xee10, 0xfa80, 0xfe00, 0xe186,
    0xce00, 0xc44a, 0xce00, 0x0001, 0xce00, 0xcc2a, 0x9eeb, 0xfcc1,
    0xee10, 0xfa80, 0xfe00, 0xe186, 0xce00, 0xe441, 0xce00, 0x0001,
    0xce00, 0xec21, 0xce00, 0xda8b, 0xd389, 0x0204, 0xce00, 0xdb0b,
    0xce00, 0xea41, 0x9e13, 0x0001, 0xd389, 0x0209, 0xce00, 0xd80b,
    0xce00, 0x98ca, 0x9eec, 0xf8c1, 0x9eea, 0xe241, 0xce00, 0xfc81,
    0xae15, 0xfe80, 0x9eec, 0xd8dc, 0xee15, 0xd85c, 0x9e13, 0xfec1,
    0xae14, 0xf880, 0x9eed, 0xd8dd, 0xee94, 0xd85d, 0xee0c, 0x874a,
    0xeebc, 0xf880, 0xce00, 0x9c9d, 0xee94, 0x0010, 0xbe14, 0xfcc0,
    0xd381, 0x0222, 0xbe14, 0xfcc0, 0xd381, 0x0220, 0xeebc, 0xfb00,
    0xce00, 0x9c9d, 0xee94, 0x0010, 0xbe14, 0xfcc0, 0xd389, 0x021b,
    0xee0c, 0x874b, 0xeebc, 0xf880, 0xee15, 0xdd7c, 0xbe15, 0xfcc0,
    0xd389, 0x020f, 0xd388, 0x02d2, 0x9eeb, 0xf8c1, 0xce00, 0xe4c0,
    0xae13, 0xfff0, 0xce00, 0xeac1, 0x9eed, 0xfff0, 0xbe13, 0xe240,
    0xbe13, 0xf998, 0xbe13, 0xfcc0, 0x9eed, 0xf8c1, 0xee10, 0xfa80,
    0xfe00, 0xe186, 0xce00, 0xc44a, 0xce00, 0x0001, 0xce00, 0xcc2a,
    0x9eeb, 0xfcc1, 0xee10, 0xfa80, 0xfe00, 0xe186, 0xce00, 0xe441,
    0xce00, 0x0001, 0xce00, 0xec21, 0xce00, 0xda8b, 0xd389, 0x023d,
    0xce00, 0xdb0b, 0xce00, 0xea41, 0x9e13, 0x0001, 0xd389, 0x0242,
    0xce00, 0xd80b, 0xce00, 0x98ca, 0xce00, 0xdb6a, 0xce00, 0xdb6b,
    0x9eec, 0xf8c1, 0x9eea, 0xe241, 0xce00, 0xfc81, 0xae15, 0xfe80,
    0x9eec, 0xd8dc, 0xee15, 0xd85c, 0x9e13, 0xfec1, 0xae14, 0xf880,
    0x9eed, 0xd8dd, 0xee94, 0xd85d, 0xee0c, 0x86ca, 0xeebc, 0xf880,
    0xce00, 0x9c9d, 0xee94, 0x0010, 0xbe14, 0xfcc0, 0xd381, 0x025d,
    0xbe14, 0xfcc0, 0xd381, 0x025b, 0xeebc, 0xf900, 0xce00, 0x9c9d,
    0xee94, 0x0010, 0xbe14, 0xfcc0, 0xd389, 0x0256, 0xee0c, 0x86cb,
    0xeebc, 0xf880, 0xee15, 0xdd7c, 0xbe15, 0xfcc0, 0xd389, 0x024a,
    0xd388, 0x02d2, 0x9eeb, 0xf8c1, 0xce00, 0xe4c0, 0xae13, 0xfff0,
    0xce00, 0xeac1, 0x9eed, 0xfff0, 0xbe13, 0xe240, 0xbe13, 0xf998,
    0xbe13, 0xfcc0, 0x9eed, 0xf8c1, 0xee10, 0xfa80, 0xfe00, 0xe186,
    0xce00, 0xc44a, 0xce00, 0x0001, 0xce00, 0xcc2a, 0x9eeb, 0xfcc1,
    0xee10, 0xfa80, 0xfe00, 0xe186, 0xce00, 0xe441, 0xce00, 0x0001,
    0xce00, 0xec21, 0xce00, 0xda8b, 0xd389, 0x0278, 0xce00, 0xdb0b,
    0xce00, 0xea41, 0x9e13, 0x0001, 0xd389, 0x027d, 0xce00, 0xd80b,
    0xce00, 0x98ca, 0x9eec, 0xf8c1, 0x9eea, 0xe241, 0xce00, 0xfc81,
    0xae15, 0xfe80, 0x9eec, 0xd8dc, 0xee15, 0xd85c, 0x9e13, 0xfec1,
    0xae14, 0xf880, 0x9eed, 0xd8dd, 0xee94, 0xd85d, 0xee0c, 0x870a,
    0xeebc, 0xf880, 0xce00, 0x9c9d, 0xee94, 0x0010, 0xbe14, 0xfcc0,
    0xd381, 0x0297, 0xbe14, 0xfcc0, 0xd381, 0x0295, 0xee0c, 0xfac1,
    0xeebc, 0xf880, 0xce00, 0x9c9d, 0xee94, 0x0010, 0xbe14, 0xfcc0,
    0xd389, 0x028f, 0xee0c, 0x870b, 0xeebc, 0xf880, 0xee15, 0xdd7c,
    0xbe15, 0xfcc0, 0xd389, 0x0283, 0xd388, 0x02d2, 0x9edd, 0xf8c1,
    0xae11, 0xf880, 0x9e0f, 0xe281, 0x9eed, 0xc209, 0xd383, 0x029c,
    0x9edc, 0xf8c1, 0xaee8, 0xf880, 0x9e10, 0xe281, 0x9eec, 0xc209,
    0xd383, 0x02a1, 0x9ed8, 0xf8c1, 0xaee5, 0xf880, 0x9ed7, 0xf8c1,
    0xaee4, 0xf880, 0x9ee8, 0xf8c1, 0x9e10, 0xe281, 0xce00, 0xe440,
    0xaee6, 0x0001, 0x9e11, 0xf8c1, 0xaee9, 0xf880, 0x9e0f, 0xe281,
    0xce00, 0xe440, 0xaee7, 0x0001, 0xd188, 0x0656, 0x9ee7, 0xf8c1,
    0x9e0f, 0xe281, 0xaee7, 0xf880, 0x9ee9, 0xf8c1, 0x9e0f, 0xe281,
    0xaee9, 0xf880, 0x9eeb, 0xc249, 0xd38b, 0x02b2, 0x9ee6, 0xf8c1,
    0x9e10, 0xe281, 0xaee6, 0xf880, 0x9ee8, 0xf8c1, 0x9e10, 0xe281,
    0xaee8, 0xf880, 0x9eea, 0xc249, 0xd38b, 0x02ad, 0x9e12, 0xf8c1,
    0xbed8, 0xe280, 0xce00, 0xea41, 0x9ed8, 0x0400, 0xd383, 0x02d0,
    0xaed8, 0xf880, 0xbed7, 0xfcc0, 0xce00, 0xea41, 0x9ed7, 0x0400,
    0xd383, 0x02c5, 0xce00, 0xf8e0, 0xaed7, 0x036e, 0xd388, 0x02c5,
    0x9e12, 0xf999, 0xbed9, 0xe280, 0x9e07, 0xf8c1, 0xaeed, 0xf880,
    0x9e08, 0xf8c1, 0xaeec, 0xf880, 0x9e09, 0xf8c1, 0xaeeb, 0xf880,
    0x9e0a, 0xf8c1, 0xaeea, 0xf880, 0xd388, 0x00c3, 0x9eb6, 0xf8c1,
    0xd389, 0x02f0, 0x9ee3, 0xf8c1, 0xd381, 0x02f0, 0x9ebf, 0xf8c1,
    0xd181, 0x0341, 0x9ebf, 0xf8c1, 0xd381, 0x02f0, 0xd188, 0x0481,
    0xee0c, 0xf8c1, 0xaeb7, 0xf880, 0x9ed8, 0xf8c1, 0xaeba, 0xf880,
    0x9ed7, 0xf8c1, 0xaeb9, 0xf880, 0x9ed9, 0xf8c1, 0xaeb8, 0xf880,
    0xaeb6, 0xfb00, 0xd188, 0x03e0, 0xaed9, 0xf900, 0xd388, 0x00c3,
    0xd188, 0x0481, 0xee0c, 0xfcc1, 0xce00, 0xec81, 0xbed9, 0xe280,
    0xce00, 0xe99d, 0xbed8, 0xe280, 0xce00, 0xea41, 0x9ed8, 0x0400,
    0xd383, 0x00c3, 0xaed8, 0xf880, 0xbed7, 0xfcc0, 0xce00, 0xea41,
    0x9ed7, 0x0400, 0xd383, 0x02f6, 0xce00, 0xf8e0, 0xaed7, 0x036e,
    0xd388, 0x02f6, 0x9eba, 0xf8c1, 0xaed8, 0xf880, 0x9eb9, 0xf8c1,
    0xaed7, 0xf880, 0x9eb8, 0xf8c1, 0xaed9, 0xf880, 0xaeb6, 0xf900,
    0x9eb7, 0xf8c1, 0xd389, 0x030e, 0xce00, 0xf8e0, 0xaec2, 0x0050,
    0xd188, 0x03f9, 0xca00, 0x7140, 0xd188, 0x0481, 0xee0c, 0xf8c1,
    0xaefe, 0xf880, 0x9eb7, 0xf8c1, 0xd383, 0x0319, 0xbefe, 0xf198,
    0xce00, 0xea00, 0xbeb7, 0x0001, 0xd381, 0x031c, 0xaeb6, 0xfb00,
    0xd388, 0x0323, 0xbeb7, 0xfec0, 0xce00, 0xea40, 0xbeb7, 0x0001,
    0x9ed8, 0xf8c1, 0xaeba, 0xf880, 0x9ed7, 0xf8c1, 0xaeb9, 0xf880,
    0x9ed9, 0xf8c1, 0xaeb8, 0xf880, 0xaeb6, 0xfb00, 0xce00, 0xeac1,
    0x9efe, 0x00ff, 0xaebe, 0xf880, 0xce00, 0xa81d, 0xce00, 0x9cdd,
    0xee14, 0x03ff, 0xce00, 0xf59d, 0x9ebf, 0xc29c, 0xce00, 0x9cdc,
    0xce00, 0x03ff, 0xee95, 0xd85c, 0xee0c, 0xf8c1, 0xaebd, 0xf880,
    0xd181, 0x0388, 0xce00, 0xe441, 0xce00, 0x0001, 0xd381, 0x0301,
    0x9ebd, 0xf8c1, 0xd381, 0x0301, 0xce00, 0xeac1, 0x9ebd, 0x003f,
    0xaed8, 0xe99c, 0x9ebd, 0xf599, 0xce00, 0xe4c0, 0xaed7, 0x03ff,
    0xd188, 0x0481, 0xd188, 0x0481, 0xee0c, 0xf8c1, 0xaed9, 0xfe80,
    0xca00, 0x7140, 0xce00, 0xf8e0, 0xee12, 0x0001, 0xce00, 0xf8e1,
    0xce00, 0x0001, 0xee73, 0xf880, 0xce00, 0xf8e0, 0xee12, 0x0002,
    0x9ee3, 0xf8c1, 0xee73, 0xf880, 0xce00, 0xf8e0, 0xee12, 0x0000,
    0xee73, 0xf900, 0xce00, 0xf8e0, 0xee12, 0x0000, 0xee40, 0x7140,
    0xee03, 0xf8c1, 0xd381, 0x034d, 0xce00, 0xf8e0, 0xee52, 0x0001,
    0xee03, 0xf8c1, 0xda01, 0x7140, 0x9ec0, 0xfd99, 0xce00, 0xe480,
    0xaebf, 0x034e, 0xce00, 0xf8e0, 0xee15, 0x034d, 0x9ec0, 0xf8c1,
    0xee94, 0xf880, 0xee0c, 0xf8c1, 0xd381, 0x0360, 0xd188, 0x053e,
    0xce00, 0xf8e0, 0xee15, 0x034d, 0x9ec0, 0xf8c1, 0xee94, 0xf880,
    0x9ee3, 0xf8c1, 0xeebc, 0xf880, 0xce00, 0xd8f8, 0xee52, 0x0002,
    0x9ebf, 0xd8c5, 0xee15, 0xd845, 0xce00, 0xf8e0, 0xaeef, 0xfffc,
    0xee14, 0xd904, 0xee03, 0xf8c1, 0xee18, 0xf880, 0xee52, 0xdd78,
    0xee03, 0xf8c1, 0xee19, 0xf880, 0xee52, 0xdd78, 0xee03, 0xf8c1,
    0xee1a, 0xf880, 0xee52, 0xdd78, 0xee03, 0xf8c1, 0xeebb, 0xf880,
    0xee52, 0xdd78, 0xce00, 0x9c84, 0xee14, 0x0040, 0xce00, 0x8444,
    0xce00, 0x0400, 0xd383, 0x036d, 0xee15, 0xdd65, 0xbeef, 0xfcc0,
    0xd383, 0x036c, 0xce00, 0xea80, 0xbec0, 0x0010, 0xce00, 0xea41,
    0x9ec0, 0x0060, 0xda03, 0x7140, 0xaec0, 0xf900, 0xca00, 0x7140,
    0xeebc, 0xfd00, 0xce00, 0xf8e0, 0xee12, 0x0001, 0xce00, 0xf8e1,
    0xce00, 0x0002, 0xee73, 0xf880, 0xce00, 0xf8e0, 0xee12, 0x0002,
    0x9ee3, 0xf8c1, 0xee73, 0xf880, 0xce00, 0xf8e0, 0xee12, 0x0003,
    0x9ebe, 0xf8c1, 0xee73, 0xf880, 0xce00, 0xf8e0, 0xee12, 0x0000,
    0xee73, 0xf900, 0xce00, 0xf8e0, 0xee12, 0x0000, 0xee40, 0x7140,
    0xee03, 0xf8c1, 0xd381, 0x0399, 0xce00, 0xf8e0, 0xee52, 0x0001,
    0xee03, 0xf8c1, 0xda01, 0x7140, 0x9ebc, 0xb827, 0x9ebb, 0xed99,
    0xce00, 0x8087, 0xeebc, 0xf880, 0xaebd, 0xf880, 0x8e07, 0x99fd,
    0xce00, 0x8d9c, 0xce00, 0x9887, 0x9ebc, 0xd8dd, 0xee14, 0xd85d,
    0x9ebb, 0xd8dc, 0xee95, 0xd85c, 0xee0c, 0xf8c1, 0xd381, 0x03b1,
    0xd188, 0x0561, 0xeebc, 0xd847, 0xce00, 0xd8f8, 0xee52, 0x0002,
    0xee03, 0xf8c1, 0xce00, 0xdc86, 0xce00, 0xdf66, 0xce00, 0xea80,
    0xbebc, 0x0010, 0xce00, 0xea41, 0x9ebc, 0x0400, 0xd383, 0x03c3,
    0xaebc, 0xf900, 0xbebb, 0xfcc0, 0xce00, 0xea41, 0x9ebb, 0x0400,
    0xd383, 0x03c3, 0xce00, 0xf8e0, 0xaebb, 0x036e, 0xce00, 0xd806,
    0xda01, 0x7140, 0x9ebc, 0xd8dd, 0xee14, 0xd85d, 0x9ebb, 0xd8dc,
    0xee95, 0xd85c, 0xee52, 0xd858, 0xee0c, 0xf8c1, 0xd381, 0x03d1,
    0xd188, 0x0561, 0x9ebc, 0xf8c1, 0xee14, 0xf880, 0x9ebb, 0xf8c1,
    0xee95, 0xf880, 0xee03, 0xf8c1, 0xeebc, 0xf880, 0xce00, 0xdd78,
    0xce00, 0xdd66, 0xd388, 0x03b7, 0xce00, 0xea01, 0x9ec2, 0x0370,
    0xd383, 0x00c3, 0xd188, 0x03e0, 0xd388, 0x00c3, 0xce00, 0xea41,
    0x9ec2, 0x00a0, 0xd383, 0x00c3, 0xd188, 0x03f9, 0xd388, 0x00c3,
    0xce00, 0xf8e0, 0xee15, 0x034c, 0x9ec2, 0xf8c1, 0xee94, 0xd884,
    0x9eff, 0xf8c1, 0xeebc, 0xf880, 0xce00, 0x9c84, 0xee94, 0x0010,
    0x9ee2, 0xf8c1, 0xeebc, 0xf880, 0xce00, 0x9c84, 0xee94, 0x0010,
    0x9ee0, 0xf8c1, 0xeebc, 0xf880, 0xce00, 0x9c84, 0xee94, 0x0010,
    0x9ee1, 0xf8c1, 0xeebc, 0xf880, 0xce00, 0x9c84, 0xee94, 0x0010,
    0x9ee3, 0xf8c1, 0xeebc, 0xf880, 0xce00, 0xea80, 0xbec2, 0x0050,
    0xca00, 0x7140, 0xce00, 0xf8e0, 0xee15, 0x034c, 0xce00, 0xca44,
    0x9ec2, 0x0010, 0xee94, 0xd844, 0xee0c, 0xf8c1, 0xaee3, 0xf880,
    0xce00, 0x9c44, 0xee94, 0x0010, 0xee0c, 0xf8c1, 0xaee1, 0xf880,
    0xce00, 0x9c44, 0xee94, 0x0010, 0xee0c, 0xf8c1, 0xaee0, 0xf880,
    0xce00, 0x9c44, 0xee94, 0x0010, 0xee0c, 0xf8c1, 0xaee2, 0xf880,
    0xce00, 0x9c44, 0xee94, 0x0010, 0xee0c, 0xf8c1, 0xaeff, 0xf880,
    0xce00, 0xea40, 0xbec2, 0x0050, 0xd188, 0x052b, 0xd188, 0x0b77,
    0x9ee0, 0xf8c1, 0xaec3, 0xfe80, 0xca00, 0x7140, 0xd188, 0x0481,
    0xee0c, 0xd8ce, 0x9edf, 0x868e, 0x9edd, 0xe281, 0xaed4, 0xf880,
    0x9edf, 0x864e, 0x9edd, 0xe281, 0xaed6, 0xf880, 0x9ede, 0x868e,
    0x9edc, 0xe281, 0xaed3, 0xf880, 0x9ede, 0x864e, 0x9edc, 0xe281,
    0xaed5, 0xf880, 0xd188, 0x063e, 0x9edf, 0xf8c1, 0x9edd, 0xe281,
    0xae0a, 0xf880, 0x9ede, 0xf8c1, 0x9edc, 0xe281, 0xae0b, 0xf880,
    0xce00, 0xd90f, 0xce00, 0xd80e, 0xce00, 0xd890, 0xce00, 0xd8f1,
    0xce00, 0x0003, 0xce00, 0xec01, 0xce00, 0x9851, 0xce00, 0xdd1b,
    0xaefe, 0xfb00, 0xd188, 0x044d, 0xce00, 0xd811, 0xd383, 0x0441,
    0xce00, 0xc491, 0xce00, 0x000a, 0xce00, 0xd80f, 0xce00, 0x8010,
    0xce00, 0xe59d, 0xce00, 0x9891, 0xce00, 0x9c50, 0xce00, 0x0001,
    0xd388, 0x0445, 0xce00, 0xc491, 0xce00, 0x0006, 0xce00, 0x858f,
    0xce00, 0x9891, 0xce00, 0xdd6f, 0xce00, 0xd80f, 0xce00, 0x8010,
    0xd38b, 0x044c, 0xd188, 0x044d, 0xd188, 0x0462, 0xd388, 0x0436,
    0xd388, 0x00c3, 0x9e0a, 0x868f, 0xce00, 0xd89d, 0x9e0b, 0x8690,
    0xce00, 0xd89c, 0xd188, 0x0477, 0x9e0a, 0x8690, 0xce00, 0xd89d,
    0x9e0b, 0x864f, 0xce00, 0xd89c, 0xd188, 0x0477, 0x9e0a, 0x864f,
    0xce00, 0xd89d, 0x9e0b, 0x8650, 0xce00, 0xd89c, 0xd188, 0x0477,
    0x9e0a, 0x8650, 0xce00, 0xd89d, 0x9e0b, 0x868f, 0xce00, 0xd89c,
    0xd188, 0x0477, 0xca00, 0x7140, 0x9e0a, 0x8690, 0xce00, 0xd89d,
    0x9e0b, 0x868f, 0xce00, 0xd89c, 0xd188, 0x0477, 0x9e0a, 0x868f,
    0xce00, 0xd89d, 0x9e0b, 0x8650, 0xce00, 0xd89c, 0xd188, 0x0477,
    0x9e0a, 0x8650, 0xce00, 0xd89d, 0x9e0b, 0x864f, 0xce00, 0xd89c,
    0xd188, 0x0477, 0x9e0a, 0x864f, 0xce00, 0xd89d, 0x9e0b, 0x8690,
    0xce00, 0xd89c, 0xd188, 0x0477, 0xca00, 0x7140, 0x9eed, 0x861d,
    0xda03, 0x7140, 0x9eeb, 0x865d, 0xda03, 0x7140, 0x9eec, 0x861c,
    0xda03, 0x7140, 0x9eea, 0x865c, 0xda03, 0x7140, 0xd188, 0x0b5b,
    0xca00, 0x7140, 0x9ed8, 0xf8c1, 0xee14, 0xf880, 0x9ed7, 0xf8c1,
    0xee95, 0xf880, 0xbed9, 0xfcc0, 0xce00, 0xea80, 0xbed8, 0x0010,
    0xce00, 0xea41, 0x9ed8, 0x0400, 0xda03, 0x7140, 0xaed8, 0xf900,
    0xbed7, 0xfcc0, 0xce00, 0xea41, 0x9ed7, 0x0400, 0xda03, 0x7140,
    0xce00, 0xf8e0, 0xaed7, 0x036e, 0xca00, 0x7140, 0xce00, 0xeac1,
    0x9eff, 0x0005, 0xce00, 0xec01, 0xce00, 0xbee0, 0x9eff, 0x0005,
    0xce00, 0xe400, 0xaeff, 0x000f, 0xca00, 0x7140, 0xce00, 0xf8e0,
    0xee52, 0x0002, 0xee03, 0xf8c1, 0xaece, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0003, 0xee03, 0xf8c1, 0xaecd, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0004, 0xee03, 0xf8c1, 0xaecc, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0005, 0xee03, 0xf8c1, 0xaecb, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0006, 0xee03, 0xf8c1, 0xaec9, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0007, 0xee03, 0xf8c1, 0xaeca, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0008, 0xee03, 0xf8c1, 0xaeff, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0009, 0xee03, 0xf8c1, 0xaedb, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x000a, 0xee03, 0xf8c1, 0x9edb, 0xe241, 0xd189, 0x0493,
    0xd188, 0x0b77, 0x9ece, 0xf8c1, 0xaed6, 0xf880, 0x9ec9, 0xe281,
    0xce00, 0xe440, 0xaed4, 0x0001, 0x9ecd, 0xf8c1, 0xaed5, 0xf880,
    0x9eca, 0xe281, 0xce00, 0xe440, 0xaed3, 0x0001, 0xd188, 0x063e,
    0x9ef0, 0xf8c1, 0xd389, 0x04d9, 0x9ecc, 0xf8c1, 0xaed6, 0xf880,
    0x9ec9, 0xe281, 0xce00, 0xe440, 0xaed4, 0x0001, 0x9ecb, 0xf8c1,
    0xaed5, 0xf880, 0x9eca, 0xe281, 0xce00, 0xe440, 0xaed3, 0x0001,
    0xd188, 0x063e, 0xd188, 0x08c4, 0x9ef0, 0xf8c1, 0xd381, 0x001a,
    0xd188, 0x060f, 0xaef0, 0xf900, 0xd388, 0x001a, 0xce00, 0xf8e0,
    0xee52, 0x0002, 0xee03, 0xf8c1, 0xaebe, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0003, 0xee03, 0xf8c1, 0xaec7, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0004, 0xee03, 0xf8c1, 0xaec6, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0005, 0xee03, 0xf8c1, 0xaec5, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0006, 0xee03, 0xf8c1, 0xd381, 0x04f4, 0xce00, 0xfb01,
    0xaec4, 0xf880, 0x9ebe, 0xf8c1, 0xaed6, 0xf880, 0x9ec6, 0xe281,
    0xce00, 0xe440, 0xaed4, 0x0001, 0x9ec7, 0xf8c1, 0xaed5, 0xf880,
    0x9ec5, 0xe281, 0xce00, 0xe440, 0xaed3, 0x0001, 0xd188, 0x063e,
    0xce00, 0xd8f8, 0xce00, 0x0001, 0xd188, 0x0afc, 0x9ef0, 0xf8c1,
    0xd381, 0x001a, 0xd188, 0x060f, 0xaef0, 0xf900, 0xd388, 0x001a,
    0xce00, 0xf8e0, 0xee52, 0x0002, 0xee03, 0xf8c1, 0xaee3, 0xf880,
    0xd188, 0x052b, 0x9ebf, 0xf8c1, 0xd189, 0x053e, 0xd388, 0x001a,
    0xce00, 0xf8e0, 0xee52, 0x0002, 0xee03, 0xf8c1, 0xaee3, 0xf880,
    0xce00, 0xf8e0, 0xee52, 0x0003, 0xee03, 0xf8c1, 0xaebe, 0xf880,
    0xd188, 0x052b, 0x9ebf, 0xf8c1, 0xd381, 0x001a, 0x9ebe, 0xf8c1,
    0xee14, 0xe99c, 0xce00, 0xf59d, 0x9ebf, 0xe281, 0xee95, 0xf880,
    0xee0c, 0xfec1, 0xd38b, 0x001a, 0x9ebd, 0xf8c1, 0xce00, 0xeac1,
    0x9ebd, 0x003f, 0xee14, 0xa81d, 0x9ebd, 0xf599, 0xce00, 0xc4dc,
    0xee15, 0x03ff, 0xd188, 0x0561, 0xd388, 0x001a, 0xce00, 0xf8e0,
    0xee15, 0x034d, 0xce00, 0xf8e0, 0xaebf, 0x034e, 0xce00, 0xd904,
    0xce00, 0xf8e0, 0xaeef, 0xfff8, 0xee94, 0xd844, 0xee0c, 0xf8c1,
    0x9ee3, 0xe241, 0xda01, 0x7140, 0xce00, 0x9c84, 0xce00, 0x0010,
    0xce00, 0xea80, 0xbebf, 0x0004, 0xbeef, 0xfcc0, 0xd383, 0x0532,
    0xaebf, 0xf900, 0xca00, 0x7140, 0xce00, 0xea41, 0x9ebf, 0x034e,
    0xce00, 0xf99d, 0xee14, 0xf880, 0xce00, 0xf8e0, 0xee95, 0x034d,
    0xeebc, 0xf900, 0x9ebf, 0xd8c7, 0xce00, 0xd906, 0xce00, 0xf8e0,
    0xaeef, 0xfffc, 0xee14, 0xd846, 0xee95, 0xd847, 0xee0c, 0xf8c1,
    0xd381, 0x0558, 0xaebd, 0xf880, 0xce00, 0xe441, 0xce00, 0x0001,
    0xd381, 0x0558, 0xce00, 0xeac1, 0x9ebd, 0x003f, 0xee14, 0xa81d,
    0x9ebd, 0xf599, 0xce00, 0xc4dc, 0xee15, 0x03ff, 0xd188, 0x0561,
    0xce00, 0x9c86, 0xce00, 0x0010, 0xce00, 0x8446, 0xce00, 0x0400,
    0xd383, 0x0549, 0xce00, 0xdd67, 0xbeef, 0xfcc0, 0xd383, 0x0549,
    0xca00, 0x7140, 0xee14, 0xd85d, 0xee95, 0xd85c, 0xee0c, 0xd8c4,
    0xeebc, 0xf900, 0xce00, 0x84c4, 0xce00, 0x003f, 0xee14, 0xa81f,
    0xce00, 0x9584, 0xce00, 0xc4de, 0xee95, 0x03ff, 0xeebc, 0xf900,
    0xce00, 0x9c9d, 0xee14, 0x0010, 0xce00, 0x845d, 0xce00, 0x0400,
    0xd383, 0x0578, 0xce00, 0xd91d, 0xce00, 0xdd7c, 0xce00, 0x845c,
    0xce00, 0x0400, 0xd383, 0x0578, 0xce00, 0xd8fc, 0xce00, 0x036e,
    0xee95, 0xd85c, 0xee0c, 0xdec5, 0xeebc, 0xf900, 0xce00, 0xd805,
    0xda01, 0x7140, 0xce00, 0x9c9d, 0xee14, 0x0010, 0xce00, 0x845d,
    0xce00, 0x0400, 0xd383, 0x0589, 0xce00, 0xd91d, 0xee15, 0xdd7c,
    0xce00, 0x845c, 0xce00, 0x0400, 0xd383, 0x0589, 0xce00, 0xd8fc,
    0xee15, 0x036e, 0xee80, 0x7140, 0xeebc, 0xf900, 0xce00, 0xdd65,
    0xd388, 0x057b, 0x9ef1, 0xf8c1, 0xd189, 0x05ee, 0xce00, 0xf8e0,
    0xee52, 0x0002, 0xee03, 0xf8c1, 0xaefb, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0003, 0xee03, 0xf8c1, 0xaefa, 0xf880, 0xce00, 0xd8f8,
    0xee52, 0x0004, 0xce00, 0xd8fc, 0xee15, 0x0320, 0xce00, 0xf8e0,
    0xaeef, 0xfff8, 0xee14, 0xd91d, 0xce00, 0xf8e0, 0xaeee, 0xfff0,
    0xee03, 0xf8c1, 0xee18, 0xf880, 0xee52, 0xdd78, 0xee03, 0xf8c1,
    0xee19, 0xf880, 0xee52, 0xdd78, 0xee03, 0xf8c1, 0xee1a, 0xf880,
    0xee52, 0xdd78, 0xee03, 0xf8c1, 0xeebb, 0xf880, 0xee52, 0xdd78,
    0xce00, 0x9c9d, 0xee14, 0x0040, 0xbeee, 0xfcc0, 0xd383, 0x05a0,
    0xee15, 0xdd7c, 0xbeef, 0xfcc0, 0xd383, 0x059d, 0xd188, 0x05cf,
    0x9ef1, 0xf8c1, 0xd189, 0x060f, 0xd388, 0x001a, 0x9ef1, 0xf8c1,
    0xd389, 0x001a, 0xd188, 0x060f, 0xaef1, 0xfb00, 0xd388, 0x001a,
    0x9ef1, 0xf8c1, 0xd381, 0x001a, 0xd188, 0x05ee, 0xaef1, 0xf900,
    0xd388, 0x001a, 0x9ef1, 0xf8c1, 0xd189, 0x05ee, 0xce00, 0xf8e0,
    0xee52, 0x0002, 0xee03, 0xf8c1, 0xaefd, 0xf880, 0xce00, 0xf8e0,
    0xee52, 0x0003, 0xee03, 0xf8c1, 0xaefc, 0xf880, 0xd188, 0x05cf,
    0x9ef1, 0xf8c1, 0xd189, 0x060f, 0xd388, 0x001a, 0x9efd, 0xf8c1,
    0x9efb, 0xe201, 0xaef9, 0xf880, 0xaef5, 0xf880, 0xd38b, 0x05d5,
    0xaef5, 0xf900, 0xce00, 0xe481, 0xce00, 0x003f, 0xaef7, 0xf880,
    0xaef3, 0xf880, 0xce00, 0xe401, 0xce00, 0x03ff, 0xd38b, 0x05de,
    0xce00, 0xf8e0, 0xaef3, 0x03ff, 0x9efc, 0xf8c1, 0x9efa, 0xe201,
    0xaef8, 0xf880, 0xaef4, 0xf880, 0xd38b, 0x05e4, 0xaef4, 0xf900,
    0xce00, 0xe481, 0xce00, 0x003f, 0xaef6, 0xf880, 0xaef2, 0xf880,
    0xce00, 0xe401, 0xce00, 0x031f, 0xda0b, 0x7140, 0xce00, 0xf8e0,
    0xaef2, 0x031f, 0xca00, 0x7140, 0x9eed, 0xd8c9, 0x9eec, 0xd8ca,
    0x9eeb, 0xd8cb, 0x9eea, 0xd8cc, 0x9eff, 0xd8cd, 0xce00, 0xf8e0,
    0xaeff, 0x000c, 0xd188, 0x0b77, 0xaeed, 0xf900, 0xaeec, 0xf900,
    0xce00, 0xf8e0, 0xaeeb, 0x03ff, 0xce00, 0xf8e0, 0xaeea, 0x031f,
    0x9ef5, 0xf8c1, 0xaee9, 0xf880, 0x9ef4, 0xf8c1, 0xaee8, 0xf880,
    0x9ef3, 0xf8c1, 0xaee7, 0xf880, 0x9ef2, 0xf8c1, 0xaee6, 0xf880,
    0xce00, 0xf8e0, 0xaee4, 0x0328, 0xaee5, 0xf900, 0xd188, 0x0656,
    0xaeed, 0xd849, 0xaeec, 0xd84a, 0xaeeb, 0xd84b, 0xaeea, 0xd84c,
    0xaeff, 0xd84d, 0xd188, 0x0b77, 0xca00, 0x7140, 0x9ef3, 0xf8c1,
    0x9ef5, 0xe201, 0xda03, 0x7140, 0x9ef2, 0xf8c1, 0x9ef4, 0xe201,
    0xda03, 0x7140, 0xd188, 0x0b27, 0x9eed, 0xd8c9, 0x9eec, 0xd8ca,
    0x9eeb, 0xd8cb, 0x9eea, 0xd8cc, 0x9eff, 0xd8cd, 0xaeed, 0xf900,
    0xaeec, 0xf900, 0xce00, 0xf8e0, 0xaeeb, 0x03ff, 0xce00, 0xf8e0,
    0xaeea, 0x031f, 0x9ef9, 0xf8c1, 0xaee9, 0xf880, 0x9ef8, 0xf8c1,
    0xaee8, 0xf880, 0x9ef7, 0xf8c1, 0xaee7, 0xf880, 0x9ef6, 0xf8c1,
    0xaee6, 0xf880, 0xce00, 0xf8e0, 0xaeff, 0x0002, 0xd188, 0x0b77,
    0xce00, 0xf8e0, 0xaee4, 0x0320, 0xaee5, 0xf900, 0xd188, 0x0656,
    0xce00, 0xf8e0, 0xaeff, 0x000e, 0xd188, 0x0b77, 0xce00, 0xf8e0,
    0xaee4, 0x0324, 0xaee5, 0xf900, 0xd188, 0x0656, 0xaeed, 0xd849,
    0xaeec, 0xd84a, 0xaeeb, 0xd84b, 0xaeea, 0xd84c, 0xaeff, 0xd84d,
    0xd188, 0x0b77, 0xca00, 0x7140, 0x9ef1, 0xf8c1, 0xda01, 0x7140,
    0x9ef9, 0xf8c1, 0x9ed4, 0xe241, 0xd383, 0x0651, 0x9ed6, 0xf8c1,
    0x9ef7, 0xe241, 0xd383, 0x0651, 0x9ef8, 0xf8c1, 0x9ed3, 0xe241,
    0xd383, 0x0651, 0x9ed5, 0xf8c1, 0x9ef6, 0xe241, 0xd383, 0x0651,
    0x9ef0, 0xf8c1, 0xda09, 0x7140, 0xd188, 0x05ee, 0xaef0, 0xfb00,
    0xca00, 0x7140, 0x9ef0, 0xf8c1, 0xda01, 0x7140, 0xd188, 0x060f,
    0xaef0, 0xf900, 0xca00, 0x7140, 0xae01, 0xf900, 0x9ee9, 0xd8c4,
    0x9eed, 0x8604, 0xd38b, 0x065c, 0x9eed, 0xd8c4, 0xae01, 0xfb00,
    0x9ee8, 0xd8c5, 0x9eec, 0x8605, 0xd38b, 0x0661, 0x9eec, 0xd8c5,
    0xae01, 0xfb00, 0x9ee7, 0xd8c6, 0x9eeb, 0x8646, 0xd38b, 0x0666,
    0x9eeb, 0xd8c6, 0xae01, 0xfb00, 0x9ee6, 0xd8c7, 0x9eea, 0x8647,
    0xd38b, 0x066b, 0x9eea, 0xd8c7, 0xae01, 0xfb00, 0x9e01, 0xf8c1,
    0xd389, 0x075f, 0x9ee9, 0xf8c1, 0x9ee7, 0xe241, 0xce00, 0xdc88,
    0xce00, 0xe441, 0xce00, 0x0010, 0xd38b, 0x06de, 0x9ee7, 0xf8c1,
    0xce00, 0xe4c0, 0xae01, 0xfff0, 0xce00, 0xeac1, 0x9ee9, 0xfff0,
    0xbe01, 0xe240, 0xd389, 0x06a4, 0xee10, 0xde48, 0xfe00, 0xe186,
    0xce00, 0xc440, 0xce00, 0x0001, 0xce00, 0xdb60, 0x9ee9, 0xf8c1,
    0xee10, 0xfe80, 0xfe00, 0x81e0, 0x9ee6, 0xfcc1, 0x9ee8, 0xe201,
    0xae01, 0xfe80, 0xda01, 0x7140, 0x9ee5, 0xd8df, 0xee16, 0xd85f,
    0x9ee4, 0xd8de, 0xeed7, 0xd85e, 0x9ee9, 0xd8dd, 0xee14, 0xd85d,
    0x9ee8, 0xd8dc, 0xee15, 0xd85c, 0xee1f, 0xd85a, 0xee8d, 0xf8c1,
    0xfe00, 0xa017, 0xd108, 0x7140, 0x9eb5, 0xa100, 0xeebc, 0xf880,
    0xce00, 0x9c9f, 0xee16, 0x0010, 0xce00, 0x845f, 0xce00, 0x0400,
    0xd383, 0x06a0, 0xce00, 0xd91f, 0xee17, 0xdd7e, 0xce00, 0x845e,
    0xce00, 0x0400, 0xd383, 0x06a0, 0xce00, 0xd8fe, 0xee17, 0x036e,
    0xeed5, 0xdd7c, 0xbe01, 0xfcc0, 0xd389, 0x068f, 0xca00, 0x7140,
    0x9ee9, 0xf8c1, 0xee10, 0xfa80, 0xfe00, 0xe186, 0xce00, 0xc440,
    0xce00, 0x0001, 0xce00, 0xcc20, 0x9ee7, 0xfcc1, 0xee10, 0xfa80,
    0xfe00, 0xe186, 0xce00, 0xe441, 0xce00, 0x0001, 0xce00, 0xec21,
    0xce00, 0xda81, 0xd389, 0x06b3, 0xce00, 0xdb01, 0x9ee9, 0xfec1,
    0xee10, 0xf880, 0x9ee6, 0xfcc1, 0x9ee8, 0xe201, 0xae01, 0xfe80,
    0xda01, 0x7140, 0x9ee5, 0xd8df, 0xee16, 0xd85f, 0x9ee4, 0xd8de,
    0xeed7, 0xd85e, 0x9ee8, 0xd8dc, 0xee15, 0xd85c, 0x9ee9, 0xd8dd,
    0xee1f, 0xd85a, 0xee14, 0xd85d, 0xee8d, 0xf8c1, 0xfe00, 0xa002,
    0x8e17, 0xd962, 0xd108, 0x7140, 0x9eb5, 0xa100, 0xeebc, 0xf880,
    0xce00, 0x9c9f, 0xee16, 0x0010, 0xce00, 0x845f, 0xce00, 0x0400,
    0xd383, 0x06d4, 0xce00, 0xd91f, 0xee17, 0xdd7e, 0xce00, 0x845e,
    0xce00, 0x0400, 0xd383, 0x06d4, 0xce00, 0xd8fe, 0xee17, 0x036e,
    0xce00, 0x949d, 0xee94, 0x0010, 0x8e17, 0xd962, 0xd108, 0x7140,
    0x9eb5, 0xa101, 0xeebc, 0xf880, 0xeed5, 0xdd7c, 0xbe01, 0xfcc0,
    0xd389, 0x06c1, 0xca00, 0x7140, 0x9ee7, 0xf8c1, 0xce00, 0xe4c0,
    0xae02, 0xfff0, 0xce00, 0xeac1, 0x9ee9, 0xfff0, 0xbe02, 0xe240,
    0xbe02, 0xf998, 0xbe02, 0xfcc0, 0x9ee9, 0xf8c1, 0x9ee7, 0xe241,
    0xce00, 0xe4c1, 0xce00, 0xfff0, 0xce00, 0xf99d, 0x9e02, 0xe201,
    0xae01, 0xfc80, 0x9ee9, 0xf8c1, 0xee10, 0xfa80, 0xfe00, 0xe186,
    0xce00, 0xc440, 0xce00, 0x0001, 0xce00, 0xcc20, 0x9ee7, 0xfcc1,
    0xee10, 0xfa80, 0xfe00, 0xe186, 0xce00, 0xe441, 0xce00, 0x0001,
    0xce00, 0xec21, 0xce00, 0xda81, 0xd389, 0x06fc, 0xce00, 0xdb01,
    0x9ee9, 0xfec1, 0xee10, 0xf880, 0xfe00, 0xe186, 0xce00, 0xc442,
    0xce00, 0x0001, 0xd389, 0x0703, 0xce00, 0xdb02, 0xce00, 0xdb62,
    0x9ee6, 0xfcc1, 0x9ee8, 0xe201, 0xae03, 0xfe80, 0xda01, 0x7140,
    0x9ee5, 0xd8df, 0xee16, 0xd85f, 0x9ee4, 0xd8de, 0xee17, 0xd85e,
    0x9ee8, 0xd8dc, 0xee15, 0xd85c, 0xee1f, 0xd85a, 0x9e02, 0xfec1,
    0xae04, 0xf880, 0x9ee9, 0xd8dd, 0xeed4, 0xd85d, 0xee8d, 0xf8c1,
    0xfe00, 0xe19d, 0xae05, 0xf880, 0xce00, 0xd897, 0xd108, 0x7140,
    0x9eb5, 0xa100, 0xeebc, 0xf880, 0xce00, 0x9c9f, 0xee16, 0x0010,
    0xce00, 0x845f, 0xce00, 0x0400, 0xd383, 0x0726, 0xce00, 0xd91f,
    0xee17, 0xdd7e, 0xce00, 0x845e, 0xce00, 0x0400, 0xd383, 0x0726,
    0xce00, 0xd8fe, 0xee17, 0x036e, 0xce00, 0x9c9d, 0xeed4, 0x0010,
    0xbe04, 0xfcc0, 0xd381, 0x075b, 0xbe04, 0xfcc0, 0xd381, 0x0746,
    0xee0d, 0xf8c1, 0xfe00, 0xe19d, 0x9e05, 0xa102, 0xce00, 0xd897,
    0xee8d, 0xf8c1, 0xfe00, 0xe19d, 0xae05, 0xf880, 0xd108, 0x7140,
    0x9eb5, 0xf8c1, 0xeebc, 0xf880, 0xce00, 0x9c9f, 0xee16, 0x0010,
    0xce00, 0x845f, 0xce00, 0x0400, 0xd383, 0x0742, 0xce00, 0xd91f,
    0xee17, 0xdd7e, 0xce00, 0x845e, 0xce00, 0x0400, 0xd383, 0x0742,
    0xce00, 0xd8fe, 0xee17, 0x036e, 0xce00, 0x9c9d, 0xeed4, 0x0010,
    0xbe04, 0xfcc0, 0xd389, 0x072c, 0xee0d, 0xf8c1, 0xfe00, 0xe19d,
    0x9e05, 0xa102, 0xee80, 0xd897, 0xd108, 0x7140, 0x9eb5, 0xa101,
    0xeebc, 0xf880, 0x9e01, 0xf8c1, 0xd389, 0x075b, 0xce00, 0x9c9f,
    0xee16, 0x0010, 0xce00, 0x845f, 0xce00, 0x0400, 0xd383, 0x075b,
    0xce00, 0xd91f, 0xee17, 0xdd7e, 0xce00, 0x845e, 0xce00, 0x0400,
    0xd383, 0x075b, 0xce00, 0xd8fe, 0xee17, 0x036e, 0xee15, 0xdd7c,
    0xbe03, 0xfcc0, 0xd389, 0x070f, 0xca00, 0x7140, 0xce00, 0xd806,
    0xce00, 0x8004, 0xda03, 0x7140, 0xce00, 0xd807, 0xce00, 0x8005,
    0xda03, 0x7140, 0x9ee5, 0xd8df, 0x9ee4, 0xd8de, 0x9ee7, 0xfcc1,
    0x9ee9, 0xe201, 0xce00, 0xe481, 0xce00, 0x000f, 0xce00, 0xe4c0,
    0xae03, 0xfff0, 0xce00, 0xd805, 0x9ee8, 0xe201, 0xd381, 0x0775,
    0xaeef, 0xfe80, 0x9e03, 0x9e9f, 0xd188, 0x07c0, 0xbeef, 0xfcc0,
    0xd383, 0x0771, 0xce00, 0xd804, 0x9ee9, 0xe201, 0xce00, 0xe4c0,
    0xae04, 0xfff0, 0xce00, 0xe4c0, 0xae05, 0x000f, 0x9e04, 0x9e9f,
    0xd188, 0x07c0, 0xce00, 0xdc06, 0x9ee9, 0xe201, 0xce00, 0xe481,
    0xce00, 0x000f, 0xce00, 0xe4c1, 0xce00, 0xfff0, 0x9e03, 0xe241,
    0xbe04, 0xe280, 0x8e1c, 0xd965, 0x8e1d, 0xd964, 0x8e08, 0xdd66,
    0xce00, 0xd804, 0xce00, 0x9848, 0xae02, 0xf900, 0xae06, 0xfb00,
    0xee14, 0xd85f, 0xee95, 0xd85e, 0x9e06, 0xf8c1, 0xd381, 0x079e,
    0xae06, 0xf900, 0x9e05, 0xf8c1, 0xee10, 0xf880, 0xee0c, 0xf8c1,
    0xfe00, 0xe19d, 0xaefe, 0xf880, 0xce00, 0xca1b, 0x9e05, 0x0010,
    0xce00, 0xd81b, 0xce00, 0x8008, 0xd383, 0x07a7, 0x8e1b, 0xd968,
    0xae02, 0xfb00, 0xd388, 0x07a7, 0xee0c, 0xf8c1, 0xaefe, 0xf880,
    0xce00, 0xd8fb, 0xce00, 0x0010, 0xce00, 0x8408, 0xce00, 0x0010,
    0xd383, 0x07a7, 0x8e1b, 0xd968, 0xae02, 0xfb00, 0xd188, 0x0b5b,
    0xce00, 0x9c9f, 0xce00, 0x0010, 0xce00, 0x845f, 0xce00, 0x0400,
    0xd383, 0x07b4, 0xce00, 0xd91f, 0xce00, 0xdd7e, 0xce00, 0x845e,
    0xce00, 0x0400, 0xd383, 0x07b4, 0xce00, 0xd8fe, 0xce00, 0x036e,
    0xce00, 0xd81b, 0xce00, 0x989d, 0xce00, 0x9848, 0xbe02, 0xf8c0,
    0xd381, 0x078c, 0x9e04, 0x9e9f, 0xd188, 0x07c0, 0xce00, 0xdd7c,
    0xce00, 0xd81c, 0xce00, 0x8047, 0xd38b, 0x0786, 0xca00, 0x7140,
    0xce00, 0x845f, 0xce00, 0x0400, 0xda03, 0x7140, 0xce00, 0xd89f,
    0xce00, 0xdd7e, 0xce00, 0x845e, 0xce00, 0x0400, 0xd383, 0x07c0,
    0xce00, 0xd8fe, 0xce00, 0x036e, 0xd388, 0x07c0, 0x9ed0, 0xf8c1,
    0xaed4, 0xf880, 0x9ed2, 0xf8c1, 0xaed6, 0xf880, 0x9ed0, 0xc204,
    0xd383, 0x07d4, 0xaed4, 0xf880, 0x9ed0, 0xf8c1, 0xaed6, 0xf880,
    0x9ecf, 0xf8c1, 0xaed3, 0xf880, 0x9ed1, 0xf8c1, 0xaed5, 0xf880,
    0x9ecf, 0xc204, 0xd383, 0x07dd, 0xaed3, 0xf880, 0x9ecf, 0xf8c1,
    0xaed5, 0xf880, 0xd188, 0x063e, 0xae03, 0xfd00, 0x9ed0, 0xd8c5,
    0x9ed2, 0x9e05, 0xd38b, 0x07e4, 0xce00, 0xdf65, 0xbe03, 0xfec0,
    0xae04, 0xfd00, 0x9ecf, 0xd8c6, 0x9ed1, 0x9e06, 0xd38b, 0x07ea,
    0xce00, 0xdf66, 0xbe04, 0xfec0, 0x9eed, 0xf8c1, 0x9ed6, 0xe241,
    0xd383, 0x087d, 0x9eec, 0xf8c1, 0x9ed5, 0xe241, 0xd383, 0x087d,
    0x9eeb, 0xf8c1, 0x9ed4, 0xe201, 0xd383, 0x087d, 0x9eea, 0xf8c1,
    0x9ed3, 0xe201, 0xd383, 0x087d, 0x9ee0, 0xf8c1, 0xd389, 0x087d,
    0xce00, 0xeac1, 0x9eff, 0x000c, 0xd381, 0x082b, 0xce00, 0xe441,
    0xce00, 0x0004, 0xd381, 0x0854, 0xce00, 0xe441, 0xce00, 0x0004,
    0xda01, 0x7140, 0xd388, 0x0802, 0x9ed2, 0xd8dd, 0xee14, 0xd85d,
    0x9ed1, 0xd8dc, 0xee95, 0xd85c, 0xce00, 0xd805, 0xce00, 0x8006,
    0xd383, 0x081a, 0xae02, 0xda45, 0xce00, 0xd805, 0xae01, 0xec80,
    0xee10, 0xda5d, 0xee0c, 0xf8c1, 0xfe00, 0xe182, 0xeebc, 0xf880,
    0xbe01, 0x9646, 0xd38b, 0x0815, 0xbe01, 0x9685, 0x9e04, 0x9e9c,
    0xee15, 0xd85c, 0x9e03, 0x9e9d, 0xee94, 0xd85d, 0xbe02, 0xfcc0,
    0xd383, 0x080c, 0xca00, 0x7140, 0xae02, 0xda46, 0xce00, 0xd806,
    0xae01, 0xec80, 0xee10, 0xda5d, 0xee0c, 0xf8c1, 0xfe00, 0xe182,
    0xeebc, 0xf880, 0xbe01, 0x9645, 0xd38b, 0x0826, 0xbe01, 0x9686,
    0x9e03, 0x9e9d, 0xee14, 0xd85d, 0x9e04, 0x9e9c, 0xee95, 0xd85c,
    0xbe02, 0xfcc0, 0xd383, 0x081d, 0xca00, 0x7140, 0x9ed2, 0xd8dd,
    0xee14, 0xd85d, 0x9ed1, 0xd8dc, 0xee95, 0xd85c, 0xce00, 0xd805,
    0xce00, 0x8006, 0xd383, 0x0843, 0xae02, 0xda45, 0xce00, 0xd805,
    0xae01, 0xec80, 0xee10, 0xda5d, 0xee0c, 0xf8c1, 0xfe00, 0xe181,
    0xeebc, 0xf880, 0xbe01, 0x9646, 0xd38b, 0x083e, 0xbe01, 0x9685,
    0x9e04, 0x9e9c, 0xee15, 0xd85c, 0x9e03, 0x9e9d, 0xee94, 0xd85d,
    0xbe02, 0xfcc0, 0xd383, 0x0835, 0xca00, 0x7140, 0xae02, 0xda46,
    0xce00, 0xd806, 0xae01, 0xec80, 0xee10, 0xda5d, 0xee0c, 0xf8c1,
    0xfe00, 0xe181, 0xeebc, 0xf880, 0xbe01, 0x9645, 0xd38b, 0x084f,
    0xbe01, 0x9686, 0x9e03, 0x9e9d, 0xee14, 0xd85d, 0x9e04, 0x9e9c,
    0xee95, 0xd85c, 0xbe02, 0xfcc0, 0xd383, 0x0846, 0xca00, 0x7140,
    0x9ed2, 0xd8dd, 0xee14, 0xd85d, 0x9ed1, 0xd8dc, 0xee95, 0xd85c,
    0xce00, 0xd805, 0xce00, 0x8006, 0xd383, 0x086c, 0xae02, 0xda45,
    0xce00, 0xd805, 0xae01, 0xec80, 0xee10, 0xda5d, 0xfe00, 0xe186,
    0xee0c, 0xe301, 0xeebc, 0xf880, 0xbe01, 0x9646, 0xd38b, 0x0867,
    0xbe01, 0x9685, 0x9e04, 0x9e9c, 0xee15, 0xd85c, 0x9e03, 0x9e9d,
    0xee94, 0xd85d, 0xbe02, 0xfcc0, 0xd383, 0x085e, 0xca00, 0x7140,
    0xae02, 0xda46, 0xce00, 0xd806, 0xae01, 0xec80, 0xee10, 0xda5d,
    0xfe00, 0xe186, 0xee0c, 0xe301, 0xeebc, 0xf880, 0xbe01, 0x9645,
    0xd38b, 0x0878, 0xbe01, 0x9686, 0x9e03, 0x9e9d, 0xee14, 0xd85d,
    0x9e04, 0x9e9c, 0xee95, 0xd85c, 0xbe02, 0xfcc0, 0xd383, 0x086f,
    0xca00, 0x7140, 0x9ed2, 0xd8dd, 0x9ed1, 0xd8dc, 0xce00, 0xdd1b,
    0x9ee1, 0xf8c1, 0xbee0, 0xf8c0, 0xd389, 0x0884, 0xce00, 0xfb01,
    0xaefe, 0xf880, 0xce00, 0xd805, 0xce00, 0x8006, 0xd383, 0x08a5,
    0xae02, 0xda45, 0xce00, 0xd805, 0xae01, 0xec80, 0x9eed, 0x861d,
    0xd383, 0x0894, 0x9eeb, 0x865d, 0xd383, 0x0894, 0x9eec, 0x861c,
    0xd383, 0x0894, 0x9eea, 0x865c, 0xd383, 0x0894, 0xd188, 0x0b5b,
    0x9ee0, 0xf8c1, 0xd381, 0x089d, 0xbefe, 0xe398, 0xbec3, 0xfcc0,
    0xd383, 0x089d, 0xee10, 0xfe80, 0xaec3, 0xfe80, 0xfefe, 0xe199,
    0xaefe, 0xf880, 0xbe01, 0x9646, 0xd38b, 0x08a1, 0xbe01, 0x9685,
    0x9e04, 0x9e9c, 0x9e03, 0x9e9d, 0xbe02, 0xfcc0, 0xd383, 0x088b,
    0xd388, 0x08c1, 0xae02, 0xda46, 0xce00, 0xd806, 0xae01, 0xec80,
    0x9eed, 0x861d, 0xd383, 0x08b1, 0x9eeb, 0x865d, 0xd383, 0x08b1,
    0x9eec, 0x861c, 0xd383, 0x08b1, 0x9eea, 0x865c, 0xd383, 0x08b1,
    0xd188, 0x0b5b, 0x9ee0, 0xf8c1, 0xd381, 0x08ba, 0xbefe, 0xe398,
    0xbec3, 0xfcc0, 0xd383, 0x08ba, 0xee10, 0xfe80, 0xaec3, 0xfe80,
    0xfefe, 0xe199, 0xaefe, 0xf880, 0xbe01, 0x9645, 0xd38b, 0x08be,
    0xbe01, 0x9686, 0x9e03, 0x9e9d, 0x9e04, 0x9e9c, 0xbe02, 0xfcc0,
    0xd383, 0x08a8, 0x9efe, 0xf8c1, 0xaee1, 0xf880, 0xca00, 0x7140,
    0x9ecd, 0xf8c1, 0x9ecb, 0xe201, 0xd383, 0x09d6, 0xce00, 0xf881,
    0xd389, 0x08cc, 0x9ece, 0xf8c1, 0x9ecc, 0xe201, 0xd383, 0x09d6,
    0x9ecc, 0xf8c1, 0x9ec9, 0xe281, 0xce00, 0xe441, 0xce00, 0x0001,
    0xce00, 0xe4c0, 0xae02, 0xfff0, 0xce00, 0xeac1, 0x9ecc, 0xfff0,
    0xbe02, 0xe240, 0xbe02, 0xf998, 0xbe02, 0xfcc0, 0x9ecc, 0xf8c1,
    0xee10, 0xfa80, 0xfe00, 0xe186, 0xce00, 0xc440, 0xce00, 0x0001,
    0xce00, 0xcc20, 0x9ecc, 0xf8c1, 0x9ec9, 0xe281, 0xee10, 0xfa80,
    0xfe00, 0xe186, 0xce00, 0xe441, 0xce00, 0x0001, 0xce00, 0xec21,
    0xce00, 0xda81, 0xd389, 0x08e7, 0xce00, 0xdb01, 0xce00, 0xea41,
    0x9e02, 0x0001, 0xd389, 0x08ec, 0xce00, 0xd801, 0xce00, 0x98c0,
    0xce00, 0xeac1, 0x9ece, 0x000f, 0xae01, 0xf880, 0xce00, 0xeac1,
    0x9ecc, 0x000f, 0xbe01, 0xe240, 0xd381, 0x097a, 0x9e01, 0xf8c1,
    0xd383, 0x093c, 0x9e01, 0xf8c1, 0xee10, 0xf880, 0xfe00, 0xe186,
    0xce00, 0xc442, 0xce00, 0x0001, 0xce00, 0xdb62, 0x9eca, 0xfec1,
    0xae03, 0xf880, 0xda01, 0x7140, 0x9ecd, 0xd8de, 0xee17, 0xd85e,
    0x9ecb, 0xd8dc, 0xee15, 0xd85c, 0xee1f, 0xd85a, 0x9e02, 0xfec1,
    0xae04, 0xf880, 0x9ece, 0xd8df, 0xee16, 0xd85f, 0x9ecc, 0xd8dd,
    0xeed4, 0xd85d, 0xee0d, 0xf8c1, 0xfe00, 0xe19d, 0xae05, 0xf880,
    0xce00, 0x9c9f, 0xeed6, 0x0010, 0xee0d, 0xf8c1, 0xfe00, 0xe19d,
    0x9e05, 0xa102, 0xce00, 0xd897, 0xee8d, 0xf8c1, 0xfe00, 0xe19d,
    0xae05, 0xf880, 0xd108, 0x7140, 0x9eb5, 0xa100, 0xeebc, 0xf880,
    0xce00, 0x9c9f, 0xeed6, 0x0010, 0xce00, 0x9c9d, 0xee14, 0x0010,
    0xbe04, 0xfcc0, 0xd381, 0x0937, 0xbe04, 0xfcc0, 0xd381, 0x0930,
    0xee0d, 0xf8c1, 0xfe00, 0xe19d, 0x9e05, 0xa102, 0xce00, 0xd897,
    0xee8d, 0xf8c1, 0xfe00, 0xe19d, 0xae05, 0xf880, 0xd108, 0x7140,
    0x9eb5, 0xf8c1, 0xeebc, 0xf880, 0xce00, 0x9c9f, 0xeed6, 0x0010,
    0xce00, 0x9c9d, 0xee14, 0x0010, 0xbe04, 0xfcc0, 0xd389, 0x0920,
    0xee0d, 0xf8c1, 0xfe00, 0xe19d, 0x9e05, 0xa102, 0xee80, 0xd897,
    0xd108, 0x7140, 0x9eb5, 0xa101, 0xeebc, 0xf880, 0xee15, 0xdd7c,
    0xee17, 0xdd7e, 0xbe03, 0xfcc0, 0xd389, 0x0903, 0xca00, 0x7140,
    0x9e01, 0xf8c1, 0xee10, 0xf880, 0xfe00, 0xe186, 0xce00, 0xc442,
    0xce00, 0x0001, 0xce00, 0xdb62, 0x9eca, 0xfec1, 0xae03, 0xf880,
    0x9ecd, 0xd8de, 0xee17, 0xd85e, 0x9ecb, 0xd8dc, 0xee15, 0xd85c,
    0xee1f, 0xd85a, 0x9e02, 0xfec1, 0xae04, 0xf880, 0x9ece, 0xd8df,
    0xee16, 0xd85f, 0x9ecc, 0xd8dd, 0xeed4, 0xd85d, 0xee8d, 0xf8c1,
    0xfe00, 0xe19d, 0xae05, 0xf880, 0xce00, 0xd897, 0xd108, 0x7140,
    0x9eb5, 0xa100, 0xeebc, 0xf880, 0xce00, 0x9c9f, 0xeed6, 0x0010,
    0xce00, 0x9c9d, 0xee14, 0x0010, 0xbe04, 0xfcc0, 0xd381, 0x0975,
    0xbe04, 0xfcc0, 0xd381, 0x096e, 0xee0d, 0xf8c1, 0xfe00, 0xe19d,
    0x9e05, 0xa102, 0xce00, 0xd897, 0xee8d, 0xf8c1, 0xfe00, 0xe19d,
    0xae05, 0xf880, 0xd108, 0x7140, 0x9eb5, 0xf8c1, 0xeebc, 0xf880,
    0xce00, 0x9c9f, 0xeed6, 0x0010, 0xce00, 0x9c9d, 0xee14, 0x0010,
    0xbe04, 0xfcc0, 0xd389, 0x095e, 0xee0d, 0xf8c1, 0xfe00, 0xe19d,
    0x9e05, 0xa102, 0xee80, 0xd897, 0xd108, 0x7140, 0x9eb5, 0xa101,
    0xeebc, 0xf880, 0xee15, 0xdd7c, 0xee17, 0xdd7e, 0xbe03, 0xfcc0,
    0xd389, 0x0949, 0xca00, 0x7140, 0xce00, 0xea41, 0x9eff, 0x000c,
    0xd381, 0x09ab, 0x9eca, 0xfec1, 0xae03, 0xf880, 0xda01, 0x7140,
    0x9ecd, 0xd8de, 0xee17, 0xd85e, 0x9ecb, 0xd8dc, 0xee15, 0xd85c,
    0x9e02, 0xfec1, 0xae04, 0xf880, 0x9ece, 0xd8df, 0xee16, 0xd85f,
    0x9ecc, 0xd8dd, 0xeed4, 0xd85d, 0xee8d, 0xf8c1, 0xaeb5, 0xf880,
    0xee0c, 0xf8c1, 0x9eb5, 0xa100, 0xeebc, 0xf880, 0xce00, 0x9c9f,
    0xeed6, 0x0010, 0xce00, 0x9c9d, 0xee14, 0x0010, 0xbe04, 0xfcc0,
    0xd381, 0x09a6, 0xbe04, 0xfcc0, 0xd381, 0x09a1, 0xee8d, 0xd8d7,
    0xd108, 0x7140, 0x9eb5, 0xf8c1, 0xeebc, 0xf880, 0xce00, 0x9c9f,
    0xeed6, 0x0010, 0xce00, 0x9c9d, 0xee14, 0x0010, 0xbe04, 0xfcc0,
    0xd389, 0x0997, 0xee8d, 0xf8c1, 0xaeb5, 0xf880, 0xee0c, 0xf8c1,
    0x9eb5, 0xa101, 0xeebc, 0xf880, 0xee15, 0xdd7c, 0xee17, 0xdd7e,
    0xbe03, 0xfcc0, 0xd389, 0x0984, 0xca00, 0x7140, 0x9eca, 0xfec1,
    0xae03, 0xf880, 0xda01, 0x7140, 0x9ecd, 0xd8de, 0xee17, 0xd85e,
    0x9ecb, 0xd8dc, 0xee15, 0xd85c, 0xee1f, 0xd85a, 0x9e02, 0xfec1,
    0xae04, 0xf880, 0x9ece, 0xd8df, 0xee16, 0xd85f, 0x9ecc, 0xd8dd,
    0xeed4, 0xd85d, 0xee8d, 0xd8d7, 0xd108, 0x7140, 0x9eb5, 0xa100,
    0xeebc, 0xf880, 0xce00, 0x9c9f, 0xeed6, 0x0010, 0xce00, 0x9c9d,
    0xee14, 0x0010, 0xbe04, 0xfcc0, 0xd381, 0x09d1, 0xbe04, 0xfcc0,
    0xd381, 0x09cd, 0xee8d, 0xf8c1, 0xeebc, 0xf880, 0xce00, 0x9c9f,
    0xeed6, 0x0010, 0xce00, 0x9c9d, 0xee14, 0x0010, 0xbe04, 0xfcc0,
    0xd389, 0x09c5, 0xee8d, 0xd8d7, 0xd108, 0x7140, 0x9eb5, 0xa101,
    0xeebc, 0xf880, 0xee15, 0xdd7c, 0xee17, 0xdd7e, 0xbe03, 0xfcc0,
    0xd389, 0x09b3, 0xca00, 0x7140, 0x9ecc, 0xf8c1, 0x9ec9, 0xe281,
    0xce00, 0xe441, 0xce00, 0x0001, 0xce00, 0xe4c0, 0xae02, 0xfff0,
    0xce00, 0xeac1, 0x9ecc, 0xfff0, 0xbe02, 0xe240, 0xbe02, 0xf998,
    0xbe02, 0xfcc0, 0x9ecc, 0xf8c1, 0xee10, 0xfa80, 0xfe00, 0xe186,
    0xce00, 0xc440, 0xce00, 0x0001, 0xce00, 0xcc20, 0x9ecc, 0xf8c1,
    0x9ec9, 0xe281, 0xee10, 0xfa80, 0xfe00, 0xe186, 0xce00, 0xe441,
    0xce00, 0x0001, 0xce00, 0xec21, 0xce00, 0xda81, 0xd389, 0x09f1,
    0xce00, 0xdb01, 0xce00, 0xea41, 0x9e02, 0x0001, 0xd389, 0x09f6,
    0xce00, 0xd801, 0xce00, 0x98c0, 0xce00, 0xea41, 0x9ec9, 0x0001,
    0x9ece, 0xe281, 0xce00, 0xe4c0, 0xae01, 0x000f, 0xce00, 0xea41,
    0x9ec9, 0x0001, 0x9ecc, 0xe281, 0xce00, 0xe4c1, 0xce00, 0x000f,
    0xbe01, 0xe240, 0xd381, 0x0a94, 0x9e01, 0xf8c1, 0xd383, 0x0a48,
    0x9e01, 0xf8c1, 0xee10, 0xf880, 0xfe00, 0xe186, 0xce00, 0xc442,
    0xce00, 0x0001, 0x9eca, 0xfec1, 0xae03, 0xf880, 0xda01, 0x7140,
    0xce00, 0xea41, 0x9eca, 0x0001, 0x9ecd, 0xc29e, 0xee17, 0xd85e,
    0x9ecb, 0xc29c, 0xee15, 0xd85c, 0xee1f, 0xd85a, 0x9e02, 0xfec1,
    0xae04, 0xf880, 0xce00, 0xea41, 0x9ec9, 0x0001, 0x9ece, 0xc29f,
    0xee16, 0xd85f, 0x9ecc, 0xc29d, 0xeed4, 0xd85d, 0xee8d, 0xf8c1,
    0xfe00, 0xe19d, 0xae05, 0xf880, 0xce00, 0xd897, 0xd108, 0x7140,
    0x9eb5, 0xa101, 0xeebc, 0xf880, 0xce00, 0x9c5f, 0xeed6, 0x0010,
    0xce00, 0x9c5d, 0xee14, 0x0010, 0xbe04, 0xfcc0, 0xd381, 0x0a41,
    0xbe04, 0xfcc0, 0xd381, 0x0a3a, 0xee0d, 0xf8c1, 0xfe00, 0xe19d,
    0x9e05, 0xa102, 0xce00, 0xd897, 0xee8d, 0xf8c1, 0xfe00, 0xe19d,
    0xae05, 0xf880, 0xd108, 0x7140, 0x9eb5, 0xf8c1, 0xeebc, 0xf880,
    0xce00, 0x9c5f, 0xeed6, 0x0010, 0xce00, 0x9c5d, 0xee14, 0x0010,
    0xbe04, 0xfcc0, 0xd389, 0x0a2a, 0xee0d, 0xf8c1, 0xfe00, 0xe19d,
    0x9e05, 0xa102, 0xee80, 0xd897, 0xd108, 0x7140, 0x9eb5, 0xa100,
    0xeebc, 0xf880, 0xce00, 0x9c5c, 0xee15, 0x0001, 0xce00, 0x9c5e,
    0xee17, 0x0001, 0xbe03, 0xfcc0, 0xd389, 0x0a13, 0xca00, 0x7140,
    0x9e01, 0xf8c1, 0xee10, 0xf880, 0xfe00, 0xe186, 0xce00, 0xc442,
    0xce00, 0x0001, 0x9eca, 0xfec1, 0xae03, 0xf880, 0xda01, 0x7140,
    0xce00, 0xea41, 0x9eca, 0x0001, 0x9ecd, 0xc29e, 0xee17, 0xd85e,
    0x9ecb, 0xc29c, 0xee15, 0xd85c, 0xee1f, 0xd85a, 0x9e02, 0xfec1,
    0xae04, 0xf880, 0xce00, 0xea41, 0x9ec9, 0x0001, 0x9ece, 0xc29f,
    0xee16, 0xd85f, 0x9ecc, 0xc29d, 0xeed4, 0xd85d, 0xee0d, 0xf8c1,
    0xfe00, 0xe19d, 0xae05, 0xf880, 0xce00, 0x9c5f, 0xeed6, 0x0010,
    0xee0d, 0xf8c1, 0xfe00, 0xe19d, 0x9e05, 0xa102, 0xce00, 0xd897,
    0xee8d, 0xf8c1, 0xfe00, 0xe19d, 0xae05, 0xf880, 0xd108, 0x7140,
    0x9eb5, 0xa101, 0xeebc, 0xf880, 0xce00, 0x9c5f, 0xeed6, 0x0010,
    0xce00, 0x9c5d, 0xee14, 0x0010, 0xbe04, 0xfcc0, 0xd381, 0x0a8d,
    0xbe04, 0xfcc0, 0xd381, 0x0a86, 0xee0d, 0xf8c1, 0xfe00, 0xe19d,
    0x9e05, 0xa102, 0xce00, 0xd897, 0xee8d, 0xf8c1, 0xfe00, 0xe19d,
    0xae05, 0xf880, 0xd108, 0x7140, 0x9eb5, 0xf8c1, 0xeebc, 0xf880,
    0xce00, 0x9c5f, 0xeed6, 0x0010, 0xce00, 0x9c5d, 0xee14, 0x0010,
    0xbe04, 0xfcc0, 0xd389, 0x0a76, 0xee0d, 0xf8c1, 0xfe00, 0xe19d,
    0x9e05, 0xa102, 0xee80, 0xd897, 0xd108, 0x7140, 0x9eb5, 0xa100,
    0xeebc, 0xf880, 0xce00, 0x9c5c, 0xee15, 0x0001, 0xce00, 0x9c5e,
    0xee17, 0x0001, 0xbe03, 0xfcc0, 0xd389, 0x0a57, 0xca00, 0x7140,
    0xce00, 0xea41, 0x9eff, 0x000c, 0xd381, 0x0aca, 0x9eca, 0xfec1,
    0xae03, 0xf880, 0xda01, 0x7140, 0xce00, 0xea41, 0x9eca, 0x0001,
    0x9ecd, 0xc29e, 0xee17, 0xd85e, 0x9ecb, 0xc29c, 0xee15, 0xd85c,
    0xee1f, 0xd85a, 0x9e02, 0xfec1, 0xae04, 0xf880, 0xce00, 0xea41,
    0x9ec9, 0x0001, 0x9ece, 0xc29f, 0xee16, 0xd85f, 0x9ecc, 0xc29d,
    0xeed4, 0xd85d, 0xee8d, 0xd8d7, 0xd108, 0x7140, 0x9eb5, 0xa101,
    0xeebc, 0xf880, 0xce00, 0x9c5f, 0xeed6, 0x0010, 0xce00, 0x9c5d,
    0xee14, 0x0010, 0xbe04, 0xfcc0, 0xd381, 0x0ac3, 0xbe04, 0xfcc0,
    0xd381, 0x0abf, 0xee8d, 0xd8d7, 0xd108, 0x7140, 0x9eb5, 0xf8c1,
    0xeebc, 0xf880, 0xce00, 0x9c5f, 0xeed6, 0x0010, 0xce00, 0x9c5d,
    0xee14, 0x0010, 0xbe04, 0xfcc0, 0xd389, 0x0ab5, 0xee8d, 0xd8d7,
    0xd108, 0x7140, 0x9eb5, 0xa100, 0xeebc, 0xf880, 0xce00, 0x9c5c,
    0xee15, 0x0001, 0xce00, 0x9c5e, 0xee17, 0x0001, 0xbe03, 0xfcc0,
    0xd389, 0x0aa1, 0xca00, 0x7140, 0x9eca, 0xfec1, 0xae03, 0xf880,
    0xda01, 0x7140, 0xce00, 0xea41, 0x9eca, 0x0001, 0x9ecd, 0xc29e,
    0xee17, 0xd85e, 0x9ecb, 0xc29c, 0xee15, 0xd85c, 0x9e02, 0xfec1,
    0xae04, 0xf880, 0xce00, 0xea41, 0x9ec9, 0x0001, 0x9ece, 0xc29f,
    0xee16, 0xd85f, 0x9ecc, 0xc29d, 0xeed4, 0xd85d, 0xee8d, 0xf8c1,
    0xaeb5, 0xf880, 0xee0c, 0xf8c1, 0x9eb5, 0xa101, 0xeebc, 0xf880,
    0xce00, 0x9c5f, 0xeed6, 0x0010, 0xce00, 0x9c5d, 0xee14, 0x0010,
    0xbe04, 0xfcc0, 0xd381, 0x0af5, 0xbe04, 0xfcc0, 0xd381, 0x0af0,
    0xee8d, 0xf8c1, 0xeebc, 0xf880, 0xce00, 0x9c5f, 0xeed6, 0x0010,
    0xce00, 0x9c5d, 0xee14, 0x0010, 0xbe04, 0xfcc0, 0xd389, 0x0ae8,
    0xee8d, 0xf8c1, 0xaeb5, 0xf880, 0xee0c, 0xf8c1, 0x9eb5, 0xa100,
    0xeebc, 0xf880, 0xce00, 0x9c5c, 0xee15, 0x0001, 0xce00, 0x9c5e,
    0xee17, 0x0001, 0xbe03, 0xfcc0, 0xd389, 0x0ad3, 0xca00, 0x7140,
    0xce00, 0xea81, 0x9ec6, 0x000f, 0xce00, 0xe4c1, 0xce00, 0xfff0,
    0xae01, 0xf99c, 0x9ebe, 0xf8c1, 0xee10, 0xf880, 0xfe00, 0xe186,
    0xce00, 0xc440, 0xce00, 0x0001, 0xce00, 0xdb60, 0x9ec5, 0xfec1,
    0xae02, 0xf880, 0xda01, 0x7140, 0x9ec7, 0xd8de, 0xee17, 0xd85e,
    0xee12, 0xd858, 0x9e01, 0xfec1, 0xae03, 0xf880, 0x9ebe, 0xd8df,
    0xeed6, 0xd85f, 0xee0d, 0xf8c1, 0xfe00, 0xe19d, 0xae04, 0xf880,
    0xce00, 0x9c9f, 0xeed6, 0x0010, 0xee0d, 0xf8c1, 0xfe00, 0xe19d,
    0x9e04, 0xa100, 0x9ec4, 0xe301, 0xee73, 0xf880, 0xee12, 0xdd78,
    0xee0d, 0xf8c1, 0xfe00, 0xe19d, 0xae04, 0xf880, 0xce00, 0x9c9f,
    0xeed6, 0x0010, 0xbe03, 0xfcc0, 0xd389, 0x0b16, 0xee17, 0xdd7e,
    0xbe02, 0xfcc0, 0xd389, 0x0b0d, 0xca00, 0x7140, 0x9ef5, 0xf8c1,
    0x9ef3, 0xe241, 0xce00, 0xe4c1, 0xce00, 0xfff0, 0xae01, 0xf99c,
    0xbe01, 0xfcc0, 0x9ef4, 0xf8c1, 0x9ef2, 0xe241, 0xce00, 0xfc81,
    0xae02, 0xfe80, 0x9ef5, 0xf8c1, 0xee10, 0xf880, 0xfe00, 0xe186,
    0xce00, 0xc440, 0xce00, 0x0001, 0xce00, 0xdb60, 0x9ef4, 0xd8de,
    0xee17, 0xd85e, 0xee14, 0xd91d, 0xce00, 0xd8fc, 0xee15, 0x0328,
    0x9e01, 0xfec1, 0xae03, 0xf880, 0x9ef5, 0xd8df, 0xeed6, 0xd85f,
    0xee0d, 0xf8c1, 0xfe00, 0xe19d, 0xae04, 0xf880, 0xce00, 0x9c9f,
    0xeed6, 0x0010, 0xee0d, 0xf8c1, 0xfe00, 0xe19d, 0xce00, 0xd881,
    0x9e04, 0xa100, 0xee80, 0x7140, 0xeebc, 0xf880, 0xae04, 0xd841,
    0xce00, 0x9c9f, 0xeed6, 0x0010, 0xce00, 0x9c9d, 0xee14, 0x0010,
    0xce00, 0x845d, 0xce00, 0x0400, 0xd383, 0x0b55, 0xce00, 0xd91d,
    0xee15, 0xdd7c, 0xbe03, 0xfcc0, 0xd389, 0x0b45, 0xee17, 0xdd7e,
    0xbe02, 0xfcc0, 0xd389, 0x0b3c, 0xca00, 0x7140, 0xee14, 0xd85d,
    0xee95, 0xd85c, 0xee10, 0xde5b, 0xfe00, 0xe186, 0xce00, 0xde80,
    0xee10, 0xde5d, 0xfe00, 0x81e0, 0xfe00, 0xe186, 0xce00, 0xe180,
    0xd381, 0x0b66, 0xce00, 0xf901, 0xce00, 0xfe81, 0x8e02, 0x98c0,
    0xce00, 0xfa81, 0x8e01, 0x98c0, 0xfefe, 0xa037, 0xee1f, 0xd85a,
    0xd108, 0x7140, 0x9eb5, 0xa101, 0xeebc, 0xf880, 0xce00, 0xd802,
    0xda01, 0x7140, 0xce00, 0x949d, 0xee94, 0x0010, 0xd108, 0x7140,
    0x9eb5, 0xa102, 0xeebc, 0xf880, 0xca00, 0x7140, 0x9eff, 0xe599,
    0xce00, 0xe480, 0xee1f, 0x0b7b, 0xd308, 0x7140, 0xce00, 0xd8fa,
    0xce00, 0x0bbb, 0xce00, 0xd8f9, 0xca00, 0x0bbd, 0xce00, 0xd8fa,
    0xce00, 0x0bbf, 0xce00, 0xd8f9, 0xca00, 0x0bc1, 0xce00, 0xd8fa,
    0xce00, 0x0bc3, 0xce00, 0xd8f9, 0xca00, 0x0bc6, 0xce00, 0xd8fa,
    0xce00, 0x0bc9, 0xce00, 0xd8f9, 0xca00, 0x0bcb, 0xce00, 0xd8fa,
    0xce00, 0x0bcd, 0xce00, 0xd8f9, 0xca00, 0x0bd0, 0xce00, 0xd8fa,
    0xce00, 0x0bd3, 0xce00, 0xd8f9, 0xca00, 0x0bd5, 0xce00, 0xd8fa,
    0xce00, 0x0bd7, 0xce00, 0xd8f9, 0xca00, 0x0bd9, 0xce00, 0xd8fa,
    0xce00, 0x0bdb, 0xce00, 0xd8f9, 0xca00, 0x0bdd, 0xce00, 0xd8fa,
    0xce00, 0x0bdf, 0xce00, 0xd8f9, 0xca00, 0x0be1, 0xce00, 0xd8fa,
    0xce00, 0x0be3, 0xce00, 0xd8f9, 0xca00, 0x0be5, 0xce00, 0xd8fa,
    0xce00, 0x0be7, 0xce00, 0xd8f9, 0xca00, 0x0be9, 0xce00, 0xd8fa,
    0xce00, 0x0beb, 0xce00, 0xd8f9, 0xca00, 0x0bee, 0xce00, 0xd8fa,
    0xce00, 0x0bf1, 0xce00, 0xd8f9, 0xca00, 0x0bf3, 0xce00, 0xd8fa,
    0xce00, 0x0bf5, 0xce00, 0xd8f9, 0xca00, 0x0bf8, 0xce00, 0xd8fa,
    0xce00, 0x0bfb, 0xce00, 0xd8f9, 0xca00, 0x0bfd, 0xce00, 0xd8fa,
    0xce00, 0x0bff, 0xce00, 0xd8f9, 0xca00, 0x0c01, 0xee0c, 0xf8c1,
    0xaab5, 0xf900, 0xee0d, 0xf8c1, 0xaab5, 0xf900, 0xee0c, 0xf8c1,
    0xaab5, 0x9137, 0xee0d, 0xf8c1, 0xaab5, 0x9137, 0xee0c, 0xf8c1,
    0xaeb5, 0xda57, 0xbab5, 0xe2c0, 0xee0d, 0xf8c1, 0xaeb5, 0xda57,
    0xbab5, 0xe2c0, 0xee0c, 0xf8c1, 0xaab5, 0xda57, 0xee0d, 0xf8c1,
    0xaab5, 0xda57, 0xee0c, 0xf8c1, 0xaeb5, 0xfa80, 0xbab5, 0x96d7,
    0xee0d, 0xf8c1, 0xaeb5, 0xfa80, 0xbab5, 0x96d7, 0xee0c, 0xf8c1,
    0xaab5, 0xfa80, 0xee0d, 0xf8c1, 0xaab5, 0xfa80, 0xee0c, 0xf8c1,
    0xaab5, 0x9117, 0xee0d, 0xf8c1, 0xaab5, 0x9117, 0xee0c, 0xf8c1,
    0xaab5, 0x90f7, 0xee0d, 0xf8c1, 0xaab5, 0x90f7, 0xee0c, 0xf8c1,
    0xaab5, 0x90d7, 0xee0d, 0xf8c1, 0xaab5, 0x90d7, 0xee0c, 0xf8c1,
    0xaab5, 0x9177, 0xee0d, 0xf8c1, 0xaab5, 0x9177, 0xee0c, 0xf8c1,
    0xaab5, 0xf880, 0xee0d, 0xf8c1, 0xaab5, 0xf880, 0xee0c, 0xf8c1,
    0xaeb5, 0xda57, 0xbab5, 0xe340, 0xee0d, 0xf8c1, 0xaeb5, 0xda57,
    0xbab5, 0xe340, 0xee0c, 0xf8c1, 0xaab5, 0xd857, 0xee0d, 0xf8c1,
    0xaab5, 0xd857, 0xee0c, 0xf8c1, 0xaeb5, 0xda57, 0xbab5, 0xe2e0,
    0xee0d, 0xf8c1, 0xaeb5, 0xda57, 0xbab5, 0xe2e0, 0xee0c, 0xf8c1,
    0xaab5, 0x9157, 0xee0d, 0xf8c1, 0xaab5, 0x9157, 0xee0c, 0xf8c1,
    0xaab5, 0xfb00, 0xee0d, 0xf8c1, 0xaab5, 0xfb00
};
