/* 
 * (c) Copyright 1989, 1990, 1991, 1992 OPEN SOFTWARE FOUNDATION, INC. 
 * ALL RIGHTS RESERVED 
*/ 
/* 
 * Motif Release 1.2
*/ 
#ifdef REV_INFO
#ifndef lint
static char rcsid[] = "$RCSfile: MallocWrap.c,v $ $Revision: 1.3.2.2 $ $Date: 1992/03/27 18:35:50 $"
#endif
#endif
/*
 * FILE: MallocWrap.c
 *
 * This file contains a wrapper for the Xt functions that do memory
 *   allocation and management (XtMalloc, XtRealloc, XtCalloc, and
 *   XtFree).  It pulls in other support routines to track what
 *   is being allocated and freed, etc. 
 *
 * Based on: 
 *       Conor Cahill's Malloc Library - (see copyright statement)
 *       David Brook's XtMalloc wrapper
 * Condensed 910214, SMT
 *
 * (c) Copyright 1990 Conor P. Cahill (uunet!virtech!cpcahil).  
 * You may copy, distribute, and use this software as long as this
 * copyright statement is not removed.
 */

/*
 *  Includes
 */

#include <stdio.h>

#ifdef X11R5
#include <stdlib.h>
#endif

#include <fcntl.h>
#include <X11/Xlib.h>
#ifdef X11R5
#include <X11/Xlibint.h>
#else
#include <X11/Xlibos.h>
#endif
#include <X11/Intrinsic.h>
#include "Malloc.h"
#include "string.h"

#ifdef X11R5

typedef struct {
    char*	start;
    char*	current;
    int		bytes_remaining;
} Heap;

#endif

/*
 *  Global variable definitions
 */

struct mnode *malloc_root;      /* Pointer to the root		          */
Bool malloc_inited = False;     /* Whether malloc's been initialized      */
unsigned int total_bytes_allocated = 0; /* Total #of bytes ever allocated */
unsigned int total_alloc_requests = 0;/* Total # of blocks ever allocated */
int bytes_in_use = 0;           /* Outstanding # of bytes allocated       */
int alloc_requests_in_use = 0;  /* Outstanding # of blocks allocated      */
FILE *malloc_outfp;             /* File pointer to output file            */
int malloc_outfd;               /* File descriptor of output file         */
char m_alloc_char = MALLOC_ALLOC_FILL;/* Fill character for allocating    */
char m_free_char = MALLOC_FREE_FILL;  /* Fill character for freeing       */
char m_clear_char = MALLOC_CLEAR_FILL; /* Blanks */
Bool m_end_dump = True;         /* Whether to call dump when exiting      */
Bool m_log_info = False;        /* Whether to log information             */
Bool m_check_string = False;    /* Whether to check string.c, memory.c    */
Bool malloc_checking = False;

/*
 * Function:	XtMalloc()
 *
 * Purpose:	memory allocator
 *
 * Arguments:	size	- size of data area needed
 *
 * Returns:	pointer to allocated area, or NULL if unable
 *		to allocate addtional data.
 *
 * Narrative:   This wrapper assumes that malloc_init has been
 *		called.
 */

#ifdef _NO_PROTO
char *XtMalloc(size)
unsigned int size;
#else  /* _NO_PROTO */
char *XtMalloc(unsigned int size)
#endif /* _NO_PROTO */

{
  char *ptr;
  unsigned int newsize, delta;

  /*
   * Always make sure there is at least one extra byte in the malloc
   * area so that we can verify that the user does not overrun the
   * data area.
   */

  newsize = size + (4 - (size % 4)); 
  delta = newsize - size;

  /*
   * Do the allocate
   */

  if ((ptr = Xmalloc(newsize)) == NULL)
    {

      malloc_msg(MALLOC_INTERNAL_ERROR,
		 1,                   /* one parameter to substitute */
		 "Xmalloc (XtMalloc) failed when allocating %u bytes\n", 
		 newsize, NULL,0,NULL,NULL); /* parameters  to sprintf */
    }

  /*
   * Record this information - only if we are tracking (malloc_inited)
   */

  if (malloc_inited)
    record_alloc(ptr, 		 /* ptr to the memory allocated              */
		 newsize, 	 /* size of memory allocated                 */
		 MALLOC_MALLOC,  /* literal indicating calling fn = XtMalloc */
		 delta);         /* difference in requested size/actual size */

  /*
   *  Return pointer
   */

  return (ptr);
}

/*
 * Function:	XtFree()
 *
 * Purpose:	memory free routine
 *
 * Arguments:	ptr - which memory pointer to free
 *
 * Returns:	Void
 *
 */

#ifdef _NO_PROTO
void XtFree(ptr)
char *ptr;
#else  /* _NO_PROTO */
void XtFree(char *ptr)
#endif /* _NO_PROTO */

{
  if (ptr != NULL) 
    {
      if (malloc_inited)
	record_freed (ptr, MALLOC_FREE); 
      Xfree(ptr);
    }
}

/*
 * Function:	XtCalloc()
 *
 * Purpose:	To allocate memory for an array of numelem elements
 * 	        of size elsize.
 *
 * Arguments:	numelem	- number of elements in the array
 *		elsize	- size of each element in the array
 *
 * Returns:	pointer to allocated area, or NULL if unable to 
 *		to allocate.
 *
 */

#ifdef _NO_PROTO
char *XtCalloc(numelem, elsize)
unsigned int numelem, elsize;
#else  /* _NO_PROTO */
char *XtCalloc(unsigned int numelem, unsigned int elsize)
#endif

{
  char *ptr;
  unsigned int size, delta;
  
  /*
   *  Allocate one extra element array, to use as a check area
   */

  size = (numelem + 1) * elsize;
  delta = elsize; 

  if ((ptr = Xcalloc(numelem+1, elsize)) == NULL)
    {
      malloc_msg(MALLOC_INTERNAL_ERROR,
		 1,
		 "Xcalloc failed when allocating %u bytes\n",
		 size,NULL,0,NULL,NULL);
    }

  if (malloc_inited)
    record_alloc(ptr, 	   	/* Pointer to the memory allocated	*/
		 size, 		/* Size of memory allocated		*/
		 MALLOC_CALLOC, /* Tell rtn where being called from 	*/
		 delta);        /* Difference in requested/actual size	*/
  return (ptr);
}

/*
 * Function:	XtRealloc()
 *
 * Purpose:	to reallocate a memory location 
 *
 * Arguments:	ptr 	- address of memory to reallocate
 *		size	- size of data area needed
 *
 * Returns:	pointer to allocated area, or NULL if unable
 *		to allocate addtional data.
 *
 */

#ifdef _NO_PROTO
char *XtRealloc(ptr, size)
char *ptr;
unsigned int size;
#else
char *XtRealloc(char *ptr, unsigned int size)
#endif
{
  unsigned int newsize, delta;

  if (ptr == NULL) return (XtMalloc(size));

  /*
   *  As always, increment size by at least one byte for checking
   */

  newsize = size + (4 - (size%4)); 
  delta = newsize - size;

  if (malloc_inited)
    {
      if (m_log_info)
	malloc_msg(MALLOC_STAT,
		   2, "+++Realloc request of %u bytes at 0x%08x address\n",
                   size, ptr, 0, NULL, NULL);
      record_freed(ptr, MALLOC_REALLOC);  
    } 

  if ((ptr = Xrealloc(ptr, newsize)) == NULL)
    {
      malloc_msg(MALLOC_INTERNAL_ERROR,2,
	 "Xrealloc (XtRealloc) failed when reallocating %u bytes at 0x%08x\n", 
	 newsize, ptr,0,NULL,NULL);
    }

  if (malloc_inited)
    record_alloc(ptr, 		 /* Pointer to memory allocated         */
		 newsize, 	 /* Size of memory allocated            */
		 MALLOC_REALLOC, /* Tell function being called from     */
		 delta);         /* Difference in requested/actual size */
  
  return(ptr);
}

/*
 *  These routines are supplied just to resolve linker problems.  The
 *    lib/Xt/Alloc.c module contains it and thus, it needs to be
 *    included here so that that the linker doesn't try to load
 *    the Alloc.o module.
 */

#ifdef _NO_PROTO
void _XtAllocError(type)
String type;
#else
void _XtAllocError(String type)
#endif

{
  Cardinal num_params = 1;
  if (type == NULL) type = "local memory allocation";
  XtErrorMsg("allocError", type, "XtCXtToolkitError",
	     "Cannot perform %s", &type, &num_params);
}

#ifdef X11R5

#ifdef _XNEEDBCOPYFUNC
void _XtBCopy(b1, b2, length)
    register char *b1, *b2;
    register length;
{
    if (b1 < b2) {
	b2 += length;
	b1 += length;
	while (length--)
	    *--b2 = *--b1;
    } else {
	while (length--)
	    *b2++ = *b1++;
    }
}
#endif

void _XtHeapInit(heap)
    Heap*	heap;
{
    heap->start = NULL;
    heap->bytes_remaining = 0;
}

#ifndef HEAP_SEGMENT_SIZE
#define HEAP_SEGMENT_SIZE 1492
#endif

char* _XtHeapAlloc(heap, bytes)
    Heap*	heap;
    Cardinal	bytes;
{
    register char* heap_loc;
    if (heap == NULL) return XtMalloc(bytes);
    if (heap->bytes_remaining < (int)bytes) {
	if ((bytes + sizeof(char*)) >= (HEAP_SEGMENT_SIZE>>1)) {
	    /* preserve current segment; insert this one in front */
#ifdef _TRACE_HEAP
	    printf( "allocating large segment (%d bytes) on heap %#x\n",
		    bytes, heap );
#endif
	    heap_loc = XtMalloc(bytes + sizeof(char*));
	    if (heap->start) {
		*(char**)heap_loc = *(char**)heap->start;
		*(char**)heap->start = heap_loc;
	    }
	    else {
		*(char**)heap_loc = NULL;
		heap->start = heap_loc;
	    }
	    return heap_loc + sizeof(char*);
	}
	/* else discard remainder of this segment */
#ifdef _TRACE_HEAP
	printf( "allocating new segment on heap %#x\n", heap );
#endif
	heap_loc = XtMalloc((unsigned)HEAP_SEGMENT_SIZE);
	*(char**)heap_loc = heap->start;
	heap->start = heap_loc;
	heap->current = heap_loc + sizeof(char*);
	heap->bytes_remaining = HEAP_SEGMENT_SIZE - sizeof(char*);
    }
#ifdef WORD64
    /* round to nearest 8-byte boundary */
    bytes = (bytes + 7) & (~7);
#else
    /* round to nearest 4-byte boundary */
    bytes = (bytes + 3) & (~3);
#endif /* WORD64 */
    heap_loc = heap->current;
    heap->current += bytes;
    heap->bytes_remaining -= bytes; /* can be negative, if rounded */
    return heap_loc;
}

void _XtHeapFree(heap)
    Heap*	heap;
{
    char* segment = heap->start;
    while (segment != NULL) {
	char* next_segment = *(char**)segment;
	XtFree(segment);
	segment = next_segment;
    }
    heap->start = NULL;
    heap->bytes_remaining = 0;
}

#endif /* X11R5 */
