/* 
 * (c) Copyright 1989, 1990, 1991, 1992 OPEN SOFTWARE FOUNDATION, INC. 
 * ALL RIGHTS RESERVED 
*/ 
/* 
 * Motif Release 1.2
*/ 
#ifdef REV_INFO
#ifndef lint
static char rcsid[] = "$RCSfile: MallocTree.c,v $ $Revision: 1.3 $ $Date: 92/03/13 18:31:15 $"
#endif
#endif
/*
 * FILE:  malloc_tree.c
 *
 *  Contains support routine for the malloc tree structure
 *  The malloc tree implementation is a binary tree, where
 *  the contents of the tree contain the currently malloc'ed data.
 *  Created 910220  SMT
 */

/*
 *  Includes
 */

#include <stdio.h>
#include <fcntl.h>
#include "Malloc.h"

/*
 * Name:     	malloc_add_node
 *
 * Purpose:  	Adds a node to the malloc tree structure
 *
 * Arguments: 	data_ptr - address of malloced data
 *	      	data_size - size in bytes of malloced data
 *	      	m_ptr - Pointer to the node to add from
 *	      	parent - address of the parent
 *
 * Implicit input: malloc_root, the root of the tree.
 *
 * Return Value:  Ptr to the node added
 */

#ifdef _NO_PROTO

struct mnode *malloc_add_node(data_ptr, data_size, delta_size, m_ptr, parent)
char *data_ptr;
unsigned int data_size;
unsigned int delta_size;
struct mnode *m_ptr;
struct mnode *parent;

#else  /* _NO_PROTO */

struct mnode *malloc_add_node(char *data_ptr, unsigned int data_size,
			      unsigned int delta_size,
			      struct mnode *m_ptr, struct mnode *parent)
#endif /* _NO_PROTO */
{
  /*
   *  If passed a null pointer, we are at a leaf and can add the node
   */

  if (m_ptr == NULL) 
    {
      /*
       *  Allocate room for the node and initialize it
       */

      m_ptr = (struct mnode *) malloc(sizeof(struct mnode));
      m_ptr->m_address = data_ptr;
      m_ptr->m_size = data_size;
      m_ptr->d_size = delta_size;   
      m_ptr->left_child = NULL;
      m_ptr->right_child = NULL;
      m_ptr->parent = parent;

      /*
       *  If this is the root, initialize malloc_root
       */

      if (malloc_root == NULL)
	{
	  malloc_root = m_ptr;
	  m_ptr->parent = NULL;
	}
    } /* (m_ptr == NULL) */

  else /* (m_ptr != NULL) */

    /*
     *  If the pointer passed is not NULL, keep looking for a leaf
     *   of the tree, via basic binary search algorithm.  Pass m_ptr
     *   to keep track of the parent.
     */

    if (data_ptr < m_ptr->m_address)
      m_ptr->left_child = malloc_add_node(data_ptr, data_size, delta_size,
					  m_ptr->left_child, 
					  m_ptr);
    else 
      if (data_ptr > m_ptr->m_address)
	m_ptr->right_child = malloc_add_node(data_ptr, data_size, delta_size,
					     m_ptr->right_child, 
					     m_ptr);
      else  /* data_ptr == m_ptr->m_address */
	/*
	 *  If the pointer is equal to one already in the tree,
	 *  the pointer has been allocated already.  Execute a 
	 *  warning; this is bad news.
	 */
	malloc_msg(MALLOC_REALLOC_ERROR, 
		   2, "Error! Reallocating %u bytes at 0x%08x location\n",
		   data_size, data_ptr,0,NULL,NULL);
  return m_ptr;
}

/*
 * Name: 	malloc_find_node
 *
 * Purpose:     Recursively search the malloc tree structure for 
 *		the node supporting the specified address.
 *
 * Arguments:   data - pointer to the malloc'ed data
 *		m_ptr - pointer to a node to begin looking at
 *
 * Returns:     pointer to the malloc tree node containing the data or
 *		NULL if not found
 */

#ifdef _NO_PROTO
struct mnode *malloc_find_node(data, m_ptr)
char *data;
struct mnode *m_ptr;
#else  /* _NO_PROTO */
struct mnode *malloc_find_node(char *data, struct mnode *m_ptr)
#endif /* _NO_PROTO */

{
  if (m_ptr == NULL) 
    return (NULL);	/* not found */

  if (data < m_ptr->m_address)
    return (malloc_find_node(data, m_ptr->left_child));
  else if (data > m_ptr->m_address)
    return (malloc_find_node(data, m_ptr->right_child));
  else  /* data == m_ptr->m_address  */
    return (m_ptr);
}

/*
 * Function: 	malloc_delete_node
 *
 * Purpose:	Delete a node from the malloc tree structure
 * 
 * Arguments:   m_node - a pointer to the node to delete
 *
 * Returns:     Void, no value
 */

#ifdef _NO_PROTO
void malloc_delete_node(m_node)
struct mnode *m_node;
#else  /* _NO_PROTO */
void malloc_delete_node(struct  mnode *m_node)
#endif /* _NO_PROTO */
{
  /*
   *  Local variables
   */

  struct mnode *temp_node;   		/* temporary node pointer       */
  struct mnode *v_node;			/* inorder successor pointer    */
  struct mnode *s_node;                 /* parent of inorder successor  */
  struct mnode *parent_node;		/* parent of node being deleted */
  Bool is_a_left_child = False;         /* keep track of whether left/right */


  parent_node = m_node->parent;		/* Save the parent pointer */

  /* 
   * Set is_a_left_child Boolean 
   */

  if (parent_node != NULL)
    is_a_left_child = (parent_node->left_child == m_node);

  /* 
   *  If node has no children, just delete it.
   *  If node has two children, get the inorder successor.
   *  If node has one child, move it up
   */

  if ((m_node->left_child == NULL) && (m_node->right_child == NULL))
    {
      /*
       *  No children; is this a left or right child of its parent?  
       *   Nullify the correct link.  If it's the root of the tree,
       *   nullify that pointer too.
       */

      if (parent_node == NULL) 		/* top of tree */
	malloc_root = NULL;
      else
	if (is_a_left_child)
	  parent_node->left_child = NULL;
	else
	  parent_node->right_child = NULL;

      /*
       *  Free the node
       */
      
      free(m_node);
      return;
    }

  /*
   *  If the node has two children, we need to find the inorder
   *	successor.  Set v_node to the inorder successor and t_node
   *    to the parent of v_node.
   */

  if ((m_node->left_child != NULL) && (m_node->right_child != NULL))
    {
      temp_node = m_node;
      v_node = m_node->right_child;
      s_node = v_node->left_child;
      
      /*
       *  Get the inorder successor and assign it to s_node;
       *   make v_node the parent of s_node.
       */

      while (s_node != NULL)
	{
	  temp_node = v_node;
	  v_node = s_node;
	  s_node = v_node->left_child;
	}

      if (temp_node != m_node)
	{
	  temp_node->left_child = v_node->right_child;
	  if (v_node->right_child != NULL) 	
	    v_node->right_child->parent = temp_node;

	  v_node->right_child = m_node->right_child;
	  if (m_node->right_child != NULL)
	    m_node->right_child->parent = v_node;
	}

      v_node->left_child = m_node->left_child;
      if (m_node->left_child != NULL)
	m_node->left_child->parent = v_node;
      
      /*
       *  Take care of malloc_root; if necessary
       */

      if (m_node->parent == NULL)     
	{
	  malloc_root = v_node;
	  malloc_root->parent = NULL;
	}
      else
	/*
	 *  Otherwise, fix the parent's child pointer
	 */
	{
	  if (parent_node->left_child == m_node)
	    parent_node->left_child = v_node;
	  else
	    parent_node->right_child = v_node;
	  if (v_node != NULL) v_node->parent = parent_node;
	}
    }  /* End if two children */

  else
    /*
     *  If not zero child and not two children, must have one child
     *    Move that child up to where the to-be-deleted node is.
     */

    if (m_node->left_child != NULL)
      {
	if (m_node->parent == NULL)   /* m_node is root */
	  {
	    malloc_root = m_node->left_child;
	    malloc_root->parent = NULL;
	  }
	else   /* has a parent */
	  {
	    if (is_a_left_child)
	      parent_node->left_child = m_node->left_child;
	    else
	      parent_node->right_child = m_node->left_child;
	    if (m_node->left_child != NULL) 
	      m_node->left_child->parent = parent_node;
	  }
      }
    else
      if (m_node->parent == NULL)
	{
	  malloc_root = m_node->right_child;
	  malloc_root->parent = NULL;
	}
      else
	{
	  if (is_a_left_child)
	    parent_node->left_child = m_node->right_child;
	  else
	    parent_node->right_child = m_node->right_child;
	  if (m_node->right_child != NULL) 
	    m_node->right_child->parent = parent_node;
	}

    /* Now free the node */

  free(m_node);
  return;
}

/*
 * Function: 	malloc_dump_tree
 *
 * Purpose:	Recursively traverse the tree and print out 
 *		remaining data at each node in the malloc tree
 * 
 * Arguments:   Pointer to a node in the tree
 *
 * Returns:     void
 */
#ifdef _NO_PROTO
void malloc_dump_tree(m_node)
struct mnode *m_node;
#else  /* _NO_PROTO */
void malloc_dump_tree(struct mnode *m_node)
#endif /* _NO_PROTO */

{

  if (m_node == NULL)
    {
      malloc_msg(MALLOC_EMPTY_TREE,0, 
		 "Pointer to malloc data is null, returning...\n",0,0,0,0,0);
      return;
    }

  malloc_msg(MALLOC_STAT,2, "Leftover %u bytes at 0x%08x \n", 
	     m_node->m_size - m_node->d_size, m_node->m_address, 0, 
	     NULL, NULL);

  /*
   * recurse left and right children
   */

  if (m_node->left_child != NULL) malloc_dump_tree(m_node->left_child);
  if (m_node->right_child != NULL) malloc_dump_tree(m_node->right_child);
  return;
}

/*
 * Name:	malloc_least_value
 *
 * Purpose:     This function recursively looks down the left most
 *		branch on the tree to find the left most leaf, which
 *		contains the smallest address stored in the malloc tree.
 *
 * Arguments:   m_ptr - a pointer to a node in the malloc tree
 *
 * Return Val:  The address of the smallest address stored in the tree.
 */

#ifdef _NO_PROTO
char *malloc_least_value(m_ptr) 
struct mnode *m_ptr;
#else  /* _NO_PROTO */
char *malloc_least_value(struct mnode *m_ptr)
#endif /* _NO_PROTO */
{
  if (m_ptr->left_child == NULL)
    return (m_ptr->m_address);
  else
    malloc_least_value(m_ptr->left_child);
}

/*
 * Name:	malloc_great_value
 *
 * Purpose:     This function recursively looks down the right most
 *		branch on the tree to find the right most leaf, which
 *		contains the largest address stored in the malloc tree.
 *
 * Arguments:   m_ptr - a pointer to a node in the malloc tree
 *
 * Return Val:  The address of the largest address stored in the tree.
 */

#ifdef _NO_PROTO
char *malloc_great_value(m_ptr) 
struct mnode *m_ptr;
#else  /* _NO_PROTO */
char *malloc_great_value(struct mnode *m_ptr)
#endif /* _NO_PROTO */
{
  if (m_ptr->right_child == NULL)
    return (m_ptr->m_address);
  else
    malloc_great_value(m_ptr->right_child);
}
      
/*
 * Function: 	malloc_find_addr
 *
 * Purpose:     Recursively search the malloc tree structure for the 
 *	        node supporting the specified address.
 *
 * Arguments:   data - pointer to the malloc'ed data
 *		m_ptr - pointer to a node to begin looking at
 *
 * Returns:     pointer to the malloc tree node containing the data or
 *		NULL if not found
 */

#ifdef _NO_PROTO
struct mnode *malloc_find_addr(data, m_ptr)
char *data;
struct mnode *m_ptr;
#else  /* _NO_PROTO */
struct mnode *malloc_find_addr(char *data, struct mnode *m_ptr)
#endif /* _NO_PROTO */
{
  if (m_ptr == NULL) 
    return (NULL);	/* not found */

  /*
   *  Recursively look down the tree, branching left for
   *   less-than values and right for greater-than values
   *   (basic binary search)
   */

  if (data < m_ptr->m_address)
    return(malloc_find_addr(data, m_ptr->left_child));
  else if (data > m_ptr->m_address + m_ptr->m_size)
    return( malloc_find_addr(data, m_ptr->right_child));
  else  /* data is within this malloc node */
    return (m_ptr);
}

/*
 * Function: 	MallocEnable
 *
 * Purpose:	Enable Malloc Checking
 * 
 * Arguments:   none
 *
 * Returns:     void
 */

#ifdef _NO_PROTO
void MallocEnable()
#else /* _NO_PROTO */
void MallocEnable()
#endif /* _NO_PROTO */
{
  /*
   *  Currently just calls MallocInit();
   */

  MallocInit();
  return;
}

/*
 * Function: 	MallocDisable
 *
 * Purpose:	Free all nodes in malloc tree
 *		Reset root pointer, reset malloc_init flag.
 * 
 * Arguments:   None
 *
 * Returns:     void
 */
#ifdef _NO_PROTO
void MallocDisable()
#else  /* _NO_PROTO */
void MallocDisable()
#endif /* _NO_PROTO */
{
  /*
   *  Delete the tree
   */

  if (malloc_root != NULL)
    malloc_delete_tree(malloc_root);

  /*
   *  Reset vars
   */

  malloc_root = NULL;
  malloc_inited = False;
  bytes_in_use = 0;
  alloc_requests_in_use = 0;
  total_bytes_allocated = 0;
  total_alloc_requests = 0;
  return;
}

/*
 * Function: 	malloc_delete_tree
 *
 * Purpose:	Free all nodes in malloc tree
 * 
 * Arguments:   Pointer to a node in the tree
 *
 * Returns:     void
 */

#ifdef _NO_PROTO
void malloc_delete_tree(m_node)
struct mnode *m_node;
#else  /* _NO_PROTO */
void malloc_delete_tree(struct mnode *m_node)
#endif /* _NO_PROTO */
{
  /*
   * Free all the leaves
   */

  if (m_node->left_child != NULL) malloc_delete_tree(m_node->left_child);
  if (m_node->right_child != NULL) malloc_delete_tree(m_node->right_child);
  malloc_delete_node(m_node);
  return;
}
