/* 
 * (c) Copyright 1989, 1990, 1991, 1992 OPEN SOFTWARE FOUNDATION, INC. 
 * ALL RIGHTS RESERVED 
*/ 
/* 
 * Motif Release 1.2
*/ 
#ifdef REV_INFO
#ifndef lint
static char rcsid[] = "$RCSfile: MallocSup.c,v $ $Revision: 1.4 $ $Date: 92/03/13 18:31:11 $"
#endif
#endif
/*
 * FILE: MallocSup.c
 *
 * This module contains the main support routines for the malloc memory
 *  wrapper.  As memory is allocated and freed, the routines keep track
 *  of that information.  Additional support exists for looking for
 *  writing beyond the end of allocated memory, etc.
 *
 * Based on: 
 *       Conor Cahill's Malloc Library - (see copyright statement)
 *       David Brook's XtMalloc wrapper
 * Condensed 910214, SMT
 *
 * (c) Copyright 1990 Conor P. Cahill (uunet!virtech!cpcahil).  
 * You may copy, distribute, and use this software as long as this
 * copyright statement is not removed.
 */

/*
 *  Includes
 */

#include <stdio.h>
#include <fcntl.h>
#include "Malloc.h"

/*
 * FUNCTION:  record_alloc
 * 
 * PURPOSE:   To record what's being malloced/calloced/realloced
 * 
 * ARGUMENTS: ptr - a pointer to the area being malloced
 *            nbytes - the number of bytes being malloced (including the
 *			check area.
 *            wherefrom - who is calling this function
 *  	      delta - difference in number of requested bytes & given bytes
 *
 * RETURNS:   void
 * 
 */

#ifdef _NO_PROTO
void record_alloc(ptr, nbytes, wherefrom, delta)
char *ptr;
unsigned int nbytes;
unsigned int wherefrom;
unsigned int delta;
#else  /* _NO_PROTO */
void record_alloc(char *ptr, unsigned int nbytes, unsigned int wherefrom,
		  unsigned int delta)
#endif /* _NO_PROTO */
{
  /* 
   * Write the malloc fill character(s) in this memory
   */

  if (wherefrom == MALLOC_MALLOC)
    malloc_memset(ptr, m_alloc_char, nbytes);
  else 
    if (wherefrom == MALLOC_CALLOC)
      {  
	/* 
	 *  Fill it all with nulls (required by calloc) except the check 
	 *  area, which gets initialized with the alloc character.
	 */

	malloc_memset(ptr, m_clear_char, (nbytes - delta));
	malloc_memset(ptr+(nbytes-delta), m_alloc_char, delta);
      }
  else 
    if (wherefrom == MALLOC_REALLOC)  /* just initialize the check area */
      { 
	malloc_memset(ptr+(nbytes-delta), m_alloc_char, delta);
      }

  /* 
   *  Add data to the malloc tree structure
   */

  malloc_add_node(ptr, nbytes, delta, malloc_root, malloc_root);

  /*
   *  Update "total" counters
   */

  bytes_in_use += (nbytes - delta);  /* dont count the check area */
  alloc_requests_in_use++;
  total_bytes_allocated += (nbytes - delta);
  total_alloc_requests++;

  /*
   * If the user has elected to log malloc information, log this.
   */

  if (m_log_info)
    {
      if (wherefrom == MALLOC_MALLOC)
	malloc_msg(MALLOC_STAT, 2, 
		   "+++Malloc'ed %u bytes at 0x%08x address\n", 
		   nbytes - delta, ptr, 0, NULL, NULL);
      else
	if (wherefrom == MALLOC_CALLOC)
	  malloc_msg(MALLOC_STAT,
		     2, "+++Calloc'ed %u bytes at 0x%08x address\n", 
		     nbytes - delta, ptr, 0, NULL, NULL);
      else
	if (wherefrom == MALLOC_REALLOC)
	  malloc_msg(MALLOC_STAT,
		     2, "+++Realloc'ed %u bytes at 0x%08x address\n", 
		     nbytes - delta, ptr, 0, NULL, NULL);
    }
}

/*
 * FUNCTION:  record_freed
 * 
 * PURPOSE:   To record what's being free'd
 * 
 * ARGUMENTS: ptr - a pointer to the area being freed
 *            wherefrom - what is calling this function
 *
 * RETURNS:   void
 * 
 */
#ifdef _NO_PROTO
void record_freed(ptr, wherefrom)
char *ptr;
unsigned int wherefrom;
#else  /* _NO_PROTO */
void record_freed(char *ptr, unsigned int wherefrom)
#endif /* _NO_PROTO */
{
  unsigned int i = 0;
  struct mnode *m_node;
  char *temp_ptr;

  /*
   * Find the address of the node in the malloc tree which 
   *  corresponds to what's being freed.
   */

  if ((m_node = malloc_find_node(ptr,malloc_root)) == NULL)
    {
      /*
       *  If we can't find the memory in the tree, execute a warning
       *   and return.
       */

      malloc_msg(MALLOC_INVALID_FREE, 2,
    "Warning: Tried to free some (%u) memory at 0x%08x, not in malloc tree\n",
	       i,ptr,0,NULL,NULL);
      return;
    }	

  /*
   *  Check to see if user overwrote the data (we always allocate at least
   *  one more byte for a check byte).  If that byte does not have the
   *  allocate fill character in it, the user went too far.
   */

  temp_ptr = m_node->m_address;
  if (temp_ptr[m_node->m_size - m_node->d_size] != m_alloc_char)
    {
      malloc_msg(MALLOC_WRITE_ERROR, 2,
	   "Warning: User overwrote allocated data; detected on free,\n          size: %u,address = 0x%08x\n",
		 m_node->m_size - m_node->d_size, ptr,0,NULL,NULL); 
    }
  
  /* 
   * Fill the memory with the malloc freed character  - ONLY if this
   *  is being called from XtFree.  NOT XtRealloc. XtRealloc guarantees
   *  the originally malloc'ed data retains the same value.
   */

  if (wherefrom != MALLOC_REALLOC)
    malloc_memset(ptr, m_free_char, m_node->m_size);

  /*
   * fix counts
   */

  bytes_in_use -= (m_node->m_size - m_node->d_size);
  alloc_requests_in_use--;

  /*
   *  If this free is due to a realloc, fix only the 
   *  total_bytes_allocated, not the total_alloc_requests,
   *  since the Realloc counts as a request.
   */

  if (wherefrom == MALLOC_REALLOC)
    {
      total_bytes_allocated -= (m_node->m_size - m_node->d_size);
    }
 

  if (m_log_info)
    {
      if (wherefrom == MALLOC_REALLOC)
	malloc_msg(MALLOC_STAT,2, 
		   "+++Freed (due to REALLOC) %u bytes at 0x%x address\n",
		   (m_node->m_size - m_node->d_size), ptr,0,NULL,NULL);
      else
	malloc_msg(MALLOC_STAT,
		   2, "+++Freed     %u bytes at 0x%x address\n",
		   (m_node->m_size - m_node->d_size), ptr,0,NULL,NULL);
    }

  /*
   * Free the mnode
   */

  malloc_delete_node(m_node);

}

/*
 * Function:    malloc_memset
 *
 * Purpose:     Fill memory with specified fill character.
 *
 * Arguments:   ptr - pointer to memory to fill
 *		fill_char - fill character
 *              len - length to write
 *
 * Returns:     No value
 */

#ifdef _NO_PROTO
void malloc_memset(ptr, fill_char, len)
char *ptr;
char fill_char;
int len;
#else  /* _NO_PROTO */
void malloc_memset(char *ptr, char fill_char, int len)
#endif /* _NO_PROTO */
{
  while (len-- > 0)
    { *ptr++ = fill_char; }

} /* malloc_memset... */



