/* This program was developed at Hitachi Computer Products (America), Inc., in
 * its Open Systems Division, which is responsible for porting and adapting OSF
 * offerings.  Hitachi Computer Products has given the program to OSF so it can
 * use and distribute it as part of its documentation build tools.  In addition,
 * OSF licensees can adapt the program to the needs of their sites.  The program
 * is offered as is, with no warranty attached, and so forth and so on.
 */

#include "filter.h"

#define STACK_ERROR -1

/* Maintains the start states in lex as a stack.  This function could be
 * generalized to maintain general objects on a stack.  The objects to
 * which this stack points are the states maintained by the lex-generated
 * lexical analyzer.  It's all internals, unfortunately.  If this program
 * could be implemented in yacc efficiently, yacc could do the stack
 * maintenance and none of this rather shaky cleverness with internals
 * would be necessary. */
void monitor_state_stack(command)
     enum stack_op command;
{
  /* yysf copied directly from lex.yy.c */
  struct yysvf {
    int yystoff;
    struct yysvf *yyother;
    int *yystops;};

  static struct state_stack {
    struct yysvf *current;
    struct state_stack *prev;
  };

  struct state_stack *state_tmp;
  static struct state_stack *state_current=NULL;

  /* yybgin is defined by lex in lex.yy.c */
  extern struct yysvf *yybgin;
  extern char *program_name;

  switch(command) {

  case PUSH:
    if ( ( state_tmp = (struct state_stack *) malloc(sizeof(struct state_stack)) )
	 == NULL )
      {
	perror(program_name);
	exit(1);
      }
    state_tmp->current = yybgin;
    state_tmp->prev = state_current;
    state_current = state_tmp;
    break;

  case POP:
    /* If this case were called when no states were on the stack; it would
       try to get a prev member from a null state_current pointer and produce an error.
       But the function does not check the pointer here, because the caller
       only invokes this case when a state is on the stack.  In other words,
       this function places the burden on the caller of ensuring that POP and CHECK
       are invoked at the right points. */
    yybgin = state_current->current;
    state_tmp = state_current->prev;
    free (state_current);
    state_current = state_tmp;
    break;

    /* The program never calls this function with an UNWIND command. */
  }
}


/* The following function maintains a stack of ifdef's and notes when it is OK to start printing
   lines from the input files again.  The arguments are:

   command	PUSH for #if or #ifdef or #ifndef line, POP for #endif, CHECK for #else,
		UNWIND at end of program for errors
   ifdef_set	significant only during PUSH; distinguishes #ifdef from #ifndef

   The else_lineno field really has a Boolean effect (0 before an #else is encountered,
   non-zero afterward).  But instead of a random non-zero value, the function stores the line
   number of the #else statement encountered, just as with the if_lineno field.
   The program can thus issue clearer error messages, giving the line number in case of error.

   Similarly, the internally-maintained top_level_ifndef counter is overloaded so as to perform
   two functions.  The first is to report the results of #if processing back to the main parser:
   when zero, the program is allowed to print lines (the IFDEFOP_PRINT return value); when non-zero,
   the lines should be suppressed (the IFDEFOP_IGNORE return value).  But top_level_ifndef also
   stores the level at which printing was suppressed.  Many other nested #if and #endif statements can
   come before printing resumes.  By comparing top_level_ifndef with depth, this function knows
   which #else or #endif statement resumes printing.
   */
enum ifdef_op monitor_ifdef_stack(command, ifdef_set )
     enum stack_op command;
     BOOL ifdef_set;		/* whether string found, or #if expression true */
{
  struct ifdef_stack {
    int if_lineno;
    int else_lineno;
    struct ifdef_stack *prev;
  };

  static int top_level_ifndef=0;
  extern unsigned int depth;
  extern unsigned int line_number;
  extern char *program_name, *curr_file_name;

  static struct ifdef_stack *ifdef_current=NULL;

  switch(command) {
    struct ifdef_stack *ifdef_tmp;
    unsigned int tmp_depth;

  case PUSH:
    if ( ( ifdef_tmp = (struct ifdef_stack *) malloc(sizeof(struct ifdef_stack)) )
	 == NULL )
      {
	perror(program_name);
	exit(1);
      }
    ifdef_tmp->if_lineno = line_number;
    ifdef_tmp->else_lineno = 0;
    ifdef_tmp->prev = ifdef_current;
    ifdef_current = ifdef_tmp;
    depth++;

    /* If we're not suppressing lines yet, a false print should be stored at the current level. */
    if (! top_level_ifndef && ! ifdef_set)
      {
	top_level_ifndef = depth;
      }

    /* Check to see whether we're suppressing lines, and if so, make sure to keep doing so. */
    if (top_level_ifndef)
      {
	return (IFDEFOP_IGNORE);
      }
    else
      {
	return (IFDEFOP_PRINT);
      }
    /* NOTREACHED */
    break;			/* useful in case program flow changes */

  case POP:
    if ( depth == 0 )
      {
	(void) fprintf(stderr,
	"%s: ERROR in %s file, line %d -- #endif with no #if, #ifdef, or #ifndef\n",
	program_name, curr_file_name, line_number);
	return (IFDEFOP_ERROR);
      }

    /* Reason for tmp_depth: it lets me decrement depth here cleanly, in one place,
       instead of having to do it within each if/else block. */
    tmp_depth = depth;
    depth--;
    ifdef_tmp = ifdef_current->prev;
    free (ifdef_current);
    ifdef_current = ifdef_tmp;

    if (! top_level_ifndef)
      {
	/* Lines are not being suppressed at the time.  So continue as usual. */
	return (IFDEFOP_PRINT);
      }
    else if (top_level_ifndef == tmp_depth)
      {
	/* The current #ifdef/#ifndef was causing lines to be suppressed.  So
	   indicate that it is safe to start printing lines now. */
	top_level_ifndef = 0;
	return (IFDEFOP_PRINT);
      }
    else
      {
	/* A higher-level #ifdef/#ifndef (outside this one) is causing lines to be
	   suppressed.  So continue suppressing them. */
	return (IFDEFOP_IGNORE);
      }
    /* NOTREACHED */
    break;			/* useful in case program flow changes */

  case CHECK:
    if ( depth == 0 )
      {
	(void) fprintf(stderr,
	"%s: ERROR in %s file, line %d -- #else with no #if, #ifdef, or #ifndef\n",
	program_name, curr_file_name, line_number);
	return (IFDEFOP_ERROR);
      }

    if ( ifdef_current->else_lineno )
      {
	(void) fprintf(stderr,
	"%s: ERROR in %s file, line %d -- an #else was already at line %d for the #if block beginning at line %d.\n",
	program_name, curr_file_name, line_number, ifdef_current->else_lineno, ifdef_current->if_lineno);
	return(IFDEFOP_ERROR);
      }

    ifdef_current->else_lineno = line_number;

    if (! top_level_ifndef)
      {
	/* Lines are not being suppressed at the time.  So start suppressing them. */
	top_level_ifndef = depth;
	return (IFDEFOP_IGNORE);
      }
    else if (top_level_ifndef == depth)
      {
	/* The current #ifdef/#ifndef was causing lines to be suppressed.  So
	   indicate that it is safe to start printing lines now. */
	top_level_ifndef = 0;
	return (IFDEFOP_PRINT);
      }
    else
      {
	/* A higher-level #ifdef/#ifndef (outside this one) is causing lines to be
	   suppressed.  So continue suppressing them. */
	return (IFDEFOP_IGNORE);
      }
    /* NOTREACHED */
    break;			/* useful in case program flow changes */

  case UNWIND:
    /* Called only at the end of the program, and only if some #ifdef or #else lines remain open,
     * that is, a closing #else line is missing from the input. */

    while ( ifdef_current != NULL ) /* Continue so long as #ifdef's remain */
      {
	(void) fprintf(stderr,
	"%s: ERROR in %s file -- #if block starting line %d has no closing #endif statement.\n",
	program_name, curr_file_name, ifdef_current->if_lineno);
	if ( ifdef_current->else_lineno )
	  {
	    (void) fprintf(stderr, "An #else statement appears at line %d.\n",
			   ifdef_current->else_lineno);
	  }
	depth--;
	ifdef_tmp = ifdef_current->prev;
	free (ifdef_current);
	ifdef_current = ifdef_tmp;
      }

    /* NOTREACHED */
    break;			/* useful in case program flow changes */

  }
    /* NOTREACHED */
}


/* The following function maintains a stack of files for .so requests.  The arguments are:
 *
 * command	PUSH for .so request, POP at end of file
 * filename	the file to be opened and read in
 *
 * By lex/yacc conventions, the file descriptor from which the parser takes
 * input is named yyin.  Thus, this function keeps track of which files to use
 * by putting the current yyin on the stack, and then assigning the file
 * descriptor of the newly-opened .so file to yyin.
 *
 * Several different documents claim that one can switch input files in lex
 * through a disciplined, standard method, but unfortunately those various
 * authors disagree on what the disciplined, standard method is -- and the
 * competing "standards" are very complicated in any case.  So program
 * switches files in its own Dionysian, idiosyncratic way.  It plays around
 * with the internal yyin descriptor as well as the more familiar yywrap
 * function, which is in the main filter_driver.c file.  The yywrap function
 * must handle all end-of-file conditions, so it always pops the stack
 * (calling monitor_file_stack with the POP command).  But .so requests are
 * handled in the main parsing part of the program, filter.l, so that part
 * pushes the stack.
 */
BOOL monitor_file_stack(command, filename)
     enum stack_op command;
     char *filename;
{
  extern unsigned int line_number;
  struct file_stack {
    FILE *filed;
    char name[MAX_FILENAME_LEN];
    unsigned int line_number;
    unsigned int depth;
    struct file_stack *prev;
  };

  static struct file_stack *file_current=NULL;
  extern FILE *yyin;
  extern char *program_name, *curr_file_name, file_name_buf[];
  extern unsigned int sizeof_buf;
  extern unsigned int depth;

  switch(command) {
    struct file_stack *file_tmp;
    BOOL ret_val;

  case PUSH:
    if ( ( file_tmp = (struct file_stack *) malloc(sizeof(struct file_stack)) )
	 == NULL )
      {
	perror(program_name);
	exit(1);
      }

    /* Push stack. */
    file_tmp->filed = yyin;
    (void) strncpy (file_tmp->name, file_name_buf, sizeof(file_tmp->name));
    file_tmp->line_number = line_number;
    file_tmp->depth = depth;
    file_tmp->prev = file_current;
    file_current = file_tmp;

    if ( (yyin = fopen(filename, "r") ) == NULL)
      {
	perror(program_name);
	(void) fprintf(stderr,
	"%s: ERROR in %s file, line %d -- tried to open .so file %s\n",
	program_name, curr_file_name, line_number, filename);
	ret_val = (BOOL) STACK_ERROR;
	line_number++;		/* In case we still want to keep track of lines */
      }
    else
      {
	ret_val = (BOOL) FALSE;		/* This means no error occurred. */
	(void) strncpy (file_name_buf, filename, sizeof_buf);
	line_number = 1;
      }
    return (ret_val);
    /* NOTREACHED */
    break;

  case POP:
    if ( ! file_current )	/* First, are we in any .so file? */
      {
	return ( (BOOL) FALSE);		/* If not, end right here. */
      }
    if ( (fclose(yyin) ) == EOF) /* Close old (.so) file */
      {
	perror(program_name);
	(void) fprintf(stderr,
	"%s: ERROR in %s file, line %d -- tried to close .so file %s\n",
	program_name, curr_file_name, line_number, filename);
      }

    /* At end of each input file, make one last check in case an #ifdef or
     * related directive is still open (that is, an #endif is missing) or whether
     * an #endif was put in the nested file without a corresponding #if.
     * See the maintenance file for implications and justification.
     *
     * Checking for an #else statement in the nested file is too much trouble
     * (the file_stack structure, already rather weighty, would have to replicate
     * a lot of the ifdef_stack structure, and perhaps even more information).
     */
    if (depth > file_current->depth)
      {
	(void) monitor_ifdef_stack(UNWIND, NULL);
	exit (1);
      }
    else if (depth < file_current->depth)
      {
	(void) fprintf(stderr,
	"%s: ERROR in %s file -- #endif statement without a corresponding #if statement\n",
	program_name, curr_file_name);
	exit (1);
      }

    /* Pop stack. */
    yyin = file_current->filed;
    (void) strncpy (file_name_buf, file_current->name, sizeof_buf);
    line_number = file_current->line_number+1; /* we are now past the .so line */
    depth = file_current->depth;
    file_tmp = file_current->prev;
    free (file_current);
    file_current = file_tmp;

    return ( (BOOL) TRUE);
  }

  /* NOTREACHED */
  /* The program never calls this function with an UNWIND command. */
}


/* The following function tells filter.l whether the user has defined (that is, used
   a -D option to pass) a string that has been found in an #if directive.  This function
   searches through a list set up by the main program in filter_driver.c.
   */
BOOL find_ifdef_in_list(file_string)
     char *file_string;
{
  extern STRING_LIST ifdefine_list;
  STRING_LIST ifdefine_ptr=ifdefine_list;
  char *start_of_whitespace=file_string;

  /* First have to strip off trailing spaces and tabs.  Search for either the
     null byte or the first space character, then store a null byte there. */
  while (*start_of_whitespace && (! isspace(*start_of_whitespace) ) )
    ++start_of_whitespace;

  *start_of_whitespace= '\0';

  while ( ifdefine_ptr->string)
    {
      if (! strcmp ( ifdefine_ptr->string, file_string) )
	return ( (BOOL) TRUE);
      else
	ifdefine_ptr = ifdefine_ptr->next;
    }
  return ( (BOOL) FALSE);
}


/* The following functions remember all the strings, operators, and parentheses
   as the lex-generated program parses an #if directive.
   */

/* The return value indicates the final result of the parsed #if expression.
   Since the proper order of tokens, for the most part, is enforced in filter.l,
   the only parsing problem that should reach this level is an extra opening
   or closing parenthesis.   */
BOOL monitor_token_stack(op)
     enum ifdef_token op;
{
  extern IF_TOKEN_LIST token_current;
  extern void new_token(), pop_token(), resolve_token();
  extern char *ifdef_directive;
  extern char *program_name, *curr_file_name;
  extern unsigned int line_number;

  switch(op) {
    BOOL tmp_set;		/* To remember final result before freeing whole stack */

    /* If a binary operator is encountered, just store it in current op member. */
  case TOKEN_AND:				/* binary & or && operator */
    new_token(STACK_AND);
    return( (BOOL) TRUE); /* ignored by caller */

  case TOKEN_OR:				/* binary | or || operator */
    new_token(STACK_OR);
    return( (BOOL) TRUE); /* ignored by caller */

  case TOKEN_XOR:				/* binary ^ operator */
    new_token(STACK_XOR);
    return( (BOOL) TRUE); /* ignored by caller */

    /* If a negation operator is encountered, just toggle the current negate member.
       Don't just set it to TRUE -- that would mean throwing away the operator if it came
       right after an #ifndef.  Incidentally, the toggling allows this program to handle
       gracefully the strange case of someone passing multiple ! in a row -- they flip
       back and forth. */
  case TOKEN_NEGATE:				/* unary ! operator */
    token_current->negate = token_current->negate ? FALSE : TRUE;
    return( (BOOL) TRUE); /* ignored by caller */

    /* Two new tokens allocated here: one just to mark the existence of a parenthesis (so
       the TOKEN_RPAREN op can verify proper nesting later) and one to start a new level
       of parsing, just like the original level stared by the #if directive.  This double
       allocation is a little wasteful, but the simplest way to verify nesting. */
  case TOKEN_LPAREN:				/* opening parenthesis */
    new_token(STACK_PAREN);
    return( (BOOL) TRUE); /* ignored by caller */

  case TOKEN_RPAREN:				/* closing parenthesis */
    if (token_current->op != STACK_PAREN)
      {
	(void) fprintf(stderr,
	"%s: ERROR in %s file, line %d, %s directive -- right parenthesis without a matching left parenthesis\n",
	program_name, curr_file_name, line_number, ifdef_directive);
	exit(1);
      }
    tmp_set = token_current->set; /* store set from the token that is to be popped */
    pop_token();
    token_current->set = tmp_set; /* store the set back into the previous (now-current) token */
    resolve_token(token_current->op);
    return( (BOOL) TRUE); /* ignored by caller */

  case TOKEN_IFDEF:				/* #if or #ifdef directive */
    new_token(STACK_IFDEF);
    return( (BOOL) TRUE); /* ignored by caller */

  case TOKEN_IFNDEF:				/* #ifndef directive */
    /* For the purposes of parsing, #ifndef is the same as #ifdef -- just use IFDEF for both. */
    new_token(STACK_IFDEF);
    token_current->negate = TRUE;
    return( (BOOL) TRUE); /* ignored by caller */

  case TOKEN_STRING_DEFINED:			/* a string that was passed by the user */
    token_current->set = TRUE;
    resolve_token(token_current->op);
    return( (BOOL) TRUE); /* ignored by caller */

  case TOKEN_STRING_UNDEFINED:		/* a string that was not passed by the user */
    token_current->set = FALSE;
    resolve_token(token_current->op);
    return( (BOOL) TRUE); /* ignored by caller */

  case TOKEN_NLINE:				/* newline */
    if (token_current->op != STACK_IFDEF)
      {
	(void) fprintf(stderr,
	"%s: ERROR in %s file, line %d, %s directive -- left parenthesis without a matching right parenthesis\n",
	program_name, curr_file_name, line_number, ifdef_directive);
	exit(1);
      }
    tmp_set = token_current->set; /* store set from the token that is to be popped */
    pop_token();
    return (tmp_set);		/* This is the point of the whole parsing exercise */
  }
  /* NOTREACHED */
}

void new_token(op)
     enum stacking_token op;
{
  extern IF_TOKEN_LIST token_current;		 /* defined at top of filter_driver.c */
  IF_TOKEN_LIST token_tmp;
  extern char *program_name;

  if ( ( token_tmp = (struct if_token_list *) malloc(sizeof(struct if_token_list)) )
       == NULL )
    {
      perror(program_name);
      exit(1);
    }

  /* Initialize all members -- except the "set" member, which will be filled in later. */
  token_tmp->op = op;
  token_tmp->negate = FALSE;
  token_tmp->prev = token_current;
  token_current = token_tmp;
}

/* This function does not have to check whether there is a previous token before popping;
   checks in the calling function ensure that this function can never be called under
   those circumstances. */
void pop_token()
{
  extern IF_TOKEN_LIST token_current;
  IF_TOKEN_LIST token_tmp;

  token_tmp = token_current->prev;
  free(token_current);
  token_current = token_tmp;
}

/* Store the results of this token, resolved with the negate member.
   Then if this token contains an operator (AND, OR, or XOR) perform another level
   of popping and processing, thus removing the token.
   This function does not have to check whether there is a previous token before popping;
   checks in higher-level functions ensure that this function can never be called under
   those circumstances. */
void resolve_token(op)
     enum stacking_token op;
{
  extern IF_TOKEN_LIST token_current;

  /* Don't laugh; it works.  Remember that we are not dealing with bits here, but with
     full-fledged integers having a 0 or 1 value.  So we must do double toggling: first on
     the set member, then on the negate member (of course, the order could be reversed too). */
  token_current->set = token_current->set ?
    (! token_current->negate) : token_current->negate ;

  if ( (op == STACK_AND) || (op == STACK_OR) || (op == STACK_XOR) )
    {
      BOOL tmp_set = token_current->set;	/* To remember result before freeing whole stack */
      pop_token();

      switch (op) {

      case STACK_AND:
	token_current->set = token_current->set && tmp_set;
	break;

      case STACK_OR:
	token_current->set = token_current->set || tmp_set;
	break;

      case STACK_XOR:
	token_current->set = token_current->set ^ tmp_set;
	break;
      }
    }
}
