/*
 * HT - Hash Total Program.
 * Input: PT file, raw binary data
 * Output: prints 20-digit hash total
 * NOTE: Must be compiled with compact, large, or huge model.
 *
 * We follow the 1620 algorithm exactly.
 * First sub is {0,                 TU of char 0-8,   T digit of char 9}
 * Next sub is  {U digit of char 9, TU of char 10-18, T digit of char 19}
 * Record Marks are initially placed in last byte of each 20, which
 * is the Tens position of chars 9, 19, etc.  Since RAPT never writes
 * a RM in the Tens position, we can use these RMs (I call them stop
 * markers) to figure out _approximately_ where the last RAPT ended.
 * We go through the buffer, subtracting groups of twenty digits from
 * the accumulator, until we subtract a group with a stop marker in the LSD.
 * Each time we subtract a group, we place a stop marker in the LSD,
 * so the buffer is set up for the next RAPT.
 * This algorithm puts some chars into the hash total more than once.
 * For example, if a 25-char record is followed by a 15-char record,
 * chars 15-18 are subtracted twice (char 19's T digit was overwritten
 * by a stop marker).
 */

	#include <stdio.h>
	#include <string.h>
	#include <process.h>

#define EOL	0x80
#define X	0x40
#define O	0x20
#define C	0x10
#define RM	0x0A

#define BUFSZ	60000U
#define INSZ	1024

unsigned char	in[INSZ];
char		buf[BUFSZ];
char		acc[20];
int		verbose = 0;

void init(void)
{	/* Stop markers go in Tens digit - RAPT never puts a RM there */
    unsigned int i;

    for (i=0; i<20; i++) acc[i] = 0;
    for (i=0; i<BUFSZ; i++) buf[i] = 0;
    for (i=19; i<BUFSZ; i+=20) buf[i] = RM;
}

unsigned char parity(unsigned char c)	/* Takes char, returns C bit */
{
    unsigned char d = 0;

    while (c)
    {
	d += c & 1;		/* Count number of 1s */
	c = c >> 1;
    }
    return d & 1 ? 0 : C;	/* If odd, return 0 else C */
}

const unsigned char cvt[] = {
C,		0,0,	/* Space */
X|O|8|2|1,	0,3,	/* . */
X|O|C|8|4,	0,4,	/* ) */
X|O|C,		2,0,	/* + (Special, fold XOC into X) */
X|C|8|2|1,	1,3,	/* $ */
X|8|4,		1,4,	/* * */
X,		2,0,	/* - */
O|C|1,		2,1,	/* / */
O|C|8|2|1,	2,3,	/* , */
O|8|4,		2,4,	/* ( */
8|2|1,		3,3,	/* = */
C|8|4,		3,4,	/* @ */
O,		7,0,	/* 0 */
O|8|2,		0,RM,	/* # */
EOL,		0,RM,	/* E */
};

/*
 * RAPT - read chars, cvt to 1620 core codes
 * As in 1620, Tens digit goes to *(p-1), Units digit goes to *p, p+=2.
 * Returns number of chars read.
 * Folds XOC char into X char.
 * Reads 1024 chars at a time, converts until it hits EOL, and
 * fseek()s back to just after EOL.
 * Panics out if buffer overflow is imminent.
 */
long int RAPT(char *p, FILE *f)
{
    int			i, j, amt;
    long int		rc;
    unsigned char	c;

reRead:		/* Restart after skipping a record containing XOC82 */
    rc = 0;				/* Init to 0 chars read */
    while ((amt = fread(in, 1, INSZ, f)) != 0)	/* Read up to INSZ chars */
    {
	rc += amt;			/* Log total chars read */
	if (rc > BUFSZ/2 - 10)		/* Overflow - long record */
	{
	    fprintf(stderr, "\nPANIC: Record too long near %ld\n", ftell(f));
	    exit(1);
	}

	for (i=0; i<amt; i++)		/* Scan for the dreaded XOC82 */
	{
	    c = in[i];
	    if (c == EOL) break;	/* No XOC82 in _this_ record */
	    if (c == (X|O|C|8|2))
	    {
		printf("\nFlagged RM (XOC82) at byte %ld, skipping record\n",
		    ftell(f) - amt + i);
		if (amt < rc)
		{
		    printf("\nPANIC: XOC82 in partially processed record!\n");
		    exit(1);
		}
		do
		{	/* Swallow rest of this record, then read next */
		    for (i=0; i<amt; i++)
			if ((c = in[i]) == EOL)
			{
			    fseek(f, -(amt - i) + 1, SEEK_CUR);
			    goto reRead;
			}
		}
		while ((amt = fread(in, 1, INSZ, f)) != 0);
	    }
	}
	
	for (i=0; i<amt; i++, p+=2)	/* For as many as read */
	{
	    c = in[i];

	    for (j=0; j<sizeof(cvt); j+=3) /* Try special cases first */
		if (c == cvt[j])
		{
		    p[-1] = cvt[j+1];	/* Tens digit */
		    p[0] = cvt[j+2];	/* Units digit */
		    goto nextChar;
		}

	    if (c == (X|O|C|8|2))
	    {
		printf("\nFlagged RM (XOC82) at byte %ld, treating as RM\n",
		    ftell(f) - amt + i);
		p[-1] = 0;
		p[0] = RM;
		goto nextChar;
	    }
	    
	    if (parity(c) || (c & 0xF) > 9)	/* Error checks */
	    {
		fprintf(stderr, "\nPANIC: Bad character %2.2X at byte %ld\n",
		    c, ftell(f) - amt + i);
		exit(1);
	    }

	    switch (c & (X|O))		/* Tens digit */
	    {
	    case X|O: p[-1] = 4; break;	/* A-I */
	    case X: p[-1] = 5; break;	/* J-R */
	    case O: p[-1] = 6; break;	/* S-Z */
	    case 0: p[-1] = 7; break;	/* 1-9 */
	    }
	    p[0] = c & 0xF;		/* Units digit straight from tape */

nextChar:
	    if (c == EOL)		/* Break off at EOL */
	    {	/* Back up to just after it */
		if (verbose)
		    printf("%d ", rc - amt + i + 1);
		fseek(f, -(amt - i) + 1, SEEK_CUR);
		return rc;
	    }
	}
    }
    return rc;
}

void sub(char *acc, char *buf)	/* 20-digit BCD subtraction */
/*
 * Since (a) the 1620 is a sign-magnitude machine, (b) we're subtracting
 * only positive numbers from an initial zero, and (c) we're not interested
 * in the sign of the result, we can add instead of subtracting, and
 * get the same magnitude result.
 */
{
    int		i;
    char	cy = 0;
    char	c;

    for (i=0; i<20; i++, acc--, buf--)	/* Twenty digits */
    {
	c = *buf;
	if (c == RM) c = 0;	/* RM subtracts as 0 */
	if (*acc < 0 || *acc > 9 || c < 0 || c > 9)
	{
	    fprintf(stderr, "\nPANIC: Non-BCD acc or buf digit\n");
	    exit(1);
	}
	*acc += c + cy;
	if (*acc > 9)
	{
	    *acc -= 10;
	    cy = 1;
	}
	else
	    cy = 0;
    }
}

int main(int argc, char *argv[])
{
    FILE		*f;
    unsigned int	i;

    if (stricmp(argv[1], "-v") == 0)
    {
	verbose = 1;
	argv++;
    }
    
    f = fopen(argv[1], "rb");
    if (f == NULL)
    {
	fprintf(stderr, "\nUnable to open file\n");
	exit(1);
    }

    init();				/* Set up buffer with stop marks */
    while (RAPT(buf+2, f) > 0)		/* Got something off tape */
    {
	i = 19;				/* Point to LSD of 1st sub */
	while (1)
	{
	    sub(acc+19, buf+i);
	    if (buf[i] == RM)		/* Past current record */
		break;
	    else
	    {
		buf[i] = RM;		/* Replace stop marker */
		i += 20;		/* Advance to next cluster */
	    }
	}
    }
    putchar('\n');
    for (i=0; i<20; i++) printf("%d", acc[i]);
    putchar('\n');
    
    return 0;
}
