/****************************************************************************

		ISIS disk interrogator

****************************************************************************
*/

#include	stdio.h
#include	errno.h

/****************************

	declarations

*****************************/
struct isis_link {
	char p_sector;
	char p_track;
	};
struct isis_link *lptr;		/* ptr to link bytes */
unsigned
char	l_count,	/* Link count */
	nxt_lsector,	/* Next link block sector to read */
	nxt_ltrack;	/* Next link block track to read */


struct isis_dir_struct {
	char flag;
	char name[6];
	char ext[3];
	char attrib;
	char last_n;
	unsigned int n_blocks;
	char l_sector;
	char l_track;
	};
struct	isis_dir_struct *entry;	/* ptr to directory entry structure */

/***	bios call vectors   ***/
#define seldsk 9
#define settrk 10
#define setsec 11
#define setdma 12
#define read_sector 13
#define write_sector 14
#define sectran 16

/***   isis_dir.flag values   ***/	
#define active 0
#define never_used 0x7f
#define deleted 0xff

/***   isis_dir.attrib bit masks   ***/
#define invisible 1
#define system 2
#define protected 4
#define format 0x80

char	drive,		/* Drive number to use for ISIS disk */
	track,		/* Track # to xfer {0..76} */
	sector,		/* Sector # to xfer {1..52} */
	fn_buf[11],	/* Filename buffer */
	map_buf[256],	/* Allocation bit map buffer */
	getuser(),
	link_buf[128],	/* Link block buffer */
	in_buf[80],	/* User input buffer */
	isis_fn[10],	/* ISIS filename buffer */
	dir_buf[128],	/* Directory buffer */
	cpm_fspec[16],	/* CP/M filename buffer */
	read_buf[128];	/* disk sector buffer */

unsigned
	block_cnt;	/* Count of blocks to xfer */

int	diag,		/* Diagnostic on boolean */
	density = 0;	/* Double density boolean */

FILE	*fopen(),	/* CP/M file open function */
	*cpm_file;	/* CP/M file to xfer to/from */

#define	FIRST 0		/* First occurrence of something */
#define REMAINDER 1	/* Non-first occurrence of something */
#define SINGLE 0	/* Single density select */
#define DOUBLE 1	/* Double density select */




/****************************************************************************

	ISIS file utility

****************************************************************************/
main() {
   static int err, c;
   static unsigned tot_blocks, i, j, k;

   printf("\nISIS Disk Interrogator  Vers. 1.0\n\n");
   p_signon();

   while (TRUE) {
      printf("\n- ");
      c = toupper(getuser());
      crlf();
      switch (c) {


	 case '1':		/* Select single density */
	 density = SINGLE;
	 printf("\nSingle density selected.");
	 break;


	 case '2':		/* Select double density */
	 density = DOUBLE;
	 printf("\nDouble density selected.");
	 break;


         case 'S':		/* Select drive */
		/* Get user requirement */
	 printf("\nEnter drive # {A..P} : ");
	 c = toupper(getuser());
	 crlf();

		/* If not reasonable selection then error */
	 if (c < 'A' || c > 'P') {
	    printf("\nYou're not listening, A..P!\n");
	    break;
	    }

		/* Calculate drive number */
	 drive = c - 'A';

		/* Select drive to see if available */
         i = bdos(25,0);		/* save cp/m disk number */
         if (bdos(14,drive))
		printf("\nBios tells me drive doesn't exist???\n");

		/* Restore the drive */
         bdos(14,i);
         break;


         case 'M':		/* Display allocation bit map */
	 if (read_ISIS(drive,2,2,&map_buf[0]))   break;
	 if (read_ISIS(drive,2,3,&map_buf[128]))   break;
	 printf("\nAllocation Map of :F%1d: (1 bit = allocated)\n\
   0  1  2  3  4  5  6  7   8  9  A  B  C  D  E  F", drive);
	 for (i=0; i<16; i++) {
	    printf("\n%1x ",i);
	    for (j=0; j<16; j++){
               printf(" %02x",map_buf[j+(i*16)]);
	       if (j == 7)   printf(" ");
               }
 	    }
	 crlf();
         break;


         case 'D':		/* Display ISIS directory & flags */
         sector = 2;
	 k = 1;
	 tot_blocks = 0;
	 read_ISIS(drive,0,26,&read_buf[0]);
	 printf("\nDIRECTORY OF :F%1d:%s\n\
NAME  .EXT  BLKS   LENGTH ATTR      NAME  .EXT  BLKS   LENGTH ATTR\n",
		drive, (read_buf[0] > 0x7f) ? "\0" : &read_buf);
	 for (sector = 2; sector <= 26; sector++) {
	    entry = &dir_buf[0];
	    if (read_ISIS(drive,1,sector,&dir_buf[0]))   break;
	    for (i=0; i<8; i++,entry++) {
	       if (entry->flag == 0) {
		  fmt_name(6,entry->name,fn_buf);
		  fmt_name(3,entry->ext,&fn_buf[7]);
		  printf("%-6s.%-3s  %4u   %6u %c%c%c%c ",
			fn_buf,
			&fn_buf[7],
			entry->n_blocks + 1 + (entry->n_blocks / 62),
			(((entry->n_blocks - 1) * 128) + entry->last_n),
			(entry->attrib & invisible) ? 'I' : ' ',
			(entry->attrib & system)    ? 'S' : ' ',
			(entry->attrib & protected) ? 'P' : ' ',
			(entry->attrib & format)    ? 'F' : ' '
			);
	          if (k=k^1)   crlf();
	          else   printf("  |  ");
		  tot_blocks += (entry->n_blocks + 1 + (entry->n_blocks / 62));
	          }
	       }
	    }
	 printf("\n\n%u of %s BLOCKS USED",tot_blocks,(density)?"4004":"2002");
	 crlf();
         break;


         case 'R':		/* Read ISIS file & copy to CP/M file */
		/* Get user requirement */
	 printf("\nEnter filename.extension - ");
	 gets(in_buf);

		/* Format filespec to ISIS format */
	 if (fmt_dir(in_buf))   break;

		/* Search filename in directory */
	 if (lookup(FIRST)) {
	    printf("\nISIS file not found!");
	    break;
	    }

		/* Xfer file(s) */
	 for (i = 0; i != 0; ) {

		/* Format the directory match to CP/M filespec */
	    isis2cpm(entry->name, cpm_fspec);

		/* Echo the match filename */
	    printf("\n%10s - ", cpm_fspec);

		/* Open the same filename in CP/M */
	    cpm_file = fopen(cpm_fspec,"w");
	    if (cpm_file == NULL) {
	       printf("\nCP/M open error #%d\n", errno);
	       break;
	       }

		/* Read & fetch the first link */
	    if (rd_link(FIRST))   break;

		/* Xfer the file blocks */
	    for (block_cnt = entry->n_blocks; block_cnt > 0; block_cnt--) {
	       if (diag)  printf("\n%5d T= %2d S= %2d",block_cnt,track,sector);
	       if (read_ISIS(drive,track,sector,read_buf)) break;
	       if (write_buf()) break;
	       if (rd_nxt_link()) break;
	       }

		/* Close the CP/M file */
	    if (fclose(cpm_file) == -1) {
	       printf("****  CP/M close error #%d\n",errno);
	       break;
	       }

		/* Finished this file */
	    else printf("Copied");

		/* Check for any more matches */
	    if (lookup(REMAINDER))   break;
	    }
         break;


         case 'W':		/* Write CP/M file to ISIS disk */
         break;


         case 'H':		/* Display help message */
         case '?':
	 p_signon();
         break;


         case 'X':		/* Exit to CP/M */
	 bios(seldsk,*0x4,1);
         exit(0);
         break;


	 case '*':		/* Diagnostic switch */
	 diag = !diag;	/* toggle it */
	 printf("\nDiagnostic %s.\n", (diag) ? "ON" : "OFF");
	 break;


         default:
         printf("\n??? I don't know that command - try one of these:");
         p_signon();
         break;
         }
      }
   }


/***************************************************************************

	Print sign-on message procedure

***************************************************************************/
p_signon() {
   static char *a1 = {"\n\
1= Select single density (default)\n\
2= Select double density\n\
D= Directory listing\n\
R= Read ISIS file to CP/M file\n"};
   static char *a2 = {"\
W= Write ISIS file from CP/M (6 character filename only)\n\
M= Display block allocation map\n\
S= Select disk drive\n\
? or H= Display this list\n\
X= Exit to CP/M\n"};

   printf("%s%s", a1, a2);
   }


/***************************************************************************

	Read ISIS disk sector function
	Entry-	drive= drive number for BIOS {0,1,..15}
		track= track number to read {0,1,..76}
		sector= sector number to read {1,2,..52}
		dma= transfer address ptr
	Exit -	read status from BIOS {0: ok, /0: error}

***************************************************************************/
int read_ISIS(drive, track, sector, dma)
   	 char drive, track, sector;   unsigned int dma; {
   static int err, cpmdsk;

   cpmdsk = bdos(25,0);		/* save cp/m disk number */
   if (err = bdos(14,drive)) {	/* Temporarily select new drive */
      printf("\nDrive select error!\n");
      bdos(14,cpmdsk);
      return err;
      }
   bios(settrk,track,0);
   bios(setsec,sector,0);
   bios(setdma,dma,0);
   if (err = bios(read_sector,0,0))   printf("\nISIS read error!\n");
   bdos(14,cpmdsk);
   return err;
   }


/***************************************************************************

	Get next character from user function
	exit -	character from keyboard

***************************************************************************/
char getuser() {
   return bdos(1,0);
   }


/***************************************************************************

	Output new line procedure

***************************************************************************/
crlf() {
   printf("\n");
   }


/***************************************************************************

	Create string of n characters long function
	Entry-	cnt= number of characters allowed
		sptr= ptr to source string
	Exit -	ptr to new string terminated by '\0'

****************************************************************************/
fmt_name(cnt,sptr,bptr)   unsigned cnt; char *sptr, *bptr; {

   for ( ; cnt > 0; cnt--)   *bptr++ = *sptr++;
   *bptr = '\0';
   }


/****************************************************************************

	Format filename input into ISIS directory format
	entry-	sptr= ptr to user filename input
	exit -	0: ok
		-1: format error
		isis_fn[]= file name in ISIS directory format

****************************************************************************/
fmt_dir(sptr)   char *sptr; {
   static int i;
   static char *iptr;

   for (i = 0, iptr = isis_fn; i < 6; i++) {
      if (*sptr == '\0' ||
	  *sptr == '.' ||
	  *sptr == '\n' ||
	  *sptr == '*')   break;
      *iptr++ = *sptr++;
      }
   if (i == 6 && *sptr != '.' && *sptr != '\0' && *sptr != '\n') {
      printf("\nFile name error!\n");
      return -1;
      }
   for ( ; i < 6; i++)   *iptr++ = (*sptr == '*') ? '?' : '\0';
   if (*sptr == '.')   sptr++;
   for (i = 0, iptr = &isis_fn[6]; i < 3; i++) {
      if (*sptr == '\0' ||
	  *sptr == '\n' ||
	  *sptr == '*')   break;
      *iptr++ = *sptr++;
      }
   if (i == 3 && *sptr != '\0' && *sptr != '\n') {
      printf("\nFile extension error!\n");
      return -1;
      }
   for ( ; i < 3; i++)   *iptr++ = (*sptr == '*') ? '?' : '\0';
   crlf();
   for (i=0; i < 9; i++) {	/*** Output constructed filename ***/
      if (i==6)   putchar('.');
      if (isis_fn[i] == '\0')   putchar(' ');
      else   putchar(isis_fn[i]);
      }
   crlf();
   return 0;
   }


/****************************************************************************

	Lookup filename in ISIS directory function
	entry-	FIRST= read in directory from start
		REMAINDER= continue to check for match
	exit -	0: ok
		-1: error
		entry -> directory buffer file

****************************************************************************/
lookup(flag)   int flag; {
   static int dsec, j, ents;

	/* Select operation */
   if (flag == FIRST) {
      dsec = 2;	/* Start at sector 2 */
      ents = 8;	/* Force read */
      }

	/* Search all directory sectors */
   for (j = 0, entry++; dsec <= 26 && j != 9); dsec++) {

	/* If nothing in buffer then read a sector */
      if (ents >= 8) {
         if (read_ISIS(drive,1,dsec,dir_buf))   return -1;
	 ents = 0;
	 entry = dir_buf;
	 }

		/* Check each of 8 entries for match */
      for ( ; ents < 8 && j != 9; ents++, entry++) {

		/* If entry is deleted or unused then skip entry */
	 if (entry->flag == 0) {

		/* If no match and not wildcard then skip entry */
            for (j = 0; j < 9; j++) {
               if (entry->name[j] != isis_fn[j] && isis_fn[j] != '?')   break;
	       }
            }
	 }
      }

	/* compensate for for loop post increment */
   entry--;

	/* Return ok */
   return 0;
   }


/****************************************************************************

	Read 1st link block function
	exit -	0: ok
		/0: error
		sector= file sector to read
		track= file track to read

****************************************************************************/
rd_link(flag)   int flag; {

	/* If initial link block read then use link in directory entry */
   if (flag == 0) {
      nxt_lsector = entry->l_sector;
      nxt_ltrack = entry->l_track;
      }

	/* Read the link into the link block buffer */
printf("\n    LT= %2d LS= %2d",nxt_ltrack,nxt_lsector);
   if (read_ISIS(drive,nxt_ltrack,nxt_lsector,link_buf))   return -1;

	/* Save the next link block link for later */
   lptr = &link_buf[2];
   nxt_lsector = lptr->p_sector;
   nxt_ltrack = (lptr++)->p_track;

	/* Set 1st file block to read */
   sector = lptr->p_sector;
   track = (lptr++)->p_track;

	/* Set link count (already used first of 62 links) */
   l_count = 61;

	/* Return ok */
   return 0;
   }


/****************************************************************************

	Set up next link function
	exit -	0: ok
		/0: error
		sector = next file sector to read
		track = next file track to read

****************************************************************************/
rd_nxt_link() {

	/* If no more links then read next link & return result */
   if (l_count-- == 0)   return (rd_link(1)) ?  -1 : 0;

	/* Else fetch next link from link block & return ok */
   sector = lptr->p_sector;
   track = (lptr++)->p_track;
   return 0;
   }


/****************************************************************************

	write ISIS buffer to CP/M file
	exit -	0: ok
		-1: error

****************************************************************************/
write_buf() {
   static int i;

	/* Write sector data to CP/M file */
   for (i = 0; i < 128; i++) {
      if (putc(read_buf[i], cpm_file) == -1) {
	 printf("\nCP/M file write error #%d!\n", errno);
         return -1;
	 }
      }

	/* If successful then return ok */
   return 0;
   }


/****************************************************************************

	Convert ISIS filename to CP/M filespec string procedure
	Entry-	iptr= ptr to ISIS directory format filename
		bptr= ptr to buffer to store the CP/M filespec string
	Exit -	bptr= CP/M filespec ("nnn.nn\0")

****************************************************************************/
isis2cpm(iptr, bptr)   char *iptr, *bptr; {
   static int i;

	/* Pull out name *.
   for (i = 1; i <= 6; i++) {
      if (*iptr == '\0')   break;
      *bptr++ = *iptr++;
      }

	/* Separate name from extension */
   *bptr++ = '.';

	/* Position ISIS ptr to extension field */
   if (i < 6)   iptr = iptr + (7 - i);

	/* Pull out extension */
   for (i = 1; i <= 3; i++) {
      if (*iptr == '\0')   break;
      *bptr++ = *iptr++;
      }

	/* Terminate filespec string */
   *bptr = '\0';
   }
