ORIGIN	EQU	000H
	ORG	ORIGIN

;
;	ANOS DEADSTART LOADER
;
;	THIS PROGRAM IS LOADED FROM A FIXED GROUP OF BLOCKS ON A MASS
;	STORAGE DEVICE DURING A COLD START.  THE PROGRAM SEARCHES
;	DIRECTORY 0 OF UNIT 0 FOR ALL EXTENTS OF THE FILE ANOS.IMG.  IT
;	LOADS ANOS.IMG INTO MEMORY AND RELOCATES IT TO THE GREATEST 
;	ADDRESS POSSIBLE.  FINALLY, ONCE THE RELOCATION PROCESS IS 
;	COMPLETE, IT INITIALIZES THE ANOS ENTRY POINT JUMP AT ADDRESS 5
;	AND INVOKES THE ANOS COLD START SERVICE REQUEST.
;

;
;	ASSEMBLY CONSTANTS
;

TRUE	EQU	0FFFFH
FALSE	EQU	NOT TRUE

RPB	EQU	8		;RECORDS PER BLOCK
RPS	EQU	1		;RECORDS PER SECTOR
SPT	EQU	26		;SECTORS PER TRACK
HPU	EQU	1		;HEADS PER UNIT
BPD	EQU	2		;BLOCKS PER DIRECTORY
DPU	EQU	1		;DIRECTORIES PER UNIT
BPU	EQU	0F3H		;BLOCKS PER UNIT
FTK	EQU	2		;FIRST TRACK OF LOGICAL BLOCKS

SKEW	EQU	TRUE		;SINGLE DENSITY FLOPPIES REQUIRE SOFTWARE SKEWING

COLD	EQU	39		;COLD START REQUEST

DM	EQU	16		;OFFSET TO DISK MAP
XT	EQU	12		;OFFSET TO EXTENT
FCBLEN	EQU	32		;LENGTH OF A FILE CONTROL BLOCK

DEPB	EQU	RPB*4		;DIRECTORY ENTRIES PER BLOCK
SPB	EQU	RPB/RPS		;SECTORS PER BLOCK
BPS	EQU	RPS*128		;BYTES PER SECTOR

;
;	BOOTSTRAP LINKAGE CONSTANTS
;

BIOS		EQU	100H		;ORIGIN OF BIOS DURING COLD START
BIOS$SELHEAD	EQU	BIOS+2DH
BIOS$SELTRACK	EQU	BIOS+1EH
BIOS$SELSECTOR	EQU	BIOS+21H
BIOS$SETDMA	EQU	BIOS+24H
BIOS$READ	EQU	BIOS+27H

;
;	MAIN CONTROL SEQUENCE
;

	LXI	SP,100H
	CALL	BUILDTABLE	;SEARCH DIRECTORY FOR ANOS.IMG AND BUILD BLOCK TABLE
	CALL	GETSIZE		;GET MEMORY SIZE
	CALL	LOAD		;LOAD ANOS INTO THE MEMORY BUFFER
	CALL	RELOC		;RELOCATE ANOS
	MVI	A,JMP		;CONSTRUCT ANOS ENTRY POINT JUMP
	LHLD	ANOSBASE
	LXI	D,6
	DAD	D
	STA	5
	SHLD	6
	MVI	C,COLD		;COLD START SYSTEM
	CALL	5
	JMP	$		;HANG IF SYSTEM RETURNS, IT SHOULD NOT

;
;	BUILDTABLE - SEARCH FOR ANOS.IMG AND BUILD BLOCK TABLE
;
;	SEARCH FINDS ALL EXTENTS OF ANOS.IMG ON UNIT 0, DIRECTORY 0
;	AND BUILDS A TABLE OF BLOCK NUMBERS COMPRISING THE FILE.
;
;	EXIT	(BLOCKTABLE) - BUILT
;
;	USES	ALL
;
;	CALLS	SEARCH
;

BUILDTABLE:CALL	SEARCH		;SEARCH FOR NEXT EXTENT OF ANOS.IMG
	INR	A
	RZ			;IF ALL EXTENTS FOUND
	LXI	D,DM		;SET ADDRESS OF DISK MAP
	DAD	D
	PUSH	H
	CALL	GET8BLOCKS	;GET 8 BLOCK NUMBERS FROM FCB
	LXI	H,BPU
	LXI	D,256
	MOV	A,E
	SUB	L
	MOV	A,D
	SBB	H
	POP	H
	JC	BUILD1		;IF ENTIRE DISK MAP SCANNED
	LXI	D,8
	DAD	D
	CALL	GET8BLOCKS	;GET NEXT 8 BLOCK NUMBERS

BUILD1:	LXI	H,ANOSFCB+XT	;INCREMENT EXTENT NUMBER
	INR	M
	JMP	BUILDTABLE	;FETCH NEXT EXTENT'S FCB

;
;	SEARCH - SEARCH FOR DIRECTORY ENTRY
;
;	ENTRY	(ANOSFCB) - ADDRESS OF FCB TO SEARCH FOR
;
;	EXIT	(A) - 0 IF A DIRECTORY ENTRY FOUND
;		      0FFH OTHERWISE
;		(HL) - ADDRESS OF DIRECTORY IF FOUND
;
;	USES	ALL
;
;	CALLS	READBLOCK, COMPARE
;

SEARCH:	LXI	H,0		;INITIALIZE BLOCK TO READ
	SHLD	BLOCK

SEARCH1:LXI	H,BUFFER	;RESET DMA ADDRESS
	SHLD	DMA
	CALL	READBLOCK	;READ NEXT DIRECTORY BLOCK
	MVI	A,DEPB		;INITIALIZE MAXIMUM NUMBER OF DIRECTORY ENTRIES PER BLOCK
	STA	DELEFT
	LXI	H,BUFFER	;COMPARE FCB'S

SEARCH2:LXI	D,ANOSFCB
	MVI	B,13
	PUSH	H
	CALL	COMPARE
	POP	H
	ORA	A
	JNZ	SEARCH3		;IF MATCH FOUND
	LXI	D,FCBLEN	;ADVANCE TO NEXT FCB
	DAD	D
	LDA	DELEFT
	DCR	A
	STA	DELEFT
	JNZ	SEARCH2		;IF ALL ENTRIES IN BLOCK NOT YET EXAMINED
	LHLD	BLOCK		;READ NEXT BLOCK OF DIRECTORY
	INX	H
	SHLD	BLOCK
	LXI	D,BPD		;SET BLOCKS PER DIRECTORY
	MOV	A,L
	SUB	E
	MOV	A,H
	SBB	D
	JC	SEARCH1		;IF ALL BLOCKS NOT YET EXAMINED
	MVI	A,0FFH
	RET

SEARCH3:XRA	A
	RET

;
;	GET8BLOCKS - GET 8 BLOCKS FROM FCB DISK MAP
;
;	ENTRY	(HL) - ADDRESS OF DISK MAP
;		(BTPTR) - BLOCK TABLE POINTER
;
;	EXIT	(BTPTR) - UPDATED
;		BLOCKTABLE - APPENDED WITH BLOCKS NUMBERS FROM DISK MAP
;
;	USES	ALL
;

GET8BLOCKS:XCHG
	LXI	H,BPU
	LXI	B,256
	MOV	A,C
	SUB	L
	MOV	A,B
	SBB	H
	MVI	B,0FFH
	JC	GET8B1		;IF 2 BYTES PER BLOCK NUMBER
	MVI	B,0

GET8B1:	MVI	C,8
	LHLD	BTPTR

;
;	FETCH NEXT BLOCK NUMBER
;

GET8B2:	LDAX	D
	MOV	M,A
	INX	D
	INX	H
	MOV	A,B
	ORA	A
	JZ	GET8B3		;IF 1 BYTE PER BLOCK NUMBER
	LDAX	D
	INX	D

GET8B3:	MOV	M,A
	INX	H
	DCR	C
	JNZ	GET8B2		;IF 8 BLOCKS NOT YET RETRIEVED
	SHLD	BTPTR
	RET

;
;	READBLOCK - READ LOGICAL BLOCK FROM MASS STORAGE UNIT
;
;	ENTRY	(BLOCK) - BLOCK TO READ
;		(DMA) - ADDRESS OF DMA BUFFER
;
;	USES	ALL
;
;	CALLS	SEEK, READ
;

READBLOCK:CALL	SEEK		;INITIALIZE HEAD, TRACK, SECTOR
	LXI	H,SPB		;INITIALIZE SECTORS PER BLOCK
	PUSH	H

READB1:	CALL	READ		;READ NEXT SECTOR
	LHLD	DMA		;ADVANCE DMA ADDRESS
	LXI	D,BPS		;BYTES PER SECTOR
	DAD	D
	SHLD	DMA

	POP	H
	DCX	H
	MOV	A,H
	ORA	L
	RZ			;IF BLOCK COMPLETELY READ
	PUSH	H

	LHLD	SECTOR		;INCREMENT SECTOR
	INX	H
	SHLD	SECTOR
	LXI	D,SPT		;SECTORS PER TRACK
	MOV	A,L
	SUB	E
	MOV	A,H
	SBB	D
	JC	READB1		;IF TRACK NOT EXHAUSTED

	LXI	H,0		;RESET SECTOR
	SHLD	SECTOR
	LDA	HEAD		;INCREMENT HEAD
	INR	A
	STA	HEAD
	CPI	HPU		;HEADS PER UNIT
	JC	READB1		;IF CYLINDER NOT EXHAUSTED

	XRA	A		;RESET HEAD
	STA	HEAD
	LHLD	TRACK		;INCREMENT TRACK
	INX	H
	SHLD	TRACK
	JMP	READB1

;
;	SEEK - COMPUTE FIRST HEAD, TRACK, SECTOR OF BLOCK
;
;	ENTRY	(BLOCK) - BLOCK TO SEEK
;
;	EXIT	(HEAD, TRACK, SECTOR) - INITIALIZED
;
;	USES	ALL
;
;	CALLS	DIVIDE, MULTIPLY
;

SEEK:	LXI	H,0		;INITIALIZE LOGICAL TRACK AND SECTOR
	SHLD	LSECTOR
	SHLD	LTRACK

SEEK1:	LHLD	BLOCK
	LXI	D,1023
	MOV	A,E
	SUB	L
	MOV	A,D
	SBB	H
	JNC	SEEK3		;IF BLOCK <= 1023
	LHLD	LTRACK		;LTRACK = LTRACK + (SPB * 1024) / SPT
	LXI	D,(SPB * 1024) / SPT
	DAD	D
	SHLD	LTRACK
	LHLD	LSECTOR		;LSECTOR = LSECTOR + (SPB * 1024) MOD SPT
	LXI	D,(SPB * 1024) MOD SPT
	DAD	D
	SHLD	LSECTOR
	LXI	D,SPT
	MOV	A,L
	SUB	E
	MOV	A,H
	SBB	D
	JC	SEEK2		;IF LSECTOR < SECTORS PER TRACK
	LXI	D,-SPT
	DAD	D
	SHLD	LSECTOR
	LHLD	LTRACK
	INX	H
	SHLD	LTRACK

SEEK2:	LHLD	BLOCK		;BLOCK = BLOCK - 1024
	LXI	D,-1024
	DAD	D
	SHLD	BLOCK
	JMP	SEEK1

SEEK3:	LHLD	BLOCK		;VSECTOR = BLOCK * SECTORS PER BLOCK
	LXI	B,SPB
	CALL	MULTIPLY
	SHLD	VSECTOR
	LXI	B,SPT		;LTRACK = LTRACK + (VSECTOR / SECTORS PER TRACK)
	CALL	DIVIDE
	PUSH	H		;SAVE REMAINDER
	LHLD	LTRACK
	DAD	D
	SHLD	LTRACK
	LHLD	LSECTOR		;LSECTOR = LSECTOR + (VSECTOR MOD SECTORS PER TRACK)
	POP	D
	DAD	D
	SHLD	LSECTOR
	LXI	D,SPT
	MOV	A,L
	SUB	E
	MOV	A,H
	SBB	D
	JC	SEEK4		;IF LSECTOR < SECTORS PER TRACK
	LXI	D,-SPT
	DAD	D
	SHLD	LSECTOR
	LHLD	LTRACK
	INX	H
	SHLD	LTRACK

SEEK4:	LHLD	LTRACK		;COMPUTE PHYSICAL TRACK AND HEAD
	LXI	B,HPU		;HEADS PER UNIT
	CALL	DIVIDE
	SHLD	HEAD
	XCHG
	SHLD	TRACK
	LHLD	LSECTOR		;SET PHYSICAL SECTOR
	SHLD	SECTOR
	RET

;
;	READ - READ A SECTOR FROM A MASS STORAGE UNIT
;
;	ENTRY	(HEAD, TRACK, SECTOR, DMA) - SET APPROPRIATELY
;
;	EXIT	DMA BUFFER FILLED
;
;	USES	ALL
;
;	CALLS	SELECTHEAD, SELECTTRACK, SELECTSECTOR, SETDMA, READSECTOR
;

READ:	LDA	HEAD		;SELECT HEAD
	CALL	SELECTHEAD

	LHLD	TRACK		;SELECT TRACK
	CALL	SELECTTRACK

	LHLD	SECTOR		;SELECT SECTOR
	CALL	SELECTSECTOR

	LHLD	DMA		;SET DMA ADDRESS
	CALL	SETDMA

	CALL	READSECTOR	;READ SECTOR

	RET

;
;	COMPARE - COMPARE BLOCKS
;
;	ENTRY	(DE), (HL) - ADDRESSES OF BLOCKS TO COMPARE
;		(B) - LENGTH OF BLOCKS
;
;	EXIT	(A) - 0FFH IF BLOCKS MATCH
;		      0 OTHERWISE
;
;	USES	ALL
;

COMPARE:LDAX	D
	CMP	M
	INX	H
	INX	D
	JNZ	COMP1		;IF BLOCKS DO NOT MATCH
	DCR	B
	JNZ	COMPARE		;IF COMPARE NOT COMPLETE
	MVI	A,0FFH
	RET

COMP1:	XRA	A
	RET

;
;	GETSIZE - GET MEMORY SIZE
;
;	EXIT	(NEXTDST) - LAST VALID BYTE ADDRESS PLUS 1
;
;	USES	A, HL
;

GETSIZE:LXI	H,0FFFFH

GETS1:	XRA	A
	MOV	M,A
	CMP	M
	JZ	GETS2		;IF POSSIBLE RAM
	DCX	H
	JMP	GETS1

GETS2:	CMA
	MOV	M,A
	CMP	M
	JZ	GETS3		;IF RAM
	DCX	H
	JMP	GETS1

GETS3:	INX	H
	MVI	L,0
	SHLD	NEXTDST
	RET

;
;	LOAD - LOAD THE OS INTO THE MEMORY BUFFER
;

LOAD:	LXI	H,BUFFER	;INITIALIZE THE DMA ADDRESS
	SHLD	DMA
	LXI	H,BLOCKTABLE	;INITIALIZE THE BLOCK TABLE POINTER
	SHLD	BTPTR

LOAD1:	LHLD	BTPTR		;FETCH NEXT BLOCK NUMBER
	MOV	E,M
	INX	H
	MOV	D,M
	INX	H
	SHLD	BTPTR
	XCHG
	MOV	A,H
	ORA	L
	RZ			;IF ALL BLOCKS READ
	SHLD	BLOCK
	CALL	READBLOCK	;READ NEXT BLOCK
	JMP	LOAD1

;
;	RELOC - RELOCATE OPERATING SYSTEM
;
;	ENTRY	(BUFFER) - PAGE RELOCATABLE OPERATING SYSTEM IMAGE
;		(NEXTDST) - LAST VALID BYTE ADDRESS PLUS 1
;

RELOC:	LHLD	NEXTDST		;FETCH LAST VALID ADDRESS PLUS 1
	XCHG

;
;	COMPUTE DESTINATION ADDRESS OF OS
;

	LHLD	BUFFER		;FETCH IMAGE LENGTH
	MOV	A,E
	SUB	L
	MOV	L,A
	MOV	A,D
	SBB	H
	MOV	H,A
	SHLD	ANOSBASE
	STA	RELOCA

;
;	COMPUTE LENGTH OF BIT MAP
;

	LHLD	BUFFER		;BIT MAP LENGTH = IMAGE LENGTH / 8
	MVI	B,3

;
;	SHIFT LENGTH RIGHT 3 BIT POSITIONS
;

RELOC1:	MOV	A,H
	ORA	A
	RAR
	MOV	H,A
	MOV	A,L
	RAR
	MOV	L,A
	DCR	B
	JNZ	RELOC1		;IF SHIFT NOT COMPLETE

	LXI	D,BUFFER+2	;COMPUTE POSITION OF LAST BIT PLUS 1
	DAD	D
	SHLD	NEXTBIT
	XCHG			;COMPUTE POSITION OF LAST BYTE OF IMAGE PLUS 1
	LHLD	BUFFER
	DAD	D
	SHLD	NEXTSRC
	LHLD	NEXTDST		;INITIALIZE NEXT DESTINATION ADDRESS
	XCHG
	LHLD	NEXTSRC		;INITIALIZE NEXT SOURCE ADDRESS

RELOC2:	PUSH	H
	LHLD	NEXTBIT		;FETCH NEXT BYTE OF MAP BITS
	DCX	H
	SHLD	NEXTBIT
	MOV	C,M
	POP	H
	MVI	B,8		;SET NUMBER OF BITS PER BYTE

RELOC3:	DCX	D
	DCX	H
	MOV	A,C		;TEST NEXT MAP BIT
	RLC
	MOV	C,A
	MOV	A,M		;FETCH NEXT CODE BYTE
	JNC	RELOC4		;IF BYTE NOT RELOCATABLE
	ADI	0		;ADD RELOCATION BASE
RELOCA	EQU	$-1

RELOC4:	STAX	D		;STORE BYTE
	DCR	B
	JNZ	RELOC3		;IF MAP BYTE NOT EXHAUSTED
	PUSH	H
	LHLD	BUFFER		;DECREMENT COUNT OF BYTES TO RELOCATE
	LXI	B,-8
	DAD	B
	SHLD	BUFFER
	MOV	A,H
	ORA	L
	POP	H
	JNZ	RELOC2		;IF RELOCATION NOT COMPLETE
	RET

;
;	BIOS LINKAGE ROUTINES
;


;
;	SELECTHEAD - SELECT HEAD TO READ
;
;	ENTRY	(A) - HEAD NUMBER
;

SELECTHEAD:
	MOV	C,A
	MVI	A,HPU
	DCR	A
	RZ			;IF ONLY ONE HEAD (NO NEED TO SELECT IT)
	JMP	BIOS$SELHEAD

;
;	SELECTTRACK - SELECT TRACK TO READ
;
;	ENTRY	(HL) - TRACK NUMBER
;

SELECTTRACK:
	LXI	B,FTK		;ADD OFFSET TO FIRST TRACK OF LOGICAL BLOCKS
	DAD	B
	MOV	B,H
	MOV	C,L
	JMP	BIOS$SELTRACK

;
;	SELECTSECTOR - SELECT SECTOR TO READ
;
;	ENTRY	(HL) - SECTOR NUMBER
;

SELECTSECTOR:
	IF	SKEW		;USE LOGICAL SECTOR AS INDEX INTO TRANSLATION TABLE
	DAD	H
	LXI	B,TRANSLATE
	DAD	B
	MOV	C,M	
	INX	H
	MOV	B,M
	ENDIF
	IF	NOT SKEW	;ELSE, LOGICAL SECTOR IS PHYSICAL SECTOR
	MOV	B,H
	MOV	C,L
	ENDIF
	JMP	BIOS$SELSECTOR

;
;	SETDMA - SET DMA ADDRESS
;
;	ENTRY	(HL) - DMA ADDRESS
;

SETDMA:	MOV	B,H
	MOV	C,L
	JMP	BIOS$SETDMA

;
;	READSECTOR - READ A SECTOR
;

READSECTOR	EQU	BIOS$READ

;
;	DIVIDE - 16 BIT INTEGER DIVIDE
;
;	ENTRY	(BC) - DIVISOR
;		(HL) - DIVIDEND
;
;	EXIT	(DE) - QUOTIENT
;		(HL) - REMAINDER
;
;	USES	ALL
;

DIVIDE:	MOV	A,B		;NEGATE
	CMA
	MOV	B,A
	MOV	A,C
	CMA
	MOV	C,A
	INX	B
	LXI	D,0		;INITIALIZE 32 BIT DIVIDEND
	XCHG
	MVI	A,16		;INITIALIZE DIVISOR BIT COUNT

DIV1:	DAD	H		;LEFT SHIFT DIVIDEND
	JC	DIV4		;IF MSB WAS 1
	XCHG
	DAD	H
	XCHG
	JNC	DIV2		;PROPAGATE MSB INTO HL
	INR	L

DIV2:	PUSH	H
	DAD	B		;SUBTRACT DIVISOR
	JC	DIV3		;IF HL .GT. BC
	POP	H
	DCR	A
	JNZ	DIV1		;IF DIVISION NOT COMPLETE
	RET

DIV3:	INR	E
	INX	SP		;CLEAN UP STACK
	INX	SP
	DCR	A
	JNZ	DIV1		;IF DIVISION NOT COMPLETE
	RET

DIV4:	XCHG
	DAD	H
	XCHG
	JNC	DIV5
	INR	L

DIV5:	DAD	B		;SUBTRACT DIVISOR
	INR	E
	DCR	A
	JNZ	DIV1		;IF DIVISION NOT COMPLETE
	RET 

;
;	MULTIPLY - 16 BIT MULTIPLY
;
;	ENTRY	(BC) - MULTIPLIER
;		(HL) - MULTIPLICAND
;
;	EXIT	(HL) - PRODUCT
;
;	USES	ALL
;

MULTIPLY:LXI	D,0		;PRESET RESULT
	MVI	A,16		;BIT COUNT
	XCHG

MULT1:	DAD	H		;MULTIPLY RESULT BY 2
	XCHG
	DAD	H
	XCHG
	JNC	MULT2		;IF NO DIGIT IN MULTIPLICAND
	DAD	B

MULT2:	DCR	A
	JNZ	MULT1		;IF MULTIPLICATION NOT COMPLETE
	RET

;
;	VARIABLES, BUFFERS, AND TABLES
;

ANOSBASE:DW	0		;BASE OF OPERATING SYSTEM

ANOSFCB:DB	0,'ANOS    IMG',0	;ANOS.IMG FILE CONTROL BLOCK

BLOCK:	DW	0		;CURRENT BLOCK

DMA:	DW	BUFFER		;CURRENT DMA ADDRESS

DELEFT:	DB	0		;DIRECTORY ENTRIES LEFT IN CURRENT BLOCK

BTPTR:	DW	BLOCKTABLE	;BLOCK TABLE POINTER

BLOCKTABLE:			;TABLE OF BLOCK NUMBERS
	DW	0,0,0,0,0,0,0,0
	DW	0,0,0,0,0,0,0,0
	DW	0,0,0,0,0,0,0,0
	DW	0,0,0,0,0,0,0,0

	IF	SKEW
TRANSLATE:			;LOGICAL TO PHYSICAL SECTOR TRANSLATION TABLE
	DW	01H,07H,0DH,13H,19H,05H,0BH,11H
	DW	17H,03H,09H,0FH,15H,02H,08H,0EH
	DW	14H,1AH,06H,0CH,12H,18H,04H,0AH
	DW	10H,16H
	ENDIF

HEAD:	DB	0		;CURRENT HEAD

TRACK:	DW	0		;CURRENT TRACK

SECTOR:	DW	0		;CURRENT SECTOR

LTRACK:	DW	0		;CURRENT LOGICAL TRACK

LSECTOR:DW	0		;CURRENT LOGICAL SECTOR

VSECTOR:DW	0		;CURRENT VIRTUAL SECTOR

NEXTDST:DW	0		;NEXT DESTINATION ADDRESS

NEXTSRC:DW	0		;NEXT SOURCE ADDRESS

NEXTBIT:DW	0		;NEXT BIT MAP ADDRESS

BUFFER:	DS	0		;MASS STORAGE BUFFER

	END
