TS
;
; TTY AND CRT STATUS BITS
;
TRDY	EQU	00000001B	; TRANSMIT READY
RBR	EQU	00000010B	; RECEIVE BUFFER READY
TBE	EQU	00000100B	; TRANSMIT EMPTY
RPAR	EQU	00001000B	; RECEIVE PARITY ERROR
ROV	EQU	00010000B	; RECEIVE OVERRUN ERROR
RFR	EQU	00100000B	; RECEIVE FRAMING ERROR
DSR	EQU	10000000B	; DATA SET READY
;
; PTR, PTP, AND TTY READER CONTROLS
;
PTPREV	EQU	00010000B	; PUNCH REVERSE
PTPADV	EQU	00100000B	; PUNCH ADVANCE
PTRREV	EQU	00000100B	; READER REVERSE
PTRADV	EQU	00001000B	; READER ADVANCE
TTYADV	EQU	00000010B	; TTY ADVANCE
;
; LPT,PTR, AND PTP STATUS BITS
;
LPTRY	EQU	00000001B	; LPT READY
PTRDY	EQU	00000001B	; PTR READY WITH DATA
PTPRY	EQU	00000100B	; PTP READY FOR DATA
;
; TTY I/O CONSTANTS
;
TTI	EQU	0F4H	; TTY INPUT DATA PORT
TTO	EQU	0F4H	; TTY OUTPUT DATA PORT
TTS	EQU	0F5H	; TTY INPUT STATUS PORT
TTC	EQU	0F5H	; TTY OUTPUT CONTROL PORT
;
; CRT I/O CONSTANTS
;
CRTI	EQU	0F6H	; CRT INPUT DATA PORT
CRTO	EQU	0F6H	; CRT OUTPUT DATA PORT
CRTS	EQU	0F7H	; CRT INPUT STATUS PORT
CRTC	EQU	0F7H	; CRT OUTPUT CONTROL PORT
;
; PTR I/O CONSTANTS
;
PTRI	EQU	0F8H	; PTR INPUT DATA PORT
PTRS	EQU	0F9H	; PTR INPUT STATUS PORT
PTRC	EQU	0F9H	; PTR OUTPUT COMMAND PORT
;
; PTP I/O CONSTANTS
;
PTPO	EQU	0F8H	; PTP OUTPUT DATA PORT
PTPS	EQU	0F9H	; PTP INPUT STATUS PORT
PTPC	EQU	0F9H	; PTP OUTPUT COMMAND PORT
;
; LPT I/O CONSTANTS
;
LPTO	EQU	0FAH	; LPT OUTPUT DATA PORT
LPTS	EQU	0FBH	; LPT INPUT STATUS PORT
LPTC	EQU	0FBH	; LPT OUTPUT COMMAND PORT
;
; GLOBAL CONSTANTS
;
TOUT	EQU	250	; 250MS COUNTER FOR READER TIMEOUT
CR	EQU	0DH	; ASCII VALUE OF CARRIAGE RETURN
LF	EQU	0AH	; ASCII VALUE OF LINE FEED
ETX	EQU	03H	; MONITOR BREAK CHARACTER (CNTRL C)
;
; I/O STATUS BYTE MASKS AND VALUES
;
CMSK	EQU	11111100B	; MASK FOR CONSOLE I/O
RMSK	EQU	11110011B	; MASK FOR READER INPUT
PMSK	EQU	11001111B	; MASK FOR PUNCH OUTPUT
LMSK	EQU	00111111B	; MASK FOR LIST OUTPUT
;
CTTY	EQU	00000000B	; CONSOLE I/O = TTY
CCRT	EQU	00000001B	; CONSOLE I/O = CRT
BATCH	EQU	00000010B	; BATCH, IN = READER, OUT = LIST
CUSE	EQU	00000011B	; USER DEFINED CONSOLE I/O
RTTY	EQU	00000000B	; READER = TTY
RPTR	EQU	00000100B	; READER = PTR
RUSE1	EQU	00001000B	; USER DEFINED READER (1)
RUSE2	EQU	00001100B	; USER DEFINED READER (2)
PTTY	EQU	00000000B	; PUNCH = TTY
PPTP	EQU	00010000B	; PUNCH = PTP
PUSE1	EQU	00100000B	; USER DEFINED PUNCH 1
PUSE2	EQU	00110000B	; USER DEFINED PUNCH 2
LTTY	EQU	00000000B	; LIST = TTY
LCRT	EQU	01000000B	; LIST = CRT
LLPT	EQU	10000000B	; LIST = LPT
LUSE	EQU	11000000B	; USER DEFINED LIST
;
; PAGE 0 DEDICATED RAM LOCATIONS
;
	ORG	0
;
RESET:	DS	3	;TRAP TO MONITOR RESTART
IOBYT:	DS	1	;I/O SYSTEM STATUS BYTE
MEMTOP:	DS	2	;TOP OF RAM, ONLY H SAVED
INITIO:	DS	1	;INITIAL I/O CONFIGURATION
;
; SELECTION CODES FOR USER I/O ENTRY POINTS
;
UCI	EQU	(CILOC-XTBL)/3
UCO	EQU	(COLOC-XTBL)/3
UR1	EQU	(R1LOC-XTBL)/3
UR2	EQU	(R2LOC-XTBL)/3
UP1	EQU	(P1LOC-XTBL)/3
UP2	EQU	(P2LOC-XTBL)/3
UL1	EQU	(L1LOC-XTBL)/3
UCS	EQU	(CSLOC-XTBL)/3
;
	ORG	0F800H
;
BEGIN:	JMP	START	; RESET ENTRY POINT
	JMP	CI	; CONSOLE INPUT
	JMP	RI	; READER INPUT
	JMP	CO	; CONSOLE OUTPUT
	JMP	PO	; PUNCH OUTPUT
	JMP	LO	; LIST OUTPUT
	JMP	CSTS	; CONSOLE INPUT STATUS
	JMP	IOCHK	; I/O SYSTEM STATUS
	JMP	IOSET	; SET I/O CONFIGURATION
	JMP	MEMCK	; COMPUTE SIZE OF MEMORY
	JMP	IODEF	; DEFINE USER I/O ENTRY POINTS
	JMP	SPCL	; I/O CONTROL
	DW	DATE	; DATE STAMP FOR MONITOR ROM
;
; ERROR EXIT
;
ERROR:	LHLD	MEMTOP
	MVI	L,TOS AND 0FFH
	SPHL
	CALL	COMC
	DB	'#'
;
; MAIN COMMAND LOOP
;
START:	IN	0FFH	; LOOP UNTIL BOOTSTRAP MODE IS DISABLED
	ANI	2
	JNZ	START
	EI		; ENABLE INTERRUPTS
	CALL	CRLF	; TYPE <CR>,<LF>
	CALL	COMC	; OUTPUT A PERIOD
	DB	'.'
	CALL	TI	; GET A CHARACTER, ECHO IT
	SUI	'A'	; TEST FOR A=Z
	JM	START	; LT A, IGNORE
	MVI	C,2	; C IS SET UP FOR 2 PARAMETER COMMAND
	LXI	D,START	; SET UP FOR PSEUDO RETURN ADDRESS
	PUSH	D	; TO SIMULATE EFFECT OF CALL
	LXI	H,CTBL	; BRANCH TO PROCESSING ROUTINE
	CPI	LCT	; TEST FOR OVERRUN
	JP	ERROR
	MOV	E,A	; MOVE INDEX TO DE
	MVI	D,0
	DAD	D	; ADD BASE + 2 * INDEX => HL
	DAD	D
	MOV	A,M	; GET LSB OF BRANCH LOCATION
	INX	H
	MOV	H,M	; GET MSB OF BRANCH LOCATION
	MOV	L,A
	PCHL		; TAKE THE BRANCH
;
CTBL:	DW	ASSIGN	; A = ASSIGN I/O UNITS
	DW	ERROR	; B =
	DW	ERROR	; C =
	DW	DISP	; D = DISPLAY RAM MEMORY
	DW	EOF	; E = ENDFILE A HEXIDECIMAL FILE
	DW	FILL	; F = FILL MEMORY
	DW	GOTO	; G = GO TO MEMORY ADDRESS
	DW	HEXN	; H = HEXIDECIMAL SUM AND DIFFERENCE
	DW	ERROR	; I =
	DW	ERROR	; J =
	DW	ERROR	; K =
	DW	ERROR	; L =
	DW	MOVE	; M = MOVE MEMORY
	DW	NULL	; N = PUNCH NULLS FOR LEADER
	DW	ERROR	; O =
	DW	ERROR	; P =
	DW	QUERY	; Q = QUERY I/O SYSTEM STATUS
	DW	READ	; R = READ HEXIDECIMAL FILE
	DW	SUBS	; S = SUBSTITUTE MEMORY
	DW	ERROR	; T =
	DW	ERROR	; U =
	DW	ERROR	; V =
	DW	WRITE	; W = WRITE HEX TAPE
	DW	X	; X = EXAMINE AND MODIFY REGISTERS
LCT	EQU	($-CTBL)/2
;
; PROCESS I/O DEVICE ASSIGNMENT COMMANDS
;
ASSIGN:	CALL	TI	; GET LOGICAL DEVICE CHARACTER
	LXI	H,LTBL	; ADDRESS OF MASTER TABLE
	MVI	C,4	; MAXIMUM OF 4 ENTRIES
AS0:	CMP	M	; TEST FOR IDENTIFING CHARACTER
	INX	H
	JZ	AS1	; FOUND IT
	INX	H	; POINT TO NEXT ENTRY
	INX	H
	INX	H
	DCR	C	; DECREMENT LOOP COUNT
	JNZ	AS0	; TRY NEXT ENTRY
	JMP	ERROR	; NO MATCH, ERROR
;
AS1:	MOV	B,M	; GET SELECT BIT MASK
	INX	H
	MOV	E,M	; GET PHYSICAL DEVICE TABLE
	INX	H
	MOV	D,M
	XCHG
AS1A:	CALL	TI
	CPI	'#'
	JNZ	AS1A
AS1B:	CALL	TI
	CPI	' '
	JZ	AS1B
	MVI	C,4	; SET TABLE LENGTH
AS2:	CMP	M	; INDEX THROUGH PHYSICAL TABLE
	INX	H	; COMPARE DEVICE CHAR WITH LEGAL VALUE
	JZ	AS3	; RETURN WITH HL => DEVICE SELECT BITS
	INX	H
	DCR	C
	JNZ	AS2	; CONTINUE LOOKUP
	JMP	ERROR	; ERROR RETURN
;
AS3:	CALL	TI
	CPI	CR
	JNZ	AS3
	LDA	IOBYT	; GET I/O STATUS
	ANA	B	; CLEAR FIELD
	ORA	M	; SET NEW STATUS
	STA	IOBYT	; RETURN TO MEMORY
	RET
;
; MASTER I/O DEVICE TABLE
;
;	BYTE 0 = IDENTIFING CHARACTER
;	BYTE 1 = LOGICAL DEVICE MASK
;	BYTE 2,3 = SUBORDINATE PHYSICAL DEVICE TABLE
;
LTBL:	DB	'C',CMSK
	DW	ACT
	DB	'R',RMSK
	DW	ART
	DB	'P',PMSK
	DW	APT
	DB	'L',LMSK
	DW	ALT
;
; I/O SYSTEM PHYSICAL DEVICE TABLES
;
;	BYTE 0 = IDENTIFING CHARACTER
;	BYTE 1 = DEVICE SELECT BIT PATTERN
;
ACT:	DB	'T',CTTY	; CONSOLE = TTY
	DB	'C',CCRT	; CONSOLE = CRT
	DB	'B',BATCH	; BATCH MODE CONSOLE = READ, LIST
	DB	'1',CUSE	; USER DEFINED CONSOLE DEVICE
;
ART:	DB	'T',RTTY	; READER = TTY
	DB	'P',RPTR	; READER = PTR
	DB	'1',RUSE1	; USER DEFINED READER DEVICE 1
	DB	'2',RUSE2	; USER DEFINED READER DEVICE 2
;
APT:	DB	'T',PTTY	; PUNCH = TTY
	DB	'P',PPTP	; PUNCH = PTP
	DB	'1',PUSE1	; USER DEFINED PUNCH DEVICE 1
	DB	'2',PUSE2	; USER DEFINED PUNCH DEVICE 2
;
ALT:	DB	'T',LTTY	; LIST = TTY
	DB	'C',LCRT	; LIST = CRT
	DB	'L',LLPT	; LIST = LPT
	DB	'1',LUSE	; USER DEFINED LIST DEVICE
;
; DISPLAY MEMORY IN HEX ON LIST DEVICE
;
DISP:	CALL	EXPR
	POP	D
	POP	H
DI0:	CALL	LCRLF
	CALL	DADR
DI1:	MVI	C,' '
	CALL	LOM
	MOV	A,M
	CALL	DBYTE
	CALL	HILO
	JC	DI2
	MOV	A,L
	ANI	0FH
	JNZ	DI1
	JMP	DI0
DI2:	CALL	LCRLF
	MVI	C,0
	CALL	LOM
	RET
;
; END OF FILE COMMAND
;
EOF:	DCR	C	; GET ONE PARAMETER
	CALL	EXPR
	CALL	POC
	DB	':'	; OUTPUT RECORD MARK
	XRA	A	; ZERO CHECKSUM
	MOV	D,A
	CALL	PBYTE	; OUTPUT RECORD LENGTH
	POP	H	; RETRIEVE PARAMETER
	CALL	PADR	; OUTPUT IT
	MVI	A,1	; RECORD TYPE
	CALL	PBYTE	; OUTPUT IT
	XRA	A
	SUB	D	; COMPUTE CHECKSUM
	CALL	PBYTE	; OUTPUT IT
	JMP	NU0	; PUNCH TRAILER
;
; FILL RAM MEMORY BLOCK WITH CONSTANT
;
FILL:	INR	C	; GET 3 PARAMETERS
	CALL	EXPR
	POP	B	; GET DATA IN C
	POP	D	; GET HIGH ADDRESS
	POP	H	; GET LOW ADDRESS
FI0:	MOV	M,C	; STORE CONSTANT IN MEMORY
	CALL	HILO	; TEST FOR COMPLETION
	JNC	FI0	; CONTINUE LOOPING
	RET
;
; GO TO <ADDRESS>, OPTIONALLY SET BREAKPOINTS
;
GOTO:	LHLD	MEMTOP
	MVI	L,EXIT AND 0FFH
	XTHL
	CALL	PCHK
	JZ	GO0
	CALL	PA0
	XCHG
	LHLD	MEMTOP
	MVI	L,PLOC AND 0FFH
	MOV	M,D
	DCX	H
	MOV	M,E
GO0:	JC	GO4
	LXI	D,2
GO1:	CALL	COMC
	DB	'='
	CALL	PARAM
	PUSH	H
	INR	D
	JC	GO2
	DCR	E
	JNZ	GO1
GO2:	JNC	ERROR
	LHLD	MEMTOP
	MVI	L,TLOC AND 0FFH
GO3:	POP	B
	MOV	M,C
	INX	H
	MOV	M,B
	INX	H
	LDAX	B
	MOV	M,A
	INX	H
	MVI	A,0C7H	; RST 0	; SET TRAP CODE
	STAX	B
	DCR	D
	JNZ	GO3
GO4:	CALL	CRLF
	RET		; RETURN TO USER CODE
;
; COMPUTE HEXIDECIMAL SUM AND DIFFERENCE
;
; <P1+P2> <P1-P2>
;
HEXN:	CALL	EXPR	; GET TWO NUMBERS
	CALL	CRLF
	POP	D
	POP	H
	PUSH	H
	DAD	D	; COMPUTE HL+DE
	CALL	LADR	; DISPLAY SUM
	CALL	BLK	; TYPE A SPACE
	POP	H
	MOV	A,L	; COMPUTE HL-DE
	SUB	E
	MOV	L,A
	MOV	A,H
	SBB	D
	MOV	H,A
	CALL	LADR	; DISPLAY DIFFERENCE
	RET
;
; MOVE A BLOCK OF MEMORY
;
MOVE:	INR	C	; GET THREE ADDRESSES
	CALL	EXPR
	POP	B	; DESTINATION
	POP	D	; SOURCE END
	POP	H	; SOURCE BEGIN
MV0:	MOV	A,M	; GET DATA BYTE
	STAX	B	; STORE A DESTINATION
	INX	B	; MOVE DESTINATION POINTER
	CALL	HILO	; TEST FOR COMPLETION
	JNC	MV0
	RET
;
; PUNCH LEADER OR TRAILER
;
NULL:	CALL	TI	; REQUIRE CR
	CPI	CR
	JNZ	ERROR
NU0:	CALL	LEAD
	CALL	POC
	DB	0
	RET
;
; SYSTEM QUERY COMMAND
;
QUERY:	CALL	TI	; REQUIRE CR
	CPI	CR
	JNZ	ERROR
	MVI	B,4	; SET UP OUTER LOOP COUNTER
	LXI	H,LTBL	; POINT HL AT LOGICAL DEVICE TABLE
Q0:	CALL	CRLF	; START A NEW LINE
	MOV	C,M	; DISPLAY LOGICAL DEVICE IDENTIFIER
	CALL	COM
	CALL	COMC	; DISPLAY '='
	DB	'='
	INX	H	; POINT AT MASK FOR LOGICAL DEVICE
	MOV	A,M	; FETCH MASK
	CMA
	MOV	C,A
	INX	H
	MOV	E,M
	INX	H
	MOV	D,M
	INX	H
	XCHG
	LDA	3
	ANA	C
	PUSH	B
	MVI	B,4
Q1:	MOV	C,M
	INX	H
	CMP	M
	JZ	Q2
	INX	H
	DCR	B
	JNZ	Q1
Q2:	CALL	COM
	XCHG
	POP	B
	DCR	B
	JNZ	Q0
	RET
;
; READ ROUTINE
;
READ:	DCR	C
	CALL	EXPR
	CALL	CRLF
RED0:	CALL	RIX
	CPI	':'
	JNZ	RED0
	XRA	A
	MOV	D,A
	CALL	BYTE
	JZ	RED3
	MOV	E,A
	CALL	BYTE
	MOV	H,A
	CALL	BYTE
	MOV	L,A
	CALL	BYTE
	MOV	C,E
	PUSH	H
	LXI	H,0FF00H
	DAD	SP
RED1:	CALL	BYTE
	MOV	M,A
	INX	H
	DCR	E
	JNZ	RED1
	CALL	BYTE
	JNZ	ERROR
	POP	D	; DE = LOAD ADRESS
	XTHL		; HL = BIAS ADDRESS, TOS = BUFFER
	XCHG		; DE = BIAS ADDRESS, HL = LOAD ADDRESS
	DAD	D
	MVI	B,0
	DAD	B
	XCHG
	XTHL
RED2:	DCX	H
	MOV	A,M
	DCX	D
	STAX	D
	DCR	C
	JNZ	RED2
	JMP	RED0
RED3:	LHLD	MEMTOP
	MVI	L,PLOC AND 0FFH
	CALL	BYTE
	MOV	M,A
	DCX	H
	CALL	BYTE
	MOV	M,A
	POP	H
	RET
;
; SUBSTITUTE MEMORY CONTENTS ROUTINE
;
SUBS:	CALL	PARAM
	RC
SU0:	MOV	A,M
	CALL	LBYTE
	CALL	COMC
	DCR	L
	CALL	PCHK
	RC
	JZ	SU1
	XCHG
	CALL	PA0
	XCHG
	MOV	M,E
	RC
SU1:	INX	H
	JMP	SU0
;
; WRITE ROUTINE
;
WRITE:	CALL	EXPR
	CALL	CRLF
	POP	D
	POP	H
WR0:	CALL	POC	; EMIT RECORD MARK
	DB	':'
	LXI	B,16	; INITIALIZE B=0, C=16
	PUSH	H	; SAVE HL
WR1:	INR	B
	DCR	C
	JZ	WR2
	CALL	HILO
	JNC	WR1
WR2:	POP	H
	PUSH	D
	MVI	D,0
	MOV	A,B
	CALL	PBYTE
	CALL	PADR
	XRA	A
	CALL	PBYTE
WR3:	MOV	A,M
	CALL	PBYTE
	INX	H
	DCR	B
	JNZ	WR3
	SUB	D
	CALL	PBYTE
	POP	D
	DCX	H
	CALL	PEOL
	CALL	HILO
	JNC	WR0
	RET
;
; EXAMINE AND MODIFY CPU REGISTERS
;
X:	LXI	H,ACTBL
	CALL	PCHK
	JC	X5
	MVI	C,NREGS
X0:	CMP	M
	JZ	X1
	INX	H
	INX	H
	INX	H
	DCR	C
	JNZ	X0
	JMP	ERROR
;
X1:	CALL	BLK
X2:	CALL	DREG
	CALL	COMC
	DB	'-'	; TYPE PROMPT
	CALL	PCHK
	RC
	JZ	X4
	PUSH	H
	PUSH	B
	CALL	PA0
	MOV	A,L
	STAX	D
	POP	PSW
	ORA	A
	JM	X3
	INX	D
	MOV	A,H
	STAX	D
X3:	POP	H
X4:	XRA	A
	ORA	M
	RM
	MOV	A,B
	CPI	0
	RZ
	JMP	X2
;
X5:	CALL	CRLF
X6:	CALL	BLK
	XRA	A
	ORA	M
	RM
	MOV	C,M
	CALL	COM
	CALL	COMC
	DB	'='
	CALL	DREG
	JMP	X6
;
; TABLE FOR ACCESSING REGISTERS
; TABLE CONTAINS:
;	(1) REGISTER IDENTIFIER
;	(2) LOCATION ON STORAGE PAGE
;	(3) PRECISION
;
ACTBL:	DB	'A',ALOC,0
	DB	'B',BLOC,0
	DB	'C',CLOC,0
	DB	'D',DLOC,0
	DB	'E',ELOC,0
	DB	'F',FLOC,0
	DB	'H',HLOC,0
	DB	'I',ILOC,0
	DB	'L',LLOC,0
	DB	'M',MLOC,1
	DB	'P',PLOC,1
	DB	'S',SLOC,1
	DB	-1
NREGS	EQU	($-ACTBL)/3	; LENGTH OF ACCESS TABLE
;
; END OF MONITOR COMMANDS
;
;
; I/O SUBROUTINES
;
;
; CONSOLE INPUT
;
CI:	LDA	IOBYTE	; GET STATUS BYTE
	ANI	NOT CMSK	; GET CONSOLE BITS
	JNZ	CI0	; TEST FOR CRT
TTYIN:	IN	TTS	; TTY STATUS PORT
	ANI	RBR	; CHECK FOR RECEIVE BUFFER READY
	JZ	TTYIN
	IN	TTI	; READ THE CHARACTER
	RET
;
CI0:	CPI	CCRT	; CONSOLE = CRT
	JNZ	CI1	; TEST FOR BATCH
CRTIN:	IN	CRTS	; CRT STATUS PORT
	ANI	RBR	; CHECK FOR RECEIVE BUFFER READY
	JZ	CRTIN	; NOT READY CONTINUE LOOPING
	IN	CRTI	; READ THE CHARACTER
	RET
;
CI1:	CPI	BATCH
	JZ	RI	; BATCH MODE, INPUT = READER
	MVI	A,CILOC	; USER DEFINED CONSOLE INPUT
	JMP	@USER
;
; TEST FOR OPERATOR INTERRUPTION OF COMMAND
; BY DEPRESSING BREAK KEY
;
BREAK:	CALL	CSTS	; SEE IF KEY WAS DEPRESSED
	ORA	A
	RZ		; NO CHARACTER READY
	JMP	TI	; GET THE CHARACTER
;
RI:	PUSH	H
	LXI	H,IOBYT	; POINT HL AT IOBYT
	MOV	A,M
	ANI	NOT RMSK	; READER = PTR?
	JNZ	RI3	; BRANCH TO PTR ROUTINE
	MVI	A,TTYADV	; READER = TTY
	OUT	PTRC
	MVI	H,TOUT	; SET READER TIMEOUT TIMER
RI0:	RC
	PUSH	PSW
	ANI	2
	JNZ	RI2	; DATA IS READY
	CALL	DELAY	; DELAY 10 MS
	DCR	H
	JNZ	RI0
RI1:	XRA	A
	STC		; SET CARRY INDICATING EOF
	POP	H
	RET
;
RI2:	IN	TTI
	ORA	A	; CLEAR CARRY
	POP	H
	RET
;
RI3:	CPI	RPTR
	JNZ	RI6
	MVI	A,PTRADV	; START PTR
	OUT	PTRC
	MVI	H,IOUT	; SET READER TIMEOUT TIMER
RI4:	IN	PTRS
	ANI	PTRDY
	JNZ	RI5
	CALL	DELAY
	DCR	H
	JNZ	RI4
	JMP	RI1
;
RI5:	IN	PTRI	; GET THE DATA
	ORA	A
	POP	H
	RET
;
RI6:	POP	H
	CPI	RUSE1
	MVI	A,R1LOC
	JZ	@USER
	MVI	A,R2LOC
;
; USER DEFINED I/O ENTRY POINT TRANSFER LOGIC
;
@USER:	PUSH	H	; SAVE HL, CREATE A STACK ENTRY
	LHLD	MEMTOP
	MOV	L,A
	XTHL
	RET
;
; CONSOLE OUTPUT CODE
;
BLK:	MVI	C,' '	; PRINT A BLANK
COM:	LDA	IOBYT
	ANI	NOT CMSK
	CPI	BATCH	; DON'T HONOR BREAK KEY IN BATCH MODE
	CNZ	BREAK	; TEST FOR BREAK KEY
CO:	LDA	IOBYT	; GET STATUS BYTE
	ANI	NOT CMSK	; GET CONSOLE BITS
	JNZ	CO0	; TEST FOR CRT
TTYOUT:	IN	TTS	; CONSOLE = TTY
	ANI	TRDY
	JZ	TTYOUT	; LOOP UNTIL READY
	MOV	A,C
	OUT	TTO	; OUTPUT CHARACTER
	RET
;
CO0:	CPI	BATCH
	JZ	LO
	CPI	CCRT	; CONSOLE = CRT?
	MVI	A,COLOC
	JNZ	@USER	; USER DEFINED CONSOLE OUTPUT
CRTOUT:	IN	CRTS	; CONSOLE = CRT
	ANI	TRDY
	JZ	CRTOUT	; LOOP UNTIL READY
	MOV	A,C
	OUT	CRTO
	RET
;
; CONSOLE OUTPUT OF CONSTANT DATA
;
COMC:	XTHL		; GET RETURN ADDRESS
	MOV	C,M	; GET PARAMETER
	INX	H	; BUMP RETURN ADDRESS
	XTHL
	JMP	COM	; OUTPUT IT
;
; EXTERNALLY REFERENCED ROUTINE
; PUNCH OUTPUT CODE, VALUE EXPECTED IN C
; A, FLAGS, AND C MODIFIED
; STACK USAGE: 2 BYTES
;
POC:	XTHL	; PUNCH A CONSTANT
	MOV	C,M
	INX	H
	XTHL
PO:	LDA	IOBYT	; GET STATUS BYTE
	ANI	NOT PMSK	; GET PUNCH BITS
	JZ	TTYOUT	; NO, PUNCH = TTY
	CPI	PPTP	; TEST FOR PTP
	JNZ	PO1	; TEST FOR USER DEVIVE(S)
PO0:	IN	PTPS	; GET STATUS
	ANI	PTPRY	; CHECK STATUS
	JZ	PO0	; LOOP UNTIL READY
	MOV	A,C
	OUT	PTPO
	MVI	A,PTPADV	; START PUNCH
	OUT	PTPC
	RET
;
PO1:	CPI	PUSE1
	MVI	A,P1LOC
	JZ	@USER	; USER DEFINED PUNCH 1
	MVI	A,P2LOC
	JMP	@USER	; USER DEFINED PUNCH 2
;
; EXTERNALLY REFERENCED ROUTINE
; LIST OUTPUT CODE
; VALUE EXPECTED IN C, A AND FLAGS MODIFIED
; STACK USAGE: 2 BYTES
;
LOM:	LDA	IOBYT
	ANI	NOT CMSK
	CPI	BATCH	; DON'T HONOR BREAK KEY IN BATCH MODE
	CNZ	BREAK	; TEST FOR BREAK KEY
LO:	LDA	IOBYT	; GET STATUS BYTE
	ANI	NOT LMSK	; GET STATUS BITS
	JZ	TTYOUT	; LIST = TTY
	CPI	LCRT
	JZ	CRTOUT	; LIST = CRT
	CPI	LUSE	; TEST FOR USER DEFINED LIST DEVICE
	MVI	A,L1LOC
	JZ	@USER	; USER DEFINED LIST DEVICE
LP0:	IN	LPTS
	ANI	LPTRY
	JZ	LP0
	MOV	A,C
	CMA
	OUT	LPTO
	RET
;
; EXTERNALLY REFERENCED ROUTINE
; CONSOLE INPUT STATUS CODE
; A, FLAGS MODIFIED
; STACK USAGE: 2 BYTES
;
CSTS:	LDA	IOBYT	; GET STATUS BYTE
	ANI	NOT CMSK	; CONSOLE = TTY?
	JNZ	CS0	; CONSOLE = CRT
	IN	TTS	; GET TTY STATUS
	JMP	CS1
;
CS0:	CPI	CCRT
	JNZ	CS3
	IN	CRTS	; GET CRT STATUS
CS1:	ANI	RBR
	MVI	A,FALSE	; RETURN FALSE IF NO DATA AVAILABLE
CS2:	RZ
	CMA
	RET
;
CS3:	CPI	BATCH
	MVI	A,TRUE
	JZ	CS2
	MVI	A,CSLOC	; USER DEFINE CONSOLE STATUS
	JMP	@USER
;
; EXTERNALLY REFERENCED ROUTINE
; I/O SYSTEM STATUS CODE
; STATUS BYTE RETURNED IN A
; STACK USAGE: 2 BYTES
;
IOCHK:	LDA	IOBYT	; GET STATUS BYTE
	RET
;
; EXTERNALLY REFERENCED ROUTINE
; SET I/O CONFIGURATION
; VALUE EXPECTED IN C, A MODIFIED
; STACK USAGE: 2 BYTES
;
IOSET:	MOV	A,C
	STA	IOBYT	; PUT NEW BYTE IN MEMORY
	RET
;
; EXTERNALLY REFERENCED ROUTINE
; RETURN ADDRESS OF END OF MEMORY TO USER
; VALUE RETURNED IN (B,A)
; STACK USAGE: 8 BYTES
;
MEMCK:	LDA	MEMTOP+1
	DCR	A
	MOV	B,A
	MVI	A,USER
	RET
;
; EXTERNALLY REFERENCED ROUTINE
; DEFINE USER I/O ENTRY POINTS
; SELECTION CODE IN C
; ENTRY POINT IN D,E
; A,FLAGS MODIFIED
; STACK USAGE 8 BYTES
;
IODEF:	PUSH	H
	PUSH	B
	LHLD	MEMTOP
	MVI	L,XTBL+1 AND 0FFH
	MOV	A,C
	CPI	UCS+1
	JNC	ERROR	; INVALID SELECTION CODE
	ADD	C	; DOUBLE INDEX
	ADD	C	; TRIPLE INDEX
	MOV	C,A
	MVI	B,0
	DAD	B	; COMPUTE LOCATION OF I/O OPERAND
	MOV	M,E	; STORE BRANCH OPERAND
	INX	H
	MOV	M,D
	POP	B
	POP	H
	RET
;
; END I/O SUBROUTINES
;
;
; BEGIN MONITOR SUBROUTINES
;
; READ TWO ASCII CHARACTERS, DECODE INTO 8 BITS
;
BYTE:	PUSH	B
	CALL	RIX
	CALL	NIBBLE
	RLC
	RLC
	RLC
	RLC
	MOV	C,A
	CALL	RIX
	CALL	NIBBLE
	ORA	C
	MOV	C,A
	ADD	D
	MOV	D,A
	MOV	A,C
	POP	B
	RET
;
; CONVERT 4 BIT HEX VALUE TO ASCII CHARACTER
;	INPUT = 0...0FH
;	OUTPUT = 30H...39H,41H...46H
;
CONV:	ANI	0FH
	ADI	90H
	DAA
	ACI	40H
	DAA
	MOV	C,A
	RET
;
; TYPE A CARRAIGE RETURN AND LINE FEED ON CONSOLE
;
CRLF:	CALL	COMC
	DCR	C
	CALL	COMC
	LDAX	B
	RET
;
; PRINT CONTENTS OF HL IN HEX ON LIST DEVICE
;
DADR:	MOV	A,H
	CALL	DBYTE
	MOV	A,L
	JMP	DBYTE
;
; LIST A BYTE ON THE LIST DEVICE AS 2 ASCII CHARACTERS
;
DBYTE:	PUSH	PSW	; SAVE A COPY OF A
	RRC
	RRC
	RRC
	RRC
	CALL	CONV
	CALL	LOM
	POP	PSW	; RETRIEVE ORIGINAL VALUE
	CALL	CONV
	JMP	LOM
;
; 1.0 MS DELAY
;
DELAY:	IN	0FFH
	ANI	1
	JZ	DELAY
	RET
;
; DISPLAY THE CONTENTS OF A USER REGISTER
;
DREG:	INX	H
	MOV	E,M
	LDA	5
	MOV	D,A
	INX	H
	MOV	B,M
	INX	H
	LDAX	D
	CALL	LBYTE
	DCR	B
	RM
	DCX	D
	LDAX	D
	JMP	LBYTE
;
; THE C REGISTER CONTAINS THE NUMBER OF PARAMETERS REQUIRED
; (1,2, OR 3). PARAMETERS ARE RETURNED ON THE STACK
;
EXPR:	CALL	PARAM
	XTHL
	PUSH	H
	DCR	C
	JNC	EX0
	JNZ	ERROR
	RET
;
EX0:	JNZ	EXPR
	JMP	ERROR
;
; COMPARE HL WITH DE
; IF HL <= DE THEN CARRY = 0
; IF HL >  DE THEN CARRY = 1
;
HILO:	INX	H	; BUMP HL
	MOV	A,H	; TEST FOR HL = 0
	ORA	L
	STC
	RZ
	MOV	A,E	; DE - HL, SET/RESET CARRY
	SUB	L
	MOV	A,D
	SBB	H
	RET
;
; CONVERT NIBBLE IN A-REGISTER TO A ASCII IN A-REGISTER
; AND PRINT ON CONSOLE DEVICE
;
HXD:	CALL	CONV
	JMP	COM
;
; PRINT CONTENTS OF HL IN HEX ON CONSOLE DEVICE
;
LADR:	MOV	A,H	; PRINT MSB
	CALL	LBYTE
	MOV	A,L	; PRINT LSB
	JMP	LBYTE
;
; LIST A BYTE AS 2 ASCII CHARACTERS
;
LBYTE:	PUSH	PSW	; SAVE A COPY IN A
	RRC
	RRC
	RRC
	RRC
	CALL	HXD
	POP	PSW	; RETRIEVE ORIGINAL VALUE
	JMP	HXD
;
; PRINT CR, LF ON LIST DEVICE
;
LCRLF:	MVI	C,CR
	CALL	LOM
	MVI	C,LF
	JMP	LOM
;
; PUNCH 6 INCHES OF LEADER
;
LEAD:	MVI	B,60	; SET TO PUNCH 6 INCHES OF NULLS
LE0:	CALL	POC
	NOP
	DCR	B
	JNZ	LE0
	RET
;
; COLLECT A HEXIDECIMAL PARAMETER
;
PARAM:	CALL	PCHK
	JZ	ERROR
PA0:	LXI	H,0
PA1:	MOV	B,A
	CALL	NIBBLE
	JC	PA2
	DAD	H
	DAD	H
	DAD	H
	DAD	H
	ORA	L
	MOV	L,A
	CALL	TI
	JMP	PA1
;
PA2:	MOV	A,B
	CALL	P2C
	JNZ	ERROR
	RET
;
; DECODE ASCII CHAR IN A-REGISTER INTO HEX DIGIT IN A-REGISTER
; FILTER OUT ALL CHARACTERS NOT IN THE SEQUENCE (0...9,A...F),
; RETURN CARRY = 1 FOR ILLEGAL CHARACTERS
;
NIBBLE:	SUI	'0'
	RC		; FILTER OUT 0-2FH
	ADI	'0'-'G'
	RC		; FILTER OUT 47H-0FFH
	ADI	6
	JP	NI0	; TAKE BRANCH FOR A-F
	ADI	7
	RC		; FILTER OUT 3AH-40H
NI0:	ADI	10
	ORA	A	; CLEAR ERROR FLAG
	RET
;
; PUNCH CONTENTS OF HL IN HEX ON PUNCH DEVICE
;
PADR:	MOV	A,H
	CALL	PBYTE
	MOV	A,L
	JMP	PBYTE
;
; PUNCH A BYTE AS 2 ASCII CHARACTERS
;
PBYTE:	MOV	E,A	; SAVE VALUE
	RRC
	RRC
	RRC
	RRC
	CALL	CONV
	CALL	PO
	MOV	A,E
	CALL	CONV
	CALL	PO
	MOV	A,E
	ADD	D
	MOV	D,A
	RET
;
; TEST FOR NULL INPUT PARAMETER
;
PCHK:	CALL	TI	; GET A CHARACTER
P2C:	CPI	' '
	RZ
	CPI	','
	RZ
	CPI	CR
	STC
	RZ
	CMC
	RET
;
; PUNCH CR,LF
;
PEOL:	CALL	POC
	DB	CR
	CALL	POC
	DB	LF
	RET
;
; RESTART 2 CODE, (PROGRAMMED BREAKPOINT)
;
RESTART:	DI
	PUSH	H
	PUSH	D
	PUSH	B
	PUSH	PSW
	POP	H
	IN	0FCH
	PUSH	PSW
	PUSH	H
	MVI	A,NOT INT0
	OUT	0FCH
	LHLD	MEMTOP
	MVI	L,EXIT AND 0FFH
	XCHG
	LXI	H,12
	DAD	SP
	MVI	B,5
	XCHG
RST0:	DCX	H
	MOV	M,D
	DCX	H
	MOV	M,E
	POP	D
	DCR	B
	JNZ	RST0
	POP	B
	DCX	B
	SPHL
	LHLD	MEMTOP
	MVI	L,TLOC AND 0FFH
	MOV	A,M
	SUB	C
	INX	H
	JNZ	RSTA
	MOV	A,M
	SBB	B
	JZ	RST1
RSTA:	INX	H
	INX	H
	MOV	A,M
	SUB	C
	INX	H
	JNZ	RSTB
	MOV	A,M
	SBB	B
	JZ	RST1
RSTB:	MVI	A,EOI
	OUT	0FDH
	INX	B
RST1:	LHLD	MEMTOP
	MVI	L,LLOC
	MOV	M,E
	INX	H
	MOV	M,D
	MVI	L,PLOC-1
	MOV	M,C
	INX	H
	MOV	M,B
	PUSH	B
	CALL	COMC
	DB	'#'
	POP	H
	CALL	LADR
	LHLD	MEMTOP
	MVI	L,TLOC AND 0FFH
	MVI	D,2
RST2:	MOV	C,M
	XRA	A
	MOV	M,A
	INX	H
	MOV	B,M
	MOV	M,A
	INX	H
	MOV	A,C
	ORA	B
	JZ	RST3
	MOV	A,M
	STAX	B
RST3:	INX	H
	DCR	D
	JNZ	RST2
	JMP	START
;
; GET CHARACTER FROM READER, MASK OFF PARITY BIT
;
RIX:	CALL	RI
	JC	ERROR	; READER TIMEOUT ERROR
	ANI	7FH
	RET
;
; I/O CONTROL ENTRY POINT (LATER EXPANSION)
;
SPCL:	RET
;
; INPUT FROM CONSOLE, ECHOED AND RETURNED IN A
;
TI:	PUSH	B
	CALL	CI
	ANI	7FH	; MASK PARITY
	CALL	UC	; FORCE UPPER CASE
	CPI	3	; TEST FOR BREAK
	JZ	ERROR	; ABORT COMMAND
	MOV	C,A
	CALL	CO	; ECHO
	MOV	A,C
	POP	B
	RET
;
UC:	CPI	'A'+20H
	RM		; CHAR < LC(A)
	CPI	'Z'+20H+1
	RP		; CHAR > LC(Z)
	ANI	NOT 20H	; FORCE UPPER CASE
	RET
;
	END
;
            