(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified On Tue Jun 16 13:12:42 PDT 1992 by muller *)
(*      modified On Thu Jul 18 11:26:00 PDT 1991 by meehan *)
(*      Modified On Tue Dec 18 09:18:23 1990 by jdd *)

(* Management of VT intervals. *)

MODULE VTInterval;

IMPORT VTReal;
(* exported operations *)


PROCEDURE New
  (vt: T; hl, hr: Index; READONLY options: IntervalOptions): Interval
   RAISES {} =
  VAR interval: Interval;
  BEGIN
    interval :=  NEW (Interval);
    interval.vt := vt;
    interval.l := hl;
    interval.r := hr;
    interval.options := options;
    interval.state := OnOffState.Off;
    Insert (interval);
    RETURN interval;
  END New;

PROCEDURE MakeOptions
  (style: IntervalStyle; whiteBlack, whiteStroke: ColorScheme; leading: Tint)
   : IntervalOptions RAISES {} =
  VAR options: IntervalOptions;
  BEGIN
    options.style := style;
    options.whiteBlack := whiteBlack;
    options.whiteStroke := whiteStroke;
    options.leading := leading;
    RETURN options;
  END MakeOptions;

PROCEDURE Switch (interval: Interval; state: OnOffState) RAISES {} =
  BEGIN
    IF interval.state # state THEN
      Invalidate (interval.vt, interval.l, interval.r);
      interval.state := state;
    END;
  END Switch;

PROCEDURE Move (interval: Interval; hl, hr: Index) RAISES {} =
  VAR oldLeft, oldRight, newLeft, newRight: I;
  BEGIN
    oldLeft := interval.l;
    oldRight := interval.r;
    newLeft := hl;
    newRight := hr;
    IF (newLeft = oldLeft) AND (newRight = oldRight) THEN RETURN ;  END;
    interval.l := newLeft;
    interval.r := newRight;
    IF (interval.state = OnOffState.On)
      AND (interval.options.style # IntervalStyle.NoStyle) THEN
      IF (newLeft >= oldRight) OR (newRight <= oldLeft) THEN
        Invalidate (interval.vt, oldLeft, oldRight);
        Invalidate (interval.vt, newLeft, newRight);
      ELSE
        IF newLeft > oldLeft THEN
          Invalidate (interval.vt, oldLeft, newLeft);
        ELSIF newLeft < oldLeft THEN
          Invalidate (interval.vt, newLeft, oldLeft);
        END;
        IF newRight > oldRight THEN
          Invalidate (interval.vt, oldRight, newRight);
        ELSIF newRight < oldRight THEN
          Invalidate (interval.vt, newRight, oldRight);
        END;
      END;
    END;
  END Move;

PROCEDURE ChangeOptions
  (interval: Interval; READONLY options: IntervalOptions) RAISES {} =
  BEGIN
    IF interval.state = OnOffState.On THEN
      Invalidate (interval.vt, interval.l, interval.r);
    END;
    interval.options := options;
  END ChangeOptions;

PROCEDURE Close (interval: Interval) RAISES {} =
  VAR vt: T;
  BEGIN
    vt := interval.vt;
    IF vt = NIL THEN RETURN ;  END;
    Switch (interval, OnOffState.Off);
    Remove (interval);
  END Close;
(* internal VT operations *)
(* Fix bubble-sorts the intervals into order by start. *)

PROCEDURE Fix (vt: T) RAISES {} =
  VAR i, ii, iii: Interval; needScan: BOOLEAN;
  BEGIN
    i := vt.intervals;
    needScan := TRUE;
    WHILE needScan DO
      needScan := FALSE;
      i := vt.intervals;
      ii := NIL;
      iii := NIL;
      WHILE i # NIL DO
        IF (ii # NIL) AND (ii.l > i.l) THEN
          IF iii = NIL THEN
            vt.intervals := i;
            ii.next := i.next;
            i.next := ii;
          ELSE
            iii.next := i;
            ii.next := i.next;
            i.next := ii;
          END;
          needScan := TRUE;
          iii := i;
          i := ii.next;
        ELSE
          iii := ii;
          ii := i;
          i := i.next;
        END;
      END;
    END;
  END Fix;

PROCEDURE CurrentOptions
  (view: View; at: I; VAR (*OUT*) from, to: I): IntervalOptions RAISES {} =
  VAR interval: Interval; opt: IntervalOptions;
  BEGIN
    opt.style := IntervalStyle.NoStyle;
    from := 0;
    to := view.vt.length;
    interval := view.vt.intervals;
    WHILE interval # NIL DO
      WITH z_24 = interval^ DO
        IF z_24.state = OnOffState.On THEN
          IF (z_24.l <= at) THEN from := MAX (z_24.l, from);  END;
          IF (z_24.r <= at) THEN from := MAX (z_24.r, from);  END;
          IF (at < z_24.l) THEN to := MIN (z_24.l, to);  END;
          IF (at < z_24.r) THEN to := MIN (z_24.r, to);  END;
          IF (z_24.l <= at) THEN
            IF (at < z_24.r) THEN
              IF opt.style = IntervalStyle.NoStyle THEN
                opt := z_24.options;
              ELSIF z_24.options.style = IntervalStyle.NoStyle THEN
              ELSIF (opt.style = IntervalStyle.SlugStyle)
                OR (opt.style = IntervalStyle.OverlapStyle) THEN
              ELSIF (z_24.options.style = IntervalStyle.SlugStyle)
                OR (z_24.options.style = IntervalStyle.OverlapStyle) THEN
                opt := z_24.options;
              ELSIF view.vOptions.intervalStylePrecedence # NIL THEN
                IF
                  view.vOptions.intervalStylePrecedence[opt.style,
                                                        z_24.options.style]
                  THEN
                ELSIF
                  view.vOptions.intervalStylePrecedence[z_24.options.style,
                                                        opt.style] THEN
                  opt := z_24.options;
                ELSE
                  opt.style := IntervalStyle.OverlapStyle;
                END;
              ELSE
                opt.style := IntervalStyle.OverlapStyle;
              END;
            END;
          ELSE
            RETURN opt;
          END;
        END;
        interval := z_24.next;
      END;
    END;
    RETURN opt;
  END CurrentOptions;
(* Internal procedures to manipulate the list of intervals. *)

PROCEDURE Insert (interval: Interval) RAISES {} =
  BEGIN
    WITH z_25 = interval^ DO
      z_25.next := z_25.vt.intervals;
      z_25.vt.intervals := interval;
    END;
  END Insert;

PROCEDURE Remove (interval: Interval) RAISES {} =
  VAR i: Interval;
  BEGIN
    WITH z_26 = interval^ DO
      i := z_26.vt.intervals;
      IF i = interval THEN
        z_26.vt.intervals := i.next;
      ELSE
        WHILE i.next # interval DO i := i.next;  END;
        i.next := i.next.next;
      END;
      interval.next := NIL;
    END;
  END Remove;
(* utility *)
(************************************************************************)
(*		              (Utility) 				*)
(************************************************************************)

PROCEDURE Invalidate (vt: T; b, e: I) RAISES {} =
  BEGIN
    VTReal.Change (vt, b, e, e);
  END Invalidate;

BEGIN
END VTInterval.
