(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Fri Jul  3 22:04:08 PDT 1992 by meehan *)
(*      modified on Tue Jun 16 21:55:35 PDT 1992 by muller *)

MODULE XParam;

IMPORT Axis, FlexShape, Fmt, Point, Rect, Text, Trestle, TrestleComm, VBT;

PROCEDURE ParseDisplay (t: TEXT): DisplayRec RAISES {Error} =
  VAR
    n                           := Text.Length (t);
    i, start: CARDINAL          := 0;
    z       : DisplayRec;
    buf     : REF ARRAY OF CHAR;
  PROCEDURE Err (spec: TEXT; index: CARDINAL) RAISES {Error} =
    BEGIN
      RAISE Error (NEW (DisplayInfo, spec := spec, index := index))
    END Err;
  PROCEDURE scan () RAISES {Error} =
    BEGIN
      IF n = 0 THEN Err (t, 0) END;
      buf := NEW (REF ARRAY OF CHAR, n);
      Text.SetChars (buf^, t);
      WHILE buf [i] # ':' DO INC (i); IF i = n THEN Err (t, 0) END END;
      IF i = n THEN Err (t, 0) END;
      (* IF buf [i] # ':' THEN Err (t, i) END; *)
      IF i # 0 THEN
        z.hostname := Text.FromChars (SUBARRAY (buf^, 0, i))
      END;
      INC (i);
      IF i = n THEN Err (t, i - 1) END;
      IF buf [i] = ':' THEN
        z.DECnet := TRUE;
        INC (i);
        IF i = n THEN Err (t, i - 1) END
      END;
      start := i;
      z.display := num (i, n, buf);
      IF i = start THEN Err (t, start) END;
      IF i = n THEN RETURN END;
      IF buf [i] # '.' THEN Err (t, i) END;
      INC (i);
      IF i = n THEN Err (t, i - 1) END;
      start := i;
      z.screen := num (i, n, buf);
      IF i = start THEN Err (t, start) END;
      IF i # n THEN Err (t, i) END;
    END scan;
  BEGIN
    scan ();
    RETURN z
  END ParseDisplay;

PROCEDURE DisplayText (READONLY d: DisplayRec): TEXT =
  CONST colons = ARRAY BOOLEAN OF TEXT {":", "::"};
  BEGIN
    RETURN Fmt.F ("%s%s%s.%s", d.hostname, colons [d.DECnet],
                  Fmt.Int (d.display), Fmt.Int (d.screen))
  END DisplayText;
  
PROCEDURE ParseGeometry (t: TEXT): GeoRec RAISES {Error} =
  CONST
    VertexMap = ARRAY BOOLEAN, BOOLEAN OF
                  Rect.Vertex {
                  ARRAY BOOLEAN OF
                    Rect.Vertex {Rect.Vertex.SE, Rect.Vertex.NE},
                  ARRAY BOOLEAN OF
                    Rect.Vertex {Rect.Vertex.SW, Rect.Vertex.NW}};
  VAR
    width, height: INTEGER;
    x, y                             := 0;
    i, start     : CARDINAL          := 0;
    n                                := Text.Length (t);
    buf          : REF ARRAY OF CHAR;
    xplus, yplus                     := TRUE;
  PROCEDURE Err (spec: TEXT; index: CARDINAL) RAISES {Error} =
    BEGIN
      RAISE Error (NEW (GeometryInfo, spec := spec, index := index))
    END Err;
  PROCEDURE scan () RAISES {Error} =
    BEGIN
      IF n = 0 THEN Err (t, 0) END;
      buf := NEW (REF ARRAY OF CHAR, n);
      Text.SetChars (buf^, t);
      width := num (i, n, buf);
      IF i = 0 THEN width := VBT.DefaultShape.hi END;
      IF i = n THEN RETURN END;
      IF buf [i] = 'x' THEN
        INC (i);
        start := i;
        height := num (i, n, buf);
        IF start = i THEN Err (t, start) END
      ELSE
        height := VBT.DefaultShape.hi
      END;
      IF i = n THEN RETURN END;
      IF buf [i] # '+' AND buf [i] # '-' THEN Err (t, i) END;
      INC (i);
      start := i;
      x := num (i, n, buf);
      IF i = start THEN Err (t, start) END;
      IF buf [start - 1] = '-' THEN x := -x; xplus := FALSE END;
      IF i = n THEN RETURN END;
      IF buf [i] # '+' AND buf [i] # '-' THEN Err (t, i) END;
      INC (i);
      start := i;
      y := num (i, n, buf);
      IF i = start OR i # n THEN Err (t, start) END;
      IF buf [start - 1] = '-' THEN y := -y; yplus := FALSE END;
    END scan;
  BEGIN
    scan ();
    RETURN GeoRec {VertexMap [xplus, yplus], Point.T {x, y},
                   Point.T {width, height}}
  END ParseGeometry;

PROCEDURE GeometryText (READONLY g: GeoRec): TEXT =
  CONST
    xplus = ARRAY Rect.Vertex OF TEXT {"+", "-", "+", "-"};
    yplus = ARRAY Rect.Vertex OF TEXT {"+", "+", "-", "-"};
  BEGIN
    RETURN Fmt.Int (g.size.h) & "x" & Fmt.Int (g.size.v) & xplus [g.vertex]
             & Fmt.Int (ABS (g.dp.h)) & yplus [g.vertex]
             & Fmt.Int (ABS (g.dp.v))
  END GeometryText;
  
PROCEDURE num (VAR i: CARDINAL; n: CARDINAL; buf: REF ARRAY OF CHAR):
  CARDINAL =
  CONST DIGITS = SET OF CHAR {'0'.. '9'};
  VAR v: CARDINAL := 0;
  BEGIN
    LOOP
      IF i = n OR NOT buf [i] IN DIGITS THEN RETURN v END;
      v := 10 * v + ORD (buf [i]) - ORD ('0');
      INC (i)
    END
  END num;

PROCEDURE Position (         trsl: Trestle.T;
                             id  : Trestle.ScreenID;
                    READONLY g   : GeoRec            ): Rect.T
  RAISES {TrestleComm.Failure} =
  VAR z: Rect.T;
  BEGIN
    WITH array = Trestle.GetScreens (trsl) DO
      IF array = NIL THEN RAISE TrestleComm.Failure END;
      FOR i := FIRST (array^) TO LAST (array^) DO
        IF array [i].id = id THEN
          WITH s = array [i].dom DO
            CASE g.vertex OF
            | Rect.Vertex.NW =>
                z.north := s.north + g.dp.v;
                z.west := s.west + g.dp.h
            | Rect.Vertex.SW =>
                z.north := s.south + g.dp.v - g.size.v;
                z.west := s.west + g.dp.h
            | Rect.Vertex.NE =>
                z.north := s.north + g.dp.v;
                z.west := s.east + g.dp.h - g.size.h
            | Rect.Vertex.SE =>
                z.north := s.south + g.dp.v - g.size.v;
                z.west := s.east + g.dp.h - g.size.h
            END;
            z.south := z.north + g.size.v;
            z.east := z.west + g.size.h;
            RETURN z
          END
        END
      END
    END;
    RAISE TrestleComm.Failure
  END Position;

PROCEDURE PrefShape (         trsl: Trestle.T;
                              id  : Trestle.ScreenID;
                     READONLY g   : GeoRec;
                     shrink, stretch := Point.T {0, 0}): FlexShape.Shape
  RAISES {TrestleComm.Failure} =
  VAR hNatural, vNatural: REAL;
  BEGIN
    WITH array = Trestle.GetScreens (trsl) DO
      IF array # NIL THEN
        FOR i := FIRST (array^) TO LAST (array^) DO
          IF array [i].id = id THEN
            WITH res  = array [i].type.res,
                 hRes = res [Axis.T.Hor],
                 vRes = res [Axis.T.Ver]    DO
              IF g.size.h >= VBT.DefaultShape.hi THEN
                hNatural := FlexShape.Missing
              ELSE
                hNatural := PixelsToPts (g.size.h, hRes)
              END;
              IF g.size.v >= VBT.DefaultShape.hi THEN
                vNatural := FlexShape.Missing
              ELSE
                vNatural := PixelsToPts (g.size.v, hRes)
              END;
              RETURN
                FlexShape.Shape {
                  FlexShape.SizeRange {hNatural, PixelsToPts (shrink.h, hRes),
                                       PixelsToPts (stretch.h, hRes)},
                  FlexShape.SizeRange {vNatural, PixelsToPts (shrink.v, vRes),
                                       PixelsToPts (stretch.v, vRes)}}
            END                 (* WITH *)
          END                   (* IF *)
        END                     (* FOR *)
      END                       (* IF *)
    END;                        (* WITH *)
    RAISE TrestleComm.Failure
  END PrefShape;


PROCEDURE PixelsToPts (pixels: CARDINAL; pixelsPerMM: REAL): REAL =
  CONST
    PointsPerInch = 72.0;
    MMPerInch     = 25.4;
  BEGIN
    RETURN FLOAT (pixels) / pixelsPerMM / MMPerInch * PointsPerInch
  END PixelsToPts;


BEGIN
END XParam.
