(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)
(*                                                             *)
(* Last modified on Sun Jul 19 23:05:43 PDT 1992 by meehan     *)
(*      modified on Tue Jun 16 21:55:35 PDT 1992 by muller     *)

(* This interface provides utilities for handling the "display"
   and "geometry" command-line arguments for programs that use
   an X server.  *)
   
INTERFACE XParam;

IMPORT FlexShape, Point, Rect, Trestle, TrestleComm;

TYPE
  DisplayRec = RECORD
                 hostname       : TEXT     := "";
                 display, screen: CARDINAL := 0;
                 DECnet         : BOOLEAN  := FALSE
               END;
  GeoRec = RECORD
             vertex: Rect.Vertex;  (* corner for displacement *)
             dp    : Point.T;      (* displacement *)
             size  : Point.T;      (* width, height *)
           END;

PROCEDURE ParseDisplay (spec: TEXT): DisplayRec RAISES {Error};
(* This parses a "display" parameter, e.g.,
   "myrtle.pa.dec.com:0.2". *)

PROCEDURE ParseGeometry (spec: TEXT): GeoRec RAISES {Error};
(* This parses a "geometry" parameter, e.g., 
   "1024x800-0-10" is parsed as
|     GeoRec {Rect.Vertex.SE,
|             Point.T {0, -10},
|             Point.T {1024, 80}}
   The width and height default to "VBT.DefaultShape.hi".  The
   horizontal and vertical displacements default to "0"; note
   that they are signed, even though that information is also
   represented by the vertex.  *)

PROCEDURE DisplayText (READONLY d: DisplayRec): TEXT;
PROCEDURE GeometryText (READONLY g: GeoRec): TEXT;
(* These return the text-versions of the records. *)

PROCEDURE Position (         trsl: Trestle.T;
                             id  : Trestle.ScreenID;
                    READONLY g   : GeoRec            ): Rect.T
  RAISES {TrestleComm.Failure};
(* This computes the rectangle specified by "g" in the screen
   coordinates for the screenID "id" on the window system
   connected to "trsl".  If "trsl" has no screen whose id is
   "id", the exception is raised.  See example below. *)

PROCEDURE PrefShape (         trsl: Trestle.T;
                              id  : Trestle.ScreenID;
                     READONLY g   : GeoRec;
                     shrink, stretch := Point.T {0, 0}):
  FlexShape.Shape RAISES {TrestleComm.Failure};
(* This returns a FlexShape whose "natural" field is specified
   by "g", using the pixel-resolution given by the
   "VBT.ScreenType" of the screen whose ScreenID is "id" on the
   window system connected to "trsl".  If "trsl" has no screen
   whose id is "id", the exception is raised.  The "shrink" and
   "stretch" values default to 0, e.g.,
| (Width  <g.size.h> + <stretch.h> - <shrink.h>)
| (Height <g.size.v> + <stretch.v> - <shrink.v>)
   To get the effect of "infinite" shrink or stretch, use
   "VBT.DefaultShape.hi". *)

TYPE
  Info = OBJECT
           spec : TEXT;
           index: CARDINAL
         END;
  DisplayInfo = Info BRANDED OBJECT END;
  GeometryInfo = Info BRANDED OBJECT END;

EXCEPTION Error(Info);
(* Parsing errors are reported with the text ("spec") and
   position ("index") of the first illegal character in the
   text. *)

END XParam.

(* Here is an example of intended use of this interface:
|    VAR Dspec := GetParameter ("-display");
|        Gspec := GetParameter ("-geometry");
|        d     := ParseDisplay (Dspec);
|        id    := d.screen;
|        g     := ParseGeometry (Gspec);
|        trsl  := Trestle.Connect (Dspec);
|     BEGIN
|      Trestle.Attach (vbt, trsl);
|      Trestle.Decorate (vbt, ...);
|      Trestle.Overlap (
|        vbt, id,
|        Rect.NorthWest (Position (trsl, id, g));
|      ...         `Check for defaults in` g.size
|      FlexVBT.Set (FormsVBT.GetVBT (vbt, "topShape"),
|                   PrefShape (trsl, id, g));
|      ...
*)
