(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Tue Jun 16 13:16:20 PDT 1992 by muller                       *)
(*      modified on Fri Jun 12 18:02:58 1992 by mhb                          *)

MODULE AsciiUtil;

IMPORT Rd, Scan, Text, Thread;
         
<* FATAL Rd.Failure *>

PROCEDURE StripPrefix (rd: Rd.T; prefix: Text.T): Text.T
  RAISES {Thread.Alerted, Error} =
  VAR
    prefixLen : INTEGER;
    line, head: Text.T;
  BEGIN
    LOOP
      TRY line := Rd.GetLine(rd) EXCEPT Rd.EndOfFile => RAISE Error END;
       IF Text.Length(line) > 0 THEN EXIT END;
    END;
    prefixLen := Text.Length(prefix);
    head := Text.Sub(line, 0, prefixLen);
    IF NOT Text.Equal(head, prefix) THEN RAISE Error END;
    FOR i := prefixLen TO Text.Length(line) - 1 DO
      IF Text.GetChar(line, i) # ' ' THEN
        RETURN Text.Sub(line, i, LAST(CARDINAL))
      END
    END;
    RETURN Text.Sub(line, prefixLen, LAST(CARDINAL))
  END StripPrefix;

PROCEDURE SkipLeadingBlanks (line: Text.T): Text.T =
  BEGIN
    RETURN Text.Sub(line, 1 + Text.FindChar(line, ' '), LAST(CARDINAL))
  END SkipLeadingBlanks;

PROCEDURE ScanLine (rd: Rd.T): Text.T RAISES {Thread.Alerted, Error} =
  BEGIN 
    RETURN StripPrefix(rd, "")
  END ScanLine;
  
PROCEDURE ScanRInt (rd: Rd.T): INTEGER
  RAISES {Thread.Alerted, Scan.BadFormat, Error} =
  VAR
    tail  := StripPrefix(rd, "");
    index := Text.FindCharR(tail, ' ');
  BEGIN
    RETURN Scan.Int(Text.Sub(tail, index + 1, LAST(CARDINAL)))
  END ScanRInt;

PROCEDURE GetReal (rd: Rd.T; prefix: Text.T; VAR val: REAL)
  RAISES {Thread.Alerted, Scan.BadFormat, Error} =
  VAR tail := StripPrefix(rd, prefix);
  BEGIN
    val := Scan.Real(tail)
  END GetReal;

PROCEDURE GetInt (rd: Rd.T; prefix: Text.T; VAR val: INTEGER)
  RAISES {Thread.Alerted, Scan.BadFormat, Error} =
  VAR tail := StripPrefix(rd, prefix);
  BEGIN
    val := Scan.Int(tail)
  END GetInt;

PROCEDURE Get2Int (rd: Rd.T; prefix: Text.T; VAR v1, v2: INTEGER)
  RAISES {Thread.Alerted, Scan.BadFormat, Error} =
  VAR tail := StripPrefix(rd, prefix);
  BEGIN
    v1 := Scan.Int(tail);
    v2 := Scan.Int(SkipLeadingBlanks(tail));
  END Get2Int;

PROCEDURE Get4Int (rd: Rd.T; prefix: Text.T; VAR v1, v2, v3, v4: INTEGER)
  RAISES {Thread.Alerted, Scan.BadFormat, Error} =
  VAR tail := StripPrefix(rd, prefix);
  BEGIN
    v1 := Scan.Int(tail);
    FOR i := 2 TO 4 DO
      tail := SkipLeadingBlanks(tail);
      IF i = 2 THEN
        v2 := Scan.Int(tail);
      ELSIF i = 3 THEN
        v3 := Scan.Int(tail)
      ELSE
        v4 := Scan.Int(tail)
      END
    END
  END Get4Int;

PROCEDURE GetText (rd: Rd.T; prefix: Text.T; VAR val: Text.T)
  RAISES {Thread.Alerted, Error} =
  BEGIN
    val := StripPrefix(rd, prefix)
  END GetText;

BEGIN
END AsciiUtil.
