(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Mon Aug 10  1:04:43 PDT 1992 by meehan *)
(*      modified on Tue Jun 16 13:07:59 PDT 1992 by muller *)
(*      modified on Mon Jun 15 18:58:44 1992 by mhb *)
(* modified on Wed Feb 26 00:21:41 1992 by steveg*)


(* An "ZChildVBT.T" is a "VBT" that is typically used as a subwindow.
   (A {\em subwindow}\index{subwindow} is a non-background child of
   a "ZSplit".)

   A "ZChildVBT" is a subclass of a "HighlightVBT", thereby
   insulating any highlighting done in it against highlighting
   done in other subwindows.  You need to use a "ZBackgroundVBT"
   around the background child in order to insulate highlighting
   in it against the subwindows.

   A "ZChildVBT" has two properties that are specified during
   initialization: a flag to indicate whether the subwindow is initially
   ``mapped'' (displayed); and a list that specifies the default
   location of the subwindow within the parent "ZSplit".  The list
   contains two or four numbers, which may be integers or real numbers.

   If the list contains two numbers, these specify the center of
   the subwindow as "(h,v)".  Otherwise, the list has four
   numbers, and these specify the rectangle of the subwindow, as
   "(west, north, east, south)".

   If the numbers are integers, these specify distance in points
   from the upper left corner of the subwindow.  This distance
   will be maintained regardless of how the window is
   reformatted.  Otherwise, the numbers must be real numbers
   in the range 0.0--1.0, and these specify proportional position
   of the width or height of the window.  Thus the default value,
   "(0.5 0.5)", means that the subwindow is centered in the
   "ZSplit".

   If the two-number form is used, they may be followed by a
   corner specification, one of the symbols "NW", "NE", "SW",
   or "SE".  If a corner specification is given, the position
   applies to that corner of the window rather than to the
   center.  However, the system will not pop up a subwindow with
   its northwest corner north or west of the visible portion
   of the "ZSplit"; it will override the specified position as
   necessary to bring it into view.

   It is legal, but not recommended, to mix integers and reals in
   the same list.

   Finally, in order for the reformatting to meet specifications
   above, the client must call "Inserted" after the "VBT" is
   inserted as a child of a "ZSplit"; the client must call
   "Moved" after the "VBT" is repositioned by the user; and
   the client must call "Grow" after the size of the "VBT" is
   changed by the user.  *)


INTERFACE ZChildVBT;

IMPORT HighlightVBT, List, VBT;

TYPE
  T <: Public;
  Public =
    HighlightVBT.T OBJECT
    METHODS
      init     (ch: VBT.T; open := TRUE; at: List.T := NIL): T;
      callback (READONLY cd: VBT.MouseRec);
    END;

(* The call "v.init(...)" initializes "v" as a "ZChildVBT".  It
   is assumed that "v" will be a subwindow. *)

PROCEDURE InitiallyMapped (vbt: VBT.T): BOOLEAN;
(* ``Should "vbt" be mapped initially?'' This question is
   answered as follows: If "vbt" is a "ZChild", return the value
   of "open" when it was initialzied.  Otherwise, return "TRUE".
   "VBT.Parent(vbt)" must be a subclass of "ZSplit". *)

PROCEDURE Pop (vbt: VBT.T; forcePlace := FALSE);
(* Map "vbt" and lift it to the top of its parent's children.  If
   "forcePlace" is set, position "vbt" at the "at" specification.
   "VBT.Parent(vbt)" must be a subclass of "ZSplit". *)

PROCEDURE Inserted (vbt: VBT.T);
(* Call this procedure after "vbt" has been inserted into a
   "ZSplit".  This procedure sets a "ReshapeControl" object on
   "vbt".  If "vbt" isn't a "ZChild", the "ReshapeControl" tries
   to center "vbt", subject to the contraint that its northwest
   corner is just visible.  If "vbt" is a "ZChild", the
   "ReshapeControl" will follow "vbt.at" until "vbt" is moved by
   the user (signally because "Moved" is called).  At that point,
   the northwest corner of "vbt" will remain at that position
   relative to its parent, until the user moves it again.
   "VBT.Parent(vbt)" must be a sublcass of "ZSplit". *)

PROCEDURE Moved (vbt: VBT.T);
(* Call this procedure after "vbt" has been moved by a user.  If
   "vbt" is a "ZChild", mark that "vbt" has been moved by the
   user so the next time it is reshaped, it will retain its
   current position relative to its parent.  If "vbt" isn't a T,
   this procedure is a no-op.  "VBT.Parent(vbt)" must be a
   subclass of "ZSplit". *)

PROCEDURE Grow (vbt: VBT.T; w, h: INTEGER);
(* Call this procedure after the size of "vbt" has been changed
   to "w"-by-"h" (in pixels) by a user.  If "vbt" is a "ZChild",
   this procedure marks that "vbt" has a new shape and calls
   "VBT.NewShape" to tell the parent "ZSplit".  From here on,
   "vbt" will report its shape as "w"-by-"h".  If "vbt" is not a
   "ZChildVBT", this procedure is a no-op.  "VBT.Parent(vbt)" must
   be a subclass of "ZSplit". *)

END ZChildVBT.



