(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: m3bundle.m3                                           *)
(* Last modified on Thu Sep 24 15:27:34 PDT 1992 by kalsow     *)

(* This module implements the "M3Bundle" command.  See its manpage
   for details. *)

MODULE m3bundle EXPORTS Main;

IMPORT Rd, Wr, FileStream, Params, Thread, Fmt, Stdio, Text;
<* FATAL Wr.Failure, Rd.Failure, Thread.Alerted *>

CONST 
  MaxLineWidth = 75;   (* for readability *)
  MaxBlock     = 2000; (* C limits on a TEXT constant *)

TYPE
  ElementList = REF ARRAY OF Element;
  Element = RECORD
    name   : TEXT;
    path   : TEXT;
    base   : TEXT;
    length : INTEGER;
    blocks : INTEGER;
  END;

VAR
  elts   := NEW (ElementList, 20);
  n_elts := 0;
  module : TEXT := NIL;
  wr     : Wr.T := NIL;
  max_blocks := 0;

(*--------------------------------------------------------- element sizes ---*)

PROCEDURE GetElementSizes (): BOOLEAN =
  VAR rd: Rd.T;  ok := TRUE;
  BEGIN
    FOR i := 0 TO n_elts-1 DO
      WITH z = elts[i] DO
        TRY
          rd := FileStream.OpenRead (z.path);
          z.length := Rd.Length (rd);
          z.blocks := (z.length + MaxBlock - 1) DIV MaxBlock;
          z.base   := "E" & Fmt.Int (i);
          max_blocks := MAX (max_blocks, z.blocks);
          Rd.Close (rd);
        EXCEPT Rd.Failure =>
          wr := Stdio.stderr;
          Out (Params.Get(0), ": cannot read file: ", z.path, "\n");
          ok := FALSE;
        END;
      END;
    END;
    RETURN ok;
  END GetElementSizes;

(*------------------------------------------------------------- interface ---*)

CONST Intf =
  "(* Generated by m3bundle; see its manpage. *)\n\n" &
  "IMPORT Bundle;\n\n" &
  "PROCEDURE Get(): Bundle.T;\n\n";

PROCEDURE WriteInterface () =
  BEGIN
    wr := FileStream.OpenWrite (module & ".i3");
    Out ("INTERFACE ", module, ";\n");
    Out (Intf, "END ", module, ".\n");
    Wr.Close (wr);
  END WriteInterface;

(*---------------------------------------------------------------- module ---*)

CONST Mod_0 = 
  "(* Generated by m3bundle; see its manpage. *)\n" &
  "\n" &
  "IMPORT Bundle, BundleRep, Text;\n";

CONST Mod_1 =
  "IMPORT Thread, Wr, TextWr;\n";

CONST Mod_2 =
  "\n" &
  "TYPE T = Bundle.T OBJECT OVERRIDES get := LookUp END;\n" &
  "\n" &
  "VAR bundle: T := NIL;\n" &
  "\n" &
  "PROCEDURE Get(): Bundle.T =\n" &
  "  BEGIN\n" &
  "    IF (bundle = NIL) THEN bundle := NEW (T) END;\n" &
  "    RETURN bundle;\n" &
  "  END Get;\n" &
  "\n" &
  "PROCEDURE LookUp (<*UNUSED*> self: T;  element: TEXT): TEXT = \n" &
  "  BEGIN\n" &
  "    FOR i := 0 TO LAST (Names)-1 DO\n" &
  "      IF Text.Equal (Names[i], element) THEN\n";

CONST Mod_3 =
  "        IF Elements[i] = NIL THEN Elements[i] := GetElt (i) END;\n";

CONST Mod_4 =
  "        RETURN Elements[i];\n" &
  "      END;\n" &
  "    END;\n" &
  "    RETURN NIL;\n" &
  "  END LookUp;\n" &
  "\n";

CONST Mod_5 =
  "PROCEDURE GetElt (n: INTEGER): TEXT =\n" &
  "  <*FATAL Thread.Alerted, Wr.Failure *>\n" &
  "  VAR wr := TextWr.New ();\n" &
  "  BEGIN\n" &
  "    CASE n OF\n";

CONST Mod_6 =
  "    ELSE (*skip*)\n" &
  "    END;\n" &
  "    RETURN TextWr.ToText (wr);\n" &
  "  END GetElt;\n" &
  "\n";

CONST Mod_7 =
  "\n" &
  "BEGIN\n" &
  "END ";

PROCEDURE WriteModule () =
  BEGIN
    wr := FileStream.OpenWrite (module & ".m3");
    Out ("MODULE ", module, ";\n", Mod_0);
    IF (max_blocks > 1) THEN Out (Mod_1) END;
    Out (Mod_2);
    IF (max_blocks > 1) THEN Out (Mod_3) END;
    Out (Mod_4);
    WriteNames ();
    WriteElements ();
    IF (max_blocks > 1) THEN
      Out (Mod_5);
      WriteGetElt ();
      Out (Mod_6);
    END;
    WriteLiterals ();
    Out (Mod_7, module, ".\n");
    Wr.Close (wr)
  END WriteModule;

PROCEDURE WriteGetElt () =
  BEGIN
    FOR i := 0 TO n_elts-1 DO
      WITH z = elts[i] DO
        IF (z.blocks > 1) THEN
          Out ("    | ", Fmt.Int (i), " =>\n");
          FOR j := 0 TO z.blocks-1 DO
            Out ("        Wr.PutText (wr, ", BlockName (z.base, j), ");\n");
          END;
        END;
      END;
    END;
  END WriteGetElt;

PROCEDURE WriteNames () =
  VAR name: TEXT;
  BEGIN
    Out ("CONST Names = ARRAY [0..", Fmt.Int (n_elts), "] OF TEXT {\n");
    FOR i := 0 TO n_elts-1 DO
      IF (i > 0) THEN Out (",\n") END;
      name := elts[i].name;
      Out ("  \"");
      FOR j := 0 TO Text.Length (name) - 1 DO
        EVAL OutChar (Text.GetChar (name, j));
      END;
      Out ("\"");
    END;
    IF (n_elts > 0) THEN Out (",\n") END;
    Out ("  NIL\n};\n\n");
  END WriteNames;

PROCEDURE WriteElements () =
  BEGIN
    IF (max_blocks > 1)
      THEN Out ("VAR Elements :=");
      ELSE Out ("CONST Elements =");
    END;
    Out (" ARRAY [0..", Fmt.Int (n_elts), "] OF TEXT {\n");
    FOR i := 0 TO n_elts-1 DO
      IF (i > 0) THEN Out (",\n") END;
      WITH z = elts[i] DO
        IF (z.length <= 0) THEN
          Out ("  \"\"");
        ELSIF (z.blocks <= 1) THEN
          Out ("  ", BlockName (z.base, 0));
        ELSE (* fill it in at runtime by calling GetElt *)
          Out ("  NIL (* ", BlockName (z.base, 0), " .. ");
          Out (BlockName (z.base, z.blocks-1), " *)");
        END;
      END;
    END;
    IF (n_elts > 0) THEN Out (",\n") END;
    Out ("  NIL\n};\n\n");
  END WriteElements;

PROCEDURE WriteLiterals () =
  VAR rd: Rd.T;
  BEGIN
    FOR i := 0 TO n_elts-1 DO
      WITH z = elts[i] DO
        rd := FileStream.OpenRead (z.path);
        WriteLiteral (rd, z.base);
        Rd.Close (rd);
      END;
    END;
  END WriteLiterals;

PROCEDURE WriteLiteral (rd: Rd.T;  base: TEXT) =
  <*FATAL Rd.EndOfFile*>
  VAR width, bytes, blocks := 0;  ch: CHAR;
  BEGIN
    WHILE NOT Rd.EOF (rd) DO
      IF (bytes = 0) THEN
        (* start a new block *)
        Out ("CONST ", BlockName (base, blocks), " = \n   \"");
        INC (blocks);
        width := 4;
      ELSIF (width = 0) THEN
        (* start a new line *)
        Out (" & \"");
        width := 4;
      END;

      (* write a character *)
      ch := Rd.GetChar (rd);
      INC (width, OutChar (ch));
      INC (bytes);

      IF (bytes >= MaxBlock) THEN
        (* finish this block *)
        Out ("\";\n\n");
        bytes := 0;
        width := 0;
      ELSIF (width >= MaxLineWidth) THEN
        (* finish this line *)
        Out ("\"\n");
        width := 0;
      END;
    END;

    IF (width > 0) THEN (* finish the last string *) Out ("\"") END;
    IF (bytes > 0) THEN (* finish the last block *)  Out (";\n\n") END;
  END WriteLiteral;

PROCEDURE BlockName (base: TEXT;  block: INTEGER): TEXT =
  BEGIN
    IF (block = 0) THEN RETURN base END;
    RETURN base & "_" & Fmt.Int (block - 1);
  END BlockName;

(*-------------------------------------------------- command line parsing ---*)

PROCEDURE ParseCommandLine (): BOOLEAN =
  VAR next := 0;
  PROCEDURE NextParam (): TEXT =
    BEGIN
      INC (next);
      RETURN Params.Get (next);
    END NextParam;
  BEGIN
    IF ParseOptions (NextParam) THEN RETURN TRUE END;
    wr := Stdio.stderr;
    Out ("usage: ", Params.Get (0), " -name n [-element e path] ...\n");
    RETURN FALSE;
  END ParseCommandLine;
    
PROCEDURE ParseOptions (next_arg: PROCEDURE (): TEXT): BOOLEAN =
  VAR arg: TEXT;
  BEGIN
    LOOP
      arg := next_arg ();
      IF (arg = NIL) THEN
        RETURN TRUE;
      ELSIF Text.Equal (arg, "-name") THEN
        module  := next_arg ();
        IF (module = NIL) THEN RETURN FALSE END;
      ELSIF Text.Equal (arg, "-element") THEN
        IF (n_elts > LAST (elts^)) THEN ExpandElts () END;
        WITH z = elts[n_elts] DO
          z.name := next_arg ();
          z.path := next_arg ();
          IF (z.name = NIL) OR (z.path = NIL) THEN RETURN FALSE END;
        END;
        INC (n_elts);
      ELSIF Text.Equal (Text.Sub (arg, 0, 2), "-F") THEN
        IF NOT ParseOptionFile (Text.Sub (arg, 2, LAST (CARDINAL))) THEN
          RETURN FALSE;
        END;
      ELSE
        RETURN FALSE;
      END;
    END;
  END ParseOptions;

PROCEDURE ParseOptionFile (name: TEXT): BOOLEAN =
  <* FATAL Rd.EndOfFile *>
  VAR f := FileStream.OpenRead (name);  b: BOOLEAN;
  PROCEDURE NextLine (): TEXT =
    BEGIN
      IF Rd.EOF (f) THEN RETURN NIL END;
      RETURN Rd.GetLine (f);
    END NextLine;
  BEGIN
    b := ParseOptions (NextLine);
    Rd.Close(f);
    RETURN b;
  END ParseOptionFile;

PROCEDURE ExpandElts () =
  VAR new := NEW (ElementList, 2 * NUMBER (elts^));
  BEGIN
    FOR i := 0 TO LAST (elts^) DO new[i] := elts[i] END;
    elts := new;
  END ExpandElts;

(*--------------------------------------------------------- low-level I/O ---*)

PROCEDURE Out (a, b, c, d: TEXT := NIL) =
  BEGIN
    IF (a # NIL) THEN Wr.PutText (wr, a) END;
    IF (b # NIL) THEN Wr.PutText (wr, b) END;
    IF (c # NIL) THEN Wr.PutText (wr, c) END;
    IF (d # NIL) THEN Wr.PutText (wr, d) END;
  END Out;

PROCEDURE OutChar (ch: CHAR): INTEGER =
  (* writes 'ch' as a literal and returns the output width *)
  BEGIN
    IF (ch = '\\') THEN
      Wr.PutText (wr, "\\\\");
      RETURN 2;
    ELSIF (ch = '\n') THEN
      Wr.PutText (wr, "\\n");
      RETURN 2;
    ELSIF (ch = '\'') THEN
      Wr.PutText (wr, "\\\'");
      RETURN 2;
    ELSIF (ch = '\"') THEN
      Wr.PutText (wr, "\\\"");
      RETURN 2;
    ELSIF (ch < ' ') OR (ch > '~') THEN
      Wr.PutChar (wr, '\\');
      PutC (ORD(ch) DIV 64);
      PutC (ORD(ch) MOD 64 DIV 8);
      PutC (ORD(ch) MOD 8);
      RETURN 4;
    ELSE
      Wr.PutChar (wr, ch);
      RETURN 1;
    END;
  END OutChar;

PROCEDURE PutC (i: INTEGER) =
  BEGIN
    Wr.PutChar (wr, VAL(ORD('0') + i, CHAR));
  END PutC;

(*------------------------------------------------------------- main body ---*)

BEGIN
  IF ParseCommandLine () AND GetElementSizes () THEN
    WriteInterface ();
    WriteModule ();
  END;
END m3bundle. 
