(* Copyright 1992 Digital Equipment Corporation.           *)
(* Distributed only by permission.                         *)
(* Last modified on Sat Aug  8 00:19:33 PDT 1992 by mhb    *)
(*      modified on Wed Jul 29 23:17:19 PDT 1992 by johnh  *)
<* PRAGMA LL *>

MODULE PackingView2;

IMPORT Filter, Font, Fmt, GraphVBT, HSV, List, PaintOp,
       PaintOpCache, R2, VBT, View, ZeusPanel;

REVEAL
  T = Public BRANDED OBJECT
        W: INTEGER;             (* number of weights *)
        weights: REF ARRAY OF Weight;  (* the weights *)
      OVERRIDES
        <* LL=0 *>
        oeSetup      := Setup;
        oeNewWeight  := NewWeight;
        oePack       := Pack;
        oeIgnore     := Ignore;
        createGraph  := CreateGraph;
        createWeight := CreateWeight;
        <* LL=VBT.mu *>
        ueRepackBin := RepackBin;
      END;

VAR
  font := Font.FromName (ARRAY OF TEXT{"helvetica_bold14"});

PROCEDURE New (): View.T =
  BEGIN
    RETURN NEW(T).init(NEW(GraphVBT.T).init())
  END New;

PROCEDURE Setup (view: T; nBins, nWts: INTEGER) =
  BEGIN
    view.W := nWts;
    view.weights := NEW(REF ARRAY OF Weight, nWts);
    WITH mg = view.createGraph(nBins, nWts) DO
      LOCK VBT.mu DO EVAL Filter.Replace(view, mg) END
    END
  END Setup;

PROCEDURE NewWeight (view: T; id: INTEGER; wt: REAL) =
  VAR mg: GraphVBT.T := Filter.Child(view);
  BEGIN
    view.curr := view.createWeight(id, wt);
    view.weights[id] := view.curr;
    mg.redisplay()
  END NewWeight;

PROCEDURE Pack (view: T; bin: INTEGER; total: REAL) =
  VAR mg: GraphVBT.T := Filter.Child(view);
  BEGIN
    LOCK mg.mu DO
      view.curr.move(
        R2.T{0.5 + FLOAT(bin), total - view.curr.amt / 2.0},
        animated := FALSE)
    END;
    mg.redisplay()
  END Pack;

PROCEDURE Ignore (view: T) =
  VAR mg: GraphVBT.T := Filter.Child(view);
  BEGIN
    LOCK mg.mu DO view.curr.remove() END;
    mg.redisplay()
  END Ignore;

PROCEDURE RepackBin (view          : T;
                     bin           : INTEGER;
                     old, new, amts: List.T   ) =
  <* LL = VBT.mu *>
  VAR
    mg     : GraphVBT.T := Filter.Child(view);
    total  : REAL       := 0.0;
    o, n, a: List.T;
  BEGIN
    o := old;
    WHILE o # NIL DO
      IF NOT List.Member(new, List.First(o)) THEN
        WITH id  = NARROW(List.First(o), REF INTEGER)^,
             bar = NARROW(view.weights[id], Weight)     DO
          bar.remove();
          view.weights[id] := NIL;
        END
      END;
      o := List.Tail(o)
    END;
    n := new;
    a := amts;
    WHILE n # NIL DO
      WITH id  = NARROW(List.First(n), REF INTEGER)^,
           amt = NARROW(List.First(a), REF REAL)^,
           bar = NARROW(view.weights[id], Weight)     DO
        total := total + amt;
        bar.move(R2.T{0.5 + FLOAT(bin), total - amt / 2.0},
                 animated := TRUE)
      END;
      n := List.Tail(n);
      a := List.Tail(a);
    END;
    mg.animate(0.0, 1.0);
  END RepackBin;

PROCEDURE CreateGraph (<* UNUSED *> view : T;
                                    nBins: INTEGER;
                       <* UNUSED *> nWts : INTEGER  ):
  GraphVBT.T =
  BEGIN
    RETURN NEW(GraphVBT.T, world := GraphVBT.WorldRectangle{
                                     w := 0.0, s := 0.0, e :=
                                     FLOAT(nBins), n := 1.0},
               pixelSizeDivisor :=
                 ARRAY [0 .. 1] OF CARDINAL{nBins, 1}).init()
  END CreateGraph;

PROCEDURE CreateWeight (view: T; id: INTEGER; amt: REAL): Weight =
  VAR
    mg  := NARROW(Filter.Child(view), GraphVBT.T);
    rgb := HSV.RGBFromHue(FLOAT(id + 1) / FLOAT(view.W));
    op  := PaintOpCache.FromRGB(rgb);
  BEGIN 
    RETURN NEW(Weight, 
                     id          := id, 
                     amt         := amt,
                     graph       := mg,
                     pos         := R2.T{-100.0,-100.0},
                     color       := op,
                     sizeW       := R2.T{1.0, amt},
                     borderMM    := 0.25,
                     borderColor := PaintOp.Fg,
                     label       := Fmt.Int(id),
                     font        := font,
                     fontColor   := PaintOp.Fg).init()
  END CreateWeight;

BEGIN
  ZeusPanel.RegisterView (New, "Packing", "Binpack");
END PackingView2.


