INTERFACE PathName;

(***************************************************************************)
(*                      Copyright (C) Olivetti 1989                        *)
(*                          All Rights reserved                            *)
(*                                                                         *)
(* Use and copy of this software and preparation of derivative works based *)
(* upon this software are permitted to any person, provided this same      *)
(* copyright notice and the following Olivetti warranty disclaimer are     *) 
(* included in any copy of the software or any modification thereof or     *)
(* derivative work therefrom made by any person.                           *)
(*                                                                         *)
(* This software is made available AS IS and Olivetti disclaims all        *)
(* warranties with respect to this software, whether expressed or implied  *)
(* under any law, including all implied warranties of merchantibility and  *)
(* fitness for any purpose. In no event shall Olivetti be liable for any   *)
(* damages whatsoever resulting from loss of use, data or profits or       *)
(* otherwise arising out of or in connection with the use or performance   *)
(* of this software.                                                       *)
(***************************************************************************)

IMPORT Text;

(* This interface provides text manipulation routines for file and directory
path names.
  Unless stated otherwise it is a checked runtime error to supply NIL as a
'Text.T' argument to any of the procedures in this interface *)

PROCEDURE CaseSensitive(): BOOLEAN RAISES {};
(* Are path names case sensitive? *)

PROCEDURE DirSepCh(): CHAR RAISES {};
(* The character which separates different components of a path name *)

PROCEDURE Current(): Text.T RAISES {};
(* Most systems have a special name to refer to the current directory (e.g.
"." in Unix). 'Current' returns this name *)

PROCEDURE Tail(name: Text.T): Text.T RAISES {};
(* Find the tail part of a full path name - strip off leading directory name
returning just the name of the directory entry.
  e.g. under Unix 'Tail(a/b/c)' is "c" *)

PROCEDURE Head(name: Text.T): Text.T RAISES {};
(* Find the leading part of a full path name - i.e. everything but the tail.
  e.g. under Unix 'Head(a/b/c)' is "a/b/" *)

PROCEDURE Extension(name: Text.T): Text.T RAISES {};
(* Many systems allow a filename to have an extension. Such a filename ends
with a sequence of characters, usually separated from the rest of the filename
by some special character.
  'Extension' finds the extension of a full path name. If no extension is
present the null text is returned.
  Returns the longest possible extension i.e. if a filename ends with two
extension sequences both are returned.
  e.g. under Unix 'Extension(a/b/c.d)' returns "d"
                  'Extension(a/b/c)' returns ""
                  'Extension(a/b/c.d.e)' returns "d.e" *)

PROCEDURE Name(name: Text.T): Text.T RAISES {};
(* Find the tail part of a path name and then strip off any extension leaving
just the name.
  e.g. under Unix 'Name(a/b/c.d)' returns "c"
                  'Name(a/b/c)' returns "c" *)

PROCEDURE Extend(name, ext: Text.T): Text.T RAISES {};
(* Adds given extenion to given path name. Any previous extension is removed.
If the extension is the null Text the effect is to remove any existing
extension *)


PROCEDURE Concat(head, tail: Text.T): Text.T RAISES {};
(* Concatenate the given head and tail. 'head' is usually a directory name and
'tail' is a name rooted in the directory. 'Concat' inserts a directory
separator character between 'head' and 'tail' if necessary.
  If 'head' is the null text 'Concat' just returns 'tail'. If 'tail' is the
null text 'Concat' just returns 'head' *)


PROCEDURE Valid(name: Text.T): BOOLEAN RAISES {};
(* Checks if 'name' is a valid name for an object in the filing system. Does
not check if a filing system object called 'name' exists; just checks if 'name'
could be a name for a filing system object i.e. does not contain any illegal
characters, is not too long etc. *)


PROCEDURE Full(dir, relative: Text.T): Text.T RAISES {};
(* A smarter concatenation procedure. 'dir' is a directory name. 'relative' is
either an absolute path name (one that starts at a root of the filing system)
or a path name relative to 'dir'. If 'relative' really is a relative name
'Full' constructs a full path from 'dir' and 'relative'. If 'relative' is
actually absolute, 'Full' just returns 'relative'.
  If 'dir' is a null text 'Full' returns 'relative'.
  If 'relative' is a null text 'Full' returns 'dir'. *)


PROCEDURE Relative(name1, name2: Text.T): Text.T RAISES {};
(* 'name1' and 'name2' must be absolute path names (it is a checked run time
error if they are not). 'Relative' attempts to construct a relative path name
from 'name1' to 'name2'. If it succeeds it returns the relative name; if it
fails it returns 'name2'. Relative does not guarantee to find the shortest
relative path name or, indeed, to find any relative path name at all. It is
intended for use when using a relative path name is helpful but not vital e.g.
when writing out a path name for a user to read *)

END PathName.

