(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)
(* Created by Jorge Stolfi on Fri Oct  6 23:50:05 PDT 1989     *)
(* Last modified on Wed Jun 17 11:43:56 PDT 1992 by stolfi     *)
(*      modified on Thu Jun 11 19:08:49 1992 by mhb            *)
(*      modified on Tue Feb 11 21:39:48 PST 1992 by muller     *)


(* An "HSV.T" is a color represented as a (Hue, Saturation, Value) triple.
   The HSV color model is somewhat more intuitive than the standard RGB
   model.  It is a variant of the Hue-Stauration-Value model described
   in standard computer graphics textbooks, such as Foley and Van Dam's. *)

INTERFACE HSV;

IMPORT RGB, Color;

TYPE 
  T = Color.T;
(* This type is equivalent to "ARRAY [0 .. 2] OF REAL". *)

(* An HSV.T value describes a color by the three attributes hue
   (h), saturation (s), and value (v), all real numbers in the
   range 0.0--1.0.

   Hue is the same as "RGB.Hue", and defines the position of the
   color on the spectrum.  It is meant to be roughly equivalent
   to the Munsell hue, rescaled to the range 0.0--1.0.

   Saturation is a measure of the relative distance of the color
   from the black-white axis: 0.0 means some shade of grey, 1.0
   means a ``pure'' color mixed with only black or only white.

   Value is a measure of the color's lightness.  If the
   saturation is 0.0 then the value is the same as
   "RGB.Brightness".  If the saturation is 1.0 then values below
   0.5 mean a pure color mixed with black, and values above 0.5
   mean a pure color mixed with white.  Colors with intermediate
   saturations are obtained by linear interpolation between these
   two extrema.

   This mapping has the advantage that it maps the RGB unit cube
   to the HSV unit cube in almost 1:1 fashion.  The only
   exceptions are the grey values (which have indeterminate hue)
   and white and black (which have indeterminate hue and
   saturation).  Moreover, in this HSV system it is easy to
   specify both pure colors (saturation=1, value=0.5) and grey
   levels (saturation=0, value=brightness). *)

PROCEDURE HSVFromRGB (READONLY rgb: RGB.T): T;
(* Convert from RGB to HSV coordinates.  By convention, grey
   colors get hue=0.0.  In addition, black and white get
   saturation=0.0. *)

PROCEDURE RGBFromHSV (READONLY hsv: T): RGB.T;
(* Convers from HSV to RGB coordinates.  If value=0 (black) or
   value=1 (white), saturation and hue are irrelevant.  If
   saturation=0 (grey), hue is irrelevant. *)

PROCEDURE RGBFromHue (h: REAL): RGB.T;
(* Return a ``pure'' color of the given hue.
   More precisely, "RGBFromHue" returns the color that is maximally saturated
   among all colors in the unit RGB cube with the given hue.
   Such a ``pure'' color is a triple of the form (1.0 r 0.0), or a
   permutation thereof, for some r in 0.0--1.0. *)

PROCEDURE HueFromRGB (READONLY rgb: RGB.T): REAL;
(* Return the hue of the given RGB triple, from 0.0 to 1.0. *)

(* The remainder of the interface defines the "Default" and
   "Undefined" triples, and a procedure for safely
   LOOPHOLE'ing a triple into an array of bytes.  As
   explained in the "Color" interface, "Default" is useful as a
   default value for generic modules, and "Undefined" is useful
   for applications that need a special color value meaning
   ``undefined'', ``vacant'', ``don't care'', and so on.  The
   loophole function, "ToBytes", is useful for marshalling and
   pickling color triples. *)

CONST 
  Default = Color.Default;
  Undefined = Color.Undefined;

TYPE 
  Bytes = Color.Bytes;

CONST (* PROCEDURE *) 
  ToBytes = Color.ToBytes;

END HSV.

