(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)
(* Last modified on Tue Sep 22 16:15:19 PDT 1992 by meehan     *)
(*      modified on Sun Jul 12 21:58:46 1992 by mhb            *)
(*      modified on Tue Oct 22 10:36:46 PDT 1991 by steveg     *)
(*      modified on Mon Jan 15 16:48:10 PST 1990 by brooks     *)
(*      modified on Sun May 21 17:17:10 PDT 1989 by gidi       *)
<* PRAGMA LL *>

(* FormsVBT is a system for building graphical user interfaces.
   FormsVBT provides a special-purpose language for describing
   user interfaces, an interface-builder that allows editing of
   such descriptions, and a runtime library for applications to
   make use of the user interfaces.

   The locking level for any procedure in this interface that may
   alter an installed "VBT" is "LL = VBT.mu". Most applications 
   don't need to worry about "VBT.mu" because their event-handlers 
   don't fork any threads that call FormsVBT. *)

INTERFACE FormsVBT;

IMPORT AnyEvent, Filter, Rd, Rsrc, SxSyntax, Thread, VBT, Wr, ZSplit;

EXCEPTION
  Error(TEXT);
  Unimplemented;
  Mismatch;

(* \section{Creation, allocation, and initialization} *)

(* An object "fv" of type "FormsVBT.T" (or simply, a {\it form}) is
   created by parsing an S-expression whose syntax is described
   in the {\it FormsVBT Reference Manual}.  These expressions are
   usually stored in files with the suffix ``.fv''.  The usual
   way of creating a form is to call the procedure "NewFromFile",
   or the method "fv.initFromFile", with the name of such a file;
   the expression is parsed, and if there are no errors, a new
   "VBT" is created and stored in the form, which is returned.

   It is also possible for a program to generate a description
   ``on the fly'' and then use it to create a form.  The methods
   "fv.init", "fv.initFromRd", and "fv.initFromSx" support these
   options. *)

TYPE
  T <: Public;
  S_exp = REFANY;
  Public =
    Filter.T OBJECT
    METHODS
      <* LL <= VBT.mu *>
      init (description: TEXT; raw := FALSE; path: Rsrc.Path := NIL): T
            RAISES {Error};
      initFromFile (filename: TEXT; raw := FALSE; path: Rsrc.Path := NIL): T
                    RAISES {Error, Rd.Failure, Thread.Alerted};
      initFromRd (rd: Rd.T; raw := FALSE; path: Rsrc.Path := NIL): T
                  RAISES {Error, Rd.Failure, Thread.Alerted};
      initFromSx (sx: S_exp; raw := FALSE; path: Rsrc.Path := NIL): T
                  RAISES {Error};
      initFromRsrc (name: TEXT; path: Rsrc.Path; raw := FALSE): T
                    RAISES {Error, Rd.Failure, Rsrc.NotFound, Thread.Alerted};
      realize (type, name: TEXT): VBT.T RAISES {Error};
      <* LL = VBT.mu *>
      snapshot (wr: Wr.T) RAISES {Error};
      restore  (rd: Rd.T) RAISES {Mismatch, Error};
    END;

(* The call "fv.init(description, raw)" initializes "fv" as a form and returns
   "fv".  It creates a "VBT", {\it v}, from "description", which must contain
   a single, valid S-expression.  The methods "initFromFile", "initFromRd",
   "initFromSx", and "initFromRsrc" provide analogous support for files,
   readers, S-expressions, and named resources.  Files are accessed via
| FileStream.OpenRead (Filename.ExpandTilde (filename))
   Exceptions of type "Rd.EndOfFile" and "Rsrc.NotFound" are signaled as "Error".

   "fv" is a filter; if "raw" is "TRUE", then the filter's child is {\it v}.
   Otherwise, "fv" is ``cooked'', which means there are several filters
   inserted between {\it v} and "fv", so that the filter's child has the
   following structure:

| (ZSplit.T
|   (FVTypes.FVMenuBar
|     (HighlightVBT.T
|       (ClipboardVBT.T
|         (FVTypes.FVFilter `{\it v}`)))))


   The filter above {\it v} supports the common case of making an entire form
   passive without requiring an explicit "Filter" interactor in the
   description.  It also functions to restore the keyboard focus to whichever
   of the form's descendant-"VBT"s had most recently acquired the keyboard
   focus.  The "ZSplit.T" supports menus and other pop-up operations, even if
   there is no "ZSplit" explicitly mentioned in the description.  To get the
   "ZSplit.T" that is inserted, use "GetZSplit".  Clients should not traverse
   a cooked form directly.  We reserve the right to change the filters that
   are inserted.

   The description of the form is documented in the {\em FormsVBT Reference
   Manual}.  Briefly, it is an S-expression whose first element is the name of
   a component (e.g., "HBox"), and whose other elements are either properties
   (e.g., "Color"), or other components, typically describing the
   "VBT"-children of the outer component.

   The "VBT"-tree is created during a depth-first traversal of the
   S-expression.  On the way down, each "VBT" is {\it allocated}, typically
   with a call to "NEW(...)".  Then the subexpressions, if any, are traversed.
   On the way back up, each "VBT" is {\it initialized}, typically with a call
   to "v.init(...)".  The result is returned to the caller, where it is
   typically an argument to the parent's "init" method.

   In other words, allocation occurs top-down, and initialization occurs
   bottom-up.  (For more details on allocation, see Section~\ref{realize}.)

   For each subexpression, the parser produces a "VBT" whose type is defined
   in the "FVTypes" interface, and whose name corresponds to the first element
   of the subexpression.  For example, from the S-expression "(HBox ...)", the
   parser creates an object of type "FVTypes.FVHBox". *)

PROCEDURE NewFromFile (filename: TEXT; raw := FALSE; path: Rsrc.Path := NIL): T
  RAISES {Error, Rd.Failure, Thread.Alerted};
(* Create a new form from the description in the file.  "Rd.EndOfFile" is
   signalled as "Error".

   Equivalent to "NEW(T).initFromFile (name, raw, path)" *)

PROCEDURE GetZSplit (fv: T): ZSplit.T RAISES {Error};
(* Return the "ZSplit" that ``cooked'' mode inserts.  An
   exception is raised if "fv" was not created with "raw =
   TRUE". *)

(*
\section{Events} \label{sec:programming-events}
\subsection{Attaching event-handlers}

   Most interactive components in the user interface generate
   events.  To register an event-handler for such a component,
   the component must be named, and the client must call "Attach"
   or "AttachProc", giving the name of the component and a
   procedure to be called when an event occurs in that
   component. *)

PROCEDURE Attach (fv: T; name: TEXT; cl: Closure) RAISES {Error};
(* Attach an event-handler (``callback'') to the component of
   "fv" whose name is given by "name".  If there is no such
   component, then "Error" will be raised.  If "cl" is "NIL",
   then any existing event-handler for that component is removed.
   Otherwise, when an event occurs in the named component, the
   implementation calls
|    cl.apply(fv, name, `{\it time}`)
   *)

TYPE
  Closure = OBJECT
            METHODS
              apply (fv: T; name: TEXT; time: VBT.TimeStamp);
            END;

PROCEDURE AttachProc (fv       : T;
                      name     : TEXT;
                      p        : Proc;
                      eventData: REFANY := NIL) RAISES {Error};
(* This is an alternate, somewhat simpler way to attach an
   event-handler.  When an event occurs in the named component,
   the implementation calls
| p(fv, name, eventData, `{\it time}`)
   *)

TYPE
  Proc = PROCEDURE (fv       : T;
                    name     : TEXT;
                    eventData: REFANY;
                    time     : VBT.TimeStamp);

(* These event-handlers do not provide any other details, such as
   what key or mouse button was pressed, or whether it was a
   double-click.  If such information is needed, call
   "GetTheEvent" to retrieve it. *)

(* \subsection{Access to the current event} *)

PROCEDURE MakeEvent (fv: T; name: TEXT; time: VBT.TimeStamp)
  RAISES {Error};
(* "MakeEvent" invokes the event-handler that was previously
   attached (through "Attach" or "AttachProc") to the component
   of "fv" whose name is "name".  This is useful when one part of
   a large program wishes to communicate with another part by
   pretending that the named event occurred. *)

PROCEDURE GetTheEvent (fv: T): AnyEvent.Code RAISES {Error};
PROCEDURE GetTheEventTime (fv: T): VBT.TimeStamp RAISES {Error};
(* Retrieve the details of the event that is currently in
   progress.  These routines may be called only during the
   dynamic extent of an event-handler attached to some component
   via "Attach" or "AttachProc".

   The type of the event depends on the user action that caused
   the event to be generated.  If the event was caused by
   "MakeEvent", the type of the event is
   "MakeEventMiscCodeType". *)

VAR                             (* CONST *)
  MakeEventMiscCodeType: VBT.MiscCodeType;

(* \subsection{Symbol management} *)

PROCEDURE AddSymbol (fv: T; name: TEXT) RAISES {Error};
(* Add a ``virtual'' component to "fv" with the given "name".
   The form will behave as if there was a component called "name"
   (i.e., the call "GetVBT(fv, name)" will return a valid
   "VBT").

   This procedure is most useful as a means to communicate
   between distant parts of a large program.  One part of the
   program would use "AddSymbol" to create a new symbol; another
   part would call "MakeEvent" to invoke an event-handler for the
   symbol.

   "Error" is raised if "name" is already defined in "fv". *)


PROCEDURE AddUniqueSymbol (fv: T): TEXT;
(* Just like "AddSymbol", but finds a name that has not been used
   yet.  The name is returned. *)

(* \section{Reading and Changing State}

   \label{sec:programming-state}

   In response to an event or other occurrence, a program may
   want to read or change the state of various interactors in the
   form.  This is handled by the various Get and Put procedures.
   Get procedures take the form and the name of the interactor,
   and return its value.  Put procedures take the form, the name
   of the interactor, and the new value to be set.

   There are several Get procedures and several Put procedures,
   for convenient handling of various Modula types.  These should
   be used as appropriate to the type of the interactor:
   "GetText" for a "TypeIn", "GetInteger" for a "Numeric",
   "GetBoolean" for a "Boolean" or "Choice", etc. However, some
   conversions are supported: "PutInteger" to a "TypeIn" will
   convert the integer into text; "GetInteger" will likewise
   attempt to convert the text of the "TypeIn" to an integer
   (and return 0 in case of failure).  All Get and Put
   procedures, however, will raise "Error" if applied to a
   component that does not have a value.

   \subsection{Access to the {\tt Main} and {\tt Value} properties}
*) 

PROCEDURE GetText (fv: T; name: TEXT): TEXT
  RAISES {Error, Unimplemented};
(* This is implemented for "FileBrowser", "Text", "Numeric",
   "Typescript", and the text-interactors: "TextEdit", "TypeIn",
   and "TextArea". *)
   
PROCEDURE PutText (fv: T; name: TEXT; t: TEXT; append := FALSE)
  RAISES {Error, Unimplemented};
(* This is implemented for "FileBrowser", "Text", "Typescript",
   and the text-interactors: "TextEdit", "TypeIn", and
   "TextArea".  For "Text" and the text-interactors, if "append"
   is true, then the text "t" is added to the end of the current
   text, rather than replacing it. *)

PROCEDURE GetInteger (fv: T; name: TEXT): INTEGER
  RAISES {Error, Unimplemented};
PROCEDURE PutInteger (fv: T; name: TEXT; num: INTEGER)
  RAISES {Error, Unimplemented};
(* These are implemented for "Numeric", "Scroller", and "TSplit". *)

PROCEDURE GetBoolean (fv: T; name: TEXT): BOOLEAN
  RAISES {Error, Unimplemented};
PROCEDURE PutBoolean (fv: T; name: TEXT; val: BOOLEAN)
  RAISES {Error, Unimplemented};
(* These are implemented for "Boolean" and "Choice". *)

(* \subsection{Access to arbitrary properties}

   FormsVBT provides access to properties other than "Main" and
   "Value".  The intention is to provide access to all the
   inherited and class properties.  For example, the "Scroller"
   component has an integer-valued property named "Min", so it
   should be possible to call
| GetIntegerProperty(fv, name, "Min")
   to retrieve that value, or
| PutIntegerProperty(fv, name, "Min", 6)
   to change the value to 6.

   {\bf WARNING:} The current implementation, however, provides
   access only to the inherited properties, and even that is
   limited.

   A more complete implementation of properties will be a
   byproduct of providing user-defined extensions of the FormsVBT
   language. This work is in the planning stages. *)

PROCEDURE GetTextProperty (fv: T; name, property: TEXT): TEXT
  RAISES {Error, Unimplemented};
(* This is implemented for the "Color", "BgColor", "Font",
   "LabelFont", "LightShadow", and "DarkShadow" properties. *)
  
PROCEDURE PutTextProperty (fv: T; name, property: TEXT; t: TEXT)
  RAISES {Error, Unimplemented};
(* This is implemented for the "Color", "BgColor", "Font", and
   "LabelFont" properties. *)


(* \subsection{Access to the underlying {\tt VBT}s} *)

PROCEDURE GetVBT (fv: T; name: TEXT): VBT.T RAISES {Error};
(* Return the "VBT" corresponding to a named interactor in "fv".
   "Error" is raised if there is no such "VBT". *)

(* \subsection{Radios} *)

PROCEDURE GetChoice (fv: T; radioName: TEXT): TEXT
  RAISES {Error, Unimplemented};
PROCEDURE PutChoice (fv: T; radioName, choiceName: TEXT)
  RAISES {Error, Unimplemented};
(* Get/Put the name of the selected "Choice" in a radio-button
   group. *)

PROCEDURE MakeSelected (fv: T; choiceName: TEXT) RAISES {Error};
PROCEDURE IsSelected (fv: T; choiceName: TEXT): BOOLEAN
  RAISES {Error};
(* Set/test a "Choice"-button without referring to its group. *)

PROCEDURE WhichRadio (fv: T; choiceName: TEXT): TEXT
  RAISES {Error};
(* Get the name of the group to which the "Choice"-button
   belongs. *)

(* \subsection{Generic interactors} *)

PROCEDURE GetGeneric (fv: T; genericName: TEXT): VBT.T
  RAISES {Error};
(* Retrieve the "VBT" used by the named "Generic" interactor. *)

PROCEDURE PutGeneric (fv: T; genericName: TEXT; vbt: VBT.T)
  RAISES {Error};
(* Replace the named "Generic" interactor with "vbt", which may
   be "NIL".  When "NIL" is specified, a default (and initial)
   "VBT" is used: a "TextureVBT" with 0 size and 0 stretch in
   each dimension. *)

(* \subsection{Reactivity filters}

   The "(Filter ...)" expression in FormsVBT supports a feature
   called {\em reactivity}.  This has one of four states: Active,
   Passive, Dormant, or Vanished.  The state can be specified in
   the description and changed by the application at runtime.
   The normal state is Active.  In the Passive state, the
   component and its descendants, if any, are unresponsive to
   mouse clicks.  The Dormant state is like Passive, but the
   component and descendants are ``grayed out.'' Dormant is often
   to be preferred over Passive, because it provide additional
   feedback to the user.  In the Vanished state, the component
   becomes unreactive and disappears entirely. *)


PROCEDURE MakeActive  (fv: T; name: TEXT) RAISES {Error};
PROCEDURE MakePassive (fv: T; name: TEXT) RAISES {Error};
PROCEDURE MakeDormant (fv: T; name: TEXT) RAISES {Error};
PROCEDURE MakeVanish  (fv: T; name: TEXT) RAISES {Error};
(* Find the nearest ancestor of the named component that is of
   type "FVFilter", and sets its state as indicated.  The
   exception is raised if no such ancestor can be found. *)

PROCEDURE IsActive   (fv: T; name: TEXT): BOOLEAN RAISES {Error};
PROCEDURE IsPassive  (fv: T; name: TEXT): BOOLEAN RAISES {Error};
PROCEDURE IsDormant  (fv: T; name: TEXT): BOOLEAN RAISES {Error};
PROCEDURE IsVanished (fv: T; name: TEXT): BOOLEAN RAISES {Error};
(* Find the nearest ancestor of the named component that is of
   type "FVFilter", and tests its state as indicated.  The
   exception is raised if no such ancestor can be found. *)


(* \subsection{Subwindows} *)

PROCEDURE PopUp (fv        : T;
                 name      : TEXT;
                 eventTime : VBT.TimeStamp;
                 forcePlace: BOOLEAN         := FALSE)
  RAISES {Error};
(* Assuming that "name" is the name of an element of "fv" that
   can be popped up, pop it up.  That is, the named element must
   be a non-background child of a "ZSplit", or some descendant
   thereof.  In the latter case, the ancestor that is a direct
   child of the "ZSplit" will be the thing popped up.  Call this
   ancestor {\it zchild}.  "PopUp" is equivalent to activating

| (PopButton (For `{\it zchild}`) ...)

   If the target {\it zchild} is already open or has been opened
   before and has been moved by the user (to a location that is
   now visible), it will normally be left where the user left it.
   The "forcePlace" option will force it instead to be returned
   to its canonical place. *)

PROCEDURE PopDown (fv: T; name: TEXT) RAISES {Error};
(* The inverse of "PopUp": make the named element (or suitable
   ancestor) invisible.  This is implemented using "ZSplit"'s
   unmapping.  (Unfortunately, this doesn't cause the keyboard
   focus to be lost.)  The exception is raised if "name" is not
   the name of an element of "fv". *)

(* \subsection{Special controls for text-interactors} *) 

PROCEDURE TakeFocus (fv       : T;
                     name     : TEXT;
                     eventTime: VBT.TimeStamp;
                     select                     := FALSE) RAISES {Error};
(* Give the keyboard focus to a specified interactor.  An exception is raised
   if the interactor is not of a suitable class to take it; however, no
   exception is raised if the keyboard focus cannot be taken because of a
   timeout, i.e., an invalid "eventTime".  If "select" is "TRUE" and the focus
   was taken, then select the entire contents of the interactor's "TextPort"
   as a primary selection in replace-mode. *)

(* The remaining procedures in this section are obsolete.  The same
   functionality is provided by the "realize" method; see
   Section~\ref{realize}. *)

<* OBSOLETE *> PROCEDURE AttachKeyFilter (fv: T; name: TEXT; kf: KeyFilter)
  RAISES {Error};
(* Attach an event-handler for keystrokes in the named component, which should
   be a text-interactor ("TextEdit", "TextArea", and "TypeIn").  When an event
   occurs, the implementation calls
| kf.apply(fv, name, `{\it cd}`)
   which has the same effect as "TextPort.T.filter".  For a description of the
   "filter" method, see the "TextPort" interface. *)

<* OBSOLETE *> TYPE
  KeyFilter =
    OBJECT METHODS apply (fv: T; name: TEXT; VAR cd: VBT.KeyRec) END;

<* OBSOLETE *>
  PROCEDURE AttachFocusAlert (fv: T; name: TEXT; alert: FocusAlert)
  RAISES {Error};
(* Attach an event-handler to the named component, which should be a
   text-interactor.  When the component gains or loses the keyboard focus, the
   implementation calls
| alert.apply(fv, name, `{\it gaining}`, `{\it time}`)
   If "GetTheEvent" is called while this event is in progress, then it returns
   a value "c" of type "FocusMiscCodeType".  Word 0 of "c.detail" will be 0 if
   we were losing the focus, and 1 if gaining.  For a description of the
   "focus" method, see the "TextPort" interface. *)

<* OBSOLETE *>
TYPE
  FocusAlert =
    OBJECT
    METHODS
      apply (fv: T; name: TEXT; gaining: BOOLEAN; time: VBT.TimeStamp)
    END;

<* OBSOLETE *> VAR              (* CONST *)
  FocusMiscCodeType: VBT.MiscCodeType;

(* \section{Saving and restoring state} \label{sec:programming-snapshot} *)

(* The client may sometimes wish to save and restore the entire state of a
   form.  This is supported by the snapshot facility.

   A {\em snapshot\/} is an S-expression that captures the state of components
   in a form.  The call "fv.snapshot(wr)" writes a snapshot of "fv" to the
   writer "wr", and the call "fv.restore(rd)" reads a snapshot from the reader
   "rd" and restores the state of the components of "fv" to their states in
   the snapshot.

   A snapshot produced by the default method contains only those named
   components that have a modifiable value.  More precisely, a component is
   part of a snapshot if (1) it has a name and (2) the call to GetText,
   GetInteger, GetReal, GetBoolean, or GetChoice does not raise an exception.

   The "snapshot" method raises the "Error" exception if there is a problem
   writing the snapshot to the writer.

   The "restore" method raises the "Error" exception if there is a syntax
   error in the S-expression or if there is any type of problem with the
   reader.

   When restoring, the snapshot does not have to precisely match the set of
   interactors in the form.  If the snapshot is lacking values for some fields
   that the form contains, those fields will be left alone.  If the snapshot
   has values for some fields that the form does not contain, the "restore"
   method should raise "Mismatch", but only after restoring all the values
   that do match.  If the snapshot has a value for a field that the form
   contains, but the types do not agree, this is a show-stopping error; the
   "restore" method should raise "Error".  Catching "Mismatch" is useful when
   you want to continue to tolerate snapshots from old versions of a form.

   The default "snapshot" and "restore" methods write s-expressions in the
   following format:

|    ((name1 value1)
|     (name2 value2) ...)

   The typical reason why you would override these methods is if there is a
   component (e.g., a "Generic") whose value you want to be part of the
   snapshot but it doesn't respond to "GetText", "GetInteger", etc. *)

(* \section{Dynamic Alteration of Forms} \label{sec:programming-dynamicforms}

   In some applications, it may be desirable to change a form while the
   program is running.  For example, one might want to add or change items in
   a menu.  FormsVBT provides facilities to:

   \begin{itemize}

   \item add a new component, as a child of a named Split.

   \item delete a named component, provided that it is a direct child of a
   Split.

   \item read a FormsVBT S-expression from a text.

   \item print a FormsVBT S-expression as text.

   \end{itemize}

   \subsection{Modifying forms in place}

   "Insert" and "Delete" support dynamic modification of forms.  Any resizing
   that may be appropriate after an "Insert" or "Delete" is performed
   automatically.  For the common case of menus, this is not an issue because
   the menu is (almost certainly) not visible at the time the alteration takes
   place. *)

PROCEDURE Insert (fv         : T;
                  parent     : TEXT;
                  description: TEXT;
                  n          : CARDINAL := LAST (CARDINAL)): VBT.T
  RAISES {Error};
(* "Insert" parses a description in the context of an existing form, that is,
   in "fv"'s namespace, so that names already defined in "fv" are visible
   while the description is being parsed, and with the state (color,
   resource-path, etc.) that was in effect for "parent".

   Once the new "VBT" is created, it is inserted into the named component,
   which must be a Split, as the "n"th child.  It is also returned. *)

PROCEDURE InsertFromFile (fv      : T;
                          parent  : TEXT;
                          filename: TEXT;
                          n       : CARDINAL := LAST (CARDINAL)): VBT.T
  RAISES {Error, Rd.Failure, Thread.Alerted};
PROCEDURE InsertFromRsrc (fv    : T;
                          parent: TEXT;
                          name  : TEXT;
                          path  : Rsrc.Path;
                          n     : CARDINAL    := LAST (CARDINAL)): VBT.T
  RAISES {Error, Rd.Failure, Rsrc.NotFound, Thread.Alerted};
(* "InsertFromFile" and "InsertFromRsrc" read a description from a file or
   named resource, and then call "Insert". *)

PROCEDURE InsertVBT (fv    : T;
                     name  : TEXT;
                     child : VBT.T;
                     n     : CARDINAL := LAST (CARDINAL))
  RAISES {Error};
(* Insert "child" as the "n"th child of the named component,
   which must be a Split. *)

PROCEDURE Delete (fv    : T;
                  parent: TEXT;
                  n     : CARDINAL;
                  count : CARDINAL   := 1) RAISES {Error};
(* Delete the children whose indices are in the range
   "[n .. (n + count - 1)]" from the named component, which must be a
   Split. *)

(* \subsection{Reading and printing S-expressions}

   The FormsVBT parser uses "Sx.Read" to convert text-strings
   into S-expressions, and "Sx.Print" to convert S-expressions
   into text-strings.  A special ``syntax object'' is used to
   implement the syntactic shortcuts: "%x" as an abbreviation for
   "(Name x)", and "=y" as an abbreviation for "(Value y)".  The
   syntax object also handles the single-quote, backquote, and
   comma used in FormsVBT macros.

   "FVSyntax" is the syntax object.  If you read or print an
   S-expression describing a form, you should use this as the
   "syntax" argument to "Sx.Read" or "Sx.Print".

   The parser-object used by "FVSyntax" is "FVParser"; its
   printer-object is "FVPrinter".  Consult the "Sx" interface for
   information on these objects. *)

VAR                             (* CONST *)
  FVSyntax : SxSyntax.T;
  FVParser : SxSyntax.Parser;
  FVPrinter: SxSyntax.Printer;


(* \section{Subclasses of components} \label{realize}

   As the subexpressions describing the form "fv" are being parsed, the
   "VBT"-components are created (allocated) by calling
| fv.realize(`{\it type}`, `{\it name}`)
   where {\it type} is the name of the first element of the subexpression,
   and {\it name} is the "Name" property specified in the
   subexpression, or the empty string if no such property was
   specified.  For example, if the description contains the
   expression
| (Menu %mainMenu ...)
   then the FormsVBT parser will call
| fv.realize("Menu", "mainMenu")
   to create the "VBT".

   By overriding the "realize" method of "fv", the client can
   create subtypes for any or all of the components.  For each
   type of form, there is a corresponding type in the "FVTypes"
   interface.  For example, the result of parsing "(Menu ...)" is
   an object that is a subtype of "FVTypes.FVMenu".  The
   "realize" method must allocate and return a "VBT" that is a
   subtype of the corresponding type in "FVTypes".

   Example: suppose you wanted the form to keep a count of the
   number of menus it contains, and each menu to store its own
   index.

| TYPE
|   MyWindow = FormsVBT.T OBJECT
|                count: CARDINAL := 0
|              OVERRIDES
|                realize := Realize
|              END;
|   MyMenu = FVTypes.FVMenu OBJECT
|                index: CARDINAL
|            END;
|
| PROCEDURE Realize (fv: MyWindow; type, name: TEXT):
|   VBT.T RAISES {FormsVBT.Error} =
|   BEGIN
|     IF Text.Equal (type, "Menu") THEN
|       WITH m = NEW (MyMenu, index := fv.count) DO
|         INC (fv.count);
|         RETURN m
|       END
|     ELSE                     (* use the default *)
|       RETURN FormsVBT.T.realize (fv, type, name)
|     END
|   END Realize;

   Note that the "realize" method does not {\it initialize} the
   "VBT" that it allocates.  Actually, it may initialize any {\it
   private} fields, such as the "index" field in this example,
   but the "VBT"'s "init" method should not be called inside the
   call to "fv.realize", because it will be called later during
   the ``bottom-up'', initialization phase.  Of course, the
   client may also override the "init" method to control what
   happens in that phase.

   *)

END FormsVBT.
