/*
	Copyright 1987-1990 XVT Software Inc. All rights reserved.
	May be used freely by licensed and registered users of XVT.
	May be distributed in source form only when embedded in an
	XVT user's application.

	Example XVT application, showing:
	
	- how to use abstract fonts
	- how to select graphical objects when E_MOUSE_DOWN and E_MOUSE_UP events
	  occur
*/
#include "xvt.h"					/* standard XVT header */
#if (XVTCC == MWCCC) && (XVTOS == CTOOS)
pragma Calling_convention(CTOS_CALLING_CONVENTIONS);
#endif
#include "xvtmenu.h"				/* standard menu tags */

#ifdef PROTO
STATICFCN void set_bounds(int);
STATICFCN void draw_object(int);
STATICFCN void invert_selection(void);
STATICFCN int find_obj(PNT);
STATICFCN void do_close(WINDOW);
STATICFCN void do_menu(MENU_TAG, BOOLEAN, BOOLEAN);
STATICFCN void do_update(void);
STATICFCN void do_mouse(EVENT_TYPE, PNT);
STATICFCN void do_double(EVENT_TYPE, PNT);
STATICFCN void do_font(FONT *);
STATICFCN void inflate_rect(RCT *rct, int n);
#else
STATICFCN void set_bounds();
STATICFCN void draw_object();
STATICFCN void invert_selection();
STATICFCN int find_obj();
STATICFCN void do_close();
STATICFCN void do_menu();
STATICFCN void do_update();
STATICFCN void do_mouse();
STATICFCN void do_double();
STATICFCN void do_font();
STATICFCN void inflate_rect();
#endif

/* Required application setup structure. */
APPL_SETUP appl_setup = {
	0,								/* menu bar resource ID (use default) */
	0,								/* about box resource ID (use default) */
	"font",				    		/* application's name */
	W_DOC,							/* type of initial window */
	TRUE,							/* size box on initial window? */
	FALSE,							/* vert. scroll bar on initial window? */
	FALSE,							/* horz. scroll bar on initial window? */
	TRUE,							/* close box on initial window? */
	TRUE,							/* want std. font menu? (includes sizes) */
	TRUE							/* want std. style menu? */
};

#define NUM_OBJS 4
static struct {						/* info about each object */
	char *text;						/* text */
	PNT pos;						/* starting position */
	RCT bounds;						/* bounding rectangle */
	FONT font;						/* font */
} obj[NUM_OBJS] = {
	{
		"This is the first sentence.",
		{50, 10}
	},
	{
		"This is the second sentence.",
		{125, 150}
	},
	{
		"This is the third sentence.",
		{200, 100}
	},
	{
		"This is the fourth sentence.",
		{275, 200}
	}
};

#define NO_OBJ -1					/* illegal subscript means no selection */
static int sel_obj = NO_OBJ;		/* subscript of selected object, if any */

/* Function to set bounds of an object. */
#ifdef FPROTO
static void set_bounds(int n)
#else
static void set_bounds(n)
int n;
#endif
{
	int ascent, descent, width;

	set_font(&obj[n].font, FALSE);
	get_font_metrics(NULL, (INTPTR)&ascent, (INTPTR)&descent);
	width = text_width(obj[n].text);
	set_rect(&obj[n].bounds, obj[n].pos.h, obj[n].pos.v - ascent,
	  obj[n].pos.h + width, obj[n].pos.v + descent);
}

/* Application initialization. */
BOOLEAN XVTENTRY appl_init BTCENTRY(void)
{
	int i;

	for (i = 0; i < NUM_OBJS; i++) {
		switch (i % 3) {
		case 0:
			obj[i].font = big_font;
			break;
		case 1:
			obj[i].font = small_font;
			break;
		case 2:
			obj[i].font = normal_font;
		}
		set_bounds(i);
	}
	return(TRUE);
}

/* Function to draw a text object, and its selection rectangle if appropriate. */
#ifdef FPROTO
static void draw_object(int n)
#else
static void draw_object(n)
int n;
#endif
{
	set_font(&obj[n].font, FALSE);
	draw_text(obj[n].pos.h, obj[n].pos.v, obj[n].text, -1);
}

/* Function to show selection rectangle. */
#ifdef FPROTO
static void invert_selection(void)
#else
static void invert_selection()
#endif
{
	FONT font;
	DRAW_CTOOLS tools;
	CBRUSH hollow_cbrush;
	
	hollow_cbrush.pat = PAT_HOLLOW;
	hollow_cbrush.color = COLOR_BLACK;

	if (sel_obj == NO_OBJ) {
		font = normal_font;
		font.size = -1; /* no check marks */
		set_font_menu(&font);
	}
	else {
		get_draw_ctools(&tools);
		set_draw_mode(M_XOR);
		set_cpen(&black_cpen);
		set_cbrush(&hollow_cbrush);
		draw_rect(&obj[sel_obj].bounds);
		set_draw_ctools(&tools);
		set_font_menu(&obj[sel_obj].font);
	}
}

/* Function to find a text object, given a mouse position. */
#ifdef FPROTO
static int find_obj(PNT pos)
#else
static int find_obj(pos)
PNT pos;
#endif
{
	int i;

	for (i = 0; i < NUM_OBJS; i++)
		if (pt_in_rect(&obj[i].bounds, pos))
			return(i);
	return(NO_OBJ);
}

/* Function to close a window.  We quit when the window is closed. */
#ifdef FPROTO
static void do_close(WINDOW win)
#else
static void do_close(win)
WINDOW win;
#endif
{
	NOREF(win);
	terminate();
}

/* Function to handle all menu commands. */
#ifdef FPROTO
static void do_menu(MENU_TAG cmd, BOOLEAN shift, BOOLEAN control)
#else
static void do_menu(cmd, shift, control)
MENU_TAG cmd;						/* menu tag */
BOOLEAN shift;						/* was shift key down? */
BOOLEAN control;					/* was control key down? */
#endif
{
	WINDOW win;

	NOREF(shift);
	NOREF(control);
	switch (cmd) {
	case M_FILE_CLOSE:
		if ((win = get_front_window()) != NULL_WIN)
			do_close(win);
		break;
	case M_FILE_QUIT:
		terminate();
		break;
	}
}

/*
	Function to update the window.  The call to invert_selection will always
	show the selected object (if any) as selected -- never as unselected --
	because the part of the window it draws into is guaranteed to have been
	cleared by the earlier call to draw_rect.  This is true no matter how much
	or how little of the window is part of the update region.  That is,
	invert_selection may not draw anything, but, if so, no part of the selection
	rectangle was cleared, either.
*/
#ifdef FPROTO
static void do_update(void)
#else
static void do_update()
#endif
{
	RCT rct;
	int i;

	get_client_rect(std_win, &rct);
	set_cbrush(&white_cbrush);
	set_cpen(&white_cpen);
	draw_rect(&rct);
	for (i = 0; i < NUM_OBJS; i++)
		if (needs_update(std_win, &obj[i].bounds))
			draw_object(i);
	invert_selection();
}

/*
	Function to handle mouse down and up events.  A "click" is considered to be
	an E_MOUSE_DOWN and an E_MOUSE_UP within the same object's bounds.  A double
	click generates a spurious E_MOUSE_UP, because the actual sequence of events
	due to a double click is:
			E_MOUSE_DOWN
			E_MOUSE_UP
			E_MOUSE_DBL
			E_MOUSE_UP
	and this function does not get E_MOUSE_DBL events.  The variable down_obj
	records the object in which the E_MOUSE_DOWN occurred.  We set it to NO_OBJ
	after an E_MOUSE_UP event, so the extra E_MOUSE_UP will be ignored (just as
	though it had occurred within a different object than a preceding
	E_MOUSE_DOWN).

	Note also that the call invert_selection shows an unselected object as
	selected, and vice versa.
*/
#ifdef FPROTO
static void do_mouse(EVENT_TYPE type, PNT where)
#else
static void do_mouse(type, where)
EVENT_TYPE type;
PNT where;
#endif
{
	static int down_obj = NO_OBJ;

	switch (type) {
	case E_MOUSE_DOWN:
		down_obj = find_obj(where);
		break;
	case E_MOUSE_UP: /* click counts only if down & up within same object */
		if (down_obj == find_obj(where)) {
			invert_selection(); /* unselect old object */
			if (down_obj == sel_obj)
				sel_obj = NO_OBJ; /* user wanted to unselect */
			else
				sel_obj = down_obj;
			invert_selection(); /* select new object, if any */
		}
		down_obj = NO_OBJ; /* so spurious E_MOUSE_UPs will be ignored */
	}
}

/*
	Function to handle a double click.  If it occurred in an object's bounding
	rectangle, we make sure that object is selected (regardless of its current
	state) and then open a dialog box with some information about it.
*/
#ifdef FPROTO
static void do_double(EVENT_TYPE type, PNT where)
#else
static void do_double(type, where)
EVENT_TYPE type;
PNT where;
#endif
{
	int dbl_obj;

	NOREF(type);
	if ((dbl_obj = find_obj(where)) != NO_OBJ) {
		if (sel_obj != dbl_obj) {
			invert_selection();
			sel_obj = dbl_obj;
			invert_selection();
		}
		note("%d points", obj[sel_obj].font.size);
	}
}

#ifdef FPROTO
static void inflate_rect(RCT *rct, int n)
#else
static void inflate_rect(rct, n)
RCT *rct;
int n;
#endif
{	
	rct->top -= n;
	rct->left -= n;
	rct->bottom += n;
	rct->right += n;
}

/*
	Function to handle E_FONT events.  If there is a selected object, the
	call to invert_selection that selected it set the font menu to reflect
	that object's font.  Therefore the FONT object associated with this event
	accurately reflects the new FONT for that object (the original as modified
	by the user's menu choice).

	Note that we invalidate two rectangles for update: the one corresponding
	to the old bounds, and the one corresponding to the new bounds.  Whether
	they overlap or not, this guarantees that all of the window that could
	possibly need updating will be covered by an E_UPDATE event (or several
	of them).  We could instead just invalidate the whole window, but that
	would cause unnecessary updating.
*/
#ifdef FPROTO
static void do_font(FONT *fp)
#else
static void do_font(fp)
FONT *fp;
#endif
{
	RCT rct;
	if (sel_obj != NO_OBJ) {
		rct = obj[sel_obj].bounds;	 /* old bounds */
		inflate_rect(&rct, 1);
		invalidate_rect(std_win, &rct); /* old bounds */
		obj[sel_obj].font = *fp;
		set_bounds(sel_obj);
		rct = obj[sel_obj].bounds;	 /* old bounds */
		inflate_rect(&rct, 1);
		invalidate_rect(std_win, &rct); /* old bounds */
	}
}

/* Main application entry point. */
#ifdef PROTO
void XVTENTRY main_event BTCENTRY(WINDOW win, EVENT_PTR ep)
#else
void XVTENTRY main_event BTCENTRY(win, ep)
WINDOW win;
EVENT_PTR ep;
#endif
{
	switch (ep->type) {
	case E_MOUSE_DOWN:
	case E_MOUSE_UP:
		do_mouse(ep->type, ep->v.mouse.where);
		break;
	case E_MOUSE_DBL:
		do_double(ep->type, ep->v.mouse.where);
		break;
	case E_UPDATE:
		do_update();
		break;
	case E_COMMAND:
		do_menu(ep->v.cmd.tag, ep->v.cmd.shift, ep->v.cmd.control);
		break;
	case E_CLOSE:
		do_close(win);
		break;
	case E_FONT:
		do_font(&ep->v.font.font);
		break;
	case E_QUIT:
		if (ep->v.query)
			quit_OK();
		else
			terminate();
		break;
	}
}

/* Application cleanup.  Nothing to do. */
void XVTENTRY appl_cleanup BTCENTRY(void)
{
}
