$TITLE(SRP Bus Address Mapping for CTOS v 1.1)
$MOD386

BusAddress_Srp	SEGMENT	PUBLIC 'Code'
BusAddress_Srp	ENDS

DGroup		GROUP	DATA
Data		SEGMENT	PUBLIC 'Data'
Data		ENDS

; PUBLIC interfaces exported by this module

		PUBLIC	MapRemoteBusAddress
		PUBLIC	UnmapRemoteBusAddress

; Type definitions used by this module (see CtosTypes.Edf)

; Free slot head of list (zero'th entry in map allocation segment sgRgMapAlloc)

nMapSlots	EQU	WORD PTR 0
cFreeSlots	EQU	WORD PTR 2
oNextFree	EQU	WORD PTR 4
oPrevFree	EQU	WORD PTR 6

; Free slot entry (they are all kept on a doubly linked list)
nSlots		EQU	WORD PTR 0
mapShadow	EQU	WORD PTR 2
;oNextFree	EQU	WORD PTR 4
;oPrevFree	EQU	WORD PTR 6

; Mapped (in use) slot entry (also doubly linked by user number)

;nSlots		EQU	WORD PTR 0
;mapShadow	EQU	WORD PTR 2
oNextMapped	EQU	WORD PTR 4
oPrevMapped	EQU	WORD PTR 6

; EXTERNAL interfaces imported by this module

		EXTRN	DmaAddrFRomP:FAR

; PUBLIC data exported by this module

		PUBLIC	sgRgMapAlloc
		PUBLIC	sgRgMap
		PUBLIC	paRgMap

Data		SEGMENT

sgRgMapAlloc	DW	0
sgRgMap		DW	0
paRgMap		DD	?

Data		ENDS

; Error return codes used by this module

ercBusAddressNotMapped	EQU	170
ercInvalidBusAddress	EQU	171
ercNoFreeMapSlot	EQU	172
$EJECT

BusAddress_Srp	SEGMENT

		ASSUME	CS:BusAddress_Srp
		ASSUME	DS:Data

;------------------------------------------------------------------------------
; The MapBusAddress procedure is used to establish mapper entries for a
; contiguous bus address space (typically used for DMA).  There are two flavors
; of interface used, one which passes a single pbCb and the other which passes 
; a pointer to an array of pbCb's and a count of the number of elements in the
; array.  Both styles have the same number of parameters:
; 
; 	MapBusAddress(pb, cb, dmaFence, pBusAddrRet, pCbRet)
; 
; 	MapRemoteBusAddress(pRgPbCb, nPbCb, dmaFence, pBusAddrRet, pCbRet)
;
; The high byte of the 'dmaFence' parameter is interpreted to separate the two
; cases: a zero value is for the single parameter from while a one or 0FFh in
; the high byte indicates that an array of pbCb's has been passed.  The low
; byte is still interpreted as the DMA fence type, for compatability with
; workstation MapBusAddress calls.

pCbRet		EQU	DWORD PTR 6[BP]
pBusAddrRet	EQU	DWORD PTR 10[BP]
dmaFence	EQU	WORD PTR 14[BP]
nPbCb		EQU	WORD PTR 16[BP]
pRgPbCb		EQU	DWORD PTR 18[BP]

cb		EQU	WORD PTR -2[BP]
linearAddress	EQU	DWORD PTR -6[BP]
linearAddress4K	EQU	WORD PTR -8[BP]
slotsNeeded	EQU	WORD PTR -10[BP]
baBase	EQU DWORD PTR -14[BP]

MapRemoteBusAddress	PROC	FAR

		ENTER	14,0
		PUSH	DS			;Save client's DS for return
		MOV	DX,DGroup		;Switch to our own DGroup
		MOV	DS,DX
		MOV	EAX, paRgMap
		MOV	baBase, EAX
		MOV	FS,sgRgMap		;FS ==> hardware map
		MOV	DS,sgRgMapAlloc		;Finally, switch to map segment
		CALL	CalculateSlots		;See how many slots are needed
		OR	AX,AX
		 JZ	NothingToMap
		CALL	FindFreeSlots		;Scan the allocation map...
		OR	AX,AX
		 JNZ	ExitMap
		DB	66h
		MOVZX	EAX,BX			;Compute the bus address
		SHL	EAX,9
		ADD	EAX, baBase
		OR	AX,linearAddress4K
		LES	DI,pBusAddrRet		;Copy it back to the user
		MOV	ES:[DI],EAX
		LES	DI,pCbRet
		MOV	AX,cb
		MOV	ES:[DI],AX
		CALL	MapSlots
		XOR	AX,AX			;Done! Return ercOK to user
ExitMap:	POP	DS
		XOR	DX,DX			;Guard for CTOS deficiencies
		MOV	FS,DX
		LEAVE
		RET	16

NothingToMap:	LES	BX,pBusAddrRet		;Brain damaged programmer!
		MOV	ES:[BX],EAX
		LES	BX,pCbRet
		MOV	ES:[BX],AX
		JMP	ExitMap

MapRemoteBusAddress	ENDP
$EJECT

;------------------------------------------------------------------------------
; This procedure programs the referenced mapper slots to yield invalid Address
; if they are referenced and marks the slots "free" in the mapper allocation
; control array.  Possible errors are attempts to return map slots that are
; already free or attempts to return map slots within a group of contiguously
; allocated map slots.
;
;	UnmapRemoteBusAddress: PROCEDURE(busAddress) ercType REENTRANT PUBLIC;
;	DECLARE busAddress DWORD;

busAddress	EQU	DWORD PTR 6[BP]

UnmapRemoteBusAddress	PROC	FAR

		ENTER	0,0
		PUSH	DS			;Save user's data segment
		MOV	DX,DGroup		;Switch to OS's DGroup
		MOV	DS,DX
		MOV	EDX, paRgMap
		MOV	FS,sgRgMap		;So we can get hardware...
		MOV	DS,sgRgMapAlloc		;...and allocation segments
		MOV	EAX,busAddress
		CALL	CheckBusAddress
		OR	AX,AX
		 JNZ	ExitUnmap		;Oops! Bus address no good
		MOV	CX,nSlots[BX]		;Deprogram the entire block
		XOR	DX,DX			;Also, slots aren't reserved
		PUSHF
		CLI				;Critical section
		CALL	UnmapSlots
		MOV	AX,nSlots[BX]		;Reestablish AX = slots to free
		ADD	cFreeSlots,AX		;Update free slots list head
CheckUpstairs:	MOV	CX,SI			;[SI] ==> upstairs slots...
		SHR	CX,3			;...are they in range?
		 JZ	CheckDownstairs		;Wrap-around over the top...
		CMP	CX,nMapSlots
		 JAE	CheckDownstairs		;...or just past the top
		TEST	mapShadow[SI],0C000h	;Are the upstairs slots free?
		 JNZ	CheckDownstairs
		ADD	AX,nSlots[SI]		;New, bigger free block
		MOV	nSlots[BX],AX
		MOV	DX,BX			;Save offset of free slot below
		MOV	BX,SI			;[BX] ==> upstairs free slot
		MOV	DI,oNextFree[BX]	;Unlink it from free list
		MOV	SI,oPrevFree[BX]
		MOV	oPrevFree[DI],SI
		MOV	oNextFree[SI],DI
		XOR	CX,CX			;Housekeeping for ex-free block
		MOV	nSlots[BX],CX
		MOV	oNextFree[BX],CX
		MOV	oPrevFree[BX],CX
		MOV	BX,DX			;Reestablish [BX] ==> free slot
CheckDownstairs:MOV	SI,BX			;[SI] ==> downstairs slots
		SUB	SI,8
		OR	SI,SI			;Underflow to free list head?
		 JZ	RelinkFreeSlot
		TEST	mapShadow[SI],0C000h	;Are these slots free?
		 JNZ	RelinkFreeSlot
		XCHG	BX,SI			;Reverse roles
		CMP	nSlots[BX],0		;Start of the free slot block?
		 JNE	MergeDownstairs
		MOV	DX,oPrevFree[BX]	;[DX] ==> downstairs start
		MOV	oPrevFree[BX],0		;Clear the boundary tag
		MOV	BX,DX			;[BX] ==> downstairs slots
MergeDownstairs:ADD	AX,nSlots[BX]		;Build new, bigger free block
		MOV	nSlots[BX],AX
		MOV	nSlots[SI],0		;Clear the vanishing block
		MOV	DI,oNextFree[BX]
		MOV	SI,oPrevFree[BX]
		MOV	oPrevFree[DI],SI
		MOV	oNextFree[SI],DI
RelinkFreeSlot:	MOV	SI,oPrevFree		;Put free slot(s) at end of list
		MOV	oNextFree[BX],0
		MOV	oPrevFree[BX],SI
		MOV	oPrevFree,BX
		MOV	oNextFree[SI],BX
		DEC	AX			;Need a boundary tag?
		 JZ	MapUpdated
		SHL	AX,3
		MOV	DX,BX
		ADD	BX,AX
		MOV	oPrevFree[BX],DX
MapUpdated:	POPF				;End of critical section
		XOR	AX,AX			;Return ercOK
ExitUnmap:	POP	DS			;Restore user's DS...
		XOR	DX,DX			;Guard for CTOS deficiencies
		MOV	FS,DX
		POP	BP
		RET	4
		
UnmapRemoteBusAddress	ENDP
$EJECT

;------------------------------------------------------------------------------
; This procedure processes either a single pbCb or an array of pbCb's to
; determine how many mapper slots are needed to span the DMA request.  The
; linear address of the first (or only) block of memory is needed to make this
; calculation, so (as an optimization) it is saved for later use in programming
; the map hardware and computing a bus address to return to the user.  On exit,
; AX holds the count of slots needed in the hardware map (zero to 17) and CX
; holds the count of bytes requested by the user (not always the same as the
; byte count that will be addressable through the map).  NOTE:  The local stack
; frame variables used are those created by MapBusAddress and RemapBusAddress,
; which call this procedure.

cb		EQU	WORD PTR -2[BP]
linearAddress	EQU	DWORD PTR -6[BP]
linearAddress4K	EQU	WORD PTR -8[BP]
slotsNeeded	EQU	WORD PTR -10[BP]

		PUBLIC	CalculateSlots

CalculateSlots	PROC	NEAR

		LES	BX,pRgPbCb		;Assume single pbCb case
		DB	66h
		MOVZX	EAX,nPbCb
		OR	AX,AX			;No byte count (or no array)?
		 JNZ	CheckForArray
ZeroByteCount:	MOV	ECX,EAX
		RET

CheckForArray:	TEST	dmaFence,0FF00h		;Is there an array?
		 JZ	CalcSlots
		MOV	CX,AX			;Count of array elements
		XOR	EAX,EAX			;Clear to hold total bytes
		MOV	SI,BX			;Use ES:[SI] ==> array elements
CountBytes:	ADD	AX,ES:[SI+4]		;Accumulate total byte count
		ADD	SI,6
		 LOOPNZ	CountBytes		;Continue...
		OR	AX,AX			;Anything to map?
		 JZ	ZeroByteCount
		LES	BX,DWORD PTR ES:[BX]	;Leave ES:[BX] ==> first buffer
CalcSlots:	MOV	cb,AX			;Save total byte count
		PUSH	FS			; Save from DmaAddrFRomP
		PUSH	ES			;OK, translate start of buffers
		PUSH	BX
		CALL	DmaAddrFRomP			;Address returned in DX:AX
		POP	FS
		SHL	EAX,16			;Combine DX and AX into EAX
		MOV	AX,DX
		ROR	EAX,16
		MOV	linearAddress,EAX
		AND	EAX,0FFFh		;Keep offset within 4K page
		MOV	linearAddress4K,AX	;Save this for user later
		ADD	EAX,0FFFh		;Round up if overlaps boundary
		DB	66h
		MOVZX	ECX,cb
		ADD	EAX,ECX			;Add total byte count
		SHR	EAX,12			;Number of 4K pages to map
		MOV	slotsNeeded,AX
		RET

CalculateSlots	ENDP
$EJECT

;------------------------------------------------------------------------------
; Common procedure to scan the linked list of free slots to find a contiguous
; block that may be used to satisfy a MapBusAddress or ReserveBusAddress
; request.  On entry, AX holds the number of slots desired and DS is already
; set to the map allocation segment.  Exit with AX zero and [BX] ==> block of
; free slot(s) if successful, otherwise AX holds 'ercNoFreeMapSlot' and BX is
; undefined.

		PUBLIC	FindFreeSlots

FindFreeSlots	PROC	NEAR

		XOR	BX,BX			;Start at free list head
		PUSHF
		CLI				;Critical section during scan
NextFreeSlot:	MOV	BX,oNextFree[BX]	;[BX] ==> free slot block
		OR	BX,BX
		 JZ	NoFreeMapSlot
		CMP	AX,nSlots[BX]		;Enough slots in this block?
		 JA	NextFreeSlot
		MOV	DI,oNextFree[BX]	;[DI] ==> next free slot
		MOV	SI,oPrevFree[BX]	;[SI] ==> previous free slot
		 JNE	CreateFreeSlot		;More free slots than needed
		SUB	cFreeSlots,AX		;Exact fit, update free count
		DEC	AX			;More than one slot free?
		 JZ	UnlinkFreeSlot
		MOV	DX,BX
		SHL	AX,3			;Calculate [BX] ==> ceiling
		ADD	BX,AX
		MOV	oPrevFree[BX],0		;Clear boundary tag
		MOV	BX,DX
UnlinkFreeSlot:	MOV	AX,oPrevFree[BX]	;Unlink these free slot(s)
		MOV	oPrevFree[DI],AX
		MOV	AX,oNextFree[BX]
		MOV	oNextFree[SI],AX
		JMP	MarkSlotsBusy

CreateFreeSlot:	SUB	cFreeSlots,AX		;Update free slots remaining
		MOV	CX,nSlots[BX]		;Number of slots available
		SUB	CX,AX			;Calculate slots left over
		MOV	nSlots[BX],AX		;Adjust to number needed
		MOV	DX,BX			;Save pointer to free slot
		SHL	AX,3			;Create 'oNewFreeSlot'
		ADD	BX,AX
		MOV	nSlots[BX],CX		;Create new free slot block
		MOV	oNextFree[BX],DI	;Link it into the free list
		MOV	oPrevFree[BX],SI
		MOV	oPrevFree[DI],BX
		MOV	oNextFree[SI],BX
		DEC	CX			;New free block bigger than one?
		 JZ	NoBoundaryTag
		MOV	AX,BX
		SHL	CX,3
		ADD	BX,CX
		MOV	oPrevFree[BX],AX
NoBoundaryTag:	MOV	BX,DX			;Restore [BX] ==> slot block
MarkSlotsBusy:	OR	mapShadow[BX],4000h	;Mark busy even before we map
		MOV	DI,nSlots[BX]
		DEC	DI
		 JZ	SlotsAllocated
		SHL	DI,3
		OR	mapShadow[BX][DI],4000h	;Mark the top, too!
SlotsAllocated:	POPF				;Free list is updated
		XOR	AX,AX
		MOV	oNextMapped[BX],AX
		MOV	oPrevMapped[BX],AX
		RET

NoFreeMapSlot:	POPF				;Leaving critical section
		MOV	AX,ercNoFreeMapSlot	;Unable to fulfill request
		RET

FindFreeSlots	ENDP
$EJECT

;------------------------------------------------------------------------------
; Common procedure to program the hardware map elements and leave a map
; "shadow" in the map allocation data structures.  This is used by both
; MapBusAddress and RemapBusAddress.  On entry, [BX] ==> the first of the map
; allocation slots and CX holds the total count of slots needed.  No registers
; are saved on exit.

		PUBLIC	MapSlots

MapSlots	PROC	NEAR

		MOV	DI,BX			;[BX] ==> map allocation segment
		SHR	DI,1			;FS:[DI] ==> map hardware
		TEST	dmaFence,0FF00h		;Multiple pbCb case?
		 JZ	CalcMapValue		;No, slot count correct already
		LES	SI,pRgPbCb		;Set up ES:[SI] ==> array
NextElement:	DB	66h			;Calculate slots for this entry
		MOVZX	EAX,linearAddress4K
		ADD	EAX,0FFFh		;Round up to 4K boundary
		DB	66h
		MOVZX	ECX,WORD PTR ES:[SI+4]
		ADD	EAX,ECX			;And add byte count
		SHR	EAX,12
		MOV	slotsNeeded,AX
CalcMapValue:
		MOV	EAX,linearAddress
		AND	AX, 0F000h		; round to page
		MOV	CX,slotsNeeded
NextSlot:
		MOV	FS:[DI],EAX		;Enable a mapper slot
		SHR	EAX,12
		MOV	mapShadow[BX],AX
		OR	mapShadow[BX],8000h
		INC	AX
		SHL	EAX, 12
		ADD	DI,4			;Mapper entries spaced as DWORDs
		ADD	BX,8			;Size of allocation entries
		 LOOPNZ	NextSlot		;More slots for this element?
		TEST	dmaFence,0FF00h		;Multiple pbCb case?
		 JNZ	MultiPbCb
MapProgrammed:	RET

MultiPbCb:	DEC	nPbCb			;More array elements?
		 JZ	MapProgrammed
		ADD	SI,6			;Advance to next array element
		PUSH	SI			;Save position in pbCb's
		PUSH	ES
		PUSH	DI			;Save position in hardware map
		PUSH	FS
		PUSH	BX			;Save position in allocation
		PUSH	DWORD PTR ES:[SI]	;Push address pointer...
		CALL	DmaAddrFRomP			;Linear address back in DX, AX
		SHL	EAX,16			;Combine DX and AX into EAX
		MOV	AX,DX
		ROR	EAX,16
		MOV	linearAddress,EAX
		AND	EAX,0FFFh		;Keep offset within 4K page
		MOV	linearAddress4K,AX	;Save this for user later
		POP	BX			;[BX] ==> map allocation
		POP	FS			;FS:[DI] ==> map hardware
		POP	DI
		POP	ES			;ES:[SI] ==> array of pbCb's
		POP	SI
		JMP	NextElement

MapSlots	ENDP
$EJECT

;------------------------------------------------------------------------------
; Common procedure to deprogram the hardware map so that references to the
; range of bus addresses will generate an NMI.  Enter with [BX] pointing to the
; block of slots to be invalidated, CX holding the count of slots to be
; deprogrammed (in case it is not the same size as the whole block of slots)
; and DX containing a bit mask to OR with the map and shadow map values.
; Preserve [BX] on exit but other registers may be destroyed.

		PUBLIC	UnmapSlots

UnmapSlots	PROC	NEAR

		MOV	AX,BX			;Calculate ordinal of map slot
		SHR	AX,3
		OR	AX,DX			;Add "reserved" marker if needed
		MOV	SI,BX			;Use [SI] ==> map allocation...
		MOV	DI,BX			;...and FS:[DI] ==> map hardware
		SHR	DI,1
UnmapSlot:	MOV	FS:[DI],EAX		;Unmap the slot
		MOV	mapShadow[SI],AX	;Mark its shadow unmapped
		INC	AX
		ADD	DI,4			;Mapper entries spaced as DWORDs
		ADD	SI,8			;Size of allocation entries
		 LOOPNZ	UnmapSlot
		RET

UnmapSlots	ENDP

;------------------------------------------------------------------------------
; Common validation procedure for bus address supplied by user.  Enter with the
; bus address in EAX, exit with [BX] ==> start of map slots for this bus
; address and AX zero if the bus address is OK, otherwise exit with
; 'ercInvalidBusAddress' in AX.

		PUBLIC	CheckBusAddress

CheckBusAddress	PROC	NEAR

		SUB	EAX, EDX		; -paRgMap
		 JBE	InvalidBusAddr
		SHR	EAX, 12
		CMP	AX,nMapSlots		;Within range?
		 JAE	InvalidBusAddr
		SHL	AX,3			;Offset of mapped slot(s)
		XCHG	AX, BX
		CMP	nSlots[BX], 0
		JE	InvalidBusAddr
		XOR	AX,AX			;Return 'ercOK'
		RET

InvalidBusAddr:	MOV	AX,ercInvalidBusAddress
		RET

CheckBusAddress	ENDP

BusAddress_Srp	ENDS
$EJECT

;-------------------------------------------------------------------------------
; Modification History
;
;	12 MAY 88 by PGJ	Initial release for CTOS/VM 3.0
;	07 SEP 88 by JA 	Use SetDSOsDGroup
;	15 DEC 88 by RLM	RgMapAlloc no longer in OS DGroup (oRgMapAlloc
;				now pRgMapAlloc)
;	28 FEB 89 by RLM/JA	rgMap is WORD data, but arrayed as if DWORD
;	29 MAR 89 by PGJ	Set mapper enable bits in returned bus address
;	20 MAY 89 by PGJ	Disregard 0E000$0000 in UnmapBusAddress when
;				checking for overflow out of rgMapAlloc
;	22 MAY 89 by PGJ	Modify MapBusAddress interface to permit either
;				a) a single pbCb or b) an array of pbCb's passed
;				as parameters
;	12 JUN 89 by MTR/PGJ	Fix Map/Unmap bugs related to rgPbCb's, critical
;				region and slot allocation calculations
;	03 AUG 89 by JMR	Changed ENABLE and DISABLE to use FLAGS because
;				calls were being made from raw ISR's
;	31 OCT 89 by PGJ	Add RemapBusAddress and ReserveBusAddress
;	13 DEC 89 by PGJ	Change mapper allocation data structures for
;				code optimization; translate to assembler
;	15 DEC 89 by PGJ	Fix critical section bug when updating free list
;	09 FEB 93 by  JA	Adapt for PC/Comarch remote dma
;	11 FEB 93 by  JA	CalculateSlots save FS from DmaAddrFRomP
;
;------------------------------------------------------------------------------

		END
