{$LINESIZE:96}
{$TITLE:'05 FEB 85: System Service Interface for General Synchronous Communications'}
{$PAGESIZE:60}

{$DEBUG-}
{$OCODE+}

module GenSyncService [];

{$INCLUDE:'<PascalEdf>CTOS.Edf'}
{$INCLUDE:'<PascalEdf>ProcessMgt.Edf'}
{$INCLUDE:'<PascalEdf>ParameterMgt.Edf'}
{$INCLUDE:'<PascalEdf>SystemServicesMgt.Edf'}
{$INCLUDE:'<PascalEdf>TimerMgt.Edf'}
{$INCLUDE:'<PascalEdf>InterruptHandlers.Edf'}
{$INCLUDE:'<PascalEdf>8254PIT.Edf'}
{$INCLUDE:'<PascalEdf>8274MPSC.Edf'}
{$PAGE+}
{$INCLUDE:'<PascalEdf>GenSync.Edf'}
{$PAGE+}

var [extern]
   ccb :array[0..1] of ccbType;

var [public]
   bitMask :array[0..15] of bitMap;

value
   bitMask[0] := [bit0];
   bitMask[1] := [bit1];
   bitMask[2] := [bit2];
   bitMask[3] := [bit3];
   bitMask[4] := [bit4];
   bitMask[5] := [bit5];
   bitMask[6] := [bit6];
   bitMask[7] := [bit7];
   bitMask[8] := [bit8];
   bitMask[9] := [bit9];
   bitMask[10] := [bit10];
   bitMask[11] := [bit11];
   bitMask[12] := [bit12];
   bitMask[13] := [bit13];
   bitMask[14] := [bit14];
   bitMask[15] := [bit15];

{$SYMTAB-}

procedure MaskRS232CInterrupts; extern;
procedure UnmaskRS232CInterrupts; extern;
function SetRs232C(rr0Image, ecsrImage :byte; iLine :sint) :rs232CType; extern;
procedure TerminateTx(var ccb :ccbType); extern;
procedure TerminateRx(var ccb :ccbType); extern;

{$SYMTAB+}
{$PAGE+}
{User calls to the GSCS are validated by matching the comm handle supplied
 against the relative addresses of ccb[0] and ccb[1].  If a match is made,
 a further check is performed to see if the user number in the request block
 matches the user number in the Ccb.}

function checkCommHandle(vars rq :rqType; var pCcb :pCcbType) :Boolean;
var
   pCcb0, pCcb1 :pCcbType;

begin
   pCcb := retype(pCcbType, rq.commHandle);
   pCcb0 := adr ccb[0];
   pCcb1 := adr ccb[1];
   if (pCcb = pCcb0) or (pCcb = pCcb1) then
      checkCommHandle := (pCcb^.userNum = rq.userNum)
   else
      checkCommHandle := false;
end;
{$PAGE+}
{This procedure is called when CTOS issues a Termination or Abort request for
 user processes.  If the user number in the request block matches the user
 number in any open Ccb, that channel must be freed.  Furthermore, if a request
 was in progress, that request must receive an error response so that CTOS can
 free the link blocks being used to track the request.  Lastly, the Termination
 request itself receives a response.}

function abortGenSync(vars abortRq :rqType) :ercType;
var
   erc :ercType;
   pCcb :pCcbType;

begin
   for var i := lower(ccb) to upper(ccb) do [
      pCcb := adr ccb[i];
      if pCcb^.open and (abortRq.userNum = pCcb^.userNum) then [
         Disable;				{Yes, enforce critical section}
         if pCcb^.txOn then with pCcb^.txIob do [
            TerminateTx(pCcb^);
            Enable;
            trb.ercRet := ercOK;
            pRq^.ercRet := abortRq.ercAbort;
            erc := Respond(pRq);
         ];
         Disable;
         if pCcb^.rxOn then with pCcb^.rxIob do [
            TerminateRx(pCcb^);
            Enable;
            trb.ercRet := ercOK;
            pRq^.ercRet := abortRq.ercAbort;
            erc := Respond(pRq);
         ];
         Enable;				{Just in case...}
         LockOut(pCcb^.ctrl, channelReset);	{Zap everything for channel}
         pCcb^.open := false;			{Logical close, as well}
         erc := ResetCommISR(wrd(pCcb^.iLine));
         pCcb^.userNum := 0;			{All done with this user}
      ];
   ];
   abortGenSync := ercOK;
end;
{$PAGE+}
{This procedure is called when CTOS issues a ChgUserNum request (which occurs
 when a system service is installed in another or the same partition.  Both of
 the Ccb's are examined to see if their user number matches the old user number
 in the request block, and if so it is changed to the user number.  Then
 response is made to CTOS.}

function changeUserGenSync(vars chgUserNumRq :rqType) :ercType;
var
   erc :ercType;
   pCcb :pCcbType;

begin
   for var i := lower(ccb) to upper(ccb) do [
      pCcb := adr ccb[i];
      if pCcb^.open and (chgUserNumRq.oldUserNum = pCcb^.userNum) then [
         pCcb^.userNum := chgUserNumRq.userNum;
      ];
   ];
   changeUserGenSync := ercOK;
end;
{$PAGE+}
{This process is invoked by a Swapping request from the Context Manager.  The
 process identified by the user number in the request is about to be swapped
 out by the Context Manager.  The system service process must wait until all
 outstanding service requests for the application process are satisfied before
 it responds to the swapping request itself.}

function swapUserGenSync(vars swappingRq :rqType) :ercType;
var
   erc :ercType;
   pCcb :pCcbType;

begin
   for var i := lower(ccb) to upper(ccb) do [
      pCcb := adr ccb[i];
      with pCcb^ do [
         if open and (swappingRq.userNum = userNum) then [
            while txIob.pRq^.ercRet = ercRequestInProgress do
               erc := Delay(1);
            while rxIob.pRq^.ercRet = ercRequestInProgress do
               erc := Delay(1);
         ];
      ];
   ];
   swapUserGenSync := ercOK;
end;
{$PAGE+}
{Open procedure first passes through a tedious section of code that insures
 all the parameters are correct, that the channel isn't already open, that
 the user has asked for a recognizable device ("[Comm]A" or "[Comm]B"), that
 an interrupt service routine can be established and (snore) so on.  Once all
 this brouhaha is out of the way, Gspb open parameters are transferred to the
 Ccb so they may be referenced by the interrupt routines and images of the
 communications chip initialization registers are assembled according to the
 open parameters supplied.  Lastly, interrupts are disabled and the chip is
 initialized for these character lengths, etc. but is not activated for either
 receive or transmit.}

function openGenSync(vars rq :rqType) :ercType;
var [extern]
   pTxIsr :adsMem;
   pExtIsr :adsMem;
   pRxIsr :adsMem;
   pSpRxIsr :adsMem;

var [public]
   pExtCntlReg :ads of byte;			{Copy of last ECSR write}

var [static]
   channel0ID, channel1ID :lstring(7);

var
   term :byte;
   ctrlPort, erc :word;
   deviceID :lstring(74);
   pCcb :pCcbType;
   pGspb :pGspbType;
   pTermTable :adr of array[0..15] of bitMap;

value
   channel0ID := '[Comm]A';			{The only legal devices...}
   channel1ID := '[Comm]B';

begin
   pGspb := rq.pGspb;				{Optimize Gspb references}
   if rq.cbDeviceID <= upper(deviceID) then	{Check maximum length}
      deviceID.len := rq.cbDeviceID
   else
      deviceID.len := upper(deviceID);
   for var i := 1 to ord(deviceID.len) do
      deviceID[i] := rq.pbDeviceID^[i];		{Transfer string to lstring}
   if StringsEqual(ads deviceID, ads channel0ID) then
      pCcb := adr ccb[0]
   else if StringsEqual(ads deviceID, ads channel1ID) then
      pCcb := adr ccb[1]
   else [
      openGenSync := ercUnknownDevice;
      return;
   ];
   if pCcb^.open then [
      openGenSync := ercDeviceInUse;
      return;
   ];
   with pGspb^ do				{Save on typing!!}
      if not (((charSize >= 5) and (charSize <= 8)) and
             ((nSynChar > 0) and (nSynChar <= 2)) and
             ((crcSelect = CRC_16) or (crcSelect = CCITT_CRC)) and
             ((nWires = fullDuplex) or (nWires = halfDuplex))) then [
         openGenSync := ercParameterError;
         return;
      ];
   openGenSync := SetCommISR(wrd(pCcb^.iLine), ads (pCcb^), pTxIsr,
                             pExtIsr, pRxIsr, pSpRxIsr);
   if result(openGenSync) <> ercOK then
      return;
   with pCcb^ do [
      userNum := rq.userNum;
      open := true;
      crcEnable := pGspb^.crcEnable;
      nSynChar := pGspb^.nSynChar;
      synChar[1] := pGspb^.synChar[1];
      synChar[2] := pGspb^.synChar[2];
      synStrip := pGspb^.synStrip;
      nWires := pGspb^.nWires;
      for var i := lower(termTable) to upper(termTable) do
         termTable[i] := retype(bitMap, 0);
      for var i := lower(dleTermTable) to upper(dleTermTable) do
         dleTermTable[i] := retype(bitMap, 0);
      pTermTable := adr termTable;		{First come the stand-alones}
      for var i := 1 to rq.cbTerminators do [
         term := rq.pbTerminators^[i];
         if term = dle then
            pTermTable := adr dleTermTable	{Then the two-char sequences}
         else
            pTermTable^[ord(term div 16)] := pTermTable^[ord(term div 16)] +
                                              bitMask[ord(term mod 16)];
      ];
      wr1Init := extIntEnable or txIntEnable or statusVector or rxIntAll;
      case pGspb^.charSize of
         5: wr3Init := rxChar5;
         6: wr3Init := rxChar6;
         7: wr3Init := rxChar7;
         8: wr3Init := rxChar8;
      end;
      wr4Init := (nSynChar - 1) * 16;
      if pGspb^.parityEnable then
         if odd(pGspb^.parity) then
            wr4Init := wr4Init or enableParity
         else
            wr4Init := wr4Init or (evenParity or enableParity);
      case pGspb^.charSize of
         5: wr5Init := txChar5;
         6: wr5Init := txChar6;
         7: wr5Init := txChar7;
         8: wr5Init := txChar8;
      end;
      rs232C := [];
      if pGspb^.crcSelect = CRC_16 then
         wr5Init := wr5Init or crc16;
      if dataTerminalReady in rq.control then [
         wr5Init := wr5Init or dtr;
         rs232C := rs232C + [dataTerminalReady];
      ];
      if requestToSend in rq.control then [
         wr5Init := wr5Init or rts;
         rs232C := rs232C + [requestToSend];
      ];
      rxOn := false;
      txOn := false;
      MaskRS232CInterrupts;		{Critical section with interrupts}
      if iLine = 0 then
         pExtCntlReg^ := pExtCntlReg^ or ch0ExtClock
      else
         pExtCntlReg^ := pExtCntlReg^ or ch1ExtClock;
      Output(ecsr, pExtCntlReg^);	{External clocking for this channel}
      LockOut(ctrl, channelReset);	{Clear the channel...}
      UnmaskRS232CInterrupts;
      erc := Delay(1);			{Insure delay for 8274 settling}
      MaskRS232CInterrupts;		{Enter critical section...}
      ctrlPort := ctrl;			{Optimize LockOuts below}
      LockOut(ctrlPort, wr4 or resetExtInt);	{Parity, sync length}
      LockOut(ctrlPort, wr4Init);
      LockOut(ctrlPort, wr1 or resetExtInt);	{Interrupt enable settings}
      LockOut(ctrlPort, wr1Init);
      LockOut(ctrlPort, wr3 or resetExtInt);	{Rx char size, CRC enable}
      LockOut(ctrlPort, wr3Init);
      LockOut(ctrlPort, wr5 or resetExtInt);	{Tx char size, CRC type/enable}
      LockOut(ctrlPort, wr5Init);
      LockOut(ctrlPort, wr6 or resetExtInt);	{First SYN character...}
      LockOut(ctrlPort, synChar[1]);
      LockOut(ctrlPort, wr7 or resetExtInt);	{...and second (if present)}
      LockOut(ctrlPort, synChar[2]);
      LockOut(ctrlPort, errorReset);		{Last bit of insurance}
      UnmaskRS232CInterrupts;		{Safe to exit section at last}
   ];
   rq.pCommHandleRet^ := retype(word, pCcb);	{Pass comm handle back to user}
   openGenSync := ercOK;			{Channel initialized OK}
end;
{$PAGE+}
{Communications channel is closed if the caller owned it.  This entails a
 channel reset to the MPSC (allegedly clearing all relevant status/control),
 followed by a logical clean-up of the Ccb.  Last of all, reset the interrupt
 service routines (which should also release the resource back to CTOS)}

function closeGenSync(vars rq :rqType) :ercType;
var
   pCcb :pCcbType;

begin
   if not checkCommHandle(rq, pCcb) then
      closeGenSync := ercInvalidCommHandle
   else if pCcb^.txOn or pCcb^.rxOn then
      closeGenSync := ercDeviceInUse	{Can't close yet}
   else with pCcb^ do [
      MaskRS232CInterrupts;
      LockOut(ctrl, channelReset);	{Zap everything for channel}
      open := false;			{Logical close, as well}
      rxOn := false;			{Redundant, but good for emphasis}
      txOn := false;
      UnmaskRS232CInterrupts;
      userNum := 0;			{Erase user's footprint}
      closeGenSync := ResetCommISR(wrd(iLine));
   ];
end;
{$PAGE+}
{Communications read function enables the receiver and enters "hunt" phase
 after transferring the buffer address/size to the Channel Control Block.
 Then we just settle back and let the interrrupt handler do the rest}

function readGenSync(vars rq :rqType) :ercType;
var
   ctrlPort :word;
   pCcb :pCcbType;

begin
   if not checkCommHandle(rq, pCcb) then
      readGenSync := ercInvalidCommHandle
   else if pCcb^.rxOn then
      readGenSync := ercDeviceInUse
   else with pCcb^.rxIob do [
      trb.ercRet := ercRequestInProgress;	{Clear prior values}
      pRq := ads rq;			{Completion routine needs this address}
      pBuffer := rq.pBuffer;		{Transfer request block parameters}
      sBuffer := rq.sBuffer;
      count := 0;
      rxLastChar := 0;
      rxMode := rq.mode;		{Transparent or not...}
      rxState := rxHunt;		{Start by awaiting synchronization}
      rxCrcOn := false;			{CRC will be turned on after SOH/STX}
      rxCrcDelay := 4;			{Wait two pads after CRC to verify}
      ctrlPort := pCcb^.ctrl;		{Minimize time spent disabled}
      MaskRS232CInterrupts;		{Enforce 8274 critical section}
      pCcb^.rxOn := true;		{Men working sign}
      if rq.timeout <> 0 then
         trb.counter := rq.timeout + 1;	{Allow for RTC jitter}
      LockOut(ctrlPort, resetRxCrc or wr3);	{Clear CRC, select register 3}
      LockOut(ctrlPort, pCcb^.wr3Init or (enterHunt or synLoadInh or
              rxEnable));
      UnmaskRS232CInterrupts;
      readGenSync := ercRequestInProgress;
   ];
end;
{$PAGE+}
{Communications write function transfers the buffer address/count to the
 Channel Control Block, and then enables the transmitter and RTS.  The first
 character is written to the transmitter here (if CTS is already present);
 the interrupt routines will handle the rest.}

function writeGenSync(vars rq :rqType) :ercType;
var
   ctrlPort, dataPort :word;
   pCcb :pCcbType;

begin
   if not checkCommHandle(rq, pCcb) then
      writeGenSync := ercInvalidCommHandle
   else if pCcb^.txOn then
      writeGenSync := ercDeviceInUse
   else with pCcb^.txIob do [
      trb.ercRet := ercRequestInProgress;	{Clear prior values}
      pRq := ads rq;			{Completion routine needs this address}
      pBuffer := rq.pBuffer;		{Transfer request block parameters}
      sBuffer := rq.sBuffer;
      count := 0;
      txLastChar := 0;
      txMode := rq.mode;		{Transparent or not...}
      txState := txRts;			{Start by sending SYN characters}
      txCrcOn := false;			{CRC will be turned on after SOH/STX}
      txSynCount := 2 * pCcb^.nSynChar;	{Prefix with two SYN chars}
      txPadCount := 2;			{Send two pads after CRC}
      ctrlPort := pCcb^.ctrl;		{Minimize time spent disabled}
      dataPort := pCcb^.data;
      MaskRS232CInterrupts;		{Enforce 8274 critical ssection}
      pCcb^.txOn := true;		{Men working sign}
      if rq.timeout <> 0 then
         trb.counter := rq.timeout + 1;	{Allow for RTC jitter}
      LockOut(ctrlPort, resetTxCRC or wr5);	{Clear CRC, select wr5}
      LockOut(ctrlPort, pCcb^.wr5Init or rts);	{Ask permission to send}
      if (LockIn(ctrlPort) and cts) = cts then [	{OK to send already?}
         txState := txSyn;			{Yes, switch to sync state...}
         LockOut(ctrlPort, wr5);		{...and enable the transmitter}
         LockOut(ctrlPort, pCcb^.wr5Init or (txEnable or rts));
         LockOut(dataPort, pCcb^.synChar[1]);	{Leading SYN char...}
         txSynCount := txSynCount - 1;
      ];
      pCcb^.rs232C := pCcb^.rs232C + [requestToSend];	{Make note!}
      UnmaskRS232CInterrupts;
      writeGenSync := ercRequestInProgress;
   ];
end;
{$PAGE+}
{The deinstalltion procedure works only if the GSCS was originally installed
 in a multi-partition operating system in a secondary partition and if both of
 the communications channels are closed.  If these tests are met, the service
 exchanges are restored to their original values, a check is made to verify
 that no messages are queued at the exchanges of any of the other GSCS
 processes and a "service unavailable" return code is issued to any GSCS
 requests that were queued after the deinstall request.  Finally, the deinstall
 request is given an OK return code and a token is sent to the main process---
 this awakens it and causes it to unlock the partition and remove itself.}

function deinstallGenSync(vars rq :rqType) :ercType;
var [extern]
   complExch, deinstallExch, servExch, timerExch :word;
   origExch :array[0..9] of word;
   genSyncDeinstallable :Boolean;

var
   erc, iRq, otherErc :word;
   pMsgRet :adsMem;
   pRq :pRqType;

begin
   if not genSyncDeinstallable then
      deinstallGenSync := ercUnableToDeinstall
   else if ccb[0].open or ccb[1].open then
      deinstallGenSync := ercDeviceInUse
   else [
      for var i := 0 to 9 do [
         iRq := abortGenSyncRqCode + wrd(i);
         erc := ServeRq(iRq, origExch[i]);
      ];
      repeat
         erc := Check(complExch, ads pMsgRet);
      until erc = ercNoMessageAvailable;
      repeat
         erc := Check(timerExch, ads pMsgRet);
      until erc = ercNoMessageAvailable;
      repeat [
         erc := Check(servExch, ads pRq);
         if erc = ercOK then [
            pRq^.ercRet := ercServiceNotAvailable;
            otherErc := Respond(pRq);
         ];
      ] until erc = ercNoMessageAvailable;
      erc := Send(deinstallExch, pRq);
      deinstallGenSync := ercOK;
   ];
end;
{$PAGE+}
{Status function reads the state of RLSD and CTS from the MPSC, and then
 OR's it with the "remembered" DTR and RTS states to return the composite
 to the caller}

function statusGenSync(vars rq :rqType) :ercType;
var
   rr0Image, ecsrImage :byte;
   pCcb :pCcbType;

begin
   if checkCommHandle(rq, pCcb) then with pCcb^ do [
      MaskRS232CInterrupts;		{Critical section while accessing 8274}
      LockOut(ctrl, rr0);		{Select status register 0}
      rr0Image := LockIn(ctrl);
      ecsrImage := Input(ecsr);
      rs232C := rs232C * [dataTerminalReady, requestToSend];
      rs232C := rs232C + SetRs232C(rr0Image, ecsrImage,
                       iLine);
      rq.pStatusRet^ := rs232C;
      UnmaskRS232CInterrupts;
      statusGenSync := ercOK;
   ] else
      statusGenSync := ercInvalidCommHandle;
end;
{$PAGE+}
{Control function allows the user to control the state of DTR and RTS.  Since
 these cannot be retrieved from the MPSC after they are written, the last
 output state of these signals is saved in a variable in the CCB}

function controlGenSync(vars rq :rqType) :ercType;
var
   pCcb :pCcbType;

begin
   if checkCommHandle(rq, pCcb) then with pCcb^ do [
      MaskRS232CInterrupts;
      if dataTerminalReady in rq.control then [
         wr5Init := wr5Init or dtr;
         rs232C := rs232C + [dataTerminalReady];
      ] else [
         wr5Init := wr5Init and (not dtr);
         rs232C := rs232C - [dataTerminalReady];
      ];
      if requestToSend in rq.control then [
         wr5Init := wr5Init or rts;
         rs232C := rs232C + [requestToSend];
      ] else [
         wr5Init := wr5Init and (not rts);
         rs232C := rs232C - [requestToSend];
      ];
      LockOut(ctrl, wr5);		{Select write register 5}
      LockOut(ctrl, wr5Init);
      UnmaskRS232CInterrupts;
      controlGenSync := ercOK;
   ] else
      controlGenSync := ercInvalidCommHandle;
end;
{$PAGE+}
{Request handler is the central dispatcher for all request blocks that CTOS
 routes to this exchange.  All the recognizable request codes are passed on
 to the appropriate procedures; any unrecognized requests are simply ignored.
 After being processed by the request-specific procedures, requests are in one
 of two states: completed (either OK or in error) or still in progress.  If the
 request has completed, a response is made immediately, otherwise the response
 is made later by the response handler process (see next page).}

procedure RequestHandler [public];
var [extern]
   servExch :word;

var
   erc :ercType;
   pRq :pRqType;

begin
   while true do [
      erc := Wait(servExch, ads pRq);
      with pRq^ do [
         case rqCode of
            abortGenSyncRqCode:		ercRet := abortGenSync(pRq^);
            changeUserGenSyncRqCode:	ercRet := changeUserGenSync(pRq^);
            swapUserGenSyncRqCode:	ercRet := swapUserGenSync(pRq^);
            openGenSyncRqCode:		ercRet := openGenSync(pRq^);
            closeGenSyncRqCode:		ercRet := closeGenSync(pRq^);
            readGenSyncRqCode:		ercRet := readGenSync(pRq^);
            writeGenSyncRqCode:		ercRet := writeGenSync(pRq^);
            deinstallGenSyncRqCode:	ercRet := deinstallGenSync(pRq^);
            statusGenSyncRqCode:	ercRet := statusGenSync(pRq^);
            controlGenSyncRqCode:	ercRet := controlGenSync(pRq^);
            otherwise			ercRet := ercServiceNotAvailable;
         end;
         if ercRet <> ercRequestInProgress then
            erc := Respond(pRq);
      ];
   ];
end;
{$PAGE+}
{Response handler is one of the three processes that make up the general
 synchronous communications system service.  Requests that were not immediately
 completed are eventually satisfied here, either as a result of normal
 completion or an error condition.  At present, only the read and write
 requests pass by here.}

procedure ResponseHandler [public];
var [extern]
   complExch :word;

var
   erc :word;
   pMsgRet :adsMem;
   pIob :pIobType;

begin
   while true do [
      erc := Wait(complExch, ads pMsgRet);
      pIob := retype(pIobType, pMsgRet.r);
      with pIob^.pRq^ do [			{Save a little time...}
         ercRet := pIob^.trb.ercRet;	{Copy the erc to user's}
         if (rqCode = readGenSyncRqCode) or (rqCode = writeGenSyncRqCode) then
            psDataRet^ := pIob^.count;
         erc := Respond(pIob^.pRq);		{Send it all back}
      ];
   ];
end;
{$PAGE+}
{Timer function is the third of the processes that make up the system service.
 When a read or write is requested which has a nonzero timeout value, that
 value is transferred to the counter field of the channel's timer request
 block.  Usually, the counter field is reset to zero by the interrupt service
 when the read or write completes and thus the timer is never set off, but if
 it does expire this procedure is entered.  Because the communications inter-
 rupt service routines run at a higher priority than this process, a critical
 section is enforced while the timer examines the "request in progress" flag
 and it aborts transmit and/or receive functions as appropriate.  Once this is
 done, it exits the critical section and posts the timeout error to the user.}

procedure Timeout [public];
var [extern]
   timerExch :word;

var
   erc :word;
   pMsgRet :adsMem;
   pIob :pIobType;
   pCcb :pCcbType;

begin
   while true do [
      erc := Wait(timerExch, ads pMsgRet);
      pIob := retype(pIobType, pMsgRet.r);
      pCcb := retype(pCcbType, pIob^.trb.rqCode);
      pIob^.trb.cEvents := 0;		{First, reset trigger}
      MaskRS232CInterrupts;
      if pIob^.trb.ercRet = ercRequestInProgress then [
         if pCcb^.txOn and (pIob = adr pCcb^.txIob) then
            TerminateTx(pCcb^);
         if pCcb^.rxOn and (pIob = adr pCcb^.rxIob) then
            TerminateRx(pCcb^);
         if pCcb^.wr5Init and dtr = 0 then
            pCcb^.rs232C := pCcb^.rs232C - [dataTerminalReady];
         if pCcb^.wr5Init and rts = 0 then
            pCcb^.rs232C := pCcb^.rs232C - [requestToSend];
         UnmaskRS232CInterrupts;
         pIob^.trb.ercRet := ercTimeout;
         erc := Send(pIob^.complExch, ads (pIob^));
      ];
      UnmaskRS232CInterrupts;		{In case we had nothing to do...}
   ];
end;

end.
