/*
 * Copyright (c) 1994 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Data structure declarations for extended precision rational arithmetic.
 */

#ifndef	QMATH_H
#define	QMATH_H

#include "zmath.h"


/*
 * Rational arithmetic definitions.
 */
typedef struct {
	ZVALUE num;		/* numerator (containing sign) */
	ZVALUE den;		/* denominator (always positive) */
	long links;		/* number of links to this value */
} NUMBER;

extern NUMBER _qlge_;

/*
 * Input. output, allocation, and conversion routines.
 */
extern NUMBER *qalloc PROTO((void));
extern NUMBER *qcopy PROTO((NUMBER *q));
extern NUMBER *uutoq PROTO((FULL i1, FULL i2));
extern NUMBER *iitoq PROTO((long i1, long i2));
extern NUMBER *XX_strtoq PROTO((char *str));
extern NUMBER *itoq PROTO((long i));
extern NUMBER *utoq PROTO((FULL i));
extern long qtoi PROTO((NUMBER *q));
extern FULL qtou PROTO((NUMBER *q));
extern long qparse PROTO((char *str, int flags));
extern void qfreenum PROTO((NUMBER *q));
extern void qprintnum PROTO((NUMBER *q, int mode));
extern void qprintff PROTO((NUMBER *q, long width, long precision));
extern void qprintfe PROTO((NUMBER *q, long width, long precision));
extern void qprintfr PROTO((NUMBER *q, long width, BOOL force));
extern void qprintfd PROTO((NUMBER *q, long width));
extern void qprintfx PROTO((NUMBER *q, long width));
extern void qprintfb PROTO((NUMBER *q, long width));
extern void qprintfo PROTO((NUMBER *q, long width));
extern void qprintf PROTO((char *, ...));



/*
 * Basic numeric routines.
 */
extern NUMBER *qaddi PROTO((NUMBER *q, long i));
extern NUMBER *qmuli PROTO((NUMBER *q, long i));
extern NUMBER *qdivi PROTO((NUMBER *q, long i));
extern NUMBER *qadd PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qsub PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qmul PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qdiv PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qquo PROTO((NUMBER *q1, NUMBER *q2, long rnd));
extern NUMBER *qmod PROTO((NUMBER *q1, NUMBER *q2, long rnd));
extern NUMBER *qmin PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qmax PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qand PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qor PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qxor PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qpowermod PROTO((NUMBER *q1, NUMBER *q2, NUMBER *q3));
extern NUMBER *qpowi PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qsquare PROTO((NUMBER *q));
extern NUMBER *qneg PROTO((NUMBER *q));
extern NUMBER *qsign PROTO((NUMBER *q));
extern NUMBER *qint PROTO((NUMBER *q));
extern NUMBER *qfrac PROTO((NUMBER *q));
extern NUMBER *qnum PROTO((NUMBER *q));
extern NUMBER *qden PROTO((NUMBER *q));
extern NUMBER *qinv PROTO((NUMBER *q));
extern NUMBER *qabs PROTO((NUMBER *q));
extern NUMBER *qinc PROTO((NUMBER *q));
extern NUMBER *qdec PROTO((NUMBER *q));
extern NUMBER *qshift PROTO((NUMBER *q, long n));
extern NUMBER *qtrunc PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qround PROTO((NUMBER *q, long places, long rnd));
extern NUMBER *qbtrunc PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qbround PROTO((NUMBER *q, long places, long rnd));
extern NUMBER *qscale PROTO((NUMBER *q, long i));
extern BOOL qdivides PROTO((NUMBER *q1, NUMBER *q2));
extern BOOL qcmp PROTO((NUMBER *q1, NUMBER *q2));
extern BOOL qcmpi PROTO((NUMBER *q, long i));
extern FLAG qrel PROTO((NUMBER *q1, NUMBER *q2));
extern FLAG qreli PROTO((NUMBER *q, long i));
extern BOOL qisset PROTO((NUMBER *q, long i));


/*
 * More complicated numeric functions.
 */
extern NUMBER *qcomb PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qgcd PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qlcm PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qfact PROTO((NUMBER *q));
extern NUMBER *qpfact PROTO((NUMBER *q));
extern NUMBER *qminv PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qfacrem PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qperm PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qgcdrem PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qlowfactor PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qfib PROTO((NUMBER *q));
extern NUMBER *qcfappr PROTO((NUMBER *q, NUMBER *epsilon, long R));
extern NUMBER *qcfsim PROTO((NUMBER *q, long R));
extern NUMBER *qisqrt PROTO((NUMBER *q));
extern NUMBER *qjacobi PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qiroot PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qmappr PROTO((NUMBER *q, NUMBER *e, long R));
extern NUMBER *qlcmfact PROTO((NUMBER *q));
extern NUMBER *qminmod PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qredcin PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qredcout PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qredcmul PROTO((NUMBER *q1, NUMBER *q2, NUMBER *q3));
extern NUMBER *qredcsquare PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qredcpower PROTO((NUMBER *q1, NUMBER *q2, NUMBER *q3));
extern BOOL qprimetest PROTO((NUMBER *q1, NUMBER *q2, NUMBER *q3));
extern BOOL qissquare PROTO((NUMBER *q));
extern long qilog2 PROTO((NUMBER *q));
extern long qilog10 PROTO((NUMBER *q));
extern long qilog PROTO((NUMBER *q1, NUMBER *q2));
extern BOOL qcmpmod PROTO((NUMBER *q1, NUMBER *q2, NUMBER *q3));
extern BOOL qquomod PROTO((NUMBER *q1, NUMBER *q2, NUMBER **retdiv,
	NUMBER **retmod));
extern FLAG qnear PROTO((NUMBER *q1, NUMBER *q2, NUMBER *epsilon));
extern long qdigit PROTO((NUMBER *q, long i));
extern long qprecision PROTO((NUMBER *q));
extern long qplaces PROTO((NUMBER *q));
extern long qdigits PROTO((NUMBER *q));
extern void setepsilon PROTO((NUMBER *q));
extern NUMBER *qbitvalue PROTO((long i));
extern NUMBER *qtenpow PROTO((long i));

#if 0
extern NUMBER *qmulmod PROTO((NUMBER *q1, NUMBER *q2, NUMBER *q3));
extern NUMBER *qsquaremod PROTO((NUMBER *q1, NUMBER *q2));
extern NUMBER *qaddmod PROTO((NUMBER *q1, NUMBER *q2, NUMBER *q3));
extern NUMBER *qsubmod PROTO((NUMBER *q1, NUMBER *q2, NUMBER *q3));
extern NUMBER *qreadval PROTO((FILE *fp));
extern NUMBER *qnegmod PROTO((NUMBER *q1, NUMBER *q2));
extern BOOL qbittest PROTO((NUMBER *q, long i));
extern FLAG qtest PROTO((NUMBER *q));
#endif


/*
 * Transcendental functions.  These all take an epsilon argument to
 * specify the required accuracy of the calculation.
 */
extern void qsincos PROTO((NUMBER *q, long bitnum, NUMBER **vs, NUMBER **vc));
extern NUMBER *qsqrt PROTO((NUMBER *q, NUMBER *epsilon, long R));
extern NUMBER *qpower PROTO((NUMBER *q1, NUMBER *q2, NUMBER *epsilon));
extern NUMBER *qroot PROTO((NUMBER *q1, NUMBER *q2, NUMBER *epsilon));
extern NUMBER *qcos PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qsin PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qexp PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qln PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qtan PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qsec PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qcot PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qcsc PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qacos PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qasin PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qatan PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qasec PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qacsc PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qacot PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qatan2 PROTO((NUMBER *q1, NUMBER *q2, NUMBER *epsilon));
extern NUMBER *qhypot PROTO((NUMBER *q1, NUMBER *q2, NUMBER *epsilon));
extern NUMBER *qcosh PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qsinh PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qtanh PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qcoth PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qsech PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qcsch PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qacosh PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qasinh PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qatanh PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qasech PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qacsch PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qacoth PROTO((NUMBER *q, NUMBER *epsilon));
extern NUMBER *qlegtoleg PROTO((NUMBER *q, NUMBER *epsilon, BOOL wantneg));
extern NUMBER *qpi PROTO((NUMBER *epsilon));


/*
 * external swap functions
 */
extern NUMBER *swap_b8_in_NUMBER PROTO((NUMBER *dest, NUMBER *src, BOOL all));
extern NUMBER *swap_b16_in_NUMBER PROTO((NUMBER *dest, NUMBER *src, BOOL all));
extern NUMBER *swap_HALF_in_NUMBER PROTO((NUMBER *dest, NUMBER *src, BOOL all));


/*
 * macro expansions to speed this thing up
 */
#define qiszero(q)	(ziszero((q)->num))
#define qisneg(q)	(zisneg((q)->num))
#define qispos(q)	(zispos((q)->num))
#define qisint(q)	(zisunit((q)->den))
#define qisfrac(q)	(!zisunit((q)->den))
#define qisunit(q)	(zisunit((q)->num) && zisunit((q)->den))
#define qisone(q)	(zisone((q)->num) && zisunit((q)->den))
#define qisnegone(q)	(zisnegone((q)->num) && zisunit((q)->den))
#define qistwo(q)	(zistwo((q)->num) && zisunit((q)->den))
#define qiseven(q)	(zisunit((q)->den) && ziseven((q)->num))
#define qisodd(q)	(zisunit((q)->den) && zisodd((q)->num))
#define qistwopower(q)	(zisunit((q)->den) && zistwopower((q)->num))

#define qhighbit(q)	(zhighbit((q)->num))
#define qlowbit(q)	(zlowbit((q)->num))
#define qdivcount(q1, q2)	(zdivcount((q1)->num, (q2)->num))
#define qlink(q)	((q)->links++, (q))

#define qfree(q)	{if (--((q)->links) <= 0) qfreenum(q);}


/*
 * Flags for qparse calls
 */
#define QPF_SLASH	0x1	/* allow slash for fractional number */
#define QPF_IMAG	0x2	/* allow trailing 'i' for imaginary number */


/*
 * constants used often by the arithmetic routines
 */
extern NUMBER _qzero_, _qone_, _qnegone_, _qonehalf_, _qonesqbase_;

#endif
