/*-
 * Copyright (c) 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *      BSDI $Id: asyncdaemon.c,v 2.3 1996/01/10 16:09:52 karels Exp $
 */

#include <sys/param.h>
#include <sys/malloc.h>
#include <sys/proc.h>
#include <sys/queue.h>
#include <vm/vm.h>

/*
 * This file implements asynchronous I/O daemons.
 * To create an asynchronous I/O daemon,
 * just execute the asyncdaemon() system call as root.
 *
 * Currently this code supports only async paging requests,
 * so that the page daemon need never block on I/O.
 * Only the vnode pager currently makes use of this feature.
 * Requests are queued with async_putpages().
 * An async daemon will dequeue each request and
 * execute vm_pager_put_pages() on it.
 *
 * Note that there is no MP locking in this code,
 * and it is assumed that async_putpages() will never
 * be called from interrupt level.
 */

struct async_request {
	TAILQ_ENTRY(async_request)	list;
	vm_pager_t			pager;
	vm_page_t			*mlist;
	vm_page_t			m;
	int				npages;
};

TAILQ_HEAD(async_request_queue_head, async_request);
struct async_request_queue_head async_request_queue;
struct async_request_queue_head async_request_free_queue;

int async_request_free_count;
int asyncdaemon_count;
struct async_request *async_requests;

static enum { NEEDS_INIT = 0, STARTED, READY, SHUTDOWN } phase;

/*
 * If we can do so, queue this asynchronous pageout request
 * and arrange for an async I/O daemon to execute it later.
 */
int
async_putpages(pager, mlist, npages)
	vm_pager_t pager;
	vm_page_t *mlist;
	int npages;
{
	struct async_request *a;

#ifdef DEBUG
	if (curproc == NULL)
		panic("async_putpages");
#endif
	if (phase != READY || curproc->p_flag & P_ASYNCDAEMON)
		return (FALSE);
	if (async_request_free_count == 0) {
#ifdef DEBUG
		printf("async_putpages: request queue exhausted\n");
#endif
		return (FALSE);
	}

	a = async_request_free_queue.tqh_first;
	TAILQ_REMOVE(&async_request_free_queue, a, list);
	--async_request_free_count;

	a->pager = pager;
	a->npages = npages;

	if (npages == 1) {
		a->m = *mlist;
		a->mlist = &a->m;
	} else {
		/* XXX is M_WAITOK safe here?  currently this never happens */
		MALLOC(a->mlist, vm_page_t *, npages * sizeof (vm_page_t),
		    M_TEMP, M_WAITOK);
		bcopy(mlist, a->mlist, npages * sizeof (vm_page_t));
	}

	TAILQ_INSERT_TAIL(&async_request_queue, a, list);

	wakeup(&async_request_queue);

	return (TRUE);
}

/*
 * Enter the kernel and become an async I/O daemon.
 */
int
asyncdaemon(p, uap, retval)
	struct proc *p;
	void *uap;
	int *retval;
{
	struct async_request *a;
	int i;
	int error;
	int exiting;

	if (error = suser(p->p_ucred, &p->p_acflag))
		return (error);

	switch (phase) {

	case NEEDS_INIT:
		/*
		 * There is essentially no throttling on the number
		 * of pages that can be dumped into a pager at any time,
		 * so we must budget for the maximum number of requests.
		 */

		phase = STARTED;
		TAILQ_INIT(&async_request_queue);
		TAILQ_INIT(&async_request_free_queue);

		/* Very conservative...  */
		i = last_page - first_page;

		for (;;) {
			/*
			 * Use M_NOWAIT so that we don't get
			 * lazy allocation of our memory.
			 */
			MALLOC(a, struct async_request *, i * sizeof *a,
			    M_ASYNCREQ, M_NOWAIT);
			if (a)
				break;
			tsleep(&async_request_queue, PVM, "asyncmem", 10);
		}

		async_requests = a;
		async_request_free_count = i;
		for (i = 0; i < async_request_free_count; ++i, ++a)
			TAILQ_INSERT_TAIL(&async_request_free_queue, a, list);

		phase = READY;
		break;

	case STARTED:
		/* Someone else is busy getting memory. */
		while (phase != READY)
			tsleep(&async_request_queue, PVM, "asyncstart", 30);
		break;

	case READY:
		break;
	}

	p->p_flag |= P_ASYNCDAEMON;
	++asyncdaemon_count;

	for (exiting = 0;;) {

		while ((a = async_request_queue.tqh_first) == NULL) {
			if (exiting) {
				if (--asyncdaemon_count == 0) {
					free(async_requests, M_ASYNCREQ);
					phase = NEEDS_INIT;
				}
				p->p_flag &= ~P_ASYNCDAEMON;
				return (error);
			}
			if (error = tsleep(&async_request_queue, PVM | PCATCH,
			    "asyncreq", 0)) {
				exiting = 1;
				continue;
			}
		}

		TAILQ_REMOVE(&async_request_queue, a, list);
		vm_pager_put_pages(a->pager, a->mlist, a->npages, FALSE);

		for (i = 0; i < a->npages; ++i)
			PAGE_WAKEUP(a->mlist[i]);
		--a->mlist[0]->object->paging_in_progress;

		if (a->mlist != &a->m)
			FREE(a->mlist, M_TEMP);

		/* We could re-queue failed requests here... */
		TAILQ_INSERT_TAIL(&async_request_free_queue, a, list);
		++async_request_free_count;
	}
}
