/*-
 * Copyright (c) 1993, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: ppp_lcp.c,v 2.6 1995/12/04 04:00:29 prb Exp $
 *	Urner Id: ppp_lcp.c,v 1.3.0.8 1994/12/03 00:08:29 dlu Exp
 */

/*
 * This code is partially derived from CMU PPP.
 *
 * Copyright (c) 1989 Carnegie Mellon University.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by Carnegie Mellon University.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

/*
 * PPP Link Control Protocol
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/proc.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/syslog.h>
#include <sys/vmmeter.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/netisr.h>
#include <net/route.h>
#include <net/if_llc.h>
#include <net/if_dl.h>

#if INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#endif

#include <net/ppp_proto.h>
#include <net/slcompress.h>
#include <net/pppvar.h>
#include <net/if_p2p.h>

struct p2pcom   *ppp_ifs;       /* the list of active PPP interfaces */
int              ppp_idletimer; /* idle timer is active */

extern struct mbuf *ppp_addopt();

#define	PPP_LOOP_MAX	10	/* # times dup packets -> link looped */
int ppp_loop_max = PPP_LOOP_MAX;

/*
 * Initialize the default values for PPP parameters
 */
static void
ppp_defaults(pp, name)
	struct p2pcom *pp;
	char *name;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;

	if (ppp->ppp_dflt.ppp_mru != 0)
		return;

	/* The default parameters is for the best case */
	ppp->ppp_dflt.ppp_cmap = 0x0;
	ppp->ppp_dflt.ppp_flags = PPP_FTEL;
	ppp->ppp_dflt.ppp_mru = LCP_DFLT_RMU;
	ppp->ppp_dflt.ppp_idletime = 0;   /* idle timer disabled */
	ppp->ppp_dflt.ppp_maxconf = 0;    /* connect timer disabled */
	ppp->ppp_dflt.ppp_maxterm = 3;    /* 3 attempts to terminate */
	ppp->ppp_dflt.ppp_timeout = 30;   /* 3 sec */

	/* Dial-up defauts are a bit different */
	if (!strcmp(name, "ppp")) {
		ppp->ppp_dflt.ppp_maxconf = 10;  /* 30 sec */
		ppp->ppp_dflt.ppp_flags |= PPP_PFC | PPP_ACFC | PPP_TCPC;
	}
#if defined(PPP_DEBUG) && PPP_DEBUG == 1
	ppp->ppp_dflt.ppp_flags |= PPP_TRACE;
#endif
	ppp->ppp_params = ppp->ppp_dflt;
}

/*
 * The PPP ioctl routine
 */
int
ppp_ioctl(pp, cmd, data)
	struct p2pcom *pp;
	int cmd;
	caddr_t data;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;
	struct ppp_ioctl *io = ifr_pppioctl(data);
	int s;
	int error;
	extern struct ifnet *ppp_dropif;
	struct ppp_fsm *fs;
	struct proc *p = curproc;	/* XXX should be parameter */

	/*
	 * We can get here via ppp_ipcp_setaddr when ipcp sets
	 * addresses from interrupt level, in which case curproc (p)
	 * is NULL.  Check suser only for calls implemented here
	 * that modify state.
	 */
	switch (cmd) {
	case PPPIOCNPAR:
	case PPPIOCGPAR:
	case PPPIOCSPAR:
		if (error = suser(p->p_ucred, &p->p_acflag))
			return (error);
		break;
	default:
		error = 0;
		break;
	}

	s = splimp();
	switch (cmd) {
	case PPPIOCNPAR:
		*io = ppp->ppp_params;
		if (ppp->ppp_lcp.fsm_state == CP_OPENED)
			break;
		/* FALL THROUGH */
	case PPPIOCGPAR:
		ppp_defaults(pp, pp->p2p_if.if_name);
		*io = ppp->ppp_dflt;
		break;

	case PPPIOCSPAR:
		if (io->ppp_mru < LCP_MIN_RMU || io->ppp_timeout < 1 ||
		    io->ppp_maxterm < 1)
			error = EINVAL;
		else
			ppp->ppp_dflt = *io;
		break;

	case PPPIOCWAIT:        /* wait for a dropped packet */
		if_qflush(&ppp->ppp_ipdefq);
		ppp->ppp_wdrop = 1;
		error = tsleep((caddr_t) &ppp->ppp_wdrop,
			       PCATCH|(PZERO+10), "pppwait", 0);
		break;

#ifdef INET
	case PPPIOCIPWBOS:        /* wait for tlu event from IPCP */
		if (ppp->ppp_ipcp.fsm_state != CP_OPENED) {
			dprintf(("PPPIOCIPWBOS %x: waiting\n",
			    &ppp->ppp_ipwbos));
			ppp->ppp_ipwbos = 1;
			error = tsleep((caddr_t) &ppp->ppp_ipwbos,
			    PCATCH | (PZERO + 10), "pppipbos", 0);
			if (ppp->ppp_ipcp.fsm_state != CP_OPENED) {
				dprintf(("PPPIOCIPWBOS %x: failed\n",
				    &ppp->ppp_ipwbos));
				error = ENOTCONN;
			}
		} else {
			dprintf(("PPPIOCIPWBOS %x: already opened\n",
			    &ppp->ppp_ipwbos));
		}
		break;
#endif /* INET */

	default:
		error = EINVAL;
	}
	splx(s);
	return (error);
}

/*
 * Initialize the PPP protocol on an interface
 */
int
ppp_attach(pp)
	struct p2pcom *pp;
{
	struct ppp_sc *ppp;

#ifdef	DEBUG
	if (pp->p2p_private)
		panic("ppp_attach: interface already attached to protocol");
#endif

	ppp = malloc(sizeof(struct ppp_sc), M_DEVBUF, M_WAITOK);
	bzero(ppp, sizeof(*ppp));
	MGETHDR(pp->p2p_ibuf, M_WAITOK, MT_DATA);
	pp->p2p_ibuf->m_len = 0;
	pp->p2p_private = ppp;
	ppp->ppp_vjc.vjc_mbuf = pp->p2p_ibuf;
	ppp->ppp_vjc.vjc_ifp = &pp->p2p_if;
	ppp->ppp_vjc.vjc_basement = PH_LINK;
	return (0);
}

/*
 * Shutdown protocol on an interface
 */
void
ppp_detach(pp)
	struct p2pcom *pp;
{
	struct ppp_sc *ppp;

	if (pp->p2p_ibuf)
		m_freem(pp->p2p_ibuf);
	if (ppp = (struct ppp_sc *)pp->p2p_private) {
		free(ppp, M_DEVBUF);
		pp->p2p_private = 0;
	}
#ifdef	DEBUG
	else
		printf("ppp_detach: not attached\n");
#endif
}

/*
 * Initialize the PPP protocol on an interface
 */
int
ppp_init(pp)
	struct p2pcom *pp;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;
	extern void ppp_idlepoll __P((void *));
	int s;

#ifdef INET
	ppp->ppp_ctcp = malloc(sizeof(*ppp->ppp_ctcp), M_DEVBUF, M_WAITOK);
#endif

	s = splimp();

	pp->p2p_prevp = &ppp_ifs;
	if (pp->p2p_next = ppp_ifs)
		ppp_ifs->p2p_prevp = &pp->p2p_next;
	ppp_ifs = pp;
	if (ppp_idletimer == 0) {
		ppp_idletimer++;
		timeout(ppp_idlepoll, 0, hz);
	}

	dprintf(("%s%d: init PPP link\n", pp->p2p_if.if_name,
	    pp->p2p_if.if_unit));

	ppp->ppp_wdrop = 1;      /* For any case... */
	ppp->ppp_lcp.fsm_state = CP_STARTING;
#ifdef INET
	ppp->ppp_ipcp.fsm_state = CP_STARTING;
#endif

	/*
	 * Raise DTR (assume it is ON on local lines)
	 * We should get the "got CD event"
	 */
	if (pp->p2p_mdmctl)
		(*pp->p2p_mdmctl)(pp, 1);
	else
		ppp_modem(pp, 1);
	splx(s);
	return (0);
}

/*
 * Shutdown protocol on an interface
 */
void
ppp_shutdown(pp)
	struct p2pcom *pp;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;
	int s;
	struct mbuf *m;

	dprintf(("%s%d: shutdown PPP link\n", pp->p2p_if.if_name,
	    pp->p2p_if.if_unit));

	s = splimp();
	if (pp->p2p_next)
		pp->p2p_next->p2p_prevp = pp->p2p_prevp;
	*(pp->p2p_prevp) = pp->p2p_next;

	ppp_cp_close(pp, FSM_LCP);
	if (ppp->ppp_ctcp)
		free(ppp->ppp_ctcp, M_DEVBUF);
	ppp->ppp_ctcp = NULL;

	splx(s);
}

/*
 * Turn upper levels Up
 */
void
ppp_lcp_tlu(pp)
	struct p2pcom *pp;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;

	/* Load the async control characters map */
	ppp->ppp_txcmap = ppp->Ppp_cmap;

#ifdef INET
	sl_compress_init(ppp->ppp_ctcp);
	ppp->ppp_ncps++;
	ppp_cp_up(pp, FSM_IPCP);
#endif
	pp->p2p_if.if_flags |= IFF_RUNNING;
}

/*
 * Turn upper levels Down
 */
void
ppp_lcp_tld(pp)
	struct p2pcom *pp;
{

#ifdef INET
	ppp_cp_down(pp, FSM_IPCP);
#endif
}

/*
 * Timeout
 */
void
ppp_lcp_timeout(pp)
	void *pp;
{
	ppp_cp_timeout((struct p2pcom *)pp, FSM_LCP);
}

/*
 * Handle the extra LCP packet codes
 */
struct mbuf *
ppp_lcp_xcode(pp, m, cp)
	struct p2pcom *pp;
	struct mbuf *m;
	struct ppp_cp_hdr *cp;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;
	u_short type;

	switch (cp->cp_code) {
	default:
		m = (struct mbuf *) -1;
		break;

	case LCP_PROTO_REJ:     /* Protocol-Reject */
		type = ntohs(*mtod(m, u_short *));
		switch (type) {
#ifdef INET
		case PPP_IP:
		case PPP_IPCP:
			dprintf((" IP disabled\n"));
			ppp_cp_close(pp, FSM_IPCP);
			break;
#endif

		default:
			dprintf((" <0x%x> -- dropped\n", type));
			break;
		}
		m_freem(m);
		m = 0;
		break;

	case LCP_ECHO_REQ:      /* Echo-Request */
		dprintf((" magic=%x -- send Echo-Reply\n", *mtod(m, u_long *)));
		*mtod(m, u_long *) = ppp->ppp_magic;
		m = ppp_cp_prepend(m, PPP_LCP, LCP_ECHO_REPL, cp->cp_id, 0);
		break;

	case LCP_ECHO_REPL:     /* Echo-Reply */
	case LCP_DISC_REQ:      /* Discard-Request */
		dprintf((" magic=%x -- dropped\n", *mtod(m, u_long *)));
		m_freem(m);
		m = 0;
		break;
	}
	return (m);
}

/*
 * Compose a LCP Config-Request packet
 */
struct mbuf *
ppp_lcp_creq(pp)
	struct p2pcom *pp;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;
	register struct mbuf *m;
	struct ppp_option opt;

	dprintf(("BUILD LCP_CREQ --"));
	m = 0;

	/* Max-Receive-Unit */
	if (!(ppp->ppp_lcp.fsm_rej & (1<<LCP_MRU))) {
		dprintf((" {MRU: %d}", ppp->Ppp_mru));
		opt.po_type = LCP_MRU;
		opt.po_len = 4;
		opt.po_data.s = htons(ppp->Ppp_mru);
		m = ppp_addopt(m, &opt);
	}

	/* Asynchronous-Control-Characters-Map */
	if (!(ppp->ppp_lcp.fsm_rej & (1<<LCP_ACCM))) {
		dprintf((" {ACCM: %x}", ppp->Ppp_cmap));
		opt.po_type = LCP_ACCM;
		opt.po_len = 6;
		opt.po_data.l = htonl(ppp->Ppp_cmap);
		m = ppp_addopt(m, &opt);
	}

	/* Magic-Number */
	if (!(ppp->ppp_lcp.fsm_rej & (1<<LCP_MAGIC))) {
		dprintf((" {MAGIC: 0x%x}", ppp->ppp_magic));
		opt.po_type = LCP_MAGIC;
		opt.po_len = 6;
		opt.po_data.l = htonl(ppp->ppp_magic);
		m = ppp_addopt(m, &opt);
	}

	/* Protocol-Field-Compression */
	if (!(ppp->ppp_lcp.fsm_rej & (1<<LCP_PFC)) &&
	    (ppp->Ppp_flags & PPP_PFC)) {
		dprintf((" {PFC}"));
		opt.po_type = LCP_PFC;
		opt.po_len = 2;
		m = ppp_addopt(m, &opt);
	}

	/* Address-and-Control-Field-Compression */
	if (!(ppp->ppp_lcp.fsm_rej & (1<<LCP_ACFC)) &&
	    (ppp->Ppp_flags & PPP_ACFC)) {
		dprintf((" {ACFC}"));
		opt.po_type = LCP_ACFC;
		opt.po_len = 2;
		m = ppp_addopt(m, &opt);
	}
	dprintf((" id=%d", ppp->ppp_lcp.fsm_id + 1));
	dprintf(("\n"));
	m = ppp_cp_prepend(m, PPP_LCP, CP_CONF_REQ, ++(ppp->ppp_lcp.fsm_id), 0);
	return (m);
}

/*
 * Process the rejected option
 */
void
ppp_lcp_optrej(pp, opt)
	struct p2pcom *pp;
	struct ppp_option *opt;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;

	switch (opt->po_type) {
	case LCP_MRU:
		dprintf((" {MRU: len=%d mtu=%d (use 1500)}",
			opt->po_len, ntohs(opt->po_data.s)));
		/* use the RFC default value */
		ppp->Ppp_mru = PPPMTU;
		break;

	case LCP_PFC:
		dprintf((" {PFC}"));
		/* indicate that peer won't send compressed */
		ppp->Ppp_flags &= ~PPP_PFC;
		break;

	case LCP_ACFC:
		dprintf((" {ACFC}"));
		/* indicate that peer won't send compressed */
		ppp->Ppp_flags &= ~PPP_ACFC;
		break;

	default:
		dprintf((" {%d: len=%d}", opt->po_type, opt->po_len));
	}
}

/*
 * Process the NAK-ed option
 */
void
ppp_lcp_optnak(pp, opt)
	struct p2pcom *pp;
	struct ppp_option *opt;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;

	switch (opt->po_type) {
	case LCP_MRU:
		dprintf((" {MRU: len=%d mtu=%d}",
			opt->po_len, ntohs(opt->po_data.s)));
		/* agree on what the peer says */
		ppp->Ppp_mru = ntohs(opt->po_data.s);
		break;

	case LCP_ACCM:
		dprintf((" {ACCM: len=%d m=0x%x}",
			opt->po_len, ntohl(opt->po_data.l)));
		/* agree on what the peer says */
		ppp->Ppp_cmap |= ntohl(opt->po_data.l);
		break;

	case LCP_MAGIC:
		dprintf((" {MAGIC: len=%d m=0x%x}",
			opt->po_len, ntohl(opt->po_data.l)));
		if (ppp->ppp_magic == ntohl(opt->po_data.l))
			ppp->ppp_magic = ppp_lcp_magic();
		break;

	default:
		dprintf((" {%d: len=%d}", opt->po_type, opt->po_len));
	}
}

/*
 * Process the received option
 */
int
ppp_lcp_opt(pp, opt)
	struct p2pcom *pp;
	struct ppp_option *opt;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;
	u_long mask;

	switch (opt->po_type) {
	case LCP_MRU:   /* Max-Receive-Unit */
		dprintf((" {MRU: len=%d mtu=%d}",
			opt->po_len, ntohs(opt->po_data.s)));
		if (opt->po_len != 4)
			return (OPT_REJ);
		pp->p2p_if.if_mtu = ntohs(opt->po_data.s);
		if (pp->p2p_if.if_mtu < LCP_MIN_RMU) {
			opt->po_data.s = htons(LCP_MIN_RMU);
			return (OPT_NAK);
		}
		return (OPT_OK);

	case LCP_ACCM:
		dprintf((" {ACCM: len=%d m=0x%x}",
			opt->po_len, ntohl(opt->po_data.l)));
		if (opt->po_len != 6)
			return (OPT_REJ);
		mask = ntohl(opt->po_data.l);
		if (~mask & ppp->Ppp_cmap) {
			opt->po_data.l = htonl(mask | ppp->Ppp_cmap);
			return (OPT_NAK);
		}
		ppp->Ppp_cmap = mask;
		return (OPT_OK);

	case LCP_AP:
		dprintf((" {AP}"));
		/* currently not supported */
		return (OPT_REJ);

	case LCP_QP:
		dprintf((" {QP}"));
		/* currently not supported */
		return (OPT_REJ);

	case LCP_MAGIC:
		dprintf((" {MAGIC: len=%d m=0x%x}",
			opt->po_len, ntohl(opt->po_data.l)));
		if (opt->po_len != 6)
			return (OPT_REJ);
		mask = ntohl(opt->po_data.l);
		if (ppp->ppp_magic == mask || mask == 0) {
			if (ppp->ppp_badmagic++ >= ppp_loop_max) {
				/* say it only once */
				if (ppp->ppp_badmagic == ppp_loop_max+1)
					printf("%s%d: link looped",
					    pp->p2p_if.if_name,
					    pp->p2p_if.if_unit);
				return (OPT_FATAL);
			}
			ppp->ppp_magic = ppp_lcp_magic();
			opt->po_data.l = htonl(ppp->ppp_magic);
			return (OPT_NAK);
		}
		return (OPT_OK);

	case LCP_PFC:
		dprintf((" {PFC: len=%d}", opt->po_len));
		if (opt->po_len != 2)
			return (OPT_REJ);
		if (!(ppp->Ppp_flags & PPP_PFC))
			/* we ain't need no stinking compression */
			return (OPT_REJ);
		ppp->ppp_txflags |= PPP_PFC;
		return (OPT_OK);

	case LCP_ACFC:
		dprintf((" {ACFC: len=%d}", opt->po_len));
		if (opt->po_len != 2)
			return (OPT_REJ);
		if (!(ppp->Ppp_flags & PPP_ACFC))
			/* we ain't need no stinking compression */
			return (OPT_REJ);
		ppp->ppp_txflags |= PPP_ACFC;
		return (OPT_OK);

	default:
		dprintf((" {?%d: len=%d}", opt->po_type, opt->po_len));
		return (OPT_REJ);
	}
	/*NOTREACHED*/
}

/*
 * This level finished -- drop DTR
 */
void
ppp_lcp_tlf(pp)
	struct p2pcom *pp;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;

	pp->p2p_if.if_flags &= ~IFF_RUNNING;
	if (ppp->ppp_lcp.fsm_state == CP_CLOSED ||
	    ppp->ppp_lcp.fsm_state == CP_STOPPED) {
		/*
		 * Only drop DTR is we are CLOSED (ie an "ifconfig down"
		 * has been done).
		 *
		 * XXX - what will this break?  Perhaps this should be sync
		 *       only or controlled by a link option.  Needs testing
		 *       on async links.
		 */
		if (pp->p2p_mdmctl)
			(*pp->p2p_mdmctl)(pp, 0);
	}
}

/*
 * Modem events handling routine
 * (0 - DCD dropped; 1 - DCD raised)
 */
int
ppp_modem(pp, flag)
	struct p2pcom *pp;
	int flag;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;
	int s;

	/*
	 * Got DCD -- send out Config-Request
	 */
	s = splimp();
	if (flag) {
		if (ppp->ppp_lcp.fsm_state == CP_INITIAL) {
			ppp->ppp_lcp.fsm_state = CP_CLOSED;
			splx(s);
			return (0);
		}
		if (ppp->ppp_lcp.fsm_state != CP_STARTING) {
			splx(s);
			return (0);
		}
		dprintf(("%s%d: got CD -- send Conf-Req\n", pp->p2p_if.if_name,
		    pp->p2p_if.if_unit));

		/* Initialize parameters */
		ppp_defaults(pp, pp->p2p_if.if_name);
		ppp->ppp_txcmap = 0xfffffff;       /* God save */
		ppp->ppp_params = ppp->ppp_dflt;
		ppp->ppp_txflags = 0;
		ppp->ppp_magic = ppp_lcp_magic();
		ppp->ppp_badmagic = 0;
		ppp->ppp_ncps = 0;
		ppp->ppp_idlecnt = 0; /* wait till the first data packet goes thru */

		/* Calculate timer parameters */
		ppp->ppp_ticks = (ppp->Ppp_timeout * hz) / 10;
		if (ppp->ppp_ticks < 2)
			ppp->ppp_ticks = 2;

		ppp_cp_up(pp, FSM_LCP);

	} else {
		/*
		 * Carrier lost -- turn off upper levels and return
		 * to the beginning.
		 */
		dprintf(("%s%d: CD lost\n", pp->p2p_if.if_name,
		    pp->p2p_if.if_unit));
		pp->p2p_if.if_flags &= ~IFF_RUNNING;
		ppp_cp_down(pp, FSM_LCP);
		if_qflush(&pp->p2p_isnd);
		if_qflush(&pp->p2p_if.if_snd);
#ifdef INET
		if_qflush(&ppp->ppp_ipdefq);
#endif

		/*
		 * Keep the active state on idle timeouts and
		 * carrier losses -- we may reconnect without
		 * toggling IFF_UP.
		 */
		if (pp->p2p_if.if_flags & IFF_UP)
			ppp->ppp_lcp.fsm_state = CP_STARTING;
#ifdef INET
		if (ppp->ppp_ipwbos) {
			wakeup((caddr_t) &ppp->ppp_ipwbos);
			ppp->ppp_ipwbos = 0;
		}
#endif
	}
	splx(s);
	return (0);
}

/*
 * Calculate the magic number
 */
u_long
ppp_lcp_magic()
{

	return(random());
}
