/*-
 * Copyright (c) 1993, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: if_sl.c,v 2.8 1995/12/05 20:49:44 prb Exp $
 */

/*
 * Copyright (c) 1987, 1989, 1992, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)if_sl.c	8.6 (Berkeley) 2/1/94
 */

/*
 * Serial Line interface
 *
 * Rick Adams
 * Center for Seismic Studies
 * 1300 N 17th Street, Suite 1450
 * Arlington, Virginia 22209
 * (703)276-7900
 * rick@seismo.ARPA
 * seismo!rick
 *
 * Pounded on heavily by Chris Torek (chris@mimsy.umd.edu, umcp-cs!chris).
 * N.B.: this belongs in netinet, not net, the way it stands now.
 * Should have a link-layer type designation, but wouldn't be
 * backwards-compatible.
 *
 * Converted to 4.3BSD Beta by Chris Torek.
 * Other changes made at Berkeley, based in part on code by Kirk Smith.
 * W. Jolitz added slip abort.
 *
 * Hacked almost beyond recognition by Van Jacobson (van@helios.ee.lbl.gov).
 * Added priority queuing for "interactive" traffic; hooks for TCP
 * header compression; ICMP filtering (at 2400 baud, some cretin
 * pinging you can use up all your bandwidth).  Made low clist behavior
 * more robust and slightly less likely to hang serial line.
 * Sped up a bunch of things.
 *
 * Reworked compression and bpf code by Paul Borman (prb@bsdi.com) to
 * allow larger packets, less data copy, and use common code with ppp.
 * 
 * Note that splimp() is used throughout to block both (tty) input
 * interrupts and network activity; thus, splimp must be >= spltty.
 */

#include <sys/param.h>
#include <sys/proc.h>
#include <sys/mbuf.h>
#include <sys/buf.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/tty.h>
#include <sys/kernel.h>
#include <sys/conf.h>

#include <machine/cpu.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/netisr.h>
#include <net/route.h>

#if INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#else
Huh? Slip without inet?
#endif

#include <net/slcompress.h>
#include <net/if_slvar.h>
#include <net/slip.h>
#include <net/bpf.h>

/*
 * SLMTU is a hard limit on output packet size.  To insure good
 * interactive response, SLMTU wants to be the smallest size that
 * amortizes the header cost.  (Remember that even with
 * type-of-service queuing, we have to wait for any in-progress
 * packet to finish.  I.e., we wait, on the average, 1/2 * mtu /
 * cps, where cps is the line speed in characters per second.
 * E.g., 533ms wait for a 1024 byte MTU on a 9600 baud line.  The
 * average compressed header size is 6-8 bytes so any MTU > 90
 * bytes will give us 90% of the line bandwidth.  A 100ms wait is
 * tolerable (500ms is not), so want an MTU around 296.  (Since TCP
 * will send 256 byte segments (to allow for 40 byte headers), the
 * typical packet size on the wire will be around 260 bytes).  In
 * 4.3tahoe+ systems, we can set an MTU in a route so we do that &
 * leave the interface MTU relatively high (so we don't IP fragment
 * when acting as a gateway to someone using a stupid MTU).
 *
 * Similar considerations apply to SLIP_HIWAT:  It's the amount of
 * data that will be queued 'downstream' of us (i.e., in clists
 * waiting to be picked up by the tty output interrupt).  If we
 * queue a lot of data downstream, it's immune to our t.o.s. queuing.
 * E.g., if SLIP_HIWAT is 1024, the interactive traffic in mixed
 * telnet/ftp will see a 1 sec wait, independent of the mtu (the
 * wait is dependent on the ftp window size but that's typically
 * 1k - 4k).  So, we want SLIP_HIWAT just big enough to amortize
 * the cost (in idle time on the wire) of the tty driver running
 * off the end of its clists & having to call back slstart for a
 * new packet.  For a tty interface with any buffering at all, this
 * cost will be zero.  Even with a totally brain dead interface (like
 * the one on a typical workstation), the cost will be <= 1 character
 * time.  So, setting SLIP_HIWAT to ~100 guarantees that we'll lose
 * at most 1% while maintaining good interactive response.
 */
#define	SLMTU		(256+40+12)	/* TCP with timestamp option */
#define	SLIP_HIWAT	100

#if 0
/*
 * SLIP ABORT ESCAPE MECHANISM:
 *	(inspired by HAYES modem escape arrangement)
 *	1sec escape 1sec escape 1sec escape { 1sec escape 1sec escape }
 *	within window time signals a "soft" exit from slip mode by remote end
 *	if the IFF_DEBUG flag is on.
 */
#define	ABT_ESC		'\033'	/* can't be t_intr - distant host must know it*/
#define	ABT_IDLE	1	/* in seconds - idle before an escape */
#define	ABT_COUNT	3	/* count of escapes for abort */
#define	ABT_WINDOW	(ABT_COUNT*2+2)	/* in seconds - time to count */
#endif

struct sl_softc *sl_softc;

#define FRAME_END	 	0xc0		/* Frame End */
#define FRAME_ESCAPE		0xdb		/* Frame Esc */
#define TRANS_FRAME_END	 	0xdc		/* transposed frame end */
#define TRANS_FRAME_ESCAPE 	0xdd		/* transposed frame esc */

extern struct timeval time;
extern struct ttytotals ttytotals;

int	slipflags = 0;		/* default extra IFF_* flags on open */
int	nslip;			/* number of slip interfaces (set at boot) */

static int slinit __P((struct sl_softc *, struct tty *));

#define RU(x)		(((x) + sizeof(long) - 1) & ~(sizeof(long) - 1))

/*
 * Called from boot code to establish sl interfaces.
 */
void
slattach(nsl)
	int nsl;
{
	register struct sl_softc *sc;
	register int i = 0;

	nslip = nsl;
	sl_softc = malloc(nsl * sizeof *sl_softc, M_DEVBUF, M_WAIT);
	bzero(sl_softc, nsl * sizeof *sl_softc);
	for (sc = sl_softc; i < nslip; sc++) {
		sc->sc_if.if_name = "sl";
		sc->sc_if.if_next = NULL;
		sc->sc_if.if_unit = i++;
		sc->sc_if.if_mtu = SLMTU;
		sc->sc_if.if_flags = IFF_POINTOPOINT | IFF_MULTICAST;
		sc->sc_if.if_type = IFT_SLIP;
		sc->sc_if.if_ioctl = slioctl;
		sc->sc_if.if_output = sloutput;
		sc->sc_if.if_snd.ifq_maxlen = 50;
		sc->sc_fastq.ifq_maxlen = 32;
		sc->sc_weos = 0;
		if_attach(&sc->sc_if);
		bpfattach(&sc->sc_if.if_bpf, &sc->sc_if, DLT_SLIP, SLC_BPFHDR);
	}
}

static int
slinit(sc, tp)
	register struct sl_softc *sc;
	struct tty *tp;
{
	register caddr_t p;
	struct mbuf *m;

#if 0
	sc->sc_if.if_flags &= ~IFF_UP;	/* breaks example setup */
#endif
	MGETHDR(sc->sc_vjc.vjc_mbuf, M_WAITOK, MT_DATA);
	sc->sc_vjc.vjc_mbuf->m_len = 0;
	sc->sc_vjc.vjc_ifp = &sc->sc_if;
	sc->sc_vjc.vjc_basement = PH_SLINK;

	/*
	 * Lock out others by setting sc_ttyp, in case we block below.
	 * Should reset interface flags to default so that we do not
	 * inherit flags from some random previous connection, but
	 * current users depend on this.
	 */
	sc->sc_ttyp = tp;
	/* may need to allocate compression data now */
	sc->sc_if.if_flags |= slipflags & ~IFF_CANTCHANGE;
	if (sc->sc_if.if_flags & (SC_AUTOCOMP | SC_COMPRESS)) {
		sc->sc_comp = malloc(sizeof *sc->sc_comp, M_DEVBUF,
		    M_WAITOK);
		sl_compress_init(sc->sc_comp);
	}
	return (1);
}

/*
 * Line specific open routine.
 * Attach the given tty to the first available sl unit.
 */
/* ARGSUSED */
int
slopen(dev, tp)
	dev_t dev;
	register struct tty *tp;
{
	struct proc *p = curproc;		/* XXX */
	register struct sl_softc *sc;
	register int nsl;
	int error;

	if (error = suser(p->p_ucred, &p->p_acflag))
		return (error);

	if (tp->t_line == SLIPDISC)
		return (0);

	for (nsl = nslip, sc = sl_softc; --nsl >= 0; sc++)
		if (sc->sc_ttyp == NULL) {
			if (slinit(sc, tp) == 0)
				return (ENOBUFS);
			tty_net_devices(1);	/* add one tty-net device */
			tp->t_sc = (caddr_t)sc;
			sc->sc_if.if_baudrate = tp->t_ospeed;
			ttyflush(tp, FREAD | FWRITE);
			return (0);
		}
	return (ENXIO);
}

/*
 * Line specific close routine.
 * Detach the tty from the sl unit.
 */
void
slclose(tp)
	struct tty *tp;
{
	register struct sl_softc *sc;
	int s;

	ttywflush(tp);
	s = splimp();		/* actually, max(spltty, splimp) */
	tp->t_line = 0;
	sc = (struct sl_softc *)tp->t_sc;
	if (sc != NULL) {
		if_down(&sc->sc_if);
		if (sc->sc_comp != NULL) {
			free(sc->sc_comp, M_DEVBUF);
			sc->sc_comp = NULL;
		}
		sc->sc_ttyp = NULL;
		tp->t_sc = NULL;
		if (sc->sc_vjc.vjc_mbuf)
			m_freem(sc->sc_vjc.vjc_mbuf);
		sc->sc_vjc.vjc_mbuf = 0;
		tty_net_devices(-1);	/* subtract one tty-net device */
	}
	splx(s);
}

/*
 * Line specific (tty) ioctl routine.
 * Provide a way to get the sl unit number.
 */
/* ARGSUSED */
int
sltioctl(tp, cmd, data, flag)
	struct tty *tp;
	int cmd;
	caddr_t data;
	int flag;
{
	int error = 0;
	int s;
	struct sl_softc *sc = (struct sl_softc *)tp->t_sc;
	char *abuf;
	int len;

	switch (cmd) {
	case SLIOCGUNIT:
		*(int *)data = sc->sc_if.if_unit;
		break;

	case SLIOCWEOS:
		if (tp->t_cflag & CLOCAL || tp->t_state & TS_CARR_ON) {
			sc->sc_weos = 1;
			error = tsleep((caddr_t) &sc->sc_weos,
				PCATCH|(PZERO+10), "sl_weos", 0);
		}
		break;

	case SLIOCSENDADDR:
		abuf = (char *)data;
		for (len = 0; len < SLIP_ADDRLEN && abuf[len]; ++len)
			++len;

		s = spltty();
		if (!(tp->t_state & TS_CARR_ON) && !(tp->t_cflag & CLOCAL))
			error = EWOULDBLOCK;
		else if (tp->t_outq.c_cc != 0)
			error = EINVAL;
		else if (b_to_q(abuf, len, &tp->t_outq))
			error = E2BIG;
		else
			(*tp->t_oproc)(tp);
		splx(s);
		break;

	default:
		return (-1);
	}
	return (error);
}

/*
 * Queue a packet.  Start transmission if not active.
 * Compression happens in slstart; if we do it here, IP TOS
 * will cause us to not compress "background" packets, because
 * ordering gets trashed.  It can be done for all packets in slstart.
 */
int
sloutput(ifp, m, dst, rtp)
	struct ifnet *ifp;
	register struct mbuf *m;
	struct sockaddr *dst;
	struct rtentry *rtp;
{
	register struct sl_softc *sc = &sl_softc[ifp->if_unit];
	register struct ip *ip;
	register struct ifqueue *ifq;
	int s;

	/*
	 * `Cannot happen' (see slioctl).  Someday we will extend
	 * the line protocol to support other address families.
	 */
	if (dst->sa_family != AF_INET) {
		printf("sl%d: af%d not supported\n", sc->sc_if.if_unit,
			dst->sa_family);
		m_freem(m);
		sc->sc_if.if_noproto++;
		return (EAFNOSUPPORT);
	}

	if (sc->sc_ttyp == NULL || (sc->sc_if.if_flags & IFF_UP) == 0) {
		m_freem(m);
		return (ENETDOWN);	/* sort of */
	}
	if ((sc->sc_ttyp->t_state & TS_CARR_ON) == 0 &&
	    (sc->sc_ttyp->t_cflag & CLOCAL) == 0) {
		m_freem(m);
		return (EHOSTUNREACH);
	}
	ifq = &sc->sc_if.if_snd;
	ip = mtod(m, struct ip *);
	if (sc->sc_if.if_flags & SC_NOICMP && ip->ip_p == IPPROTO_ICMP) {
		m_freem(m);
		return (ENETRESET);		/* XXX ? */
	}
	if (ip->ip_tos & IPTOS_LOWDELAY)
		ifq = &sc->sc_fastq;
	s = splimp();
	if (IF_QFULL(ifq)) {
		IF_DROP(ifq);
		m_freem(m);
		splx(s);
		sc->sc_if.if_oerrors++;
		return (ENOBUFS);
	}
	IF_ENQUEUE(ifq, m);
	sc->sc_if.if_lastchange = time;
	if (sc->sc_ttyp->t_outq.c_cc == 0)
		slstart(sc->sc_ttyp);
	else
		(*sc->sc_ttyp->t_oproc)(sc->sc_ttyp);
	splx(s);
	return (0);
}

/*
 * Start output on interface.  Get another datagram
 * to send from the interface queue and map it to
 * the interface before starting output.
 */
void
slstart(tp)
	register struct tty *tp;
{
	register struct sl_softc *sc = (struct sl_softc *)tp->t_sc;
	register u_char *cp;
	register struct ip *ip;
	struct slcompress_buf cb;
	int s;
	struct mbuf *m;
	struct mbuf *m1;
	int hl;
	int type;

	for (;;) {
		/*
		 * If there is more in the output queue, just send it now.
		 * We are being called in lieu of ttstart and must do what
		 * it would.
		 */
		if (tp->t_outq.c_cc != 0) {
			(*tp->t_oproc)(tp);
			if (tp->t_outq.c_cc > SLIP_HIWAT)
				return;
		}
		/*
		 * This happens briefly when the line shuts down.
		 */
		if (sc == NULL)
			return;

		/*
		 * Get a packet and send it to the interface.
		 */
		s = splimp();
		IF_DEQUEUE(&sc->sc_fastq, m);
		if (m)
			sc->sc_if.if_omcasts++;		/* XXX */
		else
			IF_DEQUEUE(&sc->sc_if.if_snd, m);
		splx(s);
		if (m == NULL)
			return;

		/*
		 * We do the header compression here rather than in sloutput
		 * because the packets will be out of order if we are using TOS
		 * queueing, and the connection id compression will get
		 * munged when this happens.
		 */
		cb.chlen = 0;
		cb.uhlen = 0;
		cb.bpflen = sc->sc_if.if_bpf ? SLC_BPFHDRLEN : 0;
		cb.lllen = 0;
		s = 0;
		if ((ip = mtod(m, struct ip *))->ip_p == IPPROTO_TCP &&
		    (sc->sc_if.if_flags & SC_COMPRESS)) {
			s |= sl_compress_tcp(&m, ip, sc->sc_comp,
			    SL_COMP_CID, &cb);
		} else if (sc->sc_if.if_bpf) {
			M_PREPEND(m, SLC_BPFHDR, M_DONTWAIT);
			if (m == 0)
/*XXX*/				panic("Bailed on mprepend");
		}
		if (cb.bpflen) {
			cp = mtod(m, u_char *);
			*cp++ = SLIPDIR_OUT;	/* direction */
			*cp++ = 0;		/* size of link level header */
			*cp++ = cb.chlen;	/* size of compressed header */
			*cp |= s;		/* cp now points to chdr */
			bpf_mtap(sc->sc_if.if_bpf, m);
			/* cb.bpflen += cb.chlen; */
			/*	
			 * Now move the link level data and compressed
			 * header up.  If we are lucky we can just trim
			 * mbufs, else we copy
			 */
			if (m->m_len == SLC_BPFHDR &&
			    m->m_next && m->m_next->m_len >= cb.uhlen) {
				m->m_data += SLC_BPFHDRLEN;
				m->m_pkthdr.len -= SLC_BPFHDR - cb.chlen;
				m->m_len = cb.chlen;

				m->m_next->m_len -= cb.uhlen;
				m->m_pkthdr.len -= cb.uhlen;
				m->m_next->m_data += cb.uhlen;
			} else {
				m->m_data += SLC_BPFHDRLEN;
				m->m_len -= SLC_BPFHDRLEN;
				m->m_pkthdr.len -= SLC_BPFHDRLEN;

				hl = SLC_BPFHDR + cb.uhlen - cb.chlen;
				if (hl) {
					m->m_pkthdr.len -= hl;
					m1 = m;
					while (m1 && hl) {
						if (hl >= m1->m_len) {
							hl -= m1->m_len;
							m1->m_len = 0;
							if ((m1 = m1->m_next)
							    == 0)
								panic("overran mbuf");
						} else {
							m1->m_len -= hl;
							m1->m_data += hl;
							hl = 0;
						}
					}
					hl = cb.chlen;
					if (m1->m_len < hl)
						panic("IP hdr in mbuf not contiguous");
					bcopy(cp, mtod(m1, caddr_t), hl);
				}
			}
		} else
			*mtod(m, u_char *) |= s;

		sc->sc_if.if_lastchange = time;

		/*
		 * The extra FRAME_END will start up a new packet, and thus
		 * will flush any accumulated garbage.  We do this whenever
		 * the line may have been idle for some time.
		 */
		if (tp->t_outq.c_cc == 0) {
			++sc->sc_if.if_obytes;
			++ttytotals.tty_nout;
			(void) putc(FRAME_END, &tp->t_outq);
		}

		while (m) {
			register u_char *ep;

			cp = mtod(m, u_char *); ep = cp + m->m_len;
			while (cp < ep) {
				/*
				 * Find out how many bytes in the string we can
				 * handle without doing something special.
				 */
				register u_char *bp = cp;

				while (cp < ep) {
					switch (*cp++) {
					case FRAME_ESCAPE:
					case FRAME_END:
						--cp;
						goto out;
					}
				}
				out:
				if (cp > bp) {
					/*
					 * Put n characters at once
					 * into the tty output queue.
					 */
					if (b_to_q((char *)bp, cp - bp,
					    &tp->t_outq))
						break;
					sc->sc_if.if_obytes += cp - bp;
					ttytotals.tty_nout += cp - bp;
				}
				/*
				 * If there are characters left in the mbuf,
				 * the first one must be special..
				 * Put it out in a different form.
				 */
				if (cp < ep) {
					if (putc(FRAME_ESCAPE, &tp->t_outq))
						break;
					if (putc(*cp++ == FRAME_ESCAPE ?
					   TRANS_FRAME_ESCAPE : TRANS_FRAME_END,
					   &tp->t_outq)) {
						(void) unputc(&tp->t_outq);
						break;
					}
					sc->sc_if.if_obytes += 2;
					ttytotals.tty_nout += 2;
				}
			}
			MFREE(m, m1);
			m = m1;
		}

		if (putc(FRAME_END, &tp->t_outq)) {
			/*
			 * Not enough room.  Remove a char to make room
			 * and end the packet normally.
			 * If you get many collisions (more than one or two
			 * a day) you probably do not have enough clists
			 * and you should increase "nclist" in param.c.
			 */
			(void) unputc(&tp->t_outq);
			(void) putc(FRAME_END, &tp->t_outq);
			sc->sc_if.if_collisions++;
		} else {
			++sc->sc_if.if_obytes;
			++ttytotals.tty_nout;
			sc->sc_if.if_opackets++;
		}
	}
}

/*
 * tty interface receiver interrupt.
 */
void
slinput(c, tp)
	register int c;
	register struct tty *tp;
{
	register struct sl_softc *sc;
	register struct mbuf *m;
	register struct mbuf *m1;
	register int len;
	u_char *p;
	u_char uc;
	int s, o;
	
	ttytotals.tty_nin++;
	sc = (struct sl_softc *)tp->t_sc;
	if (sc == NULL)
		return;
	if (c & TTY_ERRORMASK || ((tp->t_state & TS_CARR_ON) == 0 &&
	    (tp->t_cflag & CLOCAL) == 0)) {
		sc->sc_flags |= SC_ERROR;
		return;
	}
	c &= TTY_CHARMASK;

	++sc->sc_if.if_ibytes;

#ifdef ABT_ESC
	if (sc->sc_if.if_flags & IFF_DEBUG) {
		if (c == ABT_ESC) {
			/*
			 * If we have a previous abort, see whether
			 * this one is within the time limit.
			 */
			if (sc->sc_abortcount &&
			    time.tv_sec >= sc->sc_starttime + ABT_WINDOW)
				sc->sc_abortcount = 0;
			/*
			 * If we see an abort after "idle" time, count it;
			 * record when the first abort escape arrived.
			 */
			if (time.tv_sec >= sc->sc_lasttime + ABT_IDLE) {
				if (++sc->sc_abortcount == 1)
					sc->sc_starttime = time.tv_sec;
				if (sc->sc_abortcount >= ABT_COUNT) {
					slclose(tp);
					return;
				}
			}
		} else
			sc->sc_abortcount = 0;
		sc->sc_lasttime = time.tv_sec;
	}
#endif

	switch (c) {

	case TRANS_FRAME_ESCAPE:
		if (sc->sc_escape)
			c = FRAME_ESCAPE;
		break;

	case TRANS_FRAME_END:
		if (sc->sc_escape)
			c = FRAME_END;
		break;

	case FRAME_ESCAPE:
		sc->sc_escape = 1;
		return;

	case FRAME_END:
		if (sc->sc_flags & SC_ERROR) {
			sc->sc_flags &= ~SC_ERROR;
			goto newpack;
		}
		m1 = m = sc->sc_vjc.vjc_mbuf;
		m->m_pkthdr.len = 0;
		while (m1) {
			m->m_pkthdr.len += m1->m_len;
			m1 = m1->m_next;
		}
		if (m->m_pkthdr.len < 3) {
			/* less than min length packet - ignore */
			goto newpack;
		}

		if ((c = sc->sc_vjc.vjc_type) != TYPE_IP) {
			/*
			 * We've got something that's not an IP packet.
			 * If compression is enabled, try to decompress it.
			 * Otherwise, if `auto-enable' compression is on and
			 * it's a reasonable packet, decompress it and then
			 * enable compression.  Otherwise, drop it.
			 *
			 * If compression is still being enabled (sc_comp
			 * is NULL), ignore flags and drop (see slioctl).
			 */
			if (sc->sc_comp == NULL)
				goto error;
			if (sc->sc_if.if_flags & SC_COMPRESS) {
				if (sl_muncompress_tcp(&sc->sc_vjc.vjc_mbuf,
				    (u_int)c, sc->sc_comp, 0))
					goto error;
			} else if ((sc->sc_if.if_flags & SC_AUTOCOMP) &&
			    c == TYPE_UNCOMPRESSED_TCP && len >= 40) {
				if (sl_muncompress_tcp(&sc->sc_vjc.vjc_mbuf,
				    (u_int)c, sc->sc_comp, 0))
					goto error;
				sc->sc_if.if_flags |= SC_COMPRESS;
			} else
				goto error;
		}
		m = sc->sc_vjc.vjc_mbuf;
		if (m1 = m->m_next) {
			if (sc->sc_if.if_bpf) {
				/*
				 * Put the SLIP pseudo-"link header" in place.
				 * Then, invoke BPF.
				 */

				c = m->m_len;

				m->m_pkthdr.len += SLC_BPFHDR - c;
				m->m_len = SLC_BPFHDR;
				m->m_data -= SLC_BPFHDRLEN;
				
				p = mtod(m, u_char *);
				p[0] = SLIPDIR_IN;	/* direction */
				p[1] = 0;	/* size of link level header */
				p[2] = c;	/* size of compressed header */
				bpf_mtap(sc->sc_if.if_bpf, m);
			}
		
			m->m_next = 0;
			m1->m_pkthdr.len = m->m_pkthdr.len - m->m_len;
			m1->m_pkthdr.rcvif = &sc->sc_if;
			
			sc->sc_if.if_ipackets++;
			sc->sc_if.if_lastchange = time;
			s = splimp();
			if (IF_QFULL(&ipintrq)) {
				IF_DROP(&ipintrq);
				sc->sc_if.if_iqdrops++;
				m_freem(m1);
			} else {
				IF_ENQUEUE(&ipintrq, m1);
				schednetisr(NETISR_IP);
			}
			splx(s);
		}
		m->m_len = 0;
		vjc_hdr(&sc->sc_vjc, 0, -1);
		return;
	}

	uc = c;
	if (vjc_hdr(&sc->sc_vjc, &uc, 1) >= 0) {
		sc->sc_escape = 0;
		return;
	}

	/* can't put lower; would miss an extra frame */
	sc->sc_flags |= SC_ERROR;

error:
	sc->sc_if.if_ierrors++;
newpack:
	if (m = sc->sc_vjc.vjc_mbuf->m_next) {
		sc->sc_vjc.vjc_mbuf->m_next = 0;
		m_freem(m);
	}
	vjc_hdr(&sc->sc_vjc, 0, -1);
}

/*
 * Process an ioctl request.
 */
int
slioctl(ifp, cmd, data)
	register struct ifnet *ifp;
	int cmd;
	caddr_t data;
{
	register struct ifaddr *ifa = (struct ifaddr *)data;
	register struct ifreq *ifr;
	register struct sl_softc *sc;
	struct slcompress *p;
	register int s = splimp(), error = 0;

	switch (cmd) {

	case SIOCSIFADDR:
		if (ifa->ifa_addr->sa_family == AF_INET)
			ifp->if_flags |= IFF_UP;
		else
			error = EAFNOSUPPORT;
		break;

	case SIOCSIFDSTADDR:
		if (ifa->ifa_addr->sa_family != AF_INET)
			error = EAFNOSUPPORT;
		break;

	case SIOCADDMULTI:
	case SIOCDELMULTI:
		ifr = (struct ifreq *)data;
		if (ifr == 0) {
			error = EAFNOSUPPORT;		/* XXX */
			break;
		}
		switch (ifr->ifr_addr.sa_family) {

#ifdef INET
		case AF_INET:
			break;
#endif

		default:
			error = EAFNOSUPPORT;
			break;
		}
		break;

	case SIOCSIFFLAGS:
		/*
		 * Once allocated, this is not released until close,
		 * so that incoming data is not lost.  The input
		 * interrupt code must check sc_comp, however, as the
		 * flags can be set before the space exists.
		 */
		if ((ifp->if_flags & (SC_AUTOCOMP | SC_COMPRESS)) &&
		    (sc = &sl_softc[ifp->if_unit])->sc_comp == NULL) {
			p = malloc(sizeof *p, M_DEVBUF, M_WAITOK);
			sl_compress_init(p);
			sc->sc_comp = p;
		}
		break;

	default:
		error = EINVAL;
		break;
	}
	splx(s);
	return (error);
}

int
slmodem(tp, flag)
	register struct tty *tp;
	int flag;
{
	register struct sl_softc *sc;
	int r;

	sc = (struct sl_softc *)tp->t_sc;

	if ((r = nullmodem(tp, flag)) == 0 && sc->sc_weos)
		wakeup((caddr_t) &sc->sc_weos);
	return(r);
}
