/*-
 * Copyright (c) 1992, 1993, 1994 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: tty_subr.c,v 2.1 1995/02/03 07:54:37 polk Exp $
 */

/*-
 * Copyright (c) 1982, 1986, 1991 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	from: @(#)tty_subr.c	7.7 (Berkeley) 5/9/91
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/ioctl.h>
#include <sys/tty.h>
#include <sys/malloc.h>

#define	QUOTE(qlp, qp) 	 setbit(qlp->c_ct, ((int)qp - (int)qlp->c_cq))
#define	ISQUOTE(qlp, qp) isset(qlp->c_ct, ((int)qp - (int)qlp->c_cq))
#define	UNQUOTE(qlp, qp) clrbit(qlp->c_ct, ((int)qp - (int)qlp->c_cq))

static void
clinit(qp, size)
	register struct clist *qp;
	int size;
{
	int cltsize;

	cltsize = size / NBBY;
	qp->c_cc = 0;
#ifndef notyet
	qp->c_cf = qp->c_cl = NULL;
#endif
	qp->c_cq = (char *)malloc((u_long)size, M_CLIST, M_WAITOK);
	qp->c_ct = (char *)malloc((u_long)cltsize, M_CLIST, M_WAITOK);
	bzero(qp->c_ct, cltsize);
	qp->c_ce = (char *)(qp->c_cq + size);
	qp->c_cs = size;
#ifdef notyet
	qp->c_cf = qp->c_ce - 1;
	qp->c_cl = qp->c_cq;
#endif
	qp->c_quotes = 0;
}

/*
 * malloc space of 'size' for circular buffer for new clists
 */
void
cqttyinit(tp, size)
	struct tty *tp;
	int size;
{
	int cltsize;
	int oldpri;

	cltsize = size / NBBY;
	oldpri = spltty();
#ifdef DEBUG
	if (tp->t_rawq.c_cq)
		panic("cqttyinit");
#endif
	clinit(&tp->t_rawq, size);
	clinit(&tp->t_canq, size);
	clinit(&tp->t_outq, size);
	splx(oldpri);
}

/*
 * free space from an old clist
 */
void
cqttydestroy(tp)
	struct tty *tp;
{
	int oldpri = spltty();

#ifdef DEBUG
	if (tp->t_rawq.c_cq == NULL)
		panic("cqttydestroy");
#endif
	free((caddr_t)tp->t_rawq.c_cq, M_CLIST);
	free((caddr_t)tp->t_canq.c_cq, M_CLIST);
	free((caddr_t)tp->t_outq.c_cq, M_CLIST);
	free((caddr_t)tp->t_rawq.c_ct, M_CLIST);
	free((caddr_t)tp->t_canq.c_ct, M_CLIST);
	free((caddr_t)tp->t_outq.c_ct, M_CLIST);
	tp->t_rawq.c_cq = NULL;
	splx(oldpri);
}

getc(qp)
        register struct clist *qp;
{
        register int oldpri, a;

        oldpri = spltty();
#ifdef notyet
        if (qp->c_cc <= 0) {	/* XXX */
                qp->c_cc = 0;
                qp->c_quotes = 0;
                splx(oldpri);
                return (-1);
        }
#else
        if (qp->c_cc <= 0 || qp->c_cf == NULL) {	/* XXX */
                qp->c_cc = 0;
                qp->c_cf = qp->c_cl = NULL;	/* XXX */
                qp->c_quotes = 0;
                splx(oldpri);
                return (-1);
        }
#endif
        a = *(u_char *)qp->c_cf;
	if (qp->c_quotes && ISQUOTE(qp, qp->c_cf)) {
		a |= TTY_QUOTE;
		UNQUOTE(qp, qp->c_cf);
		qp->c_quotes--;
	}
        qp->c_cc--;
        if (++qp->c_cf >= qp->c_ce)
                qp->c_cf = qp->c_cq;
        splx(oldpri);
        return (a);
}

q_to_b(qp, buf, num)
        register struct clist *qp;
        register char *buf;
        register int num;
{
        register int oldpri;

        if (num <= 0)
                return (0);
        oldpri = spltty();
        if (num > qp->c_cc)
                num = qp->c_cc;
        if ((int)qp->c_cf + num <= (int)qp->c_ce) {
		if (qp->c_quotes) {
			if (num == 1)
				UNQUOTE(qp, qp->c_cf);
			else
				qtclr(qp, qp->c_cf, qp->c_cf + num - 1);
		}
                /*
                 * one pass bcopy
                 */
                bcopy(qp->c_cf, buf, num);
		qp->c_cf += num;
        } else {
                int c1, c2;

                /*
                 * gotta do it in 2 passes
                 */
                c1 = (int)qp->c_ce - (int) qp->c_cf;
                c2 = num - c1;
		if (qp->c_quotes) {
			if (num == 1)
				UNQUOTE(qp, qp->c_cf);
			else {
				qtclr(qp, qp->c_cf, qp->c_cf + c1 - 1);
				qtclr(qp, qp->c_cq, qp->c_cq + c2 - 1);
			}
		}
                bcopy(qp->c_cf, buf, c1);
                bcopy(qp->c_cq, buf + c1, c2);
		qp->c_cf = qp->c_cq + c2;
        }
	qp->c_cc -= num;
        splx(oldpri);
        return (num);
}

/*
 * Return pointer to initial block of contiguous characters in clist,
 * along with the size of the block.  Should be called only if the queue
 * is not empty.  The caller is responsible for ensuring that the result
 * is not invalidated (interrupts are not blocked).
 */
u_char *
getcblk(qp, retp)
        register struct clist *qp;
        int *retp;
{

        if (qp->c_cf <= qp->c_cl)
        	*retp = qp->c_cl - qp->c_cf + 1;
        else
        	*retp = qp->c_ce - qp->c_cf;
        return ((u_char *) qp->c_cf);
}

void
ndflush(qp, num)
	register struct clist *qp;
	register int num;
{
	register int oldpri;

	oldpri = spltty();
	if (num <= 0 || qp->c_cc <= 0) {
		splx(oldpri);
		return;			/* XXX */
	}
	if (qp->c_cc < num)
		num = qp->c_cc;
	qp->c_cc -= num;
	if ((int)qp->c_cf + num < (int)qp->c_ce) {
		if (qp->c_quotes) {
			if (num == 1)
				UNQUOTE(qp, qp->c_cf);
			else
				qtclr(qp, qp->c_cf, qp->c_cf + num - 1);
		}
		qp->c_cf += num;
	} else {
		if (num == 1) {
			if (qp->c_quotes)
				UNQUOTE(qp, qp->c_cf);
			qp->c_cf = qp->c_cq;
		} else {
			if (qp->c_quotes)
				qtclr(qp, qp->c_cf, qp->c_ce - 1);
			qp->c_cf += num - qp->c_cs + 1;
			if (qp->c_quotes)
				qtclr(qp, qp->c_cq, qp->c_cf - 1);
		}
	}
	splx(oldpri);
}

/*
 * unquote a range of a clist
 */
qtclr(qp, buf1, buf2)
	register struct clist *qp;
	register char *buf1, *buf2;
{
	register char *buf;
	int s1, s3, i, tot;
	int l1 = 0, l2 = 0, l3 = 0;

	/*
	 * do it in 3 stages
	 * bzero the large chunk in the middle
	 * individually clear bits on the fringe
	 */
	s1 = buf1 - qp->c_cq;
	s3 = buf2 - qp->c_cq;
	tot = s3 - s1 + 1;
	if (s1 % NBBY)
		l1 = ((l1 = (NBBY - s1 % NBBY)) > tot) ? tot : l1;
	l3 = ((l3 = s3 % NBBY + 1) > tot) ? tot : l3;
	l2 = s3 - s1 - l3 - l1 + 1;

	if (l1 > 0) {
		buf = buf1;
		for (i = 0; i < l1; i++, buf++)
			UNQUOTE(qp, buf);
	}
	if (l2 > 0)
		bzero((char *)((int)qp->c_ct + (s1+l1)/NBBY), l2/NBBY);
	if (l3 > 0) {
		buf = buf2;
		for (i = l3; i > 0; i--, buf--)
			UNQUOTE(qp, buf);
	}
}

putc(a, qp)
	register int a;
        register struct clist *qp;
{
        register int oldpri;

        oldpri = spltty();
        if (qp->c_cc >= qp->c_cs) {
                splx(oldpri);
                return (-1);
        }
#ifndef notyet
        if (qp->c_cl == NULL || qp->c_cc == 0)	/* XXX */
                qp->c_cl = qp->c_cf = qp->c_cq;
        else
#endif
        if (++qp->c_cl >= qp->c_ce)
		qp->c_cl = qp->c_cq;
        qp->c_cc++;
        *qp->c_cl = a;
	if (a & TTY_QUOTE) {
		QUOTE(qp, qp->c_cl);
		qp->c_quotes++;
	}
        splx(oldpri);
        return (0);
}

b_to_q(buf, num, qp)
        register char *buf;
        register int num;
        register struct clist *qp;
{
        register int oldpri, notdone;

        if (num <= 0)
                return (0);
        oldpri = spltty();
	if (qp->c_cc + num > qp->c_cs) {
		notdone = qp->c_cc + num - qp->c_cs;
		num -= notdone;
	} else
		notdone = 0;
	if (num) {
		if (qp->c_cc <= 0 || qp->c_cl == NULL) {
			qp->c_cc = 0;
			qp->c_cl = qp->c_cf = qp->c_cq;
		} else
			qp->c_cl++;
		if ((int)qp->c_cl + num <= (int)qp->c_ce) {
			bcopy(buf, qp->c_cl, num);
			qp->c_cc += num;
			qp->c_cl += num - 1;
		} else {
			int c1, c2;

			c1 = (int) qp->c_ce - (int) qp->c_cl;
			c2 = num - c1;
			bcopy(buf, qp->c_cl, c1);
			bcopy(buf + c1, qp->c_cq, c2);
			qp->c_cc += num;
			qp->c_cl = qp->c_cq + c2 - 1;
		}
	}
        splx(oldpri);
        return (notdone);
}

char *
nextc(qp, buf, a)
	register struct clist *qp;
	register char *buf;
	register int *a;
{

	if (qp->c_cc > 0 && qp->c_cf && buf && buf != qp->c_cl) {
		if (++buf >= qp->c_ce)
			buf = qp->c_cq;
		*a = *(u_char *)buf;
		if (ISQUOTE(qp, buf))
			*a |= TTY_QUOTE;
		return (buf);
	} else
		return (NULL);
}

unputc(qp)
	register struct clist *qp;
{
	register int oldpri, a;

	oldpri = spltty();
	if (qp->c_cc <= 0 || qp->c_cl == NULL) {
		splx(oldpri);
		return (-1);
	}
	qp->c_cc--;
	a = *(u_char *)qp->c_cl;
	if (ISQUOTE(qp, qp->c_cf)) {
		a |= TTY_QUOTE;
		UNQUOTE(qp, qp->c_cf);
		qp->c_quotes--;
	}
	if (--qp->c_cl < qp->c_cq)
		qp->c_cl = qp->c_ce - 1;
	splx(oldpri);
	return (a);
}

void
catq(src, dst)
	register struct clist *src, *dst;
{
	int oldpri, len;
	u_char *p;

	oldpri = spltty();
	while (src->c_cc > 0) {
		p = getcblk(src, &len);
		(void) b_to_q((char *) p, len, dst);
		ndflush(src, len);
	}
	splx(oldpri);
}
