/*
 * Copyright (c) 1991, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: sys_process.c,v 2.5 1996/01/12 17:52:15 karels Exp $
 */

/*-
 * Copyright (c) 1982, 1986, 1989 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	from: @(#)sys_process.c	7.22 (Berkeley) 5/11/91
 */

#include <sys/param.h>
#include <sys/proc.h>
#include <sys/vnode.h>
#include <sys/buf.h>
#include <sys/ptrace.h>
#include <sys/systm.h>
#include <sys/uio.h>
#include <sys/user.h>

#include <vm/vm.h>
#include <vm/vm_kern.h>
#include <vm/vm_page.h>

#include <sys/sysctl.h>

int	ptrace_checkvictim __P((struct proc *, struct proc *));
int	ptrace_copy __P((struct proc *, char *, vm_offset_t,
			enum uio_seg, int, int));
static int ptrace_pgcopy __P((struct proc *, char *, vm_offset_t,
			enum uio_seg, int, int));

struct ptrace_args {
	int	req;
	pid_t	pid;
	void	*addr;
	int	data;
};

int
ptrace(caller, args, retval)
	struct proc *caller;
	register struct ptrace_args *args;
	int *retval;
{
	register struct proc *victim = 0;
	vm_offset_t va;
	caddr_t v, v2;
	int error = 0;
#ifndef sparc
	int writing = 0;
#endif
	int off;

	/*
	 * 4.3 ptrace errno values:
	 *	EIO: illegal request; illegal signal; illegal address
	 *	ESRCH: no such process
	 *	EPERM: can't trace this process
	 * We add:
	 *	EBUSY: parent hasn't waited for child
	 *	plus various VM errors
	 */

	/*
	 * If we are manipulating a victim process,
	 * make sure that the process exists and is traceable.
	 */
	if (args->req != PT_TRACE_ME) {
		if ((victim = pfind(args->pid)) == 0)
			return (ESRCH);
		if (args->req == PT_ATTACH)
			return (ptrace_attach(caller, victim));
		if ((error = ptrace_checkvictim(caller, victim)) != 0)
			return (error);
	}

#ifdef notyet	/* XXX move this into p->p_md, & use a macro here? */
	/*
	 * On some architectures, it is necessary to repair
	 * (e.g.) the pc queue after a single step.
	 * The parent cleans up after the child since
	 * it was the parent that altered the child originally.
	 */
	if (victim && victim->p_flag & P_SSTEP) {
		ptrace_fix_step(victim);
		victim->p_flag &= ~P_SSTEP;
	}
#endif

	/*
	 * Process a request.
	 */
	switch (args->req) {

	/*
	 * Register us as being traceable.
	 */
	case PT_TRACE_ME:
		caller->p_flag |= P_TRACED;
		break;

	/*
	 * Execute one instruction in the victim, then make it trap.
	 * (Machine-dependent code can reject this operation, however.)
	 */
	case PT_STEP:
		if ((error = ptrace_single_step(victim)) != 0)
			break;
		/* FALL THROUGH */

	/*
	 * Continue execution.
	 */
	case PT_CONTINUE:
		if (args->data < 0 || args->data >= NSIG) {
			error = EIO;
			break;
		}
		if (args->addr != PT_ADDR_ANY &&
		    (error = ptrace_set_pc(victim, args->addr)) != 0)
			break;
		victim->p_xstat = args->data;
		if (victim->p_stat == SSTOP)
			setrunnable(victim);
		break;

	/*
	 * Terminate the child process.
	 */
	case PT_KILL:
		victim->p_xstat = SIGKILL;
		setrunnable(victim);
		break;

	case PT_DETACH:			/* stop tracing the child */
		if ((unsigned)args->data >= NSIG) {
			error = EIO;
			break;
		}
		if (args->addr != PT_ADDR_ANY &&
		    (error = ptrace_set_pc(victim, args->addr)) != 0)
			break;
		victim->p_xstat = args->data;
		victim->p_flag &= ~P_TRACED;
		if (victim->p_oppid != victim->p_pptr->p_pid) {
			register struct proc *pp = pfind(victim->p_oppid);

			if (pp)
				proc_reparent(victim, pp);
		}
		victim->p_oppid = 0;
		if (victim->p_stat == SSTOP)
			setrunnable(victim);
		break;

#ifdef sparc
	case PT_GETREGS:
		error = ptrace_getregs(victim, args->addr);
		break;

	case PT_SETREGS:
		error = ptrace_setregs(victim, args->addr);
		break;

	case PT_GETFPREGS:
		error = ptrace_getfpregs(victim, args->addr);
		break;

	case PT_SETFPREGS:
		error = ptrace_setfpregs(victim, args->addr);
		break;
#else
	/*
	 * Read or write the kernel stack / u area.
	 */
	case PT_WRITE_U:
		++writing;
		/* FALL THROUGH */

	case PT_READ_U:
		off = (int) args->addr;
		if (error = ptrace_check_u(victim, off, writing, args->data))
			break;

		/*
		 * Try to avoid copying material to u_kproc.
		 */
		v = (caddr_t) victim->p_addr + off;
		v2 = (caddr_t) &victim->p_addr->u_kproc.kp_proc;
		if (v >= v2 && v < v2 + sizeof (struct proc))
			v = (caddr_t) victim + (v - v2);
		else {
			v2 = (caddr_t) &victim->p_addr->u_kproc.kp_eproc;
			if (v >= v2 && v < v2 + sizeof (struct eproc))
				fill_eproc(victim, (struct eproc *) v2);
		}

		if (writing)
			bcopy(&args->data, v, sizeof (int));
		else
			bcopy(v, retval, sizeof (int));
		break;
#endif

	/*
	 * Read or write the victim's address space.
	 * Sorry, Mike & Keith, we don't support split I&D any more :-)!
	 */
	case PT_READ_I:
	case PT_READ_D:
		error = ptrace_copy(victim, (char *)retval,
		    (vm_offset_t)args->addr, UIO_SYSSPACE, sizeof(int), 0);
		break;

	case PT_WRITE_I:
	case PT_WRITE_D:
		error = ptrace_copy(victim, (char *)&args->data,
		    (vm_offset_t)args->addr, UIO_SYSSPACE, sizeof(int), 1);
		break;

	/*
	 * Unsupported request.
	 * EIO seems like a bogus way to report this...
	 */
	default:
		error = EIO;
		break;
	}

	return (error);
}

/*
 * Make sure the target of a ptrace operation is legal.
 */
int
ptrace_checkvictim(caller, victim)
	struct proc *caller, *victim;
{

	/* Victim must be the child of the caller, and must be traced. */
	if (victim->p_pptr != caller || (victim->p_flag & P_TRACED) == 0)
		return (EPERM);
	/* Child must be stopped and parent must wait for it. */
	if (victim->p_stat != SSTOP || (victim->p_flag & P_WAITED) == 0)
		return (EBUSY);
	return (0);
}

/*
 * Process a request by caller to attach to victim.
 * If acceptable, mark victim as traced and stop it,
 * and change its parent to be the caller while being debugged.
 */
int
ptrace_attach(caller, victim)
	struct proc *caller, *victim;
{
	int error;

	/*
	 * Must be root if the process has used set user or group
	 * privileges or does not belong to the real user.  Must
	 * not be already traced.  Can't attach to ourselves.
	 */
	if ((victim->p_flag & P_SUGID ||
	    victim->p_cred->p_ruid != caller->p_cred->p_ruid) &&
	    (error = suser(caller->p_ucred, &caller->p_acflag)) != 0)
		return (error);
	if (victim->p_flag & P_TRACED)
		return (EALREADY);	/* ??? */
	if (victim == caller)
		return (EINVAL);
	/*
	 * It would be nice if the tracing relationship were separate
	 * from the parent relationship but that would require
	 * another set of links in the proc struct or for "wait"
	 * to scan the entire proc table.  To make life easier,
	 * we just re-parent the process we're trying to trace.
	 * The old parent is remembered so we can put things back
	 * on a "detach".
	 */
	victim->p_flag |= P_TRACED;
	victim->p_oppid = victim->p_pptr->p_pid;
	proc_reparent(victim, caller);
	psignal(victim, SIGSTOP);
	return (0);
}

/*
 * Copy 'len' bytes of 'data' to or from user address 'va' in proc 'p'.
 */
int
ptrace_copy(p, data, va, seg, len, writing)
	struct proc *p;
	char *data;
	vm_offset_t va;
	enum uio_seg seg;
	int len, writing;
{
	vm_offset_t eva;
	int error, resid;

	/*
	 * We map each page separately just in case the operation spans
	 * multiple objects.  Ptrace() is permitted to puke if the address
	 * isn't aligned, but this code may be used to access longer chunks
	 * of the victim's address space (and is from, e.g., COMPAT_SUNOS).
	 */
	eva = va + len;
	if (va >= VM_MAXUSER_ADDRESS || eva > VM_MAXUSER_ADDRESS || eva < va)
		return (EINVAL);
	if (VM_MIN_ADDRESS > 0 && (va < VM_MIN_ADDRESS || eva < VM_MIN_ADDRESS))
		return (EINVAL);
	resid = len;
	while (resid > 0) {
		len = min(resid, CLBYTES - (va & CLOFSET));
		error = ptrace_pgcopy(p, data, va, seg, len, writing);
		if (error)
			return (error);
		va += len;
		data += len;
		resid -= len;
	}
	return (0);
}

/*
 * Copy, but contained within a single page.
 * We avoid a context switch by mapping the victim's page into the kernel.
 * I'm not at all sure that this is more efficient...
 */
static int
ptrace_pgcopy(p, data, va, seg, len, writing)
	struct proc *p;
	char *data;
	vm_offset_t va;
	enum uio_seg seg;
	int len, writing;
{
	vm_offset_t page;
	vm_map_t map = &p->p_vmspace->vm_map;
	vm_map_t tmap;
	vm_map_entry_t out_entry;
	vm_page_t m;
	vm_prot_t out_prot;
	vm_object_t object;
	vm_offset_t offset;
	vm_offset_t kpage = 0;
	boolean_t prot_ok;
	boolean_t wired, single_use;
	int error = 0, o;

	/*
	 * If we're writing and the page isn't writable, fix it.
	 * This is for the sake of the vm_fault() call below;
	 * vm_map_find() ensures that the kernel can write the page.
	 */
	page = va & ~CLOFSET;
	o = va & CLOFSET;
	prot_ok = 1;
	if (writing)
		prot_ok = vm_map_check_protection(map, page,
		    page + CLBYTES, VM_PROT_WRITE);
	if (!prot_ok)
		if (error = vm_map_protect(map, page,
		    page + CLBYTES, VM_PROT_ALL, 0))
			/* the page may not exist */
			return error;

	/*
	 * Identify the object backing the page.
	 * Don't retain a map lock; other stuff below will fail.
	 */
	tmap = map;
	error = vm_map_lookup(&tmap, page,
	    writing ? VM_PROT_WRITE : VM_PROT_READ,
	    &out_entry, &object, &offset, &out_prot, &wired, &single_use);
	if (!error)
		vm_map_lookup_done(tmap, out_entry);

#ifdef __i386__
	/*
	 * Handle disgusting botch in i386 code
	 * (create page-table page if not present).
	 */
	ptpage_botch(map, page);
#endif

	/*
	 * We need to be sure that only the victim sees any changes we make,
	 * not other processes mapping the same object.
	 * If there's a danger, we force a COW fault in the victim.
	 */
	if (!error && writing && object->shadow) {
		m = vm_page_lookup(object, offset);
		if (m == NULL || m->flags & PG_COPYONWRITE)
			error = vm_fault(map, page, VM_PROT_WRITE, 0);
	}

	/*
	 * Put the page into the kernel map so that
	 * we can read or write it.
	 */
	if (!error)
		error = vm_map_find(kernel_map, object, offset,
				    &kpage, CLBYTES, 1);

	if (!error) {
		/*
		 * For some reason, vm_map_find()/vm_map_insert()
		 * won't add a reference to the object...
		 * The vm_map_remove() below can destroy it
		 * if we don't add a reference here.
		 */
		vm_object_reference(object);

		/*
		 * Fault the page in and wire it down.
		 */
		error = vm_map_pageable(kernel_map, kpage, kpage + CLBYTES, 0);

		/*
		 * Alter the bits.
		 */
		if (!error) {
			if (seg == UIO_SYSSPACE) {
				if (writing)
					bcopy(data, (caddr_t)kpage + o, len);
				else
					bcopy((caddr_t)kpage + o, data, len);
			} else {
				if (writing)
					error = copyin(data,
					    (caddr_t)kpage + o, len);
				else
					error = copyout((caddr_t)kpage + o,
					    data, len);
			}
		}

#ifdef __powerpc__
		/*
		 * Flush the instruction cache for this address.
		 */
		if (!error && writing)
			pmap_cache_sync(kernel_pmap, kpage + o, len);
#endif

		vm_map_remove(kernel_map, kpage, kpage + CLBYTES);
	}

	if (!prot_ok)
		vm_map_protect(map, page, page + CLBYTES,
		    VM_PROT_READ|VM_PROT_EXECUTE, 0);

	return (error);
}

/*
 * Currently a dummy routine -- the parent manipulates the child.
 * No more need to context-switch for every ptrace() call,
 * but ptrace() calls are more expensive...
 */
int
trace_req(p)
	struct proc *p;
{

	return (1);
}
