/*
 * Copyright (c) 1994 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: sizemem.c,v 2.2 1995/03/09 20:40:05 prb Exp $
 */

#include <stand/stand.h>
#include <i386/stand/stand.h>
#include <sys/param.h>
#include <i386/include/cpu.h>
#include <i386/isa/isa.h>
#include <i386/isa/rtc.h>
#include <i386/stand/bioscall.h>
#include <sys/reboot.h>

/* convert kilobytes to a multiple of the page size, in bytes */
#define	k2truncpage(k)	(((k) * 1024) &~ (NBPG - 1))

extern struct bootparam *getbootparam __P((int, struct bootparam *));

extern int cpu;
int noflushcache = 0;		/* initialize to make patching easy */

/*
 * Quick and dirty cache flush.
 * This touches the first byte of each page for 1 MB+, which is the same
 * as the test reference pattern that we want to flush from the cache.
 * Also use wbinvd on 486 and beyond, but we don't trust the external cache
 * to be wired correctly to do the flush, and some motherboards hang :-(.
 */
flushcache()
{
	register int i;
	static int first;

	if (bootdebug && first == 0)
		printf("flush cache... ");
	if (cpu >= CPU_486SX && !noflushcache)
		asm (".byte 0x0F, 0x09	# wbinvd");
	for (i = 0; i < 0x200000; i += NBPG) {
		if (i == 0x80000)
			i = 0x100000;
		*(int *)i = *(int *)i;
	}
	if (bootdebug && first == 0) {
		first = 1;
		printf("done\n");
	}
}

/*
 * We test for memory starting at MINMEM, and continuing at most
 * to MAXMEM.  We test in groups of CS bytes, a page at a time.
 * For each group, we save the value at the start of each page,
 * and set the value to a test value.  After flushing the cache,
 * we check for mismatches, then restore the original values.
 * (The initial value is likely to be zero or garbage, but a few
 * machines preserve memory, so leave the msgbuf alone if the BIOS does!)
 * MINMEM and CS must be a power of two for wraparound testing to work
 * correctly.
 */
#define	MINMEM	0x200000		/* start test @ 2MB; /boot is below */
#define	MAXMEM	0x80000000		/* 2 GB; avoid magic high addrs */
#define CS	0x100000		/* chunk size 1 megabyte */
#define OPW	(NBPG / sizeof (u_int))	/* page offset per word */
#define PPC	(CS / NBPG)		/* pages per chunk */
#define	TESTPAT	0xaa55aa55		/* must not be zero! */

void
sizemem(o)
	struct boot_options *o;
{
	struct bootparam *bp;
	struct biosinfo *bi;
	struct bios_args ba;
	u_int zero;
	u_int *zerop, *ip;
	u_int topmost, cmos_end;
	u_int base;
	u_int *uip;
	int i, j;
	int npg;
	static int sizedmem;
	static u_int savebuf[PPC];

	if (sizedmem)
		return;
	if ((bp = getbootparam(B_BIOSINFO, NULL)) == NULL) {
		printf("no biosinfo???\n");
		bi = NULL;
	} else {
		bi = (struct biosinfo *)B_DATA(bp);

		if (o->o_basemem == 0) {
			bzero(&ba, sizeof(ba));
			bios_call(0x12, &ba);
			o->o_basemem = bi->basemem = k2truncpage(ba.ba_ax);
		} else
			bi->basemem = o->o_basemem;
	}
	printf("basemem = %dK, ", o->o_basemem / 1024);

	cmos_end = k2truncpage(rtcin(RTC_EXTLO) + (rtcin(RTC_EXTHI) << 8)) +
	    IOM_END;

	if (o->o_extmem != 0) 		/* user specified amount */
		goto out;

	zerop = (u_int *)0;
	zero = *zerop;

	switch (o->o_searchend) {
	case 0:
		topmost = MAXMEM;	 	/* 2 GB */
		break;
	case -1:
		topmost = cmos_end;
		break;
	default:
		topmost = o->o_searchend & ~(NBPG -1);
		break;
	}
	if (bootdebug)
		printf("Memory search limit: 0x%x (%dMB)\n",
		    topmost, topmost / (1024 * 1024));

	
	/*
	 * Don't need to check under 2 meg because this
	 * code is located there and we wouldn't be
	 * running if there is less.
	 */
    	*zerop = 0;
	for (base = MINMEM; base < topmost; base += CS) {
		uip = (u_int *)base;
		savebuf[0] = *uip;

		/*
		 * Check for address wraparound.
		 * We assume that this will happen only on power-of-two
		 * boundaries, which are also chunksize (1 MB) boundaries.
		 */
		*uip = TESTPAT;
		if (*zerop) {
			if (bootdebug)
				printf("Memory wrapped\n");
			goto done;
		}
		flushcache();
		if (*zerop) {
			if (bootdebug)
				printf("Memory wrapped after flushing cache\n");
			goto done;
		}

		/*
		 * calculations for partial chunk at end
		 */
		if (base + CS > topmost)
			npg = (topmost - base) / NBPG;
		else
			npg = PPC;

		for (i = 0, ip = uip; i < npg; i++, ip += OPW) {
			savebuf[i] = *ip;
			*ip = TESTPAT + i;
			/*
			 * Quick test for a mismatch without a cache flush;
			 * avoid poking locations past end of memory
			 * if we can avoid it.
			 */
			if (*ip != TESTPAT + i) {
				if (bootdebug)
				    printf("Found end of memory, %s: 0x%x\n",
				        "mismatch w/o cache flush", ip);
				npg = i;
				break;
			}
		}

		flushcache();
		for (j = 0, ip = uip; j < npg; j++, ip += OPW) {
			if (*ip != TESTPAT + j) {
				if (bootdebug)
				    printf("Found end of memory, %s: 0x%x\n",
				        "mismatch after cache flush", ip);
				break;
			}
		}

		for (i = 0, ip = uip; i < npg; i++, ip += OPW)
			*ip = savebuf[i];

		/*
		 * If j < npg or npg < PPC, we found the end of memory
		 * because of a mismatch or hitting the limit.
		 */
		if (j != PPC) {
			base += (j * NBPG);
			goto done;
		}
	}
done:
    	*zerop = zero;
	o->o_extmem = base - IOM_END;

out:
	if (bi)
		bi->extmem = o->o_extmem;
	printf("extmem = %dK, total %dK\n", o->o_extmem / 1024,
	    (o->o_extmem + IOM_END) / 1024);
	if (o->o_extmem + IOM_END != cmos_end)
		printf("NOTE: extended mem found (%dK) differs from CMOS amount (%dK)\n",
		    o->o_extmem / 1024, (cmos_end - IOM_END) / 1024);
	sizedmem = 1;
}
