/*-
 * Copyright (c) 1995 Berkeley Software Design, Inc.  All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *      BSDI $Id: wdpi.c,v 2.5 1995/12/13 03:59:31 ewv Exp $
 */
 
/*
 * ATAPI "SCSI" host adapter driver
 */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/buf.h>
#include <sys/device.h>
#include <sys/kernel.h>
#include <sys/reboot.h>

#include <dev/scsi/scsi.h>
#include <dev/scsi/scsivar.h>
#include <dev/scsi/disk.h>
#include <dev/scsi/tape.h>

#include <machine/cpu.h>
#include <machine/stdarg.h>

#include "wdreg.h"
#include "wdvar.h"

#ifdef EWVDEBUG
#include "ewvdebug.h"
#else
#define	TR(desc, parm1, parm2)
#endif

/* Config */
#define	BUSY_WAIT	6000		/* uSec */
#define	PSTART_WAIT	1000		/* uSec */
#define	DRQ_WAIT	10000		/* uSec */
#define	WDPI_WATCHDOG	30		/* seconds */
#define	RESET_WAIT	5000000		/* uSec */
#define	SETFEAT_WAIT	5000000		/* uSec */

/* config flags */
#define	WDPI_POLLDRQ	0x0001		/* Always poll for DRQ */
#define	WDPI_DLSHFT	8		/* I/O delay shift */
#define	WDPI_DLMASK	0xff00		/* Transfer mode (8 bits) */

#define	wdpi_flags	wdpi_hba.hba_dev.dv_flags

typedef struct wdpi_softc {
	struct	hba_softc wdpi_hba;	/* Common SCSI HBA device info */
	struct	device *wdpi_parent;	/* Controller device (shortcut) */
	int	wdpi_state;		/* Interrupt state */
	wdc_req_t wdpi_req;		/* Controller request block */
	int	wdpi_iobase;		/* Base I/O address */
	u_char	wdpi_polldrq;		/* Poll for DRQ flag */
	int	wdpi_selunit;		/* Unit select word */
	int	wdpi_io;		/* Read flag */
	int	wdpi_iodelay;		/* Broken drive I/O delay */

	scdgo_fn wdpi_dgo;		/* SCSI go callback */
	struct device *wdpi_scdev;	/* Calling device */
	struct scsi_cdb wdpi_cdb;	/* SCSI command block to be executed */
	scintr_fn wdpi_scintr;		/* SCSI interrupt callback */

	int	wdpi_xfr_resid;		/* Bytes left in complete transfer */
	int	wdpi_resid;		/* Bytes left in this bp */
	caddr_t	wdpi_addr;		/* Current location in memory */
	struct	buf *wdpi_curbp;
} wdpi_softc_t;

/* wdpi_softc_t.wdpi_polldrq */
#define	POLL_IRQ	0		/* Wait for interrupt */
#define	POLL_IGNORE	1		/* Poll, ignore interrupt */
#define	POLL_NOIRQ	2		/* Poll, no interrupt expected */

/* wdpi_softc_t.wdpi_state */
#define	WDPI_IDLE		0	/* Nothing happening */
#define	WDPI_WAIT_CMDDRQ	1	/* Waiting for packet DRQ */
#define	WDPI_WAIT_XFR		2	/* Waiting for data transfer */

int wdpimatch __P((struct device *parent, struct cfdata *cf, void *aux));
void wdpiattach __P((struct device *parent, struct device *self, void *aux));
void wdpi_grant __P((struct device *self));
int wdpi_intr __P((struct device *self));
void wdpi_watchdog __P((struct device *arg));
int wdpi_icmd __P((struct hba_softc *hba, int targ,
                       struct scsi_cdb *cdb, caddr_t buf,
                       int len, int rw));
int wdpi_dump __P((struct hba_softc *hba, int targ,
                       struct scsi_cdb *cdb, caddr_t buf,
                       int len));
void wdpi_start __P((struct device *self, struct sq *sq,
                         struct buf *bp, scdgo_fn dgo, struct
                         device *dev));
int wdpi_go __P((struct device *self, int targ, scintr_fn intr,
			struct device *dev, struct buf *bp, int pad));
void wdpi_release __P((struct device *self));
void wdpi_hbareset __P((struct hba_softc *hba, int resetunits));
void wdpi_block_xfr __P((wdpi_softc_t *sc, int base, int readflag));
void wdpi_prpad __P((char *p, int len, prf_t pf));
void wdpi_reset __P((wdpi_softc_t *sc));
void wdprf __P((prf_t pf, wdpi_softc_t *sc, char *fmt, ...));

struct cfdriver wdpicd = { 
	0, "wdpi", wdpimatch, wdpiattach, DV_DULL, sizeof(wdpi_softc_t)
};

struct hbadriver wdpihbadriver =
	{ wdpi_icmd, wdpi_dump, wdpi_start, wdpi_go, wdpi_release, 
	  wdpi_hbareset };

int wdpi_bsize = WD_SECSIZE*4;		/* Default block transfer size */

struct bootparam *getbootparam __P((int, struct bootparam *));

/*
 * Inlines and macros
 * ----------------------------------------------------------------------
 */

/*
 * insw() with configurable delay
 */
#define pi_insw(sc, port, ad, ln) {				\
	int _iodelay = (sc)->wdpi_iodelay;			\
								\
	if (_iodelay) {						\
		u_short *_addr = ad;				\
		int _len = ln;					\
								\
		while (_len--) {				\
			*(_addr)++ = inw(port);			\
			if (_iodelay == 0xff)			\
				continue;			\
			DELAY(_iodelay);			\
		}						\
	} else							\
		insw((port), (ad), (ln));			\
}

/*
 * outsw() with configurable delay
 */
#define pi_outsw(sc, port, ad, ln) {				\
	int _iodelay = sc->wdpi_iodelay;			\
								\
	if (_iodelay) {						\
		u_short *_addr = ad;				\
		int _len = ln;					\
								\
		while (_len--) {				\
			outw(port, *_addr++);			\
			if (_iodelay == 0xff)			\
				continue;			\
			DELAY(_iodelay);			\
		}						\
	} else							\
		outsw(port, ad, ln);				\
}

#ifdef __GNUC__
/*
 * Wait for bits to clear in status register
 */
#define	waitclr(base, bits, tout) ({				\
	int _tout = tout;					\
	int _rc = 1;						\
								\
	do {							\
		if ((inb((base) + wd_status) & (bits)) == 0) {	\
			_rc = 0;				\
			break;					\
		}						\
		DELAY(1);					\
	} while (_tout--);					\
	_rc;							\
})

/*
 * Wait for bits to set in status register
 */
#define	waitset(base, bits, tout) ({				\
	int _tout = tout;					\
	int _rc = 1;						\
								\
	do {							\
		if ((inb((base) + wd_status) & (bits)) == (bits)) { \
			_rc = 0;				\
			break;					\
		}						\
		DELAY(1);					\
	} while (_tout--);					\
	_rc;							\
})

/*
 * Wait for pattern of bits
 */
#define	waitcond(base, mask, bits, tout) ({			\
	int _tout = tout;					\
	int _rc = 1;						\
								\
	do {							\
		if ((inb((base) + wd_status) & (mask)) == (bits)) { \
			_rc = 0;				\
			break;					\
		}						\
		DELAY(1);					\
	} while (_tout--);					\
	_rc;							\
})
#endif

/*
 * Configuration
 * ----------------------------------------------------------------------
 *
 * Check for ATAPI device using the protocol from the draft ATAPI std
 *	(section 5.14.4 June 13 1994 draft, pp3)
 */
int
wdpimatch(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	wdc_attach_args_t *ap = (wdc_attach_args_t *)aux;
	int unit = ap->drive;
	int unit_loc = cf->cf_loc[LOC_DRIVE];
	int base = wdc_getiobase(parent);
	int s;

	if (ap->found || (unit != unit_loc && unit_loc != -1))
		return (0);

	TR("wdpimatch parent/unit",parent,unit);
	aprint_debug("probe wdpi%d at %s unit %d:", cf->cf_unit,
		    parent->dv_xname, unit);
	outb(base + wd_sdh, WDSD_IBM | (unit << 4));	/* Select drive */

	/* See if officially not present */
	DELAY(10000);
	if (inb(base + wd_status) == 0xff) {
		aprint_debug(" status=0xff");
		goto not_here;
	}

	/* Check signature */
	if (inb(base + wd_status) != 0 || inb(base + wd_cyl_hi) != 0xeb ||
	    inb(base + wd_cyl_lo) != 0x14) {
		int s;

		if (waitclr(base, WDCS_BUSY, BUSY_WAIT)) {
			aprint_debug(" busy not clear");
			goto not_here;
		}

		/*
		 * The identify drive command will cause an ATAPI device 
		 * to reinit its signature (and abort the command).
		 */
		s = splbio();				/* XXX needed? */
		outb(base + wd_command, WDCC_READP);
		if (wdc_pollintr(parent, 5000) ||
		    waitclr(base, WDCS_BUSY, BUSY_WAIT)) {
			splx(s);
			aprint_debug("busy t/o after READP ");
			goto not_here;
		}
		DELAY(1000);		/* Be sure interrupt clears on ATA */
		inb(base + wd_status);
		splx(s);

		/* ATA drives will return ID data */
		if (!(inb(base + wd_status) & WDCS_ERR)) {
			/* This is not an ATAPI drive, clear the ID data */
			if (!waitcond(base, WDCS_BUSY|WDCS_DRQ|WDCS_READY, 
			    WDCS_DRQ|WDCS_READY, BUSY_WAIT)) {
				int i;

				for (i = 0; i < WD_SECSIZE / 2; i++)
					inw(base + wd_data);
			}
			aprint_debug(" not ATAPI");
			goto not_here;
		}

		/* 
		 * Older non-ATA drives will abort the identify drive command
		 * but won't set the ATAPI signature so we check again.
		 */
		if ((inb(base + wd_cyl_hi) != 0xeb) ||
		    (inb(base + wd_cyl_lo) != 0x14)) {
			aprint_debug(" sig2 mismatch");
			goto not_here;
		}
	}

	/* 
	 * Looks like ATAPI, to complete the official ID sequence an ATAPI
	 * identify drive command must still be issued, we'll do this in
	 * the attach routine (since we need the data there anyway).
	 */
	ap->found = 1;
	aprint_debug(" probe good\n");
	return (1);

	/* Drive isn't present, select drive 0 to avoid spurious interrupts */
not_here:
	aprint_debug(" probe failed\n");
	outb(base + wd_sdh, WDSD_IBM);
	return (0);
}

/*
 * Attach a wdpi "SCSI" host adapter.
 */
void
wdpiattach(parent, self, aux)
	struct device *parent;
	struct device *self;
	void *aux;
{
	wdc_attach_args_t *ap = (wdc_attach_args_t *)aux;
	int base = wdc_getiobase(parent);
	wdpi_softc_t *sc = (wdpi_softc_t *)self;
	int unit = ap->drive;
	union {
		char bufr[WD_SECSIZE];
		struct wdparams p;
	} u;
	int drqt;
	int i;
	int s;

	TR("wdpiattach parent/unit",parent,unit);
	/* Set up softc */
	sc->wdpi_hba.hba_driver = &wdpihbadriver;
	sc->wdpi_iobase = base;
	sc->wdpi_parent = parent;
	sc->wdpi_selunit = WDSD_IBM|(unit << 4);
	sc->wdpi_req.next = WDC_UNALLOC;
	sc->wdpi_req.intr = wdpi_intr;
	sc->wdpi_req.tout = wdpi_watchdog;
	sc->wdpi_req.go = wdpi_grant;
	sc->wdpi_req.self = self;
	sc->wdpi_req.timeout = 0;
	sc->wdpi_state = WDPI_IDLE;

	/* Print warm fuzzies */
	aprint_naive(": ATAPI Device (IDE-CD or IDE-TAPE)");

	/* Check for flags overrides from the boot command line */
	sc->wdpi_flags = getdevconf(self->dv_cfdata, NULL, 
	    sc->wdpi_hba.hba_dev.dv_unit);

	/* Get drive ID data, this is still a normal ATA command */
	outb(base + wd_sdh, sc->wdpi_selunit);
	if (waitclr(base, WDCS_BUSY, BUSY_WAIT)) {
		printf("\n");
		wdprf(printf, sc, "wdpiattach: controller jammed\n");
		return;
	}
	s = splbio();
	outb(base + wd_command, WDCC_IDENT);
	if (wdc_pollintr(sc->wdpi_parent, 200000) ||
	    waitcond(base, WDCS_BUSY|WDCS_DRQ, WDCS_DRQ, BUSY_WAIT)) {
		printf("\n");
		wdprf(printf, sc, "wdpiattach: no response to ATAPI ident\n");
		splx(s);
		return;
	}
	pi_insw(sc, base + wd_data, (u_short *) &u, WD_SECSIZE / 2);

#ifdef PRINT_ID
	for (i = 0; i < WD_SECSIZE; i++) {
		if ((i & 0xf) == 0)
			aprint_debug("\n%3x: ", i);
		aprint_debug("%2x ", (u_char)u.bufr[i]);
	}
	aprint_debug("\n");
#endif
	inb(base + wd_status);		/* Clear possible pending interrupt */
	splx(s);
	
	/* Print device serial, type and revision */
	aprint_normal(": ");
	wdpi_prpad(u.p.wdp_model, sizeof(u.p.wdp_model), aprint_normal);
	aprint_verbose(" rev=");
	wdpi_prpad(u.p.wdp_rev, sizeof(u.p.wdp_rev), aprint_verbose);
	aprint_verbose(" ");
	wdpi_prpad(u.p.wdp_cnsn, sizeof(u.p.wdp_cnsn), aprint_verbose);
	printf("\n");

	wdprf(aprint_verbose, sc, "transfer size=%d ", wdpi_bsize);
	drqt = (u.p.wdp_config & WDPI_DRQT_MASK) >> WDPI_DRQT_SHFT;
	if (drqt == WDPI_INTR_DRQ) {
		/* We will get interrupts to start a command packet */
		if (sc->wdpi_flags & WDPI_POLLDRQ) {
			aprint_verbose("intr cmd DRQ/polled");
			sc->wdpi_polldrq = POLL_IGNORE;
		} else {
			aprint_verbose("intr cmd DRQ");
			sc->wdpi_polldrq = POLL_IRQ;
		}
	} else {
		aprint_verbose("polled cmd DRQ");
		sc->wdpi_polldrq = POLL_NOIRQ;
	}
	if ((sc->wdpi_flags & WDPI_DLMASK) != 0) {
		sc->wdpi_iodelay = (sc->wdpi_flags & WDPI_DLMASK) >>
		    WDPI_DLSHFT;
		aprint_verbose(" iodelay=%d\n", sc->wdpi_iodelay);
	} else {
		aprint_verbose("\n");
	}

	wdpi_reset(sc);

	/* Wait for unit to come ready (for a while) */
	for (i = 0; i < 50; i++) {
		if (scsi_test((struct hba_softc_t *)sc, 0, 0) == STS_GOOD)
			break;
		if (i == 3) {
			wdprf(aprint_normal, sc, "");
			printf("Delaying up to 5 seconds to allow ATAPI device "
			    "to finish self-test\n");
		}
		DELAY(100000);
	}
	if (i == 50)
		wdprf(aprint_normal, sc, "Unit did not become ready\n");

	SCSI_FOUNDTARGET(&sc->wdpi_hba, 0);	/* We only have 1 target */
	TR("wdpiattach done",0,0);
	return;
}

/*
 * Interfaces to the wdc layer (hardware controller)
 * ----------------------------------------------------------------------
 */

/*
 * Called from wdc driver when access to controller is granted (this is
 * in response to the start routine requesting access via wdc_request()).
 */
void
wdpi_grant(self)
	struct device *self;
{
	wdpi_softc_t *sc = (wdpi_softc_t *)self;
	int base = sc->wdpi_iobase;

	TR("wdpi_grant sc/cdb",self, &sc->wdpi_cdb);
	sc->wdpi_dgo(sc->wdpi_scdev, &sc->wdpi_cdb);
	TR("wdpi_grant done",0,0);
}

/*
 * Main interrupt handler
 */
int
wdpi_intr(self)
	struct device *self;
{
	wdpi_softc_t *sc = (wdpi_softc_t *)self;
	int base = sc->wdpi_iobase;
	int stat;
	int reason;
	int rc;

	sc->wdpi_req.timeout = 0;
#ifdef DIAGNOSTIC
	stat = inb(base + wd_status);
	if (stat & WDCS_BUSY)
		wdprf(printf, sc, "wdpi_intr: BUSY set, ignored\n");
#endif

again:
	stat = inb(base + wd_status);
	reason = inb(base + wd_ireason)&(WDIR_COD|WDIR_IO);
	TR("wdpi_intr loop stat/reason",stat,reason);

	/*
	 * Check for end of transfer/error
	 */
	if (!(stat & WDCS_DRQ)) {
		if (reason != (WDIR_COD|WDIR_IO))
			goto bad_reason;
		if (stat & WDCS_ERR) {
			int error = inb(base + wd_error);
			int key = error >> 4;

			if (key != SKEY_ATTENTION &&
			    key != SKEY_NOT_READY)
				wdprf(printf, sc, "ATAPI command error: "
				    "code=0x%x\n", error);
			rc = STS_CHECKCOND;
		} else {
			rc = STS_GOOD;
		}
		goto done;
	}

	/*
	 * Continue transfer in progress
	 */
	TR("wdpi_intr state", sc->wdpi_state, 0);
	switch (sc->wdpi_state) {
	case WDPI_WAIT_CMDDRQ:
		/*
		 * Ready for command packet
		 */
		if (reason != WDIR_COD)
			goto bad_reason;
		pi_outsw(sc, base + wd_data, (u_short *)&sc->wdpi_cdb, 6);
		sc->wdpi_state = WDPI_WAIT_XFR;
		break;

	case WDPI_WAIT_XFR: {
		int ioflag = sc->wdpi_io;
		if (ioflag) {
			if (reason != WDIR_IO)
				goto bad_reason;
		} else {
			if (reason != 0)
				goto bad_reason;
		}
		wdpi_block_xfr(sc, base, ioflag);
		break;
	}

	default:
		wdprf(printf, sc, "Unexpected interrupt, state=%d\n", 
		    sc->wdpi_state);
		break;
	}
	sc->wdpi_req.timeout = WDPI_WATCHDOG;
	TR("wdpi_intr done",0,0);
	return (1);

bad_reason:
	wdprf(printf, sc, "Bad interrupt reason 0x%x, stat=0x%x state=%d\n",
	    reason, stat, sc->wdpi_state);
	wdpi_reset(sc);
	rc = STS_CHECKCOND;

done:
	sc->wdpi_state = WDPI_IDLE;

	/*
	 * Release controller here to be fair to other device on bus,
	 * should be possible to use ifdefed code below instead but
	 * this will allow a device to monopolize the bus
	 */
	wdc_release(sc->wdpi_parent);

	sc->wdpi_addr = (caddr_t)~0;	/* Paranoia */
	sc->wdpi_curbp = NULL;
	if (sc->wdpi_hba.hba_intr && sc->wdpi_hba.hba_intrdev) {
		TR("calling scsi intr callback rc/resid",rc,sc->wdpi_xfr_resid);
		/* Call the SCSI interrupt handler waiting on this */
		sc->wdpi_hba.hba_intr(sc->wdpi_hba.hba_intrdev, rc,
					sc->wdpi_xfr_resid);

#ifdef DIAGNOSTIC
		/*
		 * We only support one target and one unit on this
		 * HBA so there should never be a request from
		 * another target/unit waiting here. The interrupt
		 * routine above will drive another I/O through
		 * wdpi_start() if there is more to do.
		 */
		if (sc->wdpi_hba.hba_head) {
			panic("wdpi_intr: extra unit/target queued");
		}
#endif
	}
	TR("intr done 2",0,0);
	return (1);
}

/*
 * Watchdog timeout
 */
void
wdpi_watchdog(arg)
	struct device *arg;
{
	wdpi_softc_t *sc = (wdpi_softc_t *)arg;

	switch (sc->wdpi_state) {
	case WDPI_WAIT_CMDDRQ:
	case WDPI_WAIT_XFR:
		wdprf(printf, sc, "operation timeout, state=%d\n", 
		    sc->wdpi_state);
		wdpi_reset(sc);
		wdc_release(sc->wdpi_parent);
		if (sc->wdpi_hba.hba_intr && sc->wdpi_hba.hba_intrdev) {
			/* Call the SCSI interrupt handler waiting on this */
			sc->wdpi_hba.hba_intr(sc->wdpi_hba.hba_intrdev, 
				STS_CHECKCOND, sc->wdpi_resid);
		}
		break;

	case WDPI_IDLE:
	default:
		wdprf(printf, sc, "wdpi_watchdog: invalid state %d\n", 
		    sc->wdpi_state);
		panic("wdpi_watchdog invalid state");
	}
}

/*
 * Interfaces to the SCSI layer
 * ----------------------------------------------------------------------
 */

/*
 * Immediate SCSI command
 *
 * Note: although it is possible to already have the controller this is
 *	never called when a regular operation is in progress (hence it
 *	is safe to use a single set of vars to track the operation in
 *	progress).
 */
int
wdpi_icmd(hba, targ, cdb, buf, len, rw)
	struct hba_softc *hba;
	int targ;
	struct scsi_cdb *cdb;
	caddr_t buf;
	int len;
	int rw;
{
	wdpi_softc_t *sc = (wdpi_softc_t *)hba;
	int base = sc->wdpi_iobase;
	int s = splbio();
	int rc = STS_GOOD;
	int stat;
	int reason;

	TR("wdpi_icmd entry cdb/len",cdb,len);

	/* Reject to anything but unit 0 */
	if (CDB6(cdb)->cdb_lun_lbah&0xe0)
		return (-1);

	/*
	 * Get the controller synchronously if we don't already have it.
	 * Complicated because icmd can be called with the controller 
	 * owned (xxigo) or not owned (xxintr or at attach).
	 */
	if (wdc_getctl(sc->wdpi_parent, &sc->wdpi_req)) {
		splx(s);
		return (STS_BUSY);
	}

	sc->wdpi_xfr_resid = sc->wdpi_resid = len;
	sc->wdpi_addr = buf;
	sc->wdpi_io = rw;

	/* Select the drive */
	outb(base + wd_sdh, sc->wdpi_selunit);

	/* Wait for permission to issue packet command */
	if (waitclr(base, WDCS_BUSY|WDCS_DRQ, BUSY_WAIT)) {
		wdprf(printf, sc, "immediate BUSY wait timeout\n");
		rc = STS_BUSY;
		goto err_reset;
	}

	/* Issue packet command */
	outb(base + wd_feature, 0);		/* No DMA */
	outb(base + wd_bcnt, wdpi_bsize);	/* Suggested xfer size */
	outb(base + wd_bcnt + 1, wdpi_bsize >> 8);
	outb(base + wd_command, WDCC_PACKET);

	/* Wait for the interrupt line to set if drive works that way */
	if (sc->wdpi_polldrq != POLL_NOIRQ) {
		if (wdc_pollintr(sc->wdpi_parent, DRQ_WAIT)) {
			wdprf(printf, sc, "wdpi_icmd: DRQ interrupt wait "
			    "timeout stat=0x%x\n", inb(base + wd_status));
			rc = STS_BUSY;
			goto err_reset;
		}
	} else {
		DELAY(1);			/* XXX NEC drives */
	}

	/* Wait for command DRQ */
	if (waitcond(base, WDCS_BUSY|WDCS_DRQ, WDCS_DRQ, DRQ_WAIT)) {
		wdprf(printf, sc, "immediate DRQ wait timeout, stat=0x%x\n",
		    inb(base + wd_status));
		rc = STS_BUSY;
		goto err_reset;
	}

	/* Issue the command */
	pi_outsw(sc, base + wd_data, (u_short *)cdb, 6);

	for (;;) {
		/* Wait for interrupt */
		if (wdc_pollintr(sc->wdpi_parent, 5000000)) {
			wdprf(printf, sc, "wdpi_icmd: interrupt wait timeout\n");
			rc = STS_BUSY;
			goto err_reset;
		}
		if (waitclr(base, WDCS_BUSY, BUSY_WAIT)) {
			wdprf(printf, sc, "wdpi_icmd: BUSY wait timeout - "
			    "loop\n");
			rc = STS_BUSY;
			goto err_reset;
		}

		stat = inb(base + wd_status);
		reason = inb(base + wd_ireason)&(WDIR_COD|WDIR_IO);

		/*
		 * Check for end of transfer/error
		 */
		if (!(stat & WDCS_DRQ)) {
			if (reason != (WDIR_COD|WDIR_IO))
				goto bad_reason;
			if (stat & WDCS_ERR) {
				int error = inb(base + wd_error);
				int key = error >> 4;

				if (key != SKEY_ATTENTION &&
				    key != SKEY_NOT_READY)
					wdprf(printf, sc, "ATAPI command "
					    "error: code=0x%x\n", error);
				rc = STS_CHECKCOND;
			}
			break;
		}

		/*
		 * Do transfer
		 */
		if (sc->wdpi_io) {
			if (reason != WDIR_IO)
				goto bad_reason;
		} else {
			if (reason != 0)
				goto bad_reason;
		}
		wdpi_block_xfr(sc, base, sc->wdpi_io);
	}
out:
	TR("wdpi_icmd out",0,0);
	wdc_release(sc->wdpi_parent);
	splx(s);
	TR("wdpi_icmd return",0,0);
	return (rc);

err_reset:
	wdpi_reset(sc);
	goto out;

bad_reason:
	/*
	 * Print error message and let the mainline error recovery clean
	 * up the mess on the next I/O.
	 */
	wdprf(printf, sc, "bad immediate interrupt reason, stat=0x%x "
	    "reason=0x%x\n", stat, reason);
	rc = STS_BUSY;
	goto out;
}

/*
 * We don't do dumps yet (maybe when ATAPI tapes come out), at this point
 * just return an error.
 */
int
wdpi_dump(hba, targ, cdb, buf, len)
	struct hba_softc *hba;
	int targ;
	struct scsi_cdb *cdb;
	caddr_t buf;
	int len;
{
	return (STS_CMD_TERMINATED);
}

/*
 * Request access to the wdc controller we are a child of.
 *
 * There is only one unit/target per 'host adapter' with ATAPI (2 ATAPI
 * drives on the same cable will end up being two hba's) so there is no
 * target or unit queueing. Therefore, we can assume we will not be called
 * if already queued for the controller (which will panic in wdc_request).
 * This further assumes that the controller is released before the SCSI
 * interrupt callback is called since it may call here to drive another I/O.
 */
void
wdpi_start(self, sq, bp, dgo, dev)
	struct device *self;
	struct sq *sq;
	struct buf *bp;
	scdgo_fn dgo;
	struct device *dev;
{
	wdpi_softc_t *sc = (wdpi_softc_t *)self;

	TR("wdpi_start sc",sc,0);
	sc->wdpi_dgo = dgo;
	sc->wdpi_scdev = dev;
	wdc_request(sc->wdpi_parent, &sc->wdpi_req);
}

/*
 * Start I/O on behalf of a SCSI driver, called from SCSI device go routine
 */
int
wdpi_go(self, targ, intr, dev, bp, pad)
	struct device *self;
	int targ;
	scintr_fn intr;
	struct device *dev;
	struct buf *bp;
	int pad;
{
	wdpi_softc_t *sc = (wdpi_softc_t *)self;
	int base = sc->wdpi_iobase;
	int s;

	TR("wdpi_go sc/bp",sc,bp);

	/* Set up for this BP (chain) */
	sc->wdpi_curbp = bp;
	sc->wdpi_addr = bp->b_un.b_addr;
	sc->wdpi_resid = bp->b_bcount;
	sc->wdpi_xfr_resid = bp->b_iocount;
	sc->wdpi_hba.hba_intrdev = dev;
	sc->wdpi_hba.hba_intr = intr;
	sc->wdpi_io = bp->b_flags & B_READ;

	/* Select the unit */
	outb(base + wd_sdh, sc->wdpi_selunit);

	/* Wait for permission to issue packet command */
	if (waitclr(base, WDCS_BUSY|WDCS_DRQ, PSTART_WAIT)) {
		wdprf(printf, sc, "wdpi_go: busy timeout\n");
		return (1);
	}

	/* Issue the packet command */
	outb(base + wd_feature, 0);		/* No DMA */
	outb(base + wd_bcnt, wdpi_bsize);	/* Suggested xfer size */
	outb(base + wd_bcnt + 1, wdpi_bsize >> 8);
	sc->wdpi_state = WDPI_WAIT_CMDDRQ;
	outb(base + wd_command, WDCC_PACKET);

	if (sc->wdpi_polldrq != POLL_IRQ) {
		s = splbio();

		if (sc->wdpi_polldrq == POLL_IGNORE &&
		    wdc_pollintr(sc->wdpi_parent, DRQ_WAIT)) {
			wdprf(printf, sc, "wdpi_go: cmd intr poll timeout\n");
			goto err_out;
		}

		/* Check DRQ and BUSY  */
		if (waitcond(base, WDCS_BUSY|WDCS_DRQ, WDCS_DRQ, DRQ_WAIT)) {
			wdprf(printf, sc, "wdpi_go: cmd DRQ timeout\n");
			goto err_out;
		}

		/* Send command */
		DELAY(1);			/* XXX NEC drives */
		pi_outsw(sc, base + wd_data, (u_short *)&sc->wdpi_cdb, 6);

		if (sc->wdpi_polldrq == POLL_IGNORE) {
			/*
			 * Soak up the DRQ interrupt if we're polling on
			 * a drive that would normally interrupt.
			 */
			inb(base + wd_status);
		}

		sc->wdpi_state = WDPI_WAIT_XFR;
		splx(s);

		/* 
		 * If write performance is needed we could poll for
		 * the write DRQ here and call the interrupt handler
		 * to fill the buffer.
		 */
	}
	sc->wdpi_req.timeout = WDPI_WATCHDOG;
	TR("wdpi_go done state",sc->wdpi_state,0);
	return (0);

err_out:
	wdpi_reset(sc);
	splx(s);
	return(1);
}

/*
 * Explicitly release the controller, called when the I/O request fails
 * somewhere along the way or for immediate commands (this releases the
 * outer queued request since it never get to the interrupt handler).
 */
void
wdpi_release(self)
	struct device *self;
{
	wdpi_softc_t *sc = (wdpi_softc_t *)self;

	TR("wdpi_release sc",sc,0);
	wdc_release(sc->wdpi_parent);
}

/*
 * Reset the host adapter, called with controller dedicated (implicitly)
 * during system dumps.
 */
void
wdpi_hbareset(hba, resetunits)
	struct hba_softc *hba;
	int resetunits;
{
	wdpi_softc_t *sc = (wdpi_softc_t *)hba;

	TR("wdpi_hbareset sc",sc,0);
	wdpi_reset(sc);
	if (resetunits)
		scsi_reset_units(hba);
	TR("wdpi_hbareset done",0,0);
}

/*
 * Utility functions
 * ----------------------------------------------------------------------
 */

/*
 * Transfer data in/out (handle buffer chains)
 */
void
wdpi_block_xfr(sc, base, readflag)
	wdpi_softc_t *sc;
	int base;
	int readflag;
{
	int unit_resid = inb(base + wd_bcnt) | inb(base + wd_bcnt + 1) << 8;

	TR("block_xfr, unit_resid",unit_resid,0);
	while (unit_resid) {
		int this_len;

		if (!sc->wdpi_xfr_resid) {
			/* Pad/soak if no more buffer space */
			int i;

			unit_resid += unit_resid&1;	/* Pad to word */
			TR("block_xfr: pad/soak",readflag, unit_resid);
			if (readflag) {
				/* Soak excess read data */
				for (i = 0; i < unit_resid >> 1; i++)
					inw(base + wd_data);
			} else {
				/* Pad extra write data */
				for (i = 0; i < unit_resid >> 1; i++)
					outw(base + wd_data, 0);
			}
			break;
		}

		/* Calculate maximum contiguous data for this xfer */
		this_len = min(sc->wdpi_resid, unit_resid);
		TR("block_xfr: this_len",this_len, readflag);
#ifdef DIAGNOSTIC
		if (this_len & 1) {
			wdprf(printf, sc, "block_xfr: odd transfer count %d "
				  "resid=%d unit_resid=%d\n", this_len,
				  sc->wdpi_resid, unit_resid);
			this_len += 1;
		}
#endif

		/* Transfer a chunk */
		TR("block_xfr: xfr addr/len",sc->wdpi_addr,this_len);
		if (readflag) {
			pi_insw(sc, base + wd_data, (u_short *)sc->wdpi_addr, 
			    this_len >> 1);
		} else {
			pi_outsw(sc, base + wd_data, (u_short *)sc->wdpi_addr,
			    this_len >> 1);
		}

		/* Update pointers/residuals */
		unit_resid -= this_len;
		if ((sc->wdpi_xfr_resid -= this_len) == 0)
			continue;
		if ((sc->wdpi_resid -= this_len) == 0) {
			/* move to next buffer in chain */
			struct buf *nextbp = sc->wdpi_curbp->b_chain;
			if (!nextbp)
				panic("wdpi buffer chain truncated");
			sc->wdpi_curbp = nextbp;
			sc->wdpi_addr = nextbp->b_un.b_addr;
			sc->wdpi_resid = nextbp->b_bcount;
		} else {
			sc->wdpi_addr += this_len;
		}
	}
	TR("block_xfr: done",0,0);
}

/*
 * Print a non-zero terminate space padded string (w/o the pad)
 */
void
wdpi_prpad(p, len, pf)
	char *p;
	int len;
	prf_t pf;
{
	char *last = &p[len-1];
	int i;
	u_short *sp;

	/* Most drives store ID strings big endian */
	sp = (u_short *)p;
	for (i = 0; i < len / 2; i++, sp++)
		*sp = ntohs(*sp);
		
	while (*last == ' ') {
		if (last == p)
			return;
		last--;
	}
	last++;
	while (p != last)
		pf("%c",*p++);
}

/*
 * Reset the unit using the ATAPI soft reset protocol (sync). Does not
 * wait for BUSY clear since this is used to clear a hung controller.
 */
void
wdpi_reset(sc)
	wdpi_softc_t *sc;
{
	int base = sc->wdpi_iobase;
	int xfr_resid;
	int resid;
	caddr_t addr;
	int abase = wdc_getaiobase((struct device *)sc);

	TR("wdpi_reset sc",sc,0);
	outb(base + wd_sdh, sc->wdpi_selunit);	/* Select unit (in case) */
	outb(base + wd_command, WDCC_RESET);
	DELAY(1000);
	if (waitclr(base, WDCS_BUSY, RESET_WAIT))
		wdprf(printf, sc, "wdpi_reset: Controller not responding "
		    "to reset\n");
	outb(base + wd_feature, WDFT_SETXMODE);
	outb(base + wd_seccnt, 0);
	outb(base + wd_command, WDCC_SETFEAT);
	if (waitclr(base, WDCS_BUSY, SETFEAT_WAIT))
		wdprf(printf, sc, "wdpi_reset: Set features cmd timeout\n");
	outb(abase + wda_ctlr, WDCTL_HEAD3ENB);
	outb(base + wd_feature, 0);		/* Set no DMA */

	/*
	 * The FX001DE isn't really ready at this point, on fast machines
	 * it will ignore the next packet command if it comes too soon.
	 */
	DELAY(10000);

	TR("wdpi_reset done",0,0);
}

/*
 * Print a string with an identifying prefix
 */
void
#if __STDC__
wdprf(prf_t pf, wdpi_softc_t *sc, char *fmt, ...)
#else
wdprf(pf, sc, fmt, va_alist)
	prf_t pf;
	wdpi_softc_t *sc;
	char *fmt;
#endif
{
	va_list ap;
	struct	device *dp = &sc->wdpi_hba.hba_dev;

#if __STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	pf("%s%d: %r", dp->dv_cfdata->cf_driver->cd_name, dp->dv_unit, fmt, ap);
	va_end(ap);
}
