/*
 * Soundcard driver for BSD/OS.
 *
 * BSDI $Id: soundcard.c,v 1.4 1995/11/30 20:45:53 ewv Exp $
 * 
 * Copyright by Hannu Savolainen 1993
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Id: soundcard.c,v 1.27 1995/07/28 21:40:26 jkh Exp
 */

#include "sound_config.h"
#include <sys/device.h>
#include <i386/isa/isavar.h>
#include <i386/isa/icu.h>

#ifdef CONFIGURE_SOUNDCARD

#include "dev_table.h"

#define FIX_RETURN(ret) {if ((ret)<0) return -(ret); else return 0;}

static int      timer_running = 0;

static int      soundcards_installed = 0;	/* Number of installed
						 * soundcards */
static int      soundcard_configured = 0;

static struct fileinfo files[SND_NDEVS];
struct selinfo selinfo[SND_NDEVS >> 4];

int             sndopen (dev_t dev, int flags, int devtype, struct proc *p);
int             sndclose (dev_t dev, int flags, int devtype, struct proc *p);
int             sndread (dev_t dev, struct uio *uio, int ioflag);
int             sndwrite (dev_t dev, struct uio *uio, int ioflag);
int             sndioctl (dev_t dev, int cmd, caddr_t arg, int flag,
		    struct proc *p);
int             sndselect (dev_t dev, int which, struct proc *p);
int		voxprobe(struct device *parent, struct cfdata *cf, void *aux);
void		voxattach(struct device *parent, struct device *self,
		    void *aux);
static void	sound_mem_init(void);

/*
 * Voxware meta-softc structure. The actual voxware drivers don't use
 * this (they mostly use static globals), this merely provides some
 * glue to the device driver layer.
 */
struct vox_softc {
	struct	device vx_dev;		/* base device */
	struct	isadev vx_id;		/* Isa node */
	struct	intrhand vx_ih;		/* Interrupt link */
	struct	address_info vx_config;	/* Subdriver config structure */
};

/*
 * Extra info per device driver
 */
struct vox_drvinfo {
	int	type;		/* SNDCARD_xxx */
	int	iosize;		/* Number of ports used */
	int	(*ifunc)(int);	/* Interrupt handler */
};

#define VX(name, type, iosize, ihand) \
struct vox_drvinfo vox_##name##_info = { type, iosize, ihand }; \
struct cfdriver name##cd = { NULL, # name, voxprobe, voxattach, DV_DULL, \
	sizeof(struct vox_softc), &vox_##name##_info };


#undef CDNAME

/*	Name	Type			Iosize	IntrHandler */
#if NOPL > 0
VX(	opl,	SNDCARD_ADLIB,		0x02,	NULL		)
#ifndef CDNAME
#define CDNAME oplcd
#endif
#endif

#if NSBSUPP > 0
int sbintr(int irq);
VX(	sb,	SNDCARD_SB,		0x10,	sbintr		)
VX(	sbxvi,	SNDCARD_SB16,		0x14,	sbintr		)
VX(	sbmidi,	SNDCARD_SB16MIDI, 	0x02,	sbintr		)
#ifndef CDNAME
#define CDNAME sbcd
#endif
#endif

#if NPAS > 0
int pasintr(int irq);
VX(	pas,	SNDCARD_PAS,		0x04,	pasintr		)
#ifndef CDNAME
#define CDNAME pascd
#endif
#endif

#if NMPU > 0
int mpuintr(int irq);
VX(	mpu,	SNDCARD_MPU401,		0x02,	mpuintr		)
#ifndef CDNAME
#define CDNAME mpucd
#endif
#endif

#if NGSUPP > 0
int gusintr(int irq);
int ad1848_interrupt(int irq);
VX(	gus,	SNDCARD_GUS,		0x10,	gusintr		)	
VX(	gusxvi,	SNDCARD_GUS16,		0x10,	gusintr		)	
VX(	mss,	SNDCARD_MSS,		0x10,	ad1848_interrupt )
#ifndef CDNAME
#define CDNAME guscd
#endif
#endif

#if NUART > 0
int m6859intr(int irq);
VX(	uart,	SNDCARD_UART6850,	0x02,	m6850intr	)
#ifndef CDNAME
#define CDNAME uartcd
#endif
#endif

#undef VX

struct devsw sndsw = {
	&CDNAME,				/* XXX */
	sndopen, sndclose, sndread, sndwrite, sndioctl, sndselect,
	nommap, nostrat, nodump, nopsize, 0, nostop
};


/*
 * Get time in HZ ticks from the epoch
 */
unsigned long
get_time()
{
	struct timeval timecopy;
	int x;

	microtime(&timecopy);
	return (timecopy.tv_usec / (1000000 / HZ) +
	    (unsigned long)timecopy.tv_sec * HZ);
}

int
sndread(dev_t dev, struct uio *uio, int ioflag)
{

	dev = minor (dev);
	FIX_RETURN (sound_read_sw(dev, &files[dev], uio, uio->uio_resid));
}

int
sndwrite(dev_t dev, struct uio *uio, int ioflag)
{

	dev = minor (dev);
	FIX_RETURN (sound_write_sw(dev, &files[dev], uio, uio->uio_resid));
}

int
sndopen(dev_t dev, int oflags, int devtype, struct proc *p)
{

	dev = minor(dev);
	if (!soundcard_configured && dev) {
		printf("SoundCard Error: The soundcard system has not been "
		    "configured\n");
		FIX_RETURN (-ENODEV);
	}

	files[dev].mode = 0;

	if ((oflags & (FREAD|FWRITE)) == (FREAD|FWRITE))
		files[dev].mode = OPEN_READWRITE;
	else if (oflags & FREAD)
		files[dev].mode = OPEN_READ;
	else if (oflags & FWRITE)
		files[dev].mode = OPEN_WRITE;

	selinfo[dev >> 4].si_pid = 0;
	selinfo[dev >> 4].si_flags = 0;

	FIX_RETURN (sound_open_sw(dev, &files[dev]));
}

int
sndclose(dev_t dev, int fflag, int devtype, struct proc *p)
{

	dev = minor(dev);
	sound_release_sw(dev, &files[dev]);
	FIX_RETURN (0);
}

int
sndioctl(dev_t dev, int cmd, caddr_t data, int fflag, struct proc *p)
{

	dev = minor(dev);
	FIX_RETURN (sound_ioctl_sw(dev, &files[dev], cmd, (unsigned int)data));
}

int
sndselect (dev_t dev, int which, struct proc *p)
{

	dev = minor(dev);

	DEB(printk("snd_select(dev=%d, which=%d, pid=%d)\n", dev, which,
	    p->p_pid));
#ifdef ALLOW_SELECT
	switch (dev & 0x0f) {
#ifndef EXCLUDE_SEQUENCER
	case SND_DEV_SEQ:
	case SND_DEV_SEQ2:
		return (sequencer_select(dev, &files[dev], which, p));
		break;
#endif

#ifndef EXCLUDE_MIDI
	case SND_DEV_MIDIN:
		return (MIDIbuf_select(dev, &files[dev], which, p));
		break;
#endif

#ifndef EXCLUDE_AUDIO
	case SND_DEV_DSP:
	case SND_DEV_DSP16:
	case SND_DEV_AUDIO:
		return (audio_select(dev, &files[dev], which, p));
		break;
#endif

	default:
		return (0);
	}
#endif

	return 0;
}

#define cfdata_to_dvip(cf)	((cf)->cf_driver->cd_aux)
#define dev_to_dvip(dev)	cfdata_to_dvip((dev)->dv_cfdata)

int
voxprobe(struct device *parent, struct cfdata *cf, void *aux)
{
	struct isa_attach_args *ia = (struct isa_attach_args *)aux;
	struct address_info hw_config;
	struct vox_drvinfo *dip = cfdata_to_dvip(cf);

	if (ia->ia_irq == IRQUNK)
		ia->ia_irq = IRQNONE;

	ia->ia_iosize = dip->iosize;

	hw_config.io_base = ia->ia_iobase;
	hw_config.irq = ffs(ia->ia_irq) - 1;
	hw_config.dma = ia->ia_drq;
	hw_config.dma_read = cf->cf_flags & 0x4; /* Gus read DMA chan */
  
	return (sndtable_probe(dip->type, &hw_config));
}

void
voxattach(struct device *parent, struct device *self, void *aux)
{
	struct isa_attach_args *ia = (struct isa_attach_args *)aux;
	struct vox_softc *sc = (struct vox_softc *)self;
	struct vox_drvinfo *dip = dev_to_dvip(self);
	int type;
	int ilevel;
	static int midi_initialized = 0;
	static int seq_initialized = 0;
	static int generic_midi_initialized = 0; 
	unsigned long mem_start = 0xefffffffUL;

	
	if ((type = dip->type) == 0) {
		printf(" <Unit not found>\n");
		return;
	}

	/* Init the subdriver */
	ilevel = ffs(ia->ia_irq) - 1;
	sc->vx_config.io_base = ia->ia_iobase;
	sc->vx_config.irq = ilevel;
	sc->vx_config.dma = ia->ia_drq;
	sc->vx_config.dma_read = self->dv_cfdata->cf_flags & 0x4;
	if (sndtable_init_card(type, &sc->vx_config) == 0) {
		printf (" <Driver not configured>\n");
		return;
	}

	/* Attach to isa bus */
	isa_establish(&sc->vx_id, &sc->vx_dev);

	/* Install interrupt handler (if any) */
	if (ia->ia_irq != IRQNONE && dip->ifunc != NULL) {
		sc->vx_ih.ih_fun = dip->ifunc;
		sc->vx_ih.ih_arg = (void *)ilevel;
		intr_establish(ia->ia_irq, &sc->vx_ih, DV_DULL);
	}

	/*
	 * Init the high level sound driver
	 */
	if ((soundcards_installed = sndtable_get_cardcount ()) == 0) {
		printf (" <No such hardware>\n");
		return;
	}

	printf("\n");

#ifndef EXCLUDE_AUDIO
	if (num_audiodevs != 0) {	/* Audio devices present */
		mem_start = DMAbuf_init(mem_start);
		mem_start = audio_init(mem_start);
		sound_mem_init();
	}
	soundcard_configured = 1;
#endif

#ifndef EXCLUDE_MIDI
	if (num_midis != 0 && !midi_initialized) {
		midi_initialized = 1;
		mem_start = MIDIbuf_init(mem_start);
	}
#endif

#ifndef EXCLUDE_SEQUENCER
	if ((num_midis + num_synths) != 0 && !seq_initialized) {
		seq_initialized = 1;
		mem_start = sequencer_init(mem_start);
	}
#endif
}

/*
 * Dummy attach routine for sound pseudo-device
 */
void
sndattach(int cnt)
{
}

void
tenmicrosec()
{

	DELAY(10);
}

#ifndef EXCLUDE_SEQUENCER
void
request_sound_timer(int count)
{
  static int      current = 0;
  int             tmp = count;

	if (count < 0)
		timeout ((timeout_func_t)sequencer_timer, 0, -count);
	else {
		if (count < current)
			current = 0;		/* Timer restarted */

		count = count - current;

		current = tmp;

		if (!count)
			count = 1;

		timeout ((timeout_func_t)sequencer_timer, 0, count);
	}
	timer_running = 1;
}

void
sound_stop_timer()
{
	if (timer_running)
		untimeout ((timeout_func_t)sequencer_timer, 0);
	timer_running = 0;
}
#endif

#ifndef EXCLUDE_AUDIO
static void
sound_mem_init()
{
	int i;
	int dev;
	char *tmpbuf;
	unsigned long dma_pagesize;
	struct dma_buffparms *dmap;
	static unsigned long dsp_init_mask = 0;

	for (dev = 0; dev < num_audiodevs; dev++) {	/* Enumerate devices */
		/* Check if need to allocate on this device */
		if ((dsp_init_mask & 1 << dev) != 0 ||
		    audio_devs[dev]->buffcount <= 0 ||
		    audio_devs[dev]->dmachan <= 0)
			continue;

		dsp_init_mask |= 1 << dev;
		dmap = audio_devs[dev]->dmap;

		if (audio_devs[dev]->flags & DMA_AUTOMODE)
			audio_devs[dev]->buffcount = 1;

		if (audio_devs[dev]->dmachan > 3 &&
		    audio_devs[dev]->buffsize > 65536)
			dma_pagesize = 131072;	/* 128k */
		else
			dma_pagesize = 65536;

		/* More sanity checks */

		if (audio_devs[dev]->buffsize > dma_pagesize)
			audio_devs[dev]->buffsize = dma_pagesize;
		audio_devs[dev]->buffsize &= ~0xfff;	/* Truncate to n*4k */
		if (audio_devs[dev]->buffsize < 4096)
			audio_devs[dev]->buffsize = 4096;

		/* Now allocate the buffers */

		for (dmap->raw_count = 0;
		    dmap->raw_count < audio_devs[dev]->buffcount;
		    dmap->raw_count++) {
			tmpbuf = (char *)
			    vm_page_alloc_contig(audio_devs[dev]->buffsize,
			    0ul, 0xfffffful, dma_pagesize);

			if (tmpbuf == NULL) {
				printf("snd: Unable to allocate %d bytes of "
				    "buffer\n", audio_devs[dev]->buffsize);
				return;
			}

			dmap->raw_buf[dmap->raw_count] = tmpbuf;
			dmap->raw_buf_phys[dmap->raw_count] = kvtop(tmpbuf);
		}
	}
}

#endif

int
snd_ioctl_return(int *addr, int value)
{
	if (value < 0)
		return (value);		/* Error */
	suword(addr, value);
	return (0);
}

#endif
