/*-
 * Copyright (c) 1994 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: sa.c,v 2.5 1995/12/12 22:21:55 cp Exp $
 */

/*
 * Adaptec AHA-1520/1522 SCSI host adapter driver
 */

#include <sys/param.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/device.h>
#include <sys/malloc.h>

#include <vm/vm.h>

#include <dev/scsi/scsi.h>
#include <dev/scsi/scsivar.h>
#include <dev/scsi/disk.h>
#include <dev/scsi/tape.h>

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/isa/icu.h>

#include <machine/cpu.h>

#include "sareg.h"


/*
 * information for running an operation
 */
typedef struct sa_op {
	struct	sa_op	*op_next;	/* used to put on various queues */
	caddr_t 	 op_data;	/* data buffer */
	caddr_t 	 op_savedata;	/* data buffer */
	struct buf	*op_bp;		/* original buf needed for chaining */
	u_int		 op_tlen;	/* transfer length */
	u_int		 op_savetlen;	/* transfer length */
	struct buf	*op_savebp;	/* so we can go back a buf */
	u_int		 op_target:3;
	u_int		 op_lun:3;
	u_int		 op_icmd:1;	/* operation is for an immediate cmd */
	u_int		 op_done:1;	/* icmd is done */
	u_int		 op_idsent:1;	/* have sent initial id message */
	u_char		 op_status;	/* scsi status */	
	u_char		 op_fpe;	/* fake parity error */
	phase_t		 op_phase;
	struct scsi_cdb  op_cdb;	/* real scsi cdb from caller */
	scintr_fn	 op_bintr;	/* base interrupt handler */
	struct device	*op_bintrdev;	/* base device */
} sa_op_t;

typedef struct sa_softc {
	struct	hba_softc sc_hba;
	struct	isadev sc_isa;
	struct	intrhand sc_ih;
	u_int	sc_iobase;		/* base port for device */
	sa_op_t	*sc_op_free;		/* free ops */
	sa_op_t *sc_op_issue;		/* ops which are active */
	sa_op_t *sc_op_go;		/* op currently being started */
	sa_op_t *sc_op_disconnected;	/* ops which are disconnected */
	u_char	sc_noparity;		/* bit field targs don't do parity */
	u_char	sc_id;			/* adapter's SCSI ID */
	void	(*sc_intr)();		/* routine to run interrupt service */
}sa_softc_t;

static int saprobe __P((struct device *parent, struct cfdata *cf, void *aux));
static void saattach __P((struct device *parent, struct device *self,
	    void *aux));
static int saicmd __P((struct hba_softc *hba, int targ, struct scsi_cdb *cdb,
	    caddr_t data, int len, int rw));
static int sadump __P((struct hba_softc *hba, int targ, struct scsi_cdb *cdb,
	    caddr_t addr, int len));
static void sastart __P((struct device *self, struct sq *sq, struct buf *bp,
	    scdgo_fn dgo, struct device *dev));
static int sago __P((struct device *self, int targ, scintr_fn intr,
	    struct device *dev, struct buf *bp, int pad));
static void sarel __P((struct device *self));
static void sahbareset __P((struct hba_softc *hba, int resetunits));

struct cfdriver sacd =
    { 0, "sa", saprobe, saattach, DV_DULL, sizeof (sa_softc_t) };

static struct hbadriver sahbadriver =
    { saicmd, sadump, sastart, sago, sarel, sahbareset };

/*
 * shifts to pick up values from flags field
 */
#define SA_F_NOPARITY_S 0	/* units which have parity forced off */
#define SA_F_NODISCONNECT_S 8	/* units which have disconnect forced off */
#define SA_F_IID_S 16		/* shift to get initiator id */
#define SA_F_IID_M 0xf		/* mask, includes extra bit to allow setting
					an id of 0 */
/*
 * macros to read and write the device
 */
#define OUTB_SC(off, value)	outb(sc->sc_iobase + (off), (value))
#define OUTB_IOB(off, value)	outb(iob + (off), (value))

#define INB_SC(off) (inb(sc->sc_iobase + (off)))
#define INB_IOB(off) (inb(iob + (off)))

/*
 * internal use forward references
 */
static int saintr __P((void *sc0));
static void saintr_busfree __P((sa_softc_t *sc));
static void saintr_selected __P((sa_softc_t *sc));
static void saintr_itp __P((sa_softc_t *sc));
static void saintr_reconnect __P((sa_softc_t *sc));

static sa_op_t *op_alloc __P((sa_softc_t *sc));
static void op_run __P((register sa_softc_t *sc, sa_op_t *op));
static void op_done __P((sa_softc_t *sc, sa_op_t *op));
static void op_free __P((sa_softc_t *sc, sa_op_t *op));

static void saforceintr __P((u_int *iobp));
static void sareset_hw __P((sa_softc_t *sc));
static void sadumpreg __P((sa_softc_t *sc));

/*
 * standard probe routine
 */
static int
saprobe(struct device *parent, struct cfdata *cf, void *aux)
{
	struct isa_attach_args *ia = aux;
	u_int iob = ia->ia_iobase;
	int i;

	/*
	 * use stack to test if hardware present. This is fairly
	 * unique behavior which should guarantee we really are
	 * talking to a 1520.
	 */
	OUTB_IOB(DMACNTRL1, 0);
	for (i = 1; i <= 16; i++)
		OUTB_IOB(STACK, i * 15);
	OUTB_IOB(DMACNTRL1, 0);
	for (i = 1; i <= 16; i++)
		if (INB_IOB(STACK) != i * 15)
			return (0);

	ia->ia_iosize = SA_NPORT;

	if (ia->ia_irq != IRQUNK)
		return (1);

	if ((ia->ia_irq = isa_discoverintr(saforceintr, (void *) &iob)) == IRQ0)
		return (0);
	return (1);
}

/*
 * standard attach routine.
 */
static void
saattach(struct device *parent, struct device *self, void *aux)
{
	struct isa_attach_args *ia = aux;
	register sa_softc_t *sc = (sa_softc_t *)self;
	int i;
	sa_op_t    *op;
	register u_int iob = ia->ia_iobase;

	sc->sc_iobase = iob;

	if ((sc->sc_hba.hba_dev.dv_flags >> SA_F_IID_S) & SA_F_IID_M != 0)
	    sc->sc_id = sc->sc_hba.hba_dev.dv_flags >> SA_F_IID_S;
	else
	    sc->sc_id = 7;

	sc->sc_noparity = sc->sc_hba.hba_dev.dv_flags >> SA_F_NOPARITY_S;

	aprint_naive(": Adaptec SCSI");
	aprint_verbose(", initiator id %d ", sc->sc_id); 

	sareset_hw(sc);

	printf("\nDelaying for SCSI bus reset and device self tests");
	DELAY(5000000); 	/* 5 seconds */

	sc->sc_hba.hba_driver = &sahbadriver;
	/*
	 * Link into isa and set interrupt handler.
	 */
	isa_establish(&sc->sc_isa, &sc->sc_hba.hba_dev);
	sc->sc_ih.ih_fun = saintr;
	sc->sc_ih.ih_arg = sc;

	intr_establish(ia->ia_irq, &sc->sc_ih, DV_DISK);
	printf("\n");

	/*
	 * alloc one more op than targets. This is
	 * because done does call back which may start
	 * next op before releasing op
	 */
	MALLOC(op, sa_op_t *, sizeof *op, M_DEVBUF, M_WAITOK);
	op_free(sc, op); 	/* place on free list */
	for (i = 0; i < 8; i++) {
		if (i == sc->sc_id)
			continue;
		if (scsi_test_unit_ready(&sc->sc_hba, i, 0) >= 0) {
			SCSI_FOUNDTARGET(&sc->sc_hba, i);
			MALLOC(op, sa_op_t *, sizeof *op, M_DEVBUF, M_WAITOK);
			op_free(sc, op); 	/* place on free list */
		}
	}
	OUTB_SC(DMACNTRL0, DMA0_INTEN);
}

/*
 * standard icmd routine
 */
static int
saicmd(struct hba_softc *hba, int targ, struct scsi_cdb *cdb, caddr_t data,
       int len, int rw)
{
	int status;
	int s;
	sa_op_t	saop;		/* op on stack elimnates deadlock problems */
	sa_op_t *op;
	sa_softc_t *sc = (sa_softc_t *)hba;

	s = splbio();
	op=&saop;
	bzero((caddr_t)&saop, sizeof (*op));
	bcopy(cdb, &op->op_cdb, SCSICMDLEN(cdb->cdb_bytes[0]));
	op->op_savetlen = len;
	op->op_savedata = data;
	op->op_icmd = 1;
	op->op_target = targ;
	op->op_lun = cdb->cdb_bytes[1] >> 5;
	op_run(sc, op);
	while (!op->op_done) {
		if (INB_SC(DMASTAT) & DMAS_INTSTAT)
			saintr(sc);
	}
	status = op->op_status;
	splx(s);
	if (status & 0x80)
		return (-1);		/* minus one for timeout */
	return (status);
}


/*
 * standard dump routine. This routine has to
 * map the physical address into a logical address
 * before it can run the io.
 */
extern char *dumpbufp;

static int
sadump(struct hba_softc *hba, int targ, struct scsi_cdb *cdb, caddr_t addr,
       int len)
{
	int i;

	for (i = 0; i < len; i += ctob(1)) 
		pmap_enter(kernel_pmap, (vm_offset_t)(dumpbufp + i),
		    (vm_offset_t) addr + i, VM_PROT_READ, TRUE);
	return (saicmd(hba, targ, cdb, dumpbufp, len, B_WRITE));
}

/*
 * standard start routine.
 * we can always start the io.
 */
static void
sastart(struct device *self, struct sq *sq, struct buf *bp, scdgo_fn dgo,
        struct device *dev)
{
	int s;
	sa_softc_t *sc = (sa_softc_t *)self;
	if (bp) {
		s = splbio();
		sc->sc_op_go = op_alloc(sc);
		splx(s);
		(*dgo)(dev, &sc->sc_op_go->op_cdb);
		return;
	}
	(*dgo)(dev, (struct scsi_cdb *)NULL);
}

/* 
 * standard go routine. Fill in the op
 * and run the io
 */
static int
sago(struct device *self, int targ, scintr_fn intr, struct device *dev,
     struct buf *bp, int pad)
{
	struct sa_softc *sc = (struct sa_softc *)self;
	sa_op_t *op;

	op = sc->sc_op_go;
	sc->sc_op_go = NULL;
	op->op_savetlen = bp->b_bcount;
	op->op_savedata = bp->b_un.b_addr;
	op->op_savebp = bp->b_chain;
	op->op_target = targ;
	op->op_bintr = intr;
	op->op_bintrdev = dev;
	op->op_lun = op->op_cdb.cdb_bytes[1] >> 5;
	op_run(sc, op);
	return (0);
}

/*
 * 	If start it called but not go be need to release
 *	the op saved for go. We also have to start the
 *	next operation if the queue for multiple luns is
 *	not empty.
 */
static void
sarel(struct device *self)
{
	sa_softc_t *sc = (sa_softc_t *)self;
	struct sq *sq;

	if (sc->sc_op_go) {
		op_free(sc, sc->sc_op_go);
		sc->sc_op_go = NULL;
	}
	if ((sq = sc->sc_hba.hba_head) != NULL) {
		sc->sc_op_go = op_alloc(sc);
		(*sq->sq_dgo)(sq->sq_dev, &sc->sc_op_go->op_cdb);
	}
	return;
}

/*
 * standard reset code.
 */
static void
sahbareset(struct hba_softc *hba, int resetunits)
{
	sa_softc_t *sc = (sa_softc_t *)hba;
	sa_op_t *op, *op1;

	sareset_hw(sc);
	DELAY(5000000); 	/* 5 seconds */
	op1 = sc->sc_op_issue;
	while ((op = op1) != NULL) {
		op1 = op->op_next;
		op_free(sc, op);
	}
	if (resetunits)
		scsi_reset_units(hba);
}

/*
 * base level sa interrupt handler. Just call the handler for the
 * expected interrupt. Loop as long as the interrupt condition
 * is present. Around half the time the next interrupt conditon
 * is present by the time we are ready to exit the handler.
 */
static int
saintr(void *sc0)
{
	sa_softc_t *sc = sc0;

	do {
		sc->sc_intr(sc);
	} while (INB_SC(DMASTAT)& DMAS_INTSTAT);
	return (1);
}

/*
 * this code is called when a bus free interrupt is expected.
 * If a target reconnects when we are expecting a busfree 
 * just call the reconnect handler. If this is a real busfree
 * interrupt then tell the hardware to acquire the bus.
 */
static void
saintr_busfree(sa_softc_t *sc)
{
	register int iob = sc->sc_iobase;
	sa_op_t	*op;

	if (INB_IOB(SSTAT0) & SS0_SELDI) {
		saintr_reconnect(sc);
		return;
	}

	/*
	 * we are idle, the bus is free, and we have a operation to
	 * run. Get the bus.
	 */

	OUTB_IOB(SCSIID, (sc->sc_id << SI_OID_S) | sc->sc_op_issue->op_target);

	OUTB_IOB(SXFRCTL1, XF1_ENSTIMER);

	if (sc->sc_op_disconnected != NULL)
		OUTB_IOB(SCSISEQ,  SS_ENSELO | SS_ENAUTOATNO | SS_ENRESELI);
	else 
		OUTB_IOB(SCSISEQ,  SS_ENSELO | SS_ENAUTOATNO);

	OUTB_IOB(SIMODE0, IM0_ENSELDO | IM0_ENSELDI);
	OUTB_IOB(SIMODE1, IM1_ENSELTIMO);
	OUTB_IOB(SSTAT1, SS1_BUSFREE);	/* clear interrupt */
	sc->sc_intr = saintr_selected;	/* selected interrupt handler */
}

/*
 * This routine is called when a selected interrupt is expected.
 * Sometimes when a selected interrupt is expected a reconnect
 * interrupt may occur. If this happens just call the reconnect
 * handler.
 * If this is a real selected interrupt set the hardware up for 
 * information transfer phases.
 */ 
static void
saintr_selected(sa_softc_t *sc)
{
	register int iob = sc->sc_iobase;
	sa_op_t	*op;

	if (INB_IOB(SSTAT0) & SS0_SELDI) {
		saintr_reconnect(sc);
		return;
	}

	op = sc->sc_op_issue;
	if (INB_IOB(SSTAT1) & SS1_SELTO) {
		/*
		 * The selection timed out
		 */
		OUTB_IOB(SCSISEQ, 0);	/* disable selection sequence */
		OUTB_IOB(SIMODE0, 0);
		OUTB_IOB(SIMODE1, 0);
		OUTB_IOB(SSTAT1, SS1_SELTO); /* clear interrupt last */
		op->op_status = 0x80;
		op_done(sc, op);
		return;
	}
		
	/*
	 * at this point we are here because we just got selected
	 */
	if (sc->sc_noparity & (1 << op->op_target))
	    OUTB_IOB(SXFRCTL1, 0);
	else
	    OUTB_IOB(SXFRCTL1, XF1_ENSPCHK);
	OUTB_IOB(SCSISEQ,  SS_ENAUTOATNP); /* set attn on parity error */
	OUTB_IOB(SSTAT0, SS0_SELDO);
	OUTB_IOB(SIMODE0, 0);
	OUTB_IOB(SIMODE1, IM1_ENREQINT);
	sc->sc_intr = saintr_itp;
	op->op_phase = P_SELECTED;
	op->op_data = op->op_savedata;
	op->op_tlen = op->op_savetlen;
	op->op_bp = op->op_savebp;
}

/*
 * information transfer phase
 * basically follow instructions for target on what
 * to do. 
 */
static void
saintr_itp(register sa_softc_t *sc)
{
    register u_int iob = sc->sc_iobase;
    register sa_op_t *op;
    phase_t phase;
    phase_t lastphase;
    int cntdwn, cnt;
    struct buf *bp;
    caddr_t addr;
    u_char c;

    op = sc->sc_op_issue;
    phase = INB_IOB(SCSISIG) & PHASE_M;
    lastphase = op->op_phase;
    op->op_phase = phase;

    /*
     * clean up from previous phase as required
     */
    switch (lastphase) {
    case P_SELECTED:		/* come in from saintr_selected with this */
    case P_MSG_OUT:
    case P_MSG_IN:
	/*
	 * the REQINT goes away without being cleared when the req goes
	 * away or the ack for the req is present
	 */
	break;
    case P_COMMAND:
    case P_DATA_OUT:
	/*
	 * get out of "dma" mode and fall through
	 * to clear interrupt
	 */
	OUTB_IOB(DMACNTRL0, DMA0_INTEN);
	OUTB_IOB(SXFRCTL0, XF0_CH1);
    case P_DATA_IN:
    case P_STATUS:
	/*
	 * clear phase mismatch interrupt
	 */
	OUTB_IOB(SSTAT1, IM1_ENPHASEMIS);
	break;
    }

    /*
     * set hardware up to run this phase
     */
    OUTB_IOB(SCSISIG, phase);

    /*
     *	run new phase
     */

    switch (phase) {

    case P_DATA_OUT:
	addr = op->op_data;
	cntdwn = op->op_tlen;
	bp = op->op_bp;

	OUTB_IOB(SIMODE1, IM1_ENPHASEMIS);
	OUTB_IOB(SXFRCTL0, XF0_CH1|XF0_SCSIEN|XF0_DMAEN);
	OUTB_IOB(DMACNTRL0, DMA0_RSTFIFO | DMA0_INTEN);
	OUTB_IOB(DMACNTRL0, DMA0_ENDMA | DMA0_WRITE | DMA0_INTEN);

	/*
	 * main data transfer out loop
	 */
	while (cntdwn != 0 || bp != NULL) {
	    /*
	     * wait for hardware to say it is ready or that 
	     * it is not going to become ready 
	     */
	    do {
		c = INB_IOB(DMASTAT);
	    } while ((c & (DMAS_DFIFOEMP | DMAS_INTSTAT)) == 0);

	    if (c & DMAS_INTSTAT) {
		/*
		 * we got a phase change, back up amount left in
		 * fifos
		 */
		c = INB_IOB(SSTAT2);
		if (c & SS2_SFULL)
		    cnt = 8;
		else
		    cnt = c & SS2_SFCNT_M;
		cnt += INB_IOB(FIFOSTAT);
		addr -= cnt;
		cntdwn += cnt;
		break;
	    }

	    if (cntdwn == 0) {
		cntdwn = bp->b_bcount;
		addr = bp->b_un.b_addr;
		bp = bp->b_chain;
	    }
	    /*
	     * compute how much data we can shove bus can take in one gulp
	     */
	    if (cntdwn > FIFO_MAXW)
		    cnt = FIFO_MAXW;
	    else
		    cnt = cntdwn;


	    outsw(sc->sc_iobase + DATA, addr, cnt>>1); /* gulp */
	    cntdwn -= cnt & ~1;
	    addr += cnt & ~1;
	    /*
	     * get rid of odd byte at end of buffer. This only
	     * happens during last gulp. It also never happens
	     * when doing chained buffers;
	     */
	    if (cnt & 1) {
		OUTB_IOB(DMACNTRL0,
		    DMA0_ENDMA | DMA0_WRITE | DMA0_INTEN | DMA0_8BIT);
		outsb(sc->sc_iobase + DATA, addr, 1);
		addr++;
		cntdwn--;
		}
	}
	op->op_tlen = cntdwn;
	op->op_data = addr;
	op->op_bp = bp;
	break;

    case P_DATA_IN:
	addr = op->op_data;
	cntdwn = op->op_tlen;
	bp = op->op_bp;

	OUTB_IOB(DMACNTRL0, DMA0_RSTFIFO | DMA0_INTEN);
	OUTB_IOB(SXFRCTL0, XF0_CH1|XF0_SCSIEN|XF0_DMAEN);
	OUTB_IOB(DMACNTRL0, DMA0_ENDMA | DMA0_INTEN);
	OUTB_IOB(SIMODE1, IM1_ENPHASEMIS);

	/*
	 * main loop to transfer data in
	 */
	while (cntdwn != 0 || bp != NULL) {
	    /* wait for data to show up or hardware to
	     * end the transfer
	     */
	    while (!(INB_IOB(DMASTAT) & (DMAS_DFIFOFULL | DMAS_INTSTAT)));
	    c = INB_IOB(FIFOSTAT);
	    if (c == 0) {
		break;
	    }
	    if (cntdwn == 0) {
		cntdwn = bp->b_bcount;
		addr = bp->b_un.b_addr;
		bp = bp->b_chain;
	    }
	    if (c > cntdwn) {
		/*
		 * this happens when the drive wants to transfer
		 * more data than was requested or when going
		 * from one buf to the next in chain mode
		 */
		c = cntdwn;
	    }
	    cnt = c & ~1;
	    /* if we haven't a full word to do then we have
	     * got a single byte. In the case of an odd length
	     * we will take another pass through the loop. If this
	     * wasn't the end of the buffer we will go ahead a fill
	     * the fifo again. This will cause the odd byte to
	     * be the first one pulled off in the next gulp.
	     * We only want to take the odd byte at the very end
	     * of the buffer.
	     */
	    if (cnt != 0) {
		insw(sc->sc_iobase + DATA, addr, cnt>>1);
		addr += cnt;
		cntdwn -= cnt;
	    } else {
		OUTB_IOB(DMACNTRL0, DMA0_ENDMA | DMA0_INTEN | DMA0_8BIT);
		insb(sc->sc_iobase + DATA, addr, 1);
		addr++;
		cntdwn--;
	    }
	}
	OUTB_IOB(DMACNTRL0, DMA0_INTEN);
	OUTB_IOB(SXFRCTL0, XF0_CH1);
	op->op_tlen = cntdwn;
	op->op_data = addr;
	op->op_bp = bp;
	break;

    case P_COMMAND:
	OUTB_IOB(SIMODE1, IM1_ENPHASEMIS);
	OUTB_IOB(SXFRCTL0, XF0_CH1|XF0_SCSIEN|XF0_DMAEN);
	OUTB_IOB(DMACNTRL0, DMA0_RSTFIFO | DMA0_INTEN);
	OUTB_IOB(DMACNTRL0, DMA0_ENDMA | DMA0_WRITE | DMA0_INTEN);
	outsw(sc->sc_iobase + DATA, &op->op_cdb,
	      SCSICMDLEN(op->op_cdb.cdb_bytes[0]) >> 1); /* length in words */
	break;

    case P_STATUS:
	OUTB_IOB(SXFRCTL0, XF0_CH1 | XF0_SPIOEN);
	op->op_status = INB_IOB(SCSIDAT);
	OUTB_IOB(SIMODE1, IM1_ENPHASEMIS);
	break;

    case P_MSG_OUT:
	OUTB_IOB(SXFRCTL0, XF0_CH1 | XF0_SPIOEN); 
	if (!op->op_idsent) {
	    u_int flags = sc->sc_hba.hba_dev.dv_flags;
	    OUTB_IOB(SSTAT1, SS1_ATNTARG); 	/* de-assert attention */
	    c = MSG_IDENTIFY | op->op_lun;
	    if (!((flags >> SA_F_NODISCONNECT_S) & (1 << op->op_target)))
		c |= MSG_IDENTIFY_DR ;
	    OUTB_IOB(SCSIDAT, c);
	    op->op_idsent = 1;
	    break;
	}

	/*
	 * if parity error and last phase was message
	 * send message parity error otherwise
	 * send initiator detected error
	 */
	if (INB_IOB(SSTAT1) & SS1_SCSIIPERR || op->op_fpe) {
	    OUTB_IOB(SSTAT1, SS1_SCSIIPERR);	/* clear parity error */
	    OUTB_IOB(SSTAT1, SS1_ATNTARG); 	/* de-assert attention */
	    if (lastphase == P_MSG_IN)
		OUTB_IOB(SCSIDAT, MSG_PARITY_ERROR);
	    else
		OUTB_IOB(SCSIDAT, MSG_INIT_DETECT_ERROR);
	    break;
	}
	/*
	 * apparenty unsolicited just send nop
	 */
	OUTB_IOB(SCSIDAT, MSG_NOOP);
	break;

    case P_MSG_IN:
	OUTB_IOB(SIMODE1, IM1_ENREQINT);
	OUTB_IOB(SXFRCTL0, XF0_CH1 | XF0_SPIOEN);
	/*
	 * first check for parity errors. If we got one
	 * toss until we get a phase change. We need to do
	 * this because we could miss an extended command or
	 * turn something not an extended command into an extended command
	 */
	if (INB_IOB(SSTAT1) & SS1_SCSIIPERR) {
	    while (!(INB_IOB(SSTAT1) & SS1_PHASEMIS)) {
		INB_IOB(SCSIDAT);
		/*
		 * if bus goes free here then device
		 * likely doesn't handle parity
		 * assume we just got a done message
		 */
		while (!(INB_IOB(SSTAT1) & SS1_REQINT)) {
		    if ((INB_IOB(SSTAT1) & SS1_BUSFREE)) {
			sc->sc_noparity |= 1 << op->op_target;
			OUTB_IOB(SXFRCTL0, XF0_CH1);
			OUTB_IOB(SXFRCTL1, 0);
			OUTB_IOB(SIMODE1, 0);
			OUTB_IOB(SSTAT1, SS1_SCSIIPERR);  /* clear parity err */
			OUTB_IOB(SSTAT1, SS1_ATNTARG); 	/* de-assert attn */
			op_done(sc, op);
			return;
		    }
		}
	    }
	    break;
	}
		
	c = INB_IOB(SCSIDAT);
	switch (c) {

	case MSG_CMD_COMPLETE:
	    OUTB_IOB(SXFRCTL0, XF0_CH1);
	    OUTB_IOB(SIMODE1, 0);
	    op_done(sc, op);
	    break;

	case MSG_EXT_MESSAGE:
	{
	    u_char emsg[6];
	    u_char elength;
	    u_char ecnt;

	    while (!(INB_IOB(SSTAT1) & SS1_REQINT));
	    /*
	     * first check if parity error set.
	     * if so don't believe length byte
	     * toss bytes until phase changes
	     */
	    if (INB_IOB(SSTAT1) & SS1_SCSIIPERR) {
		while (!(INB_IOB(SSTAT1) & SS1_PHASEMIS)) {
		    INB_IOB(SCSIDAT);
		    while (!(INB_IOB(SSTAT1) & SS1_REQINT));
		}
		break;
	    }
	    /*
	     * get extened message
	     */
	    elength = INB_IOB(SCSIDAT);
	    for (ecnt = 0; ecnt < elength; ecnt++) {
		while (!(INB_IOB(SSTAT1) & SS1_REQINT));
		if (ecnt < sizeof (emsg))
		    emsg[ecnt] = INB_IOB(SCSIDAT);
		else
		    INB_IOB(SCSIDAT);
	    }
	    if (INB_IOB(SSTAT1) & SS1_SCSIIPERR) {
		break;
	    }
	    /*
	     * stuff to handle extended message would go here
	     */
	    break;
	}

	case MSG_SAVE_DATA_PTR:
	    op->op_savedata = op->op_data;
	    op->op_savetlen = op->op_tlen;
	    op->op_savebp = op->op_bp;
	    break;

	case MSG_RESTORE_PTR:
	    op->op_tlen = op->op_savetlen;
	    op->op_data = op->op_savedata;
	    op->op_bp = op->op_savebp;
	    break;

	case MSG_DISCONNECT:
	    OUTB_IOB(SIMODE0, IM0_ENSELDI);
	    OUTB_IOB(SCSISEQ, SS_ENRESELI);
	    if ((sc->sc_op_issue = op->op_next) == NULL)
		sc->sc_intr = saintr_reconnect;
	    else {
		sc->sc_intr = saintr_busfree;
		OUTB_IOB(SIMODE1, IM1_ENBUSFREE);
	    }
	    op->op_next = sc->sc_op_disconnected;
	    sc->sc_op_disconnected = op;
	    break;
	     
	case MSG_REJECT:
	default:	/* default for message in */
	break;
	}
    default:		/* default for phase */
	/*
	 * should do something like abort operation in progress
	 */
	break;
    }
}

/*
 * code to handle reconnect interrupt. If we got here
 * we really should be doing reconnect. Either because
 * busfree or selected found they really were doing a reconnect or
 * because this is the only interrupt enabled and was called direct
 * from saintr.
 */
static void
saintr_reconnect(register sa_softc_t *sc)
{
	register u_int iob = sc->sc_iobase;
	sa_op_t	*op;
	sa_op_t **opp;
	int i;
	u_char c;

	/*
	 * figure out what target is trying to reconnect
	 */
	c = INB_IOB(SCSIID);
	for(i = 0; i < 8; i++) {
		if (i == sc->sc_id)
			continue;
		if ((c & (1 << i)) != 0)
			break;
	}

	/*
	 * search through the disconnected queue looking for
	 * this target
	 */
	opp = &sc->sc_op_disconnected;
	while ((op = *opp) != NULL) {
		if (op->op_target == i)
			break;
		opp = &op->op_next;
	}

	/* 
	 * Some drives appear to reconnect even though
	 * they never actually disconnected. Leave
	 * everything as is in this case. This could also
	 * be caused by not dismissing the select in interrupt
	 * correctly or missing the disconnect.
	 */
	if (op == NULL) {
	    OUTB_IOB(SSTAT0, SS0_SELDI);
	    return;
	}

	*opp = op->op_next;	/* take out of disconnect chain */

	op->op_next = sc->sc_op_issue;	/* put on head of issue queue */
	sc->sc_op_issue = op;
	/*
	 * set up hardware to go into information transfer
	 * phases
	 */
	if (sc->sc_noparity & (1 << op->op_target))
	    OUTB_IOB(SXFRCTL1, 0);
	else
	    OUTB_IOB(SXFRCTL1, XF1_ENSPCHK);
	OUTB_IOB(SCSISEQ,  SS_ENAUTOATNP); /* set attn on parity error */
	OUTB_IOB(SSTAT0, SS0_SELDI);
	OUTB_IOB(SIMODE0, 0);
	OUTB_IOB(SIMODE1, IM1_ENREQINT);
	sc->sc_intr = saintr_itp;
	op->op_phase = P_SELECTED;
}

/*
 * return op to free list. 
 * could be macro. But usefull place to
 * put in debugging code
 */
static void
op_free(sa_softc_t *sc, sa_op_t *op)
{
	op->op_next = sc->sc_op_free;
	sc->sc_op_free  = op;
}

/*
 * get op off of free list. There should always
 * be one available. If not either this driver or
 * one of the target drivers has a bug.
 */
static sa_op_t *
op_alloc(sa_softc_t *sc)
{
	sa_op_t *op;
	if (sc->sc_op_free == NULL) {
		panic("sa: op_alloc no op");
	}
	op = sc->sc_op_free;
	sc->sc_op_free = op->op_next;
	bzero(op, sizeof (*op));
	return (op);
}

/*
 * start an op running. If idle request buss free interrupt. Otherwise
 * put on the end of the issue queue. The head of the issue
 * queue is the op being currently run. Operations which have
 * been disconnect get pushed back on the head of the issue
 * queue when they reconnect.
 */
static void
op_run(register sa_softc_t *sc, sa_op_t *op)
{
	sa_op_t *op1;
	if (sc->sc_op_issue == NULL) {
		sc->sc_op_issue = op;
		sc->sc_intr = saintr_busfree;
		OUTB_SC(SIMODE1, IM1_ENBUSFREE);
		OUTB_SC(PORTA, 1);	/* turn light on */
	} else {
		for (op1 = sc->sc_op_issue; op1->op_next; op1 = op1->op_next);
		op1->op_next = op;
	}
	op->op_next = NULL;
	return;
}

/*
 * an operation has completed. Pull the command off of the issue
 * queue. Set up the hardware for the next operation if there is
 * one and free the op if this isn't an immediate command
 */
static void
op_done(sa_softc_t *sc, sa_op_t *op)
{
	int status;
	struct buf *bp;

	if(sc->sc_op_disconnected != NULL) {
	    OUTB_SC(SIMODE0, IM0_ENSELDI);
	    OUTB_SC(SCSISEQ, SS_ENRESELI);
	    sc->sc_intr = saintr_reconnect;
	}
	if((sc->sc_op_issue = op->op_next) != NULL) {
		OUTB_SC(SIMODE1, IM1_ENBUSFREE);	/* start next op */
		sc->sc_intr = saintr_busfree;
	} else
		OUTB_SC(PORTA, 0);	/* turn light off */
	op->op_next = NULL;

	/* 
	 * if an immediate command return
	 * op was allocated on the stack
	 */
	if (op->op_icmd) {
		op->op_done = 1;
		return;
	}
	status = op->op_status;
	if (status & 0x80)
		status = -1;
	for (bp = op->op_bp; bp != NULL; bp = bp->b_chain)
	    op->op_tlen += bp->b_bcount;
	(*op->op_bintr)(op->op_bintrdev, status, op->op_tlen);
	op_free(sc, op);
}

/*
 * routine to force interrupt. Use by probe to determine interrupt level
 */
static void 
saforceintr(u_int *iobp)
{
	u_int iob = *iobp;

	OUTB_IOB(DMACNTRL0, 0);
	OUTB_IOB(SSTAT0, SS0_CLRALL); 
	OUTB_IOB(SSTAT1, SS1_CLRALL);
	OUTB_IOB(DMACNTRL0, DMA0_SWINT | DMA0_INTEN);
}

/*
 * reset the hardware including doing a bus reset. If we allow
 * multiple masters then the bus reset should be conditional.
 */
static void
sareset_hw(sa_softc_t *sc)
{
	register u_int iob = sc->sc_iobase;

	OUTB_IOB(SCSISEQ, 0);
	OUTB_IOB(SXFRCTL0, XF0_CH1);
	OUTB_IOB(SXFRCTL1, 0);
	OUTB_IOB(SCSISIG, 0);
	OUTB_IOB(SCSIRATE, 0);
	OUTB_IOB(SCSIID, sc->sc_id << SI_OID_S);
	OUTB_IOB(STCNT0, 0);
	OUTB_IOB(STCNT1, 0);
	OUTB_IOB(STCNT2, 0);
	OUTB_IOB(SCSITEST, 0);
	OUTB_IOB(SIMODE0, 0);
	OUTB_IOB(SIMODE1, 0);
	OUTB_IOB(DMACNTRL0, 0);
	OUTB_IOB(DMACNTRL1, 0);
	OUTB_IOB(DATAPORTL, 0);
	OUTB_IOB(DATAPORTH, 0);
	OUTB_IOB(TESTREG, 0);

	OUTB_IOB(SCSISEQ, SS_SCSIRSTO);
	delay(300000);			/* 300 millisecond assert time */
	OUTB_IOB(SCSISEQ, 0);
	OUTB_IOB(SSTAT1, SS1_SCSIRSTI);	/* clear scsi reset */
	OUTB_IOB(DMACNTRL0, DMA0_RSTFIFO);
	OUTB_IOB(SSTAT4, SS4_SYNCERR | SS4_FWERR | SS4_FRERR);
	OUTB_IOB(BURSTCNTRL, 0xf1);
	OUTB_IOB(SXFRCTL0, XF0_CH1 | XF0_CLRSTCNT | XF0_CLRCH1);

	OUTB_IOB(SSTAT0, SS0_CLRALL); 
	OUTB_IOB(SSTAT1, SS1_CLRALL);

}

static void
sadumpreg(sa_softc_t *sc)
{
	int i;
	char da[] = { SCSISEQ,SXFRCTL0,SXFRCTL1,SCSISIG,SCSIRATE,
		      SCSIID,SBUSBUS,STCNT0,STCNT1,STCNT2,SSTAT0,
		      SSTAT1,SSTAT2,SSTAT3,SSTAT4,SIMODE0,SIMODE1,DMACNTRL0,
		      DMACNTRL1,DMASTAT,FIFOSTAT,BURSTCNTRL};
	for (i = 0; i < sizeof(da); i++)
		printf("%02x, %02x\n", da[i], INB_SC(da[i]));
}
