/***********************************************************************
		Copyright 1994 Comtrol Corporation.
			All Rights Reserved.

The following source code is subject to Comtrol Corporation's
Developer's License Agreement.

This source code is protected by United States copyright law and 
international copyright treaties.

This source code may only be used to develop software products that
will operate with Comtrol brand hardware.

You may not reproduce nor distribute this source code in its original
form but must produce a derivative work which includes portions of
this source code only.

The portions of this source code which you use in your derivative
work must bear Comtrol's copyright notice:

		Copyright 1994 Comtrol Corporation.

***********************************************************************/
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/ioctl.h>
#include <sys/tty.h>
#include <sys/proc.h>
#include <sys/user.h>
#include <sys/conf.h>
#include <sys/file.h>
#include <sys/uio.h>
#include <sys/kernel.h>
#include <sys/syslog.h>
#include <sys/device.h>
#include <sys/malloc.h>

#include <i386/isa/icu.h>
#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>

#define ROCKET_C
#include "rpreg.h"
#include "rpvar.h"

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

Byte_t RData[RDATASIZE] =
{
   0x00, 0x09, 0xf6, 0x82,
   0x02, 0x09, 0x86, 0xfb,
   0x04, 0x09, 0x00, 0x0a,
   0x06, 0x09, 0x01, 0x0a,
   0x08, 0x09, 0x8a, 0x13,
   0x0a, 0x09, 0xc5, 0x11,
   0x0c, 0x09, 0x86, 0x85,
   0x0e, 0x09, 0x20, 0x0a,
   0x10, 0x09, 0x21, 0x0a,
   0x12, 0x09, 0x41, 0xff,
   0x14, 0x09, 0x82, 0x00,
   0x16, 0x09, 0x82, 0x7b,
   0x18, 0x09, 0x8a, 0x7d,
   0x1a, 0x09, 0x88, 0x81,
   0x1c, 0x09, 0x86, 0x7a,
   0x1e, 0x09, 0x84, 0x81,
   0x20, 0x09, 0x82, 0x7c,
   0x22, 0x09, 0x0a, 0x0a 
};

Byte_t RRegData[RREGDATASIZE]=
{
   0x00, 0x09, 0xf6, 0x82,             /* 00: Stop Rx processor */
   0x08, 0x09, 0x8a, 0x13,             /* 04: Tx software flow control */
   0x0a, 0x09, 0xc5, 0x11,             /* 08: XON char */
   0x0c, 0x09, 0x86, 0x85,             /* 0c: XANY */
   0x12, 0x09, 0x41, 0xff,             /* 10: Rx mask char */
   0x14, 0x09, 0x82, 0x00,             /* 14: Compare/Ignore #0 */
   0x16, 0x09, 0x82, 0x7b,             /* 18: Compare #1 */
   0x18, 0x09, 0x8a, 0x7d,             /* 1c: Compare #2 */
   0x1a, 0x09, 0x88, 0x81,             /* 20: Interrupt #1 */
   0x1c, 0x09, 0x86, 0x7a,             /* 24: Ignore/Replace #1 */
   0x1e, 0x09, 0x84, 0x81,             /* 28: Interrupt #2 */
   0x20, 0x09, 0x82, 0x7c,             /* 2c: Ignore/Replace #2 */
   0x22, 0x09, 0x0a, 0x0a              /* 30: Rx FIFO Enable */
};

CONTROLLER_T sController[CTL_SIZE] =
{
   {-1,-1,0,0,0,0,0,0,0,{0,0,0,0},{0,0,0,0},{-1,-1,-1,-1},{0,0,0,0}},
   {-1,-1,0,0,0,0,0,0,0,{0,0,0,0},{0,0,0,0},{-1,-1,-1,-1},{0,0,0,0}},
   {-1,-1,0,0,0,0,0,0,0,{0,0,0,0},{0,0,0,0},{-1,-1,-1,-1},{0,0,0,0}},
   {-1,-1,0,0,0,0,0,0,0,{0,0,0,0},{0,0,0,0},{-1,-1,-1,-1},{0,0,0,0}}
};

#if 0
/* IRQ number to MUDBAC register 2 mapping */
Byte_t sIRQMap[16] =
{
   0,0,0,0x10,0x20,0x30,0,0,0,0x40,0x50,0x60,0x70,0,0,0x80
};
#endif

Byte_t sBitMapClrTbl[8] =
{
   0xfe,0xfd,0xfb,0xf7,0xef,0xdf,0xbf,0x7f
};

Byte_t sBitMapSetTbl[8] =
{
   0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80
};

/***************************************************************************
Function: sInitController
Purpose:  Initialization of controller global registers and controller
          structure.
Call:     sInitController(CtlP,CtlNum,MudbacIO,AiopIOList,AiopIOListSize,
                          IRQNum,Frequency,PeriodicOnly)
          CONTROLLER_T *CtlP; Ptr to controller structure
          int CtlNum; Controller number
          ByteIO_t MudbacIO; Mudbac base I/O address.
          ByteIO_t *AiopIOList; List of I/O addresses for each AIOP.
             This list must be in the order the AIOPs will be found on the
             controller.  Once an AIOP in the list is not found, it is
             assumed that there are no more AIOPs on the controller.
          int AiopIOListSize; Number of addresses in AiopIOList
          int IRQNum; Interrupt Request number.  Can be any of the following:
                         0: Disable global interrupts
                         3: IRQ 3
                         4: IRQ 4
                         5: IRQ 5
                         9: IRQ 9
                         10: IRQ 10
                         11: IRQ 11
                         12: IRQ 12
                         15: IRQ 15
          Byte_t Frequency: A flag identifying the frequency
                   of the periodic interrupt, can be any one of the following:
                      FREQ_DIS - periodic interrupt disabled
                      FREQ_137HZ - 137 Hertz
                      FREQ_69HZ - 69 Hertz
                      FREQ_34HZ - 34 Hertz
                      FREQ_17HZ - 17 Hertz
                      FREQ_9HZ - 9 Hertz
                      FREQ_4HZ - 4 Hertz
                   If IRQNum is set to 0 the Frequency parameter is
                   overidden, it is forced to a value of FREQ_DIS.
          int PeriodicOnly: TRUE if all interrupts except the periodic
                               interrupt are to be blocked.
                            FALSE is both the periodic interrupt and
                               other channel interrupts are allowed.
                            If IRQNum is set to 0 the PeriodicOnly parameter is
                               overidden, it is forced to a value of FALSE.
Return:   int: Number of AIOPs on the controller, or CTLID_NULL if controller
               initialization failed.

Comments:
          If periodic interrupts are to be disabled but AIOP interrupts
          are allowed, set Frequency to FREQ_DIS and PeriodicOnly to FALSE.

          If interrupts are to be completely disabled set IRQNum to 0.

          Setting Frequency to FREQ_DIS and PeriodicOnly to TRUE is an
          invalid combination.

          This function performs initialization of global interrupt modes,
          but it does not actually enable global interrupts.  To enable
          and disable global interrupts use functions sEnGlobalInt() and
          sDisGlobalInt().  Enabling of global interrupts is normally not
          done until all other initializations are complete.

          Even if interrupts are globally enabled, they must also be
          individually enabled for each channel that is to generate
          interrupts.

Warnings: No range checking on any of the parameters is done.

          No context switches are allowed while executing this function.

          After this function all AIOPs on the controller are disabled,
          they can be enabled with sEnAiop().
*/
int sInitController(	CONTROLLER_T *CtlP,
			int CtlNum,
			ByteIO_t MudbacIO,
			ByteIO_t *AiopIOList,
			int AiopIOListSize,
			int IRQNum,
			Byte_t Frequency,
			int PeriodicOnly)
{
	int		i;
	ByteIO_t	io;

   CtlP->CtlNum = CtlNum;
   CtlP->CtlID = CTLID_0001;        /* controller release 1 */

   CtlP->MBaseIO = MudbacIO;
   CtlP->MReg1IO = MudbacIO + 1;
   CtlP->MReg2IO = MudbacIO + 2;
   CtlP->MReg3IO = MudbacIO + 3;
#if 1
   CtlP->MReg2 = 0;                 /* interrupt disable */
   CtlP->MReg3 = 0;                 /* no periodic interrupts */
#else
   if(sIRQMap[IRQNum] == 0)            /* interrupts globally disabled */
   {
      CtlP->MReg2 = 0;                 /* interrupt disable */
      CtlP->MReg3 = 0;                 /* no periodic interrupts */
   }
   else
   {
      CtlP->MReg2 = sIRQMap[IRQNum];   /* set IRQ number */
      CtlP->MReg3 = Frequency;         /* set frequency */
      if(PeriodicOnly)                 /* periodic interrupt only */
      {
         CtlP->MReg3 |= PERIODIC_ONLY;
      }
   }
#endif
   sOutB(CtlP->MReg2IO,CtlP->MReg2);
   sOutB(CtlP->MReg3IO,CtlP->MReg3);
   sControllerEOI(CtlP);               /* clear EOI if warm init */

   /* Init AIOPs */
   CtlP->NumAiop = 0;
   for(i=0; i < AiopIOListSize; i++)
   {
      io = AiopIOList[i];
      CtlP->AiopIO[i] = (WordIO_t)io;
      CtlP->AiopIntChanIO[i] = io + _INT_CHAN;
      sOutB(CtlP->MReg2IO,CtlP->MReg2 | (i & 0x03)); /* AIOP index */
      sOutB(MudbacIO,(Byte_t)(io >> 6));	/* set up AIOP I/O in MUDBAC */
      sEnAiop(CtlP,i);                         /* enable the AIOP */

      CtlP->AiopID[i] = sReadAiopID(io);       /* read AIOP ID */
      if(CtlP->AiopID[i] == AIOPID_NULL)       /* if AIOP does not exist */
      {
         sDisAiop(CtlP,i);                     /* disable AIOP */
         break;                                /* done looking for AIOPs */
      }

      CtlP->AiopNumChan[i] = sReadAiopNumChan((WordIO_t)io); /* num channels in AIOP */
      sOutW((WordIO_t)io + _INDX_ADDR,_CLK_PRE);      /* clock prescaler */
      sOutB(io + _INDX_DATA,CLOCK_PRESC);
      CtlP->NumAiop++;                         /* bump count of AIOPs */
      sDisAiop(CtlP,i);                        /* disable AIOP */
   }

   if(CtlP->NumAiop == 0)
      return(-1);
   else
      return(CtlP->NumAiop);
}

/***************************************************************************
Function: sReadAiopID
Purpose:  Read the AIOP idenfication number directly from an AIOP.
Call:     sReadAiopID(io)
          ByteIO_t io: AIOP base I/O address
Return:   int: Flag AIOPID_XXXX if a valid AIOP is found, where X
                 is replace by an identifying number.
          Flag AIOPID_NULL if no valid AIOP is found
Warnings: No context switches are allowed while executing this function.

*/
int sReadAiopID(ByteIO_t io)
{
   Byte_t AiopID;               /* ID byte from AIOP */

   sOutB(io + _CMD_REG,RESET_ALL);     /* reset AIOP */
   sOutB(io + _CMD_REG,0x0);
   AiopID = sInB(io + _CHN_STAT0) & 0x07;
   if(AiopID == 0x06)
      return(1);
   else                                /* AIOP does not exist */
      return(-1);
}

/***************************************************************************
Function: sReadAiopNumChan
Purpose:  Read the number of channels available in an AIOP directly from
          an AIOP.
Call:     sReadAiopNumChan(io)
          WordIO_t io: AIOP base I/O address
Return:   int: The number of channels available
Comments: The number of channels is determined by write/reads from identical
          offsets within the SRAM address spaces for channels 0 and 4.
          If the channel 4 space is mirrored to channel 0 it is a 4 channel
          AIOP, otherwise it is an 8 channel.
Warnings: No context switches are allowed while executing this function.
*/
int sReadAiopNumChan(WordIO_t io)
{
   Word_t x;

   sOutDW((DWordIO_t)io + _INDX_ADDR,0x12340000L); /* write to chan 0 SRAM */
   sOutW(io + _INDX_ADDR,0);       /* read from SRAM, chan 0 */
   x = sInW(io + _INDX_DATA);
   sOutW(io + _INDX_ADDR,0x4000);  /* read from SRAM, chan 4 */
   if(x != sInW(io + _INDX_DATA))  /* if different must be 8 chan */
      return(8);
   else
      return(4);
}

/***************************************************************************
Function: sInitChan
Purpose:  Initialization of a channel and channel structure
Call:     sInitChan(CtlP,ChP,AiopNum,ChanNum)
          CONTROLLER_T *CtlP; Ptr to controller structure
          CHANNEL_T *ChP; Ptr to channel structure
          int AiopNum; AIOP number within controller
          int ChanNum; Channel number within AIOP
Return:   int: TRUE if initialization succeeded, FALSE if it fails because channel
               number exceeds number of channels available in AIOP.
Comments: This function must be called before a channel can be used.
Warnings: No range checking on any of the parameters is done.

          No context switches are allowed while executing this function.
*/
int sInitChan(	CONTROLLER_T *CtlP,
		CHANNEL_T *ChP,
		int AiopNum,
		int ChanNum)
{
   int i;
   WordIO_t AiopIO;
   WordIO_t ChIOOff;
   Byte_t *ChR;
   Word_t ChOff;
   static Byte_t R[4];

   if(ChanNum >= CtlP->AiopNumChan[AiopNum])
      return(FALSE);                   /* exceeds num chans in AIOP */

   /* Channel, AIOP, and controller identifiers */
   ChP->CtlP = CtlP;
   ChP->ChanID = CtlP->AiopID[AiopNum];
   ChP->AiopNum = AiopNum;
   ChP->ChanNum = ChanNum;

   /* Global direct addresses */
   AiopIO = CtlP->AiopIO[AiopNum];
   ChP->Cmd = (ByteIO_t)AiopIO + _CMD_REG;
   ChP->IntChan = (ByteIO_t)AiopIO + _INT_CHAN;
   ChP->IntMask = (ByteIO_t)AiopIO + _INT_MASK;
   ChP->IndexAddr = (DWordIO_t)AiopIO + _INDX_ADDR;
   ChP->IndexData = AiopIO + _INDX_DATA;

   /* Channel direct addresses */
   ChIOOff = AiopIO + ChP->ChanNum * 2;
   ChP->TxRxData = ChIOOff + _TD0;
   ChP->ChanStat = ChIOOff + _CHN_STAT0;
   ChP->TxRxCount = ChIOOff + _FIFO_CNT0;
   ChP->IntID = (ByteIO_t)AiopIO + ChP->ChanNum + _INT_ID0;

   /* Initialize the channel from the RData array */
   for(i=0; i < RDATASIZE; i+=4)
   {
      R[0] = RData[i];
      R[1] = RData[i+1] + 0x10 * ChanNum;
      R[2] = RData[i+2];
      R[3] = RData[i+3];
      sOutDW(ChP->IndexAddr,*((DWord_t *)&R[0]));
   }

   ChR = ChP->R;
   for(i=0; i < RREGDATASIZE; i+=4)
   {
      ChR[i] = RRegData[i];
      ChR[i+1] = RRegData[i+1] + 0x10 * ChanNum;
      ChR[i+2] = RRegData[i+2];
      ChR[i+3] = RRegData[i+3];
   }

   /* Indexed registers */
   ChOff = (Word_t)ChanNum * 0x1000;

   ChP->BaudDiv[0] = (Byte_t)(ChOff + _BAUD);
   ChP->BaudDiv[1] = (Byte_t)((ChOff + _BAUD) >> 8);
   ChP->BaudDiv[2] = (Byte_t)BRD9600;
   ChP->BaudDiv[3] = (Byte_t)(BRD9600 >> 8);
   sOutDW(ChP->IndexAddr,*(DWord_t *)&ChP->BaudDiv[0]);

   ChP->TxControl[0] = (Byte_t)(ChOff + _TX_CTRL);
   ChP->TxControl[1] = (Byte_t)((ChOff + _TX_CTRL) >> 8);
   ChP->TxControl[2] = 0;
   ChP->TxControl[3] = 0;
   sOutDW(ChP->IndexAddr,*(DWord_t *)&ChP->TxControl[0]);

   ChP->RxControl[0] = (Byte_t)(ChOff + _RX_CTRL);
   ChP->RxControl[1] = (Byte_t)((ChOff + _RX_CTRL) >> 8);
   ChP->RxControl[2] = 0;
   ChP->RxControl[3] = 0;
   sOutDW(ChP->IndexAddr,*(DWord_t *)&ChP->RxControl[0]);

   ChP->TxEnables[0] = (Byte_t)(ChOff + _TX_ENBLS);
   ChP->TxEnables[1] = (Byte_t)((ChOff + _TX_ENBLS) >> 8);
   ChP->TxEnables[2] = 0;
   ChP->TxEnables[3] = 0;
   sOutDW(ChP->IndexAddr,*(DWord_t *)&ChP->TxEnables[0]);

   ChP->TxCompare[0] = (Byte_t)(ChOff + _TXCMP1);
   ChP->TxCompare[1] = (Byte_t)((ChOff + _TXCMP1) >> 8);
   ChP->TxCompare[2] = 0;
   ChP->TxCompare[3] = 0;
   sOutDW(ChP->IndexAddr,*(DWord_t *)&ChP->TxCompare[0]);

   ChP->TxReplace1[0] = (Byte_t)(ChOff + _TXREP1B1);
   ChP->TxReplace1[1] = (Byte_t)((ChOff + _TXREP1B1) >> 8);
   ChP->TxReplace1[2] = 0;
   ChP->TxReplace1[3] = 0;
   sOutDW(ChP->IndexAddr,*(DWord_t *)&ChP->TxReplace1[0]);

   ChP->TxReplace2[0] = (Byte_t)(ChOff + _TXREP2);
   ChP->TxReplace2[1] = (Byte_t)((ChOff + _TXREP2) >> 8);
   ChP->TxReplace2[2] = 0;
   ChP->TxReplace2[3] = 0;
   sOutDW(ChP->IndexAddr,*(DWord_t *)&ChP->TxReplace2[0]);

   ChP->TxFIFOPtrs = ChOff + _TXF_OUTP;
   ChP->TxFIFO = ChOff + _TX_FIFO;

   sOutB(ChP->Cmd,(Byte_t)ChanNum | RESTXFCNT); /* apply reset Tx FIFO count */
   sOutB(ChP->Cmd,(Byte_t)ChanNum);  /* remove reset Tx FIFO count */
   sOutW((WordIO_t)ChP->IndexAddr,ChP->TxFIFOPtrs); /* clear Tx in/out ptrs */
   sOutW(ChP->IndexData,0);
   ChP->RxFIFOPtrs = ChOff + _RXF_OUTP;
   ChP->RxFIFO = ChOff + _RX_FIFO;

   sOutB(ChP->Cmd,(Byte_t)ChanNum | RESRXFCNT); /* apply reset Rx FIFO count */
   sOutB(ChP->Cmd,(Byte_t)ChanNum);  /* remove reset Rx FIFO count */
   sOutW((WordIO_t)ChP->IndexAddr,ChP->RxFIFOPtrs); /* clear Rx out ptr */
   sOutW(ChP->IndexData,0);
   sOutW((WordIO_t)ChP->IndexAddr,ChP->RxFIFOPtrs + 2); /* clear Rx in ptr */
   sOutW(ChP->IndexData,0);
   ChP->TxPrioCnt = ChOff + _TXP_CNT;
   sOutW((WordIO_t)ChP->IndexAddr,ChP->TxPrioCnt);
   sOutB(ChP->IndexData,0);
   ChP->TxPrioPtr = ChOff + _TXP_PNTR;
   sOutW((WordIO_t)ChP->IndexAddr,ChP->TxPrioPtr);
   sOutB(ChP->IndexData,0);
   ChP->TxPrioBuf = ChOff + _TXP_BUF;
   sEnRxProcessor(ChP);                /* start the Rx processor */

   return(TRUE);
}

/***************************************************************************
Function: sStopRxProcessor
Purpose:  Stop the receive processor from processing a channel.
Call:     sStopRxProcessor(ChP)
          CHANNEL_T *ChP; Ptr to channel structure

Comments: The receive processor can be started again with sStartRxProcessor().
          This function causes the receive processor to skip over the
          stopped channel.  It does not stop it from processing other channels.

Warnings: No context switches are allowed while executing this function.

          Do not leave the receive processor stopped for more than one
          character time.

          After calling this function a delay of 4 uS is required to ensure
          that the receive processor is no longer processing this channel.
*/
void sStopRxProcessor(CHANNEL_T *ChP)
{
   Byte_t R[4];

   R[0] = ChP->R[0];
   R[1] = ChP->R[1];
   R[2] = 0x0a;
   R[3] = ChP->R[3];
   sOutDW(ChP->IndexAddr,*(DWord_t *)&R[0]);
}

/***************************************************************************
Function: sFlushRxFIFO
Purpose:  Flush the Rx FIFO
Call:     sFlushRxFIFO(ChP)
          CHANNEL_T *ChP; Ptr to channel structure
Return:   void
Comments: To prevent data from being enqueued or dequeued in the Tx FIFO
          while it is being flushed the receive processor is stopped
          and the transmitter is disabled.  After these operations a
          4 uS delay is done before clearing the pointers to allow
          the receive processor to stop.  These items are handled inside
          this function.
Warnings: No context switches are allowed while executing this function.
*/
void sFlushRxFIFO(CHANNEL_T *ChP)
{
   int i;
   Byte_t Ch;                   /* channel number within AIOP */
   int RxFIFOEnabled;                  /* TRUE if Rx FIFO enabled */

   if(sGetRxCnt(ChP) == 0)             /* Rx FIFO empty */
      return;                          /* don't need to flush */

   RxFIFOEnabled = FALSE;
   if(ChP->R[0x32] == 0x08) /* Rx FIFO is enabled */
   {
      RxFIFOEnabled = TRUE;
      sDisRxFIFO(ChP);                 /* disable it */
      for(i=0; i < 2000/200; i++)	/* delay 2 uS to allow proc to disable FIFO*/
         sInB(ChP->IntChan);		/* depends on bus i/o timing */
   }
   sGetChanStatus(ChP);          /* clear any pending Rx errors in chan stat */
   Ch = (Byte_t)sGetChanNum(ChP);
   sOutB(ChP->Cmd,Ch | RESRXFCNT);     /* apply reset Rx FIFO count */
   sOutB(ChP->Cmd,Ch);                 /* remove reset Rx FIFO count */
   sOutW((WordIO_t)ChP->IndexAddr,ChP->RxFIFOPtrs); /* clear Rx out ptr */
   sOutW(ChP->IndexData,0);
   sOutW((WordIO_t)ChP->IndexAddr,ChP->RxFIFOPtrs + 2); /* clear Rx in ptr */
   sOutW(ChP->IndexData,0);
   if(RxFIFOEnabled)
      sEnRxFIFO(ChP);                  /* enable Rx FIFO */
}

/***************************************************************************
Function: sFlushTxFIFO
Purpose:  Flush the Tx FIFO
Call:     sFlushTxFIFO(ChP)
          CHANNEL_T *ChP; Ptr to channel structure
Return:   void
Comments: To prevent data from being enqueued or dequeued in the Tx FIFO
          while it is being flushed the receive processor is stopped
          and the transmitter is disabled.  After these operations a
          4 uS delay is done before clearing the pointers to allow
          the receive processor to stop.  These items are handled inside
          this function.
Warnings: No context switches are allowed while executing this function.
*/
void sFlushTxFIFO(CHANNEL_T *ChP)
{
   int i;
   Byte_t Ch;                   /* channel number within AIOP */
   int TxEnabled;                      /* TRUE if transmitter enabled */

   if(sGetTxCnt(ChP) == 0)             /* Tx FIFO empty */
      return;                          /* don't need to flush */

   TxEnabled = FALSE;
   if(ChP->TxControl[3] & TX_ENABLE)
   {
      TxEnabled = TRUE;
      sDisTransmit(ChP);               /* disable transmitter */
   }
   sStopRxProcessor(ChP);              /* stop Rx processor */
   for(i = 0; i < 4000/200; i++)         /* delay 4 uS to allow proc to stop */
      sInB(ChP->IntChan);	/* depends on bus i/o timing */
   Ch = (Byte_t)sGetChanNum(ChP);
   sOutB(ChP->Cmd,Ch | RESTXFCNT);     /* apply reset Tx FIFO count */
   sOutB(ChP->Cmd,Ch);                 /* remove reset Tx FIFO count */
   sOutW((WordIO_t)ChP->IndexAddr,ChP->TxFIFOPtrs); /* clear Tx in/out ptrs */
   sOutW(ChP->IndexData,0);
   if(TxEnabled)
      sEnTransmit(ChP);                /* enable transmitter */
   sStartRxProcessor(ChP);             /* restart Rx processor */
}

/***************************************************************************
Function: sWriteTxPrioByte
Purpose:  Write a byte of priority transmit data to a channel
Call:     sWriteTxPrioByte(ChP,Data)
          CHANNEL_T *ChP; Ptr to channel structure
          Byte_t Data; The transmit data byte

Return:   int: 1 if the bytes is successfully written, otherwise 0.

Comments: The priority byte is transmitted before any data in the Tx FIFO.

Warnings: No context switches are allowed while executing this function.
*/
int sWriteTxPrioByte(CHANNEL_T *ChP, Byte_t Data)
{
   Byte_t DWBuf[4];             /* buffer for double word writes */
   Word_t *WordPtr;          /* must be far because Win SS != DS */
   register DWordIO_t IndexAddr;

   if(sGetTxCnt(ChP) > 1)              /* write it to Tx priority buffer */
   {
      IndexAddr = ChP->IndexAddr;
      sOutW((WordIO_t)IndexAddr,ChP->TxPrioCnt); /* get priority buffer status */
      if(sInB((ByteIO_t)ChP->IndexData) & PRI_PEND) /* priority buffer busy */
         return(0);                    /* nothing sent */

      WordPtr = (Word_t *)(&DWBuf[0]);
      *WordPtr = ChP->TxPrioBuf;       /* data byte address */

      DWBuf[2] = Data;                 /* data byte value */
      sOutDW(IndexAddr,*((DWord_t *)(&DWBuf[0]))); /* write it out */

      *WordPtr = ChP->TxPrioCnt;       /* Tx priority count address */

      DWBuf[2] = PRI_PEND + 1;         /* indicate 1 byte pending */
      DWBuf[3] = 0;                    /* priority buffer pointer */
      sOutDW(IndexAddr,*((DWord_t *)(&DWBuf[0]))); /* write it out */
   }
   else                                /* write it to Tx FIFO */
   {
      sWriteTxByte(sGetTxRxDataIO(ChP),Data);
   }
   return(1);                          /* 1 byte sent */
}

/***************************************************************************
Function: sEnInterrupts
Purpose:  Enable one or more interrupts for a channel
Call:     sEnInterrupts(ChP,Flags)
          CHANNEL_T *ChP; Ptr to channel structure
          Word_t Flags: Interrupt enable flags, can be any combination
             of the following flags:
                TXINT_EN:   Interrupt on Tx FIFO empty
                RXINT_EN:   Interrupt on Rx FIFO at trigger level (see
                            sSetRxTrigger())
                SRCINT_EN:  Interrupt on SRC (Special Rx Condition)
                MCINT_EN:   Interrupt on modem input change
                CHANINT_EN: Allow channel interrupt signal to the AIOP's
                            Interrupt Channel Register.
Return:   void
Comments: If an interrupt enable flag is set in Flags, that interrupt will be
          enabled.  If an interrupt enable flag is not set in Flags, that
          interrupt will not be changed.  Interrupts can be disabled with
          function sDisInterrupts().

          This function sets the appropriate bit for the channel in the AIOP's
          Interrupt Mask Register if the CHANINT_EN flag is set.  This allows
          this channel's bit to be set in the AIOP's Interrupt Channel Register.

          Interrupts must also be globally enabled before channel interrupts
          will be passed on to the host.  This is done with function
          sEnGlobalInt().

          In some cases it may be desirable to disable interrupts globally but
          enable channel interrupts.  This would allow the global interrupt
          status register to be used to determine which AIOPs need service.
*/
void sEnInterrupts(CHANNEL_T *ChP,Word_t Flags)
{
   Byte_t Mask;                 /* Interrupt Mask Register */

   ChP->RxControl[2] |=
      ((Byte_t)Flags & (RXINT_EN | SRCINT_EN | MCINT_EN));

   sOutDW(ChP->IndexAddr,*(DWord_t *)&ChP->RxControl[0]);

   ChP->TxControl[2] |= ((Byte_t)Flags & TXINT_EN);

   sOutDW(ChP->IndexAddr,*(DWord_t *)&ChP->TxControl[0]);

   if(Flags & CHANINT_EN)
   {
      Mask = sInB(ChP->IntMask) | sBitMapSetTbl[ChP->ChanNum];
      sOutB(ChP->IntMask,Mask);
   }
}

/***************************************************************************
Function: sDisInterrupts
Purpose:  Disable one or more interrupts for a channel
Call:     sDisInterrupts(ChP,Flags)
          CHANNEL_T *ChP; Ptr to channel structure
          Word_t Flags: Interrupt flags, can be any combination
             of the following flags:
                TXINT_EN:   Interrupt on Tx FIFO empty
                RXINT_EN:   Interrupt on Rx FIFO at trigger level (see
                            sSetRxTrigger())
                SRCINT_EN:  Interrupt on SRC (Special Rx Condition)
                MCINT_EN:   Interrupt on modem input change
                CHANINT_EN: Disable channel interrupt signal to the
                            AIOP's Interrupt Channel Register.
Return:   void
Comments: If an interrupt flag is set in Flags, that interrupt will be
          disabled.  If an interrupt flag is not set in Flags, that
          interrupt will not be changed.  Interrupts can be enabled with
          function sEnInterrupts().

          This function clears the appropriate bit for the channel in the AIOP's
          Interrupt Mask Register if the CHANINT_EN flag is set.  This blocks
          this channel's bit from being set in the AIOP's Interrupt Channel
          Register.
*/
void sDisInterrupts(CHANNEL_T *ChP,Word_t Flags)
{
   Byte_t Mask;                 /* Interrupt Mask Register */

   ChP->RxControl[2] &=
         ~((Byte_t)Flags & (RXINT_EN | SRCINT_EN | MCINT_EN));
   sOutDW(ChP->IndexAddr,*(DWord_t *)&ChP->RxControl[0]);
   ChP->TxControl[2] &= ~((Byte_t)Flags & TXINT_EN);
   sOutDW(ChP->IndexAddr,*(DWord_t *)&ChP->TxControl[0]);

   if(Flags & CHANINT_EN)
   {
      Mask = sInB(ChP->IntMask) & sBitMapClrTbl[ChP->ChanNum];
      sOutB(ChP->IntMask,Mask);
   }
}


/*********************************************************************
  Begin BSD-specific driver code
**********************************************************************/

int rpprobe __P((struct device *, struct cfdata *, void *));
void rpattach __P((struct device *, struct device *, void *));
struct cfdriver rpcd =
    { NULL, "rp", rpprobe, rpattach, DV_TTY, sizeof(struct rp_softc) };

static int rp_controller_port = 0;
static int rp_num_ports_open = 0;

static int rp_num_ports[4];	/* Number of ports on each a controller */

#define _INLINE_ inline
#define POLL_INTERVAL 1

/* #define ROCKET_DEBUG_FLOW */
/* #define ROCKET_DEBUG_INTR */

/*
 * Here begins the interrupt/polling routine for the Rocketport!
 */
static _INLINE_ void rp_do_receive(struct rp_port *rp, struct tty *tp,
				   CHANNEL_t *cp, unsigned int ChanStatus)
{
    unsigned int CharNStat;
    int ToRecv, wRecv, ch;

    ToRecv= sGetRxCnt(cp);
    if (ToRecv == 0)
	return;
	
    /*
     * if status indicates there are errored characters in the
     * FIFO, then enter status mode (a word in FIFO holds
     * character and status).
     */
    if (ChanStatus & (RXFOVERFL | RXBREAK | RXFRAME | RXPARITY)) {
	if (!(ChanStatus & STATMODE)) {
	    ChanStatus |= STATMODE;
	    sEnRxStatusMode(cp);
	}
    }

    /* 
     * if we previously entered status mode, then read down the
     * FIFO one word at a time, pulling apart the character and
     * the status.  Update error counters depending on status
     */
    if (ChanStatus & STATMODE) {
	while (ToRecv) {
	    CharNStat= sInW(sGetTxRxDataIO(cp));	
	    ch = CharNStat & 0xff;

	    if ((CharNStat & STMBREAK) || (CharNStat & STMFRAMEH))
		ch |= TTY_FE;
	    else if (CharNStat & STMPARITYH)
		ch |= TTY_PE;
	    else if (CharNStat & STMRCVROVRH)
		rp->rp_overflows++;

	    (*linesw[tp->t_line].l_rint)(ch, tp);
	    ToRecv--;
	}

	/*
	 * after we've emptied the FIFO in status mode, turn
	 * status mode back off
	 */
	if (sGetRxCnt(cp) == 0)
	    sDisRxStatusMode(cp);
    } else {
	while (ToRecv) {
	    ch = sInB(sGetTxRxDataIO(cp));
	    (*linesw[tp->t_line].l_rint)(ch, tp);
	    ToRecv--;
	}
    }
}

/*
 * This function is called for each port which has signalled an
 * interrupt.  It checks what interrupts are pending and services
 * them. 
 */
static _INLINE_ void rp_handle_port(struct rp_port *rp)
{
    CHANNEL_t *cp;
    struct tty *tp;
    unsigned int IntMask, ChanStatus;
#if (defined(ROCKET_DEBUG_INTR) || defined(ROCKET_DEBUG_FLOW))
    int		old_cts;
#endif
    
    if (!rp)
	return;
	
    cp = &rp->rp_channel;
    tp = rp->rp_tty;

    IntMask = sGetChanIntID(cp);
#ifdef ROCKET_DEBUG_INTR
    printf("rp_handle_port ttyR%d %02x (mask=%02x)...", rp->rp_port,
	   IntMask, rp->rp_intmask);
#endif
    IntMask = IntMask & rp->rp_intmask;
    ChanStatus= sGetChanStatus(cp);
    if (IntMask & RXF_TRIG) {	/* Rx FIFO trigger level */
	rp_do_receive(rp, tp, cp, ChanStatus);
    }
    if (IntMask & SRC_INT) {	/* Special receive condition */
    }
    if (IntMask & DELTA_CD) {	/* CD change  */
#if (defined(ROCKET_DEBUG_OPEN) || defined(ROCKET_DEBUG_INTR))
	printf("CD now %s...", (ChanStatus & CD_ACT) ? "on" : "off");
#endif
	if (ChanStatus & CD_ACT) {
	    if ( !(tp->t_state & TS_CARR_ON)) {
		(void)(*linesw[tp->t_line].l_modem)(tp, 1);
	    }
	} else {
	    if (tp->t_state & TS_CARR_ON) {
		if ((*linesw[tp->t_line].l_modem)(tp, 0)) {
		    sClrDTR(cp);
		    sClrRTS(cp);
		}
	    }
	}
    }
#if (defined(ROCKET_DEBUG_INTR) || defined(ROCKET_DEBUG_FLOW))
    old_cts = rp->rp_cts;
    rp->rp_cts = ((ChanStatus & CTS_ACT) != 0);
    if (old_cts != rp->rp_cts) {
	printf("CTS change (now %s)...\n", rp->rp_cts ? "on" : "off");
    }
#endif
    if (IntMask & DELTA_DSR) {	/* DSR change */
#ifdef ROCKET_DEBUG_INTR
	printf("DSR change...\n");
#endif
    }
}

/*
 * The top level polling routine.
 */
void rp_do_poll()
{
    CONTROLLER_t *ctl;
    struct rp_softc *sc;
    struct rp_port *rp;
    struct tty *tp;
    int unit, aiop, ch, line, count;
    unsigned char CtlMask, AiopMask;

    for (unit=0; unit < rpcd.cd_ndevs; unit++) {
	sc = rpcd.cd_devs[unit];
	if (sc == 0)
	    continue;
	ctl = sc->sc_ctlp;

	CtlMask= sGetControllerIntStatus(ctl);
	for (aiop=0; CtlMask; CtlMask >>= 1, aiop++) {
	    if (CtlMask & 1) {
		AiopMask= sGetAiopIntStatus(ctl, aiop);
		for (ch=0; AiopMask; AiopMask >>= 1, ch++) {
		    if (AiopMask & 1) {
			line = aiop*MAX_PORTS_PER_AIOP + ch;
			rp = sc->sc_ports + sc->sc_map[line];
			rp_handle_port(rp);
		    }
		}
	    }
	}
	for (line = 0, rp = sc->sc_ports; line < sc->sc_nports; line++, rp++) {
	    tp = rp->rp_tty;
	    if (tp->t_state & (TS_BUSY)) {
		count = sGetTxCnt(&rp->rp_channel);
		if (count == 0)
		    tp->t_state &= ~(TS_BUSY);
		if (!(tp->t_state & TS_TTSTOP) &&
		    (count <= rp->rp_restart)) {
#ifdef ROCKET_DEBUG_FLOW
		    printf("Restarting rocket ldesc, count = %d\n", count);
#endif
		    (*linesw[tp->t_line].l_start)(tp);
	    	}
	    }
	}
    }
    /*
     * Reset the timer so we get called at the next clock tick.
     */
    if (rp_num_ports_open) {
	timeout(rp_do_poll, (void *)NULL, POLL_INTERVAL);
    }
}

/*
 * Here ends the interrupt/polling routine.
 */


/*
 * This function initializes the rp_port structure, as well as enabling
 * the port on the RocketPort board.
 */
void init_r_port(struct rp_softc *sc, int unit, int aiop, int chan, int port)
{
	struct rp_port 		*rp;
	CONTROLLER_T 		*ctlp;
	CHANNEL_t		*cp;
	int 			line, ChanStatus;
	
	ctlp= sCtlNumToCtlPtr(unit);

	rp = sc->sc_ports + port;
	sc->sc_map[aiop*MAX_PORTS_PER_AIOP + chan] = port;

	rp->rp_sc = sc;
	rp->rp_port = port;
	rp->rp_ctlp = ctlp;
	rp->rp_unit = unit;
	rp->rp_aiop = aiop;
	rp->rp_chan = chan;
	rp->rp_tty = sc->sc_ttydev.tty_ttys + port;

	rp->rp_intmask = RXF_TRIG | TXFIFO_MT | SRC_INT | DELTA_CD |
		DELTA_CTS | DELTA_DSR;
	if (sInitChan(ctlp, &rp->rp_channel, aiop, chan) == 0) {
		printf("Rockerport sInitChan(%d, %d, %d) failed!\n",
		       unit, aiop, chan);
		return;
	}
	ChanStatus= sGetChanStatus(&rp->rp_channel);
	rp->rp_cts = (ChanStatus & CTS_ACT) != 0;
}

static void
rp_write_enable(rp)
    struct rp_port *rp;
{
    rp->rp_disable_writes = 0;
    if (rp->rp_waiting) {
	rp->rp_waiting = 0;
	wakeup((caddr_t)rp);
    }
}

/*
 * The top-level routines begin here
 */

int	rpopen __P((dev_t, int, int, struct proc *));
int	rpclose __P((dev_t, int, int, struct proc *));
int	rpwrite __P((dev_t, struct uio *, int));
int	rpread __P((dev_t, struct uio *, int));
int	rpioctl __P((dev_t, int, caddr_t,  int, struct proc *));
int	rpselect __P((dev_t, int, struct proc *));
void	rpstop __P((struct tty *, int));
int	rpparam __P((struct tty *, struct termios *));
void	rpstart __P((struct tty *));

struct devsw rpsw = {
    &rpcd,
    rpopen, rpclose, rpread, rpwrite, rpioctl, rpselect, nommap,
    nostrat, nodump, nopsize, 0, rpstop
};

rpprobe(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
    register struct isa_attach_args *ia = (struct isa_attach_args *) aux;
    int controller;
    int i, aiop, num_aiops;
    unsigned int aiopio[MAX_AIOPS_PER_BOARD];
    CONTROLLER_t *ctlp;

    if (cf->cf_unit >= 4) {
	printf("rpprobe: unit number %d invalid.\n", cf->cf_unit);
	return 1;
    }
    if (rp_controller_port)
	controller = rp_controller_port;
    else {
	controller = ia->ia_iobase + 0x40;
	for (i=0; i < 4; i++)
	    rp_num_ports[i] = 0;
    }
    
    for (aiop=0; aiop<MAX_AIOPS_PER_BOARD; aiop++)
	aiopio[aiop]= ia->ia_iobase + (aiop * 0x400);

    ctlp= sCtlNumToCtlPtr(cf->cf_unit);
    num_aiops = sInitController(ctlp, cf->cf_unit, 
				controller + (cf->cf_unit*0x400),
				aiopio, MAX_AIOPS_PER_BOARD, 0,
				FREQ_DIS, 0);
    if (num_aiops <= 0)
	return 0;

    if (rp_controller_port) {
	ia->ia_iosize = 64;
    } else {
	ia->ia_iosize = 68;
	rp_controller_port = controller;
    }
    ia->ia_irq = IRQNONE;
    
    return 1;
}

void
rpattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
    register struct isa_attach_args *ia = (struct isa_attach_args *) aux;
    struct rp_softc *sc = (struct rp_softc *) self;
    int unit;
    int num_aiops, num_chan, num_ports;
    int aiop, chan, port;
    unsigned int aiopio[MAX_AIOPS_PER_BOARD];
    CONTROLLER_t *ctlp;

    /* record device information */
    sc->sc_base = ia->ia_iobase;

    unit = sc->sc_dev.dv_unit;

    for (aiop=0; aiop<MAX_AIOPS_PER_BOARD; aiop++)
	aiopio[aiop]= ia->ia_iobase + (aiop * 0x400);

    sc->sc_ctlp = ctlp = sCtlNumToCtlPtr(unit);
    num_aiops = sInitController(ctlp, unit, 
				rp_controller_port + (unit*0x400),
				aiopio, MAX_AIOPS_PER_BOARD, 0,
				FREQ_DIS, 0);

    sc->sc_nports = 0;
    for (aiop = 0; aiop < num_aiops; aiop++) {
	sResetAiopByNum(ctlp, aiop);
	sEnAiop(ctlp, aiop);
	sc->sc_nports += sGetAiopNumChan(ctlp,aiop);
    }

    printf(" %d ports\n", sc->sc_nports);
    rp_num_ports[unit] = sc->sc_nports;

    sc->sc_ports = (struct rp_port *) 
	malloc(sizeof(struct rp_port) * sc->sc_nports, M_DEVBUF, M_NOWAIT);
    if (sc->sc_ports == 0) {
	printf("rp_attach: Couldn't malloc rp_ports structures\n");
	return;
    }
    bzero(sc->sc_ports, sizeof(struct rp_port) * sc->sc_nports);

    sc->sc_ttydev.tty_ttys = (struct tty *)
	malloc(sizeof(struct tty) * sc->sc_nports, M_DEVBUF, M_NOWAIT);
    if (sc->sc_ttydev.tty_ttys == 0) {
	printf("rp_attach: Couldn't malloc ttydev structure\n");
	free(sc->sc_ports, M_DEVBUF);
	return;
    }
    bzero(sc->sc_ttydev.tty_ttys, sizeof(struct tty) * sc->sc_nports);

    strcpy(sc->sc_ttydev.tty_name, rpcd.cd_name);
    sc->sc_ttydev.tty_unit = sc->sc_dev.dv_unit;
    sc->sc_ttydev.tty_base = 0;
    sc->sc_ttydev.tty_count = sc->sc_nports;
    tty_attach(&sc->sc_ttydev);

    /* Initialize the map array */
    for (port = 0; port < MAX_PORTS_PER_BOARD; port++)
	sc->sc_map[port] = -1;

    /* Now initialize each of the ports */
    port = 0;
    for (aiop = 0; aiop < num_aiops; aiop++) {
	num_chan = sGetAiopNumChan(ctlp,aiop);
	for (chan=0; chan < num_chan; chan++, port++)
	    init_r_port(sc, unit, aiop, chan, port);
    }
    
    return;
}

int
rpopen(dev, flag, mode, p)
	dev_t dev;
	int flag, mode;
	struct proc *p;
{
    int unit = RP_UNIT(dev);
    struct rp_softc *sc;
    struct rp_port *rp;
    struct tty *tp;
    int port;
    int oldspl;
    int error;

    if (unit >= rpcd.cd_ndevs)
	return ENXIO;

    sc = rpcd.cd_devs[unit];
    if (!sc)
	return ENXIO;

    port = RP_PORT(dev);
    if (port >= sc->sc_nports)
	return ENXIO;
    
    rp = sc->sc_ports + port;
    tp = rp->rp_tty;

    oldspl = spltty();
	
    if (tp->t_state & (TS_ISOPEN|TS_WOPEN)) {
	if (tp->t_state & TS_XCLUDE && p->p_ucred->cr_uid != 0) {
	    splx(oldspl);
	    return(EBUSY);
	}
    } else {
	/* This is the first open of the tty.  Initialize stuff */
	tp->t_dev = dev;
	tp->t_driver = rp;
	tp->t_oproc = rpstart;
	tp->t_param = rpparam;
	tp->t_line = 0;
	tp->t_state |= TS_WOPEN;
	if (tp->t_ispeed == 0)
	    tp->t_termios = deftermios;
	else
	    ttychars(tp);

	sSetRxTrigger(&rp->rp_channel, TRIG_1);
	sGetChanStatus(&rp->rp_channel);
	sDisRxStatusMode(&rp->rp_channel);
	sFlushRxFIFO(&rp->rp_channel);	
	sFlushTxFIFO(&rp->rp_channel);	

	sEnInterrupts(&rp->rp_channel, 
		      (TXINT_EN|MCINT_EN|RXINT_EN|SRCINT_EN|CHANINT_EN));
	sSetRxTrigger(&rp->rp_channel, TRIG_1);

	sGetChanStatus(&rp->rp_channel);
	sDisRxStatusMode(&rp->rp_channel);
	sClrTxXOFF(&rp->rp_channel);

	sDisCTSFlowCtl(&rp->rp_channel);
	sDisTxSoftFlowCtl(&rp->rp_channel);

	sEnRxFIFO(&rp->rp_channel);
	sEnTransmit(&rp->rp_channel);

	sSetDTR(&rp->rp_channel);
	sSetRTS(&rp->rp_channel);

	(void) rpparam(tp, &tp->t_termios);
	ttsetwater(tp);

	rp_num_ports_open++;
	if (rp_num_ports_open == 1) {
		timeout(rp_do_poll, (void *)NULL, POLL_INTERVAL);
	}
    }

    if (IS_CALLOUT(dev) || (sGetChanStatusLo(&rp->rp_channel) & CD_ACT))
      tp->t_state |= TS_CARR_ON;

    while ((flag&O_NONBLOCK) == 0 && (tp->t_cflag&CLOCAL) == 0 &&
	   (tp->t_state & TS_CARR_ON) == 0) {
      tp->t_state |= TS_WOPEN;
      if (error = ttysleep(tp, (caddr_t)&tp->t_rawq, TTIPRI | PCATCH,
			   ttopen, 0))
	break;
    }
    splx(oldspl);

    error = (*linesw[tp->t_line].l_open)(dev, tp);

    return error;
}

int
rpclose(dev, flag, mode, p)
	dev_t dev;
	int flag, mode;
	struct proc *p;
{
    int oldspl;
    struct rp_port *rp;
    struct tty *tp;

    oldspl = spltty();

    rp = DEV_TO_RP(dev);
    tp = rp->rp_tty;

    /* XXX wait for output buffer to drain */

    (*linesw[tp->t_line].l_close)(tp, flag);

    if (tp->t_cflag&HUPCL || tp->t_state&TS_WOPEN ||
	(tp->t_state&TS_ISOPEN) == 0) {
	sClrDTR(&rp->rp_channel);
    }
    ttyclose(tp);

    sDisTransmit(&rp->rp_channel);
    sDisInterrupts(&rp->rp_channel, 
		   (TXINT_EN|MCINT_EN|RXINT_EN|SRCINT_EN|CHANINT_EN));
    sDisCTSFlowCtl(&rp->rp_channel);
    sDisTxSoftFlowCtl(&rp->rp_channel);
    sClrTxXOFF(&rp->rp_channel);
    sFlushRxFIFO(&rp->rp_channel);	
    sFlushTxFIFO(&rp->rp_channel);
    sClrRTS(&rp->rp_channel);

    splx(oldspl);

    return 0;
}

int
rpread(dev, uio, flag)
    dev_t dev;
    struct uio *uio;
{
    struct rp_port *rp;
    struct tty *tp;
    
    rp = DEV_TO_RP(dev);
    tp = rp->rp_tty;

    return ((*linesw[tp->t_line].l_read)(tp, uio, flag));
}

int 
rpwrite(dev, uio, flag)
	dev_t dev;
	struct uio *uio;
	int flag;
{
    struct rp_port *rp;
    struct tty *tp;
    int error = 0;
	
    rp = DEV_TO_RP(dev);
    tp = rp->rp_tty;

    /*
     * If writes are currently blocked, wait on the "real" tty
     */
    while (rp->rp_disable_writes) {
	rp->rp_waiting = 1;
	if (error = ttysleep(tp, (caddr_t)rp, TTOPRI|PCATCH,
			     "rp_write", 0))
	    return(error);
    }

    error = (*linesw[tp->t_line].l_write)(tp, uio, flag);
    return error;
}

rpselect(dev, flag, p)
	dev_t dev;
	int flag;
	struct proc *p;
{
	struct rp_softc *sc = DEV_TO_SC(dev);
	struct rp_port *rp;

	if (RP_PORT(dev) >= sc->sc_nports)
		return(1);

	rp = DEV_TO_RP(dev);

	return (ttyselect(rp->rp_tty, flag, p));
}

/*
 * Stop or flush output on a line; XXX the BSDI abstraction interface
 * really sucks here, and we need to be careful lest it be changed.
 * The rule is that the stop procedure only means stop if the TS_TTSTOP
 * bit is set in t_state.  If the TS_TTSTOP bit is *clear*, then the stop
 * procedure really means to flush the transmition FIFO.
 */
void
rpstop(tp, flag)
	register struct tty *tp;
	int flag;
{
    register int s;
    struct rp_port *rp;

    s = spltty();
    if (tp->t_state & TS_BUSY) {
	rp = tp->t_driver;
	if ((tp->t_state&TS_TTSTOP) == 0) {
	    sFlushTxFIFO(&rp->rp_channel);
	} else {
	    if (rp->rp_xmit_stopped == 0) {
		sDisTransmit(&rp->rp_channel);
		rp->rp_xmit_stopped = 1;
	    }
	}
    }
    splx(s);
}

int
rpioctl(dev, cmd, data, flag, p)
	dev_t dev;
	int cmd;
	caddr_t data;
	int flag;
	struct proc *p;
{
    struct rp_port *rp;
    struct tty *tp;
    int error = 0;
    int arg, flags, result, ChanStatus;
	
    rp = DEV_TO_RP(dev);
    tp = rp->rp_tty;

    error = (*linesw[tp->t_line].l_ioctl)(tp, cmd, data, flag, p);
    if (error >= 0)
      return (error);
    error = ttioctl(tp, cmd, data, flag, p);
    if (error >= 0)
      return (error);

    switch (cmd) {
    case TIOCSBRK:
	sSendBreak(&rp->rp_channel);
	break;

    case TIOCCBRK:
	sClrBreak(&rp->rp_channel);
	break;

    case TIOCSDTR:
	sSetDTR(&rp->rp_channel);
	sSetRTS(&rp->rp_channel);
	break;

    case TIOCCDTR:
	sClrDTR(&rp->rp_channel);
	sClrRTS(&rp->rp_channel);
	break;

    case TIOCMBIS:
    case TIOCMBIC:
    case TIOCMSET:
	arg = *(int *) data;
	flags = 0;
	if (arg & TIOCM_RTS)
	    flags |= SET_RTS;
	if (arg & TIOCM_DTR)
	    flags |= SET_DTR;
	switch (cmd) {
	case TIOCMBIS: 
	    rp->rp_channel.TxControl[3] |= flags;
	    break;
	case TIOCMBIC:
	    rp->rp_channel.TxControl[3] &= ~flags;
	    break;
	case TIOCMSET:
	    rp->rp_channel.TxControl[3] =
		((rp->rp_channel.TxControl[3] & ~(SET_RTS | SET_DTR)) | flags);
	    break;
	}
	sOutDW(rp->rp_channel.IndexAddr,
	       *(DWord_t *) &(rp->rp_channel.TxControl[0]));

	break;

    case TIOCMGET:
	ChanStatus = sGetChanStatusLo(&rp->rp_channel);
	flags = rp->rp_channel.TxControl[3];
	result =  (((flags & SET_RTS) ? TIOCM_RTS : 0)
		   | ((flags & SET_DTR) ? TIOCM_DTR : 0)
		   | ((ChanStatus  & CD_ACT) ? TIOCM_CAR : 0)
		   /* TIOCM_RNG not supported */
		   | ((ChanStatus  & DSR_ACT) ? TIOCM_DSR : 0)
		   | ((ChanStatus  & CTS_ACT) ? TIOCM_CTS : 0));
	
	*(int *)data = result;
	break;
    default:
	return ENOTTY;
    }
    return 0;
}

static struct speedtab baud_table[] = {
	B0,	0,		B50,	BRD50,		B75,	BRD75,
	B110,	BRD110,		B134,	BRD134,		B150,	BRD150,
	B200,	BRD200,		B300,	BRD300,		B600,	BRD600,
	B1200,	BRD1200,	B1800,	BRD1800,	B2400,	BRD2400,
	B4800,	BRD4800,	B9600,	BRD9600,	B19200,	BRD19200,
	B38400, BRD38400,	B7200,	BRD7200,	B14400, BRD14400,
 	/* B28800, BRD28800,*/	B57600, BRD57600,	B76800, BRD76800,
	B115200, BRD115200,	B230400, BRD230400,	
	-1,	-1
};

/*
 * This routine configures a rocketport port according to its
 * termio settings.
 */
int
rpparam(tp, t)
	struct tty *tp;
	struct termios *t;
{
	struct rp_port *rp = (struct rp_port *)tp->t_driver;
	CHANNEL_t	*cp;
	int oldspl, cflag, iflag, oflag, lflag;
	int error, ospeed;

	cp = &rp->rp_channel;

	cflag = t->c_cflag;
	iflag = t->c_iflag;
	oflag = t->c_oflag;
	lflag = t->c_lflag;

#if 0
	/* Wait for quiescent TX before setting parameters */
	error = rp_drainwait(pp, RPMSG_PARAM, SS_BLOCKWRITE);
	if (error > 0)
		goto out;
#endif

	ospeed = ttspeedtab(t->c_ispeed, baud_table);
	if (ospeed < 0 || t->c_ispeed != t->c_ospeed)
		return (EINVAL);
	
        /* copy parameters to tty */
        tp->t_ispeed = t->c_ispeed;
        tp->t_ospeed = t->c_ospeed;
        tp->t_cflag = cflag;
        tp->t_iflag = iflag;
        tp->t_oflag = oflag;
        tp->t_lflag = lflag;

	/* Hangup if ospeed == 0 */
	if (t->c_ospeed == 0) {
		sClrDTR(cp);
		goto out;
	}
	rp->rp_fifo_lw = ((t->c_ospeed*2) / 1000) + 1;

	oldspl = spltty();

	/* Set baud rate --- we only pay attention to ispeed */
	sSetDTR(cp);
	sSetRTS(cp);
	sSetBaud(cp, ospeed);

	if (cflag & CSTOPB) {
	    sSetStop2(cp);
	} else {
	    sSetStop1(cp);
	}

	if (cflag & PARENB) {
		sEnParity(cp);
		if (cflag & PARODD) {
			sSetOddParity(cp);
		} else {
			sSetEvenParity(cp);
		}
	} else {
		sDisParity(cp);
	}

	if ((cflag & CSIZE) == CS8) {
	    sSetData8(cp);
	    rp->rp_imask = 0xFF;
	} else {
	    sSetData7(cp);
	    rp->rp_imask = 0x7F;
	}

	if (iflag & ISTRIP)
	    rp->rp_imask &= 0x7F;
	
	if (cflag & CLOCAL)
	    rp->rp_intmask &= ~DELTA_CD;
	else
	    rp->rp_intmask |= DELTA_CD;

	/*
	 * Output flow control.  Data is only sent when CTS is high
	 */
	if (cflag & CCTS_OFLOW) {
	    sEnCTSFlowCtl(cp);
	} else {
	    sDisCTSFlowCtl(cp);
	}
	/*
	 * Input flow control.  RTS is raised when we can receive data
	 */
	if (cflag & CRTS_IFLOW) {
	    rp->rp_rts_iflow = 1;
	} else {
	    rp->rp_rts_iflow = 0;
	}

	if ((cflag & CLOCAL) || (sGetChanStatusLo(cp) & CD_ACT)) {
		tp->t_state |= TS_CARR_ON;
		wakeup((caddr_t)&tp->t_rawq);
	}

	splx(oldspl);
out:
	return(error);
}

void
rpstart(tp)
	struct tty *tp;
{
    struct rp_port *rp;
    CHANNEL_t *cp;    
    struct clist *qp;
    int spl, xmit_fifo_room;
    char ch;
    int count;

    rp = tp->t_driver;
    cp = &rp->rp_channel;

    spl = spltty();

    if (tp->t_state & TS_XON_PEND) {
	tp->t_state |= TS_BUSY;
#ifdef ROCKET_DEBUG_FLOW
	printf("Writing XON char: %d\n", tp->t_cc[VSTART]);
#endif
	sWriteTxPrioByte(cp, tp->t_cc[VSTART]);
	if (rp->rp_xmit_stopped) {
	    sEnTransmit(cp);
	    rp->rp_xmit_stopped = 0;
	}
	tp->t_state &= ~TS_XON_PEND;
	goto out;
    }
    if (tp->t_state & TS_XOFF_PEND) {
	tp->t_state |= TS_BUSY;
#ifdef ROCKET_DEBUG_FLOW
	printf("Writing XOFF char: %d\n", tp->t_cc[VSTOP]);
#endif
	sWriteTxPrioByte(cp, tp->t_cc[VSTOP]);
	if (rp->rp_xmit_stopped) {
	    sEnTransmit(cp);
	    rp->rp_xmit_stopped = 0;
	}
	tp->t_state &= ~TS_XOFF_PEND;
	goto out;
    }
    if (tp->t_state & (TS_TIMEOUT|TS_TTSTOP))
	goto out;
    if (rp->rp_xmit_stopped) {
	sEnTransmit(cp);
	rp->rp_xmit_stopped = 0;
    }
    count = sGetTxCnt(cp);
#ifdef BSDI_OLD_20
    if (tp->t_outq.c_cc <= tp->t_lowat) {
	if (tp->t_state & TS_ASLEEP) {
	    tp->t_state &= ~TS_ASLEEP;
	    wakeup((caddr_t)&tp->t_outq);
	}
	selwakeup(&tp->t_wsel);
    }
    if (tp->t_outq.c_cc == 0) {
	if ((tp->t_state & TS_BUSY) && (count == 0)) {
	    tp->t_state &= ~TS_BUSY;
	}
	goto out;
    }
#else
    if ((tp->t_state & TS_BUSY) && count == 0)
	    tp->t_state &= ~TS_BUSY;
    if (tp->t_outq.c_cc <= tp->t_lowat)
	    ttyowake(tp);
    if (tp->t_outq.c_cc == 0)
	    goto out;
#endif
    xmit_fifo_room = TXFIFO_SIZE - sGetTxCnt(cp);
    qp = &tp->t_outq;
    count = 0;
    if (xmit_fifo_room > 0 && qp->c_cc > 0) {
	tp->t_state |= TS_BUSY;
    }
    while (xmit_fifo_room > 0 && qp->c_cc > 0) {
	ch = getc(qp);
	sOutB(sGetTxRxDataIO(cp), ch);
	xmit_fifo_room--;
	count++;
    }
    rp->rp_restart = (qp->c_cc > 0) ? rp->rp_fifo_lw : 0;
out:
    splx(spl);
}
