/*-
 * Copyright (c) 1994, 1995 Matt Thomas (matt@lkg.dec.com)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software withough specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * BSDI $Id: lance.c,v 2.1 1995/09/16 22:21:53 ewv Exp $
 *
 * Id: lance.c,v 1.4 1995/05/24 22:04:13 thomas Exp
 *
 * Log: lance.c,v
 * Revision 1.4  1995/05/24  22:04:13  thomas
 * Use BITS not MASK
 *
 * Revision 1.3  1995/04/24  23:43:05  thomas
 * Fix how skipamount is used. Make sure txdescs is a power
 * of 2 (since the DEPCA can use only 48KB, the division will
 * not result in a power of 2).
 *
 * Revision 1.2  1995/04/24  19:03:43  thomas
 * Fix EISA DE422 support.  Correct Copyright message.
 * Remove unused fields.
 *
 * Revision 1.1  1995/04/23  22:07:13  thomas
 * Initial revision
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/mbuf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/malloc.h>

#if defined(__FreeBSD__)
#include <machine/cpufunc.h>
#include <machine/clock.h>
#elif defined(__bsdi__)
#include <machine/inline.h>
#endif

#include <net/if.h>
#include <net/if_types.h>
#include <net/if_dl.h>
#include <net/route.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>
#endif

#include <vm/vm.h>

#include "bpfilter.h"
#if NBPFILTER > 0
#include <net/bpf.h>
#endif

#include <i386/isa/ic/am7990.h>
#include <i386/isa/lance.h>

/*
 * Start of generic LANCE (AMD7990) support.  This code assumes a memory
 * mapped adapter.
 *
 *
 * Here's as good a place to describe our paritioning of the 
 * LANCE shared RAM space.  (NOTE: this driver does not yet support
 * the concept of a LANCE being able to DMA).
 *
 * First is the 24 (00:23) bytes for LANCE Initialization Block
 * Next are the recieve descriptors.  The number is calculated from
 * how many LN_BUFSIZE buffers we can allocate (this number must
 * be a power of 2).  Next are the transmit descriptors.  The amount
 * of transmit descriptors is derived from the size of the RAM
 * divided by 1K.  Now come the receive buffers (one for each receive
 * descriptor).  Finally is the transmit heap.  (no fixed buffers are
 * allocated so as to make the most use of the limited space).
 */
extern int
lance_init_adapmem(
    lance_softc_t *sc,
    const unsigned skipamount)
{
    lance_addr_t rxbufoffset;
    lance_addr_t rxdescoffset, txdescoffset;
    unsigned rxdescs, txdescs;

    LN_STAT(low_txfree = sc->lance_txinfo.ri_max);
    LN_STAT(low_txheapsize = 0xFFFFFFFF);

    /*
     * First calculate how many descriptors we heap.
     * Note this assumes the ramsize is a power of two.
     */
    sc->lance_rxbufsize = LN_BUFSIZE;
    rxdescs = 1;
    while (rxdescs * sc->lance_rxbufsize < sc->lance_ramsize)
	rxdescs *= 2;
    rxdescs /= 2;
    if (rxdescs > LN_DESC_MAX) {
	sc->lance_rxbufsize *= rxdescs / LN_DESC_MAX;
	rxdescs = LN_DESC_MAX;
    }
    txdescs = sc->lance_ramsize / LN_TXDESC_RATIO;
    if (txdescs > LN_DESC_MAX)
	txdescs = LN_DESC_MAX;
    if (txdescs & (txdescs-1)) {
	/* make sure txdesc is a power of 2 */
	int foo = txdescs;
	for (txdescs = 1; foo != 1; txdescs <<= 1, foo >>=1)
	    ;
    }

    /*
     * Now calculate where everything goes in memory
     */
    rxdescoffset = sizeof(ln_initb_t) + skipamount;
    txdescoffset = rxdescoffset + sizeof(ln_desc_t) * rxdescs;
    rxbufoffset  = txdescoffset + sizeof(ln_desc_t) * txdescs;

    /*
     * Remember these for debugging.
     */
    sc->lance_raminitb = (ln_initb_t *) (sc->lance_membase + skipamount);
    sc->lance_ramdesc = (ln_desc_t *) (sc->lance_raminitb + rxdescoffset);

    /*
     * Initialize the rings.
     */
    if (!lance_init_ring(sc, &sc->lance_initb.ln_rxring, &sc->lance_rxinfo,
		   rxdescs, rxbufoffset, rxdescoffset))
	return 0;
    sc->lance_rxinfo.ri_heap = rxbufoffset;
    sc->lance_rxinfo.ri_heapend = rxbufoffset + sc->lance_rxbufsize * rxdescs;

    if (!lance_init_ring(sc, &sc->lance_initb.ln_txring, &sc->lance_txinfo,
		   txdescs, 0, txdescoffset))
	return 0;
    sc->lance_txinfo.ri_heap = sc->lance_rxinfo.ri_heapend;
    sc->lance_txinfo.ri_heapend = sc->lance_ramsize;

    /*
     * Set CSR1 and CSR2 to the address of the init block (which
     * for us is always 0.
     */
    sc->lance_csr1 = LN_ADDR_LO(0 + sc->lance_ramoffset + skipamount);
    sc->lance_csr2 = LN_ADDR_HI(0 + sc->lance_ramoffset + skipamount);
    return 1;
}

extern int
lance_init_ring(
    lance_softc_t *sc,
    ln_ring_t *rp,
    lance_ring_t *ri,
    unsigned ndescs,
    lance_addr_t bufoffset,
    lance_addr_t descoffset)
{
    lance_descinfo_t *di;

    /*
     * Initialize the ring pointer in the LANCE InitBlock
     */
    rp->r_addr_lo = LN_ADDR_LO(descoffset + sc->lance_ramoffset);
    rp->r_addr_hi = LN_ADDR_HI(descoffset + sc->lance_ramoffset);
    rp->r_log2_size = ffs(ndescs) - 1;

    /*
     * Allocate the ring entry descriptors and initialize
     * our ring information data structure.  All these are
     * our copies and do not live in the LANCE RAM.
     */
    ri->ri_first = (lance_descinfo_t *) malloc(ndescs * sizeof(*di), M_DEVBUF, M_NOWAIT);
    if (ri->ri_first == NULL) {
	printf("lance_init_ring: malloc(%d) failed\n", ndescs * sizeof(*di));
	return 0;
    }
    ri->ri_free = ri->ri_max = ndescs;
    ri->ri_last = ri->ri_first + ri->ri_max;
    for (di = ri->ri_first; di < ri->ri_last; di++) {
	di->di_addr = sc->lance_membase + descoffset;
	di->di_mbuf = NULL;
	if (bufoffset) {
	    di->di_bufaddr = bufoffset;
	    di->di_buflen = sc->lance_rxbufsize;
	    bufoffset += sc->lance_rxbufsize;
	}
	descoffset += sizeof(ln_desc_t);
    }
    return 1;
}

static void
lance_dumpcsrs(
    lance_softc_t *sc,
    const char *id)
{
    printf("%s%d: %s: ", sc->lance_if.if_name, sc->lance_if.if_unit, id);
    LN_SELCSR(sc, LN_CSR0); printf(" csr0=%04x", LN_RDCSR(sc));
    LN_SELCSR(sc, LN_CSR1); printf(" csr1=%04x", LN_RDCSR(sc));
    LN_SELCSR(sc, LN_CSR2); printf(" csr2=%04x", LN_RDCSR(sc));
    LN_SELCSR(sc, LN_CSR3); printf(" csr3=%04x\n", LN_RDCSR(sc));
    LN_SELCSR(sc, LN_CSR0);
}

void
lance_hwreset(
    lance_softc_t *sc)
{
    LN_SELCSR(sc, LN_CSR0);
    LN_WRCSR(sc, LN_CSR0_STOP);
}

void
lance_ifreset(
    lance_softc_t *sc)
{
    register int cnt, csr;

    /* lance_dumpcsrs(sc, "lance_ifreset: start"); */

    LN_WRCSR(sc, LN_RDCSR(sc) & ~LN_CSR0_ENABINTR);
    LN_WRCSR(sc, LN_CSR0_STOP);
    DELAY(100);

    sc->lance_if.if_flags &= ~IFF_UP;
    sc->lance_if.if_flags &= ~(IFF_UP|IFF_RUNNING);

    /* initialize the multicast table */
    (*sc->lance_multicast_filter)(&sc->lance_ac, LN_MC_BITS,
			      sc->lance_initb.ln_multi_mask);
    if (sc->lance_if.if_flags & IFF_ALLMULTI) {
	sc->lance_initb.ln_multi_mask[0] = 0xFFFFU;
	sc->lance_initb.ln_multi_mask[1] = 0xFFFFU;
	sc->lance_initb.ln_multi_mask[2] = 0xFFFFU;
	sc->lance_initb.ln_multi_mask[3] = 0xFFFFU;
    }
    sc->lance_initb.ln_physaddr[0] = ((u_short *) sc->lance_ac.ac_enaddr)[0];
    sc->lance_initb.ln_physaddr[1] = ((u_short *) sc->lance_ac.ac_enaddr)[1];
    sc->lance_initb.ln_physaddr[2] = ((u_short *) sc->lance_ac.ac_enaddr)[2];
    if (sc->lance_if.if_flags & IFF_PROMISC) {
	sc->lance_initb.ln_mode |= LN_MODE_PROMISC;
    } else {
	sc->lance_initb.ln_mode &= ~LN_MODE_PROMISC;
    }
    /*
     * We force the init block to be at the start
     * of the LANCE's RAM buffer.
     */
    LN_COPYTO(sc, &sc->lance_initb, sc->lance_raminitb, sizeof(sc->lance_initb));
    LN_SELCSR(sc, LN_CSR1); LN_WRCSR(sc, sc->lance_csr1);
    LN_SELCSR(sc, LN_CSR2); LN_WRCSR(sc, sc->lance_csr2);
    LN_SELCSR(sc, LN_CSR3); LN_WRCSR(sc, sc->lance_csr3);

    /* lance_dumpcsrs(sc, "lance_ifreset: preinit"); */

    /*
     * clear INITDONE and INIT the chip
     */
    LN_SELCSR(sc, LN_CSR0);
    LN_WRCSR(sc, LN_CSR0_INIT|LN_CSR0_INITDONE);

    csr = 0;
    cnt = 100;
    while (cnt-- > 0) {
        if (((csr = LN_RDCSR(sc)) & LN_CSR0_INITDONE) != 0)
            break;
        DELAY(10000);
    }

    if ((csr & LN_CSR0_INITDONE) == 0) {    /* make sure we got out okay */
	lance_dumpcsrs(sc, "lance_ifreset: reset failure");
    } else {
	/* lance_dumpcsrs(sc, "lance_ifreset: end"); */
	sc->lance_if.if_flags |= IFF_UP;
    }
}

extern void
lance_ifinit(
    lance_softc_t *sc)
{
    lance_ring_t *ri;
    lance_descinfo_t *di;
    ln_desc_t desc;

    LN_STAT(inits++);
    if (sc->lance_if.if_flags & IFF_RUNNING) {
	lance_ifreset(sc);
	lance_tx_intr(sc);
	/*
	 * If we were running, requeue any pending transmits.
	 */
	ri = &sc->lance_txinfo;
	di = ri->ri_nextout;
	while (ri->ri_free < ri->ri_max) {
	    if (--di == ri->ri_first)
		di = ri->ri_nextout - 1;
	    if (di->di_mbuf == NULL)
		break;
	    IF_PREPEND(&sc->lance_if.if_snd, di->di_mbuf);
	    di->di_mbuf = NULL;
	    ri->ri_free++;
	}
    } else {
	lance_ifreset(sc);
    }

    /*
     * Reset the transmit ring.  Make sure we own all the buffers.
     * Also reset the transmit heap.
     */
    sc->lance_if.if_flags &= ~IFF_OACTIVE;
    ri = &sc->lance_txinfo;
    for (di = ri->ri_first; di < ri->ri_last; di++) {
	if (di->di_mbuf != NULL) {
	    m_freem(di->di_mbuf);
	    di->di_mbuf = NULL;
	}
	desc.d_flag = 0;
	desc.d_addr_lo = LN_ADDR_LO(ri->ri_heap + sc->lance_ramoffset);
	desc.d_addr_hi = LN_ADDR_HI(ri->ri_heap + sc->lance_ramoffset);
	desc.d_buflen = 0;
	LN_PUTDESC(sc, &desc, di->di_addr);
    }
    ri->ri_nextin = ri->ri_nextout = ri->ri_first;
    ri->ri_free = ri->ri_max;
    ri->ri_outptr = ri->ri_heap;
    ri->ri_outsize = ri->ri_heapend - ri->ri_heap;

    ri = &sc->lance_rxinfo;
    desc.d_flag = LN_DFLAG_OWNER;
    desc.d_buflen = 0 - sc->lance_rxbufsize;
    for (di = ri->ri_first; di < ri->ri_last; di++) {
	desc.d_addr_lo = LN_ADDR_LO(di->di_bufaddr + sc->lance_ramoffset);
	desc.d_addr_hi = LN_ADDR_HI(di->di_bufaddr + sc->lance_ramoffset);
	LN_PUTDESC(sc, &desc, di->di_addr);
    }
    ri->ri_nextin = ri->ri_nextout = ri->ri_first;
    ri->ri_outptr = ri->ri_heap;
    ri->ri_outsize = ri->ri_heapend - ri->ri_heap;
    ri->ri_free = 0;

    if (sc->lance_if.if_flags & IFF_UP) {
	sc->lance_if.if_flags |= IFF_RUNNING;
	LN_WRCSR(sc, LN_CSR0_START|LN_CSR0_INITDONE|LN_CSR0_ENABINTR);
	/* lance_dumpcsrs(sc, "lance_init: up"); */
	lance_ifstart(&sc->lance_if);
    } else {
	/* lance_dumpcsrs(sc, "lance_init: down"); */
	sc->lance_if.if_flags &= ~IFF_RUNNING;
    }
}

extern void
lance_intr(
    lance_softc_t *sc)
{
    unsigned oldcsr;

    oldcsr = LN_RDCSR(sc);
    oldcsr &= ~LN_CSR0_ENABINTR;
    LN_WRCSR(sc, oldcsr);
    LN_WRCSR(sc, LN_CSR0_ENABINTR);

    if (oldcsr & LN_CSR0_ERRSUM) {
	if (oldcsr & LN_CSR0_MISS) {
	    /*
             *  LN_CSR0_MISS is signaled when the LANCE receiver
             *  loses a packet because it doesn't own a receive
	     *  descriptor. Rev. D LANCE chips, which are no
	     *  longer used, require a chip reset as described
	     *  below.
	     */
	    LN_STAT(rx_misses++);
	}
	if (oldcsr & LN_CSR0_MEMERROR) {
	    LN_STAT(memory_errors++);
	    if (oldcsr & (LN_CSR0_RXON|LN_CSR0_TXON)) {
		lance_ifinit(sc);
		return;
	    }
	}
    }

    if ((oldcsr & LN_CSR0_RXINT) && lance_rx_intr(sc)) {
	lance_ifinit(sc);
	return;
    }

    if (oldcsr & LN_CSR0_TXINT) {
	if (lance_tx_intr(sc))
	    lance_ifstart(&sc->lance_if);
    }

    if (oldcsr == (LN_CSR0_PENDINTR|LN_CSR0_RXON|LN_CSR0_TXON))
        printf("%s%d: lance_intr: stray interrupt\n",
	       sc->lance_if.if_name, sc->lance_if.if_unit);
}

static int
lance_rx_intr(
    lance_softc_t *sc)
{
    lance_ring_t *ri = &sc->lance_rxinfo;
    lance_descinfo_t *eop;
    ln_desc_t desc;
    int ndescs, total_len, rxdescs;

    LN_STAT(rx_intrs++);

    for (rxdescs = 0;;) {
	/*
	 * Now to try to find the end of this packet chain.
	 */
	for (ndescs = 1, eop = ri->ri_nextin;; ndescs++) {
	    /*
	     * If we don't own this descriptor, the packet ain't
	     * all here so return because we are done.
	     */
	    LN_GETDESC(sc, &desc, eop->di_addr);
	    if (desc.d_flag & LN_DFLAG_OWNER)
		return 0;
	    /*
	     * In case we have missed a packet and gotten the
	     * LANCE confused, make sure we are pointing at the
	     * start of a packet. If we aren't, something is really
	     * strange so reinit the LANCE.
	     */
	    if (desc.d_flag & LN_DFLAG_RxBUFERROR) {
		LN_STAT(rx_buferror++);
		return 1;
	    }
	    if ((desc.d_flag & LN_DFLAG_SOP) && eop != ri->ri_nextin) {
		LN_STAT(rx_badsop++);
		return 1;
	    }
	    if (desc.d_flag & LN_DFLAG_EOP)
		break;
	    if (++eop == ri->ri_last)
		eop = ri->ri_first;
	}

	total_len = (desc.d_status & LN_DSTS_RxLENMASK) - 4;
	if ((desc.d_flag & LN_DFLAG_RxERRSUM) == 0) {
	    /*
	     * Valid Packet -- If the SOP is less than or equal to the EOP
	     * or the length is less than the receive buffer size, then the
	     * packet is contiguous in memory and can be copied in one shot.
	     * Otherwise we need to copy two segments to get the entire
	     * packet.
	     */
	    if (ri->ri_nextin <= eop || total_len <= ri->ri_heapend - ri->ri_nextin->di_bufaddr) {
		(*sc->lance_input)(&sc->lance_ac, sc->lance_membase + ri->ri_nextin->di_bufaddr,
			 total_len, total_len, NULL);
		LN_STAT(rx_contig++);
	    } else {
		(*sc->lance_input)(&sc->lance_ac, sc->lance_membase + ri->ri_nextin->di_bufaddr,
			 total_len,
			 ri->ri_heapend - ri->ri_nextin->di_bufaddr,
			 sc->lance_membase + ri->ri_first->di_bufaddr);
		LN_STAT(rx_noncontig++);
	    }
	} else {
	    /*
	     * If the packet is bad, increment the
	     * counters.  
	     */
	    sc->lance_if.if_ierrors++;
	    if (desc.d_flag & LN_DFLAG_RxBADCRC)
		LN_STAT(rx_badcrc++);
	    if (desc.d_flag & LN_DFLAG_RxOVERFLOW)
		LN_STAT(rx_badalign++);
	    if (desc.d_flag & LN_DFLAG_RxFRAMING)
		LN_STAT(rx_badframe++);
	}
	sc->lance_if.if_ipackets++;
	LN_STAT(rx_ndescs[ndescs-1]++);
	rxdescs += ndescs;
	while (ndescs-- > 0) {
	    LN_SETFLAG(sc, ri->ri_nextin->di_addr, LN_DFLAG_OWNER);
	    if (++ri->ri_nextin == ri->ri_last)
		ri->ri_nextin = ri->ri_first;
	}
    }
    /* LN_STAT(rx_intr_descs[rxdescs]++); */
    LN_MAXSTAT(rx_intr_hidescs, rxdescs);

    return 0;
}

extern ifstart_ret_t
lance_ifstart(
    struct ifnet *ifp)
{
    lance_softc_t *sc = (lance_softc_t *) ifp;
    struct ifqueue *ifq = &ifp->if_snd;
    lance_ring_t *ri = &sc->lance_txinfo;
    lance_descinfo_t *di;
    ln_desc_t desc;
    unsigned len, slop;
    struct mbuf *m, *m0;
    caddr_t bp;

    if ((ifp->if_flags & IFF_RUNNING) == 0)
	return;

    for (;;) {
	IF_DEQUEUE(ifq, m);
	if (m == NULL)
	    break;
    
	/*
	 * Make the packet meets the minimum size for Ethernet.
	 * The slop is so that we also use an even number of longwards.
	 */
	len = ETHERMIN + sizeof(struct ether_header);
	if (m->m_pkthdr.len > len)
	    len = m->m_pkthdr.len;

	slop = (8 - len) & 3;
	/*
	 * If there are no free ring entries (there must be always
	 * one owned by the host), or there's not enough space for
	 * this packet, or this packet would wrap around the end
	 * of LANCE RAM then wait for the transmits to empty for
	 * space and ring entries to become available.
	 */
	if (ri->ri_free == 1 || len + slop > ri->ri_outsize) {
	    /*
	     * Try to see if we can free up anything off the transit ring.
	     */
	    if (lance_tx_intr(sc) > 0) {
		LN_STAT(tx_drains[0]++);
		IF_PREPEND(ifq, m);
		continue;
	    }
	    LN_STAT(tx_nospc[0]++);
	    break;
	}
    
	if (len + slop > ri->ri_heapend - ri->ri_outptr) {
	    /*
	     * Since the packet won't fit in the end of the transmit
	     * heap, see if there is space at the beginning of the transmit
	     * heap.  If not, try again when there is space.
	     */
	    LN_STAT(tx_orphaned++);
	    slop += ri->ri_heapend - ri->ri_outptr;
	    if (len + slop > ri->ri_outsize) {
		LN_STAT(tx_nospc[1]++);
		break;
	    }
	    /*
	     * Point to the beginning of the heap
	     */
	    ri->ri_outptr = ri->ri_heap;
	    LN_STAT(tx_adoptions++);
	}
		
	/*
	 * Initialize the descriptor (saving the buffer address,
	 * buffer length, and mbuf) and write the packet out
	 * to the board.
	 */
	di = ri->ri_nextout;
	di->di_bufaddr = ri->ri_outptr;
	di->di_buflen = len + slop;
	di->di_mbuf = m;
#if NBPFILTER > 0
	if (sc->lance_if.if_bpf != NULL)
	    bpf_mtap(sc->lance_if.if_bpf, m);
#endif
	bp = sc->lance_membase + di->di_bufaddr;
	for (m0 = m; m0 != NULL; m0 = m0->m_next) {
	    LN_COPYTO(sc, mtod(m0, caddr_t), bp, m0->m_len);
	    bp += m0->m_len;
	}
	/*
	 * Zero out the remainder if needed (< ETHERMIN).
	 */
	if (m->m_pkthdr.len < len)
	    LN_ZERO(sc, bp, len - m->m_pkthdr.len);

	/*
	 * Finally, copy out the descriptor and tell the
	 * LANCE to transmit!.
	 */
	desc.d_buflen = 0 - len;
	desc.d_addr_lo = LN_ADDR_LO(di->di_bufaddr + sc->lance_ramoffset);
	desc.d_addr_hi = LN_ADDR_HI(di->di_bufaddr + sc->lance_ramoffset);
	desc.d_flag = LN_DFLAG_SOP|LN_DFLAG_EOP|LN_DFLAG_OWNER;
	LN_PUTDESC(sc, &desc, di->di_addr);
	LN_WRCSR(sc, LN_CSR0_TXDEMAND|LN_CSR0_ENABINTR);

	/*
	 * Do our bookkeeping with our transmit heap.
	 * (if we wrap, point back to the beginning).
	 */
	ri->ri_outptr += di->di_buflen;
	ri->ri_outsize -= di->di_buflen;
	LN_MAXSTAT(high_txoutptr, ri->ri_outptr);
	LN_MINSTAT(low_txheapsize, ri->ri_outsize);

	if (ri->ri_outptr == ri->ri_heapend)
	    ri->ri_outptr = ri->ri_heap;

	ri->ri_free--;
	if (++ri->ri_nextout == ri->ri_last)
	    ri->ri_nextout = ri->ri_first;
	LN_MINSTAT(low_txfree, ri->ri_free);
    }
    if (m != NULL) {
	ifp->if_flags |= IFF_OACTIVE;
	IF_PREPEND(ifq, m);
    }
}

static int
lance_tx_intr(
    lance_softc_t *sc)
{
    lance_ring_t *ri = &sc->lance_txinfo;
    unsigned xmits;

    LN_STAT(tx_intrs++);
    for (xmits = 0; ri->ri_free < ri->ri_max; ) {
	ln_desc_t desc;

	LN_GETDESC(sc, &desc, ri->ri_nextin->di_addr);
	if (desc.d_flag & LN_DFLAG_OWNER) 
	    break;

	if (desc.d_flag & (LN_DFLAG_TxONECOLL|LN_DFLAG_TxMULTCOLL))
	    sc->lance_if.if_collisions++;
	if (desc.d_flag & LN_DFLAG_TxDEFERRED)
	    LN_STAT(tx_deferred++);
	if (desc.d_flag & LN_DFLAG_TxONECOLL)
	    LN_STAT(tx_single_collisions++);
	if (desc.d_flag & LN_DFLAG_TxMULTCOLL)
	    LN_STAT(tx_multiple_ucollisions++);

	if (desc.d_flag & LN_DFLAG_TxERRSUM) {
	    if (desc.d_status & (LN_DSTS_TxUNDERFLOW|LN_DSTS_TxBUFERROR|
				 LN_DSTS_TxEXCCOLL|LN_DSTS_TxLATECOLL)) {
		if (desc.d_status & LN_DSTS_TxEXCCOLL) {
		    unsigned tdr;
		    LN_STAT(tx_excessive_collisions++);
		    if ((tdr = (desc.d_status & LN_DSTS_TxTDRMASK)) > 0) {
			tdr *= 100;
			printf("%s%d: lance: warning: excessive collisions: TDR %dns (%d-%dm)\n",
			       sc->lance_if.if_name, sc->lance_if.if_unit,
			       tdr, (tdr*99)/1000, (tdr*117)/1000);
		    }
		}
		if (desc.d_status & LN_DSTS_TxBUFERROR)
		    LN_STAT(tx_buferror++);
		sc->lance_if.if_oerrors++;
		if ((desc.d_status & LN_DSTS_TxLATECOLL) == 0) {
		    lance_ifinit(sc);
		    return 0;
		} else {
		    LN_STAT(tx_late_collisions++);
		}
	    }
	}
	m_freem(ri->ri_nextin->di_mbuf);
	ri->ri_nextin->di_mbuf = NULL;
	sc->lance_if.if_opackets++;
	ri->ri_free++;
	ri->ri_outsize += ri->ri_nextin->di_buflen;
	if (++ri->ri_nextin == ri->ri_last)
	    ri->ri_nextin = ri->ri_first;
	sc->lance_if.if_flags &= ~IFF_OACTIVE;
	xmits++;
    }
    if (ri->ri_free == ri->ri_max)
	LN_STAT(tx_emptied++);
    /* LN_STAT(tx_intr_descs[xmits]++); */
    LN_MAXSTAT(tx_intr_hidescs, xmits);
    return xmits;
}
