/*-
 * Copyright (c) 1995 Berkeley Software Design, Inc. 
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: if_te.c,v 2.5 1995/12/13 03:57:50 ewv Exp $
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/mbuf.h>
#include <sys/buf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/syslog.h>
#include <sys/device.h>
#include <sys/kernel.h>
#include <sys/proc.h>

#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>
#include <net/if_token.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>
#endif

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/isa/icu.h>
#include <machine/cpu.h>
#if __STDC__
#include <machine/stdarg.h>
#else
#include <machine/varargs.h>
#endif

#include <net/bpf.h>
#include <net/bpfdesc.h>

#include "if_tereg.h"
#include "teioctl.h"

/*
 * Debugging and tracing setup
 */
#ifdef TEDEBUG
#define PARANOID		/* shared memory bounds checks */
/* #define MAC3DEBUG		/* MAC3 state transition display */
#include "ewvdebug.h"
#else
#define TR(a,b,c)
#endif
#define ITR(a,b,c)	TR(a,b,c)

typedef enum { 
	TE_HALT=1,		/* TRC halted */
	TE_WINIT,		/* Waiting for TRC init complete */
	TE_WINRING,		/* Waiting for ring insertion complete */
	TE_INRING,		/* Running inserted in ring */
	TE_RETRY		/* Delaying before next init retry */
} te_state_t;

struct te_softc;

/*
 * Token ring software status per interface.
 *
 * Note: All pointers to shared RAM structures are in first RAM page (only
 *		data blocks reside in upper pages).
 */
typedef struct te_softc {
	struct	device te_dev;		/* base device (must be first) */
	struct	isadev te_id;		/* ISA device */
	struct	intrhand te_ih;		/* interrupt vectoring */	
	struct	arpcom te_ac;		/* Ethernet common part */
	struct	atshutdown te_ats;	/* Shutdown callback */

	/* State */
	te_state_t te_state;
	int	te_flags;
	int	te_backoff;	/* Number of seconds before next retry */
	int	te_oflags;	/* Last value of if_flags from te_if */

	/* Config/stat data */
	int	te_irq;		/* Selected IRQ number */
	int	te_iobase;	/* IO area base */
	caddr_t	te_maddr;	/* Physical (ISA) shared memory address */
	caddr_t	te_vaddr;	/* Kernel virtual shared memory address */
	int	te_cbusy_maxwait; /* Max number of CBUSY loops */
	int	te_cbusy_wait;	/* How many cbusy loops (statistic) */

	/* Shared memory pointers (int's are shared memory offsets) */
	int	te_ctlfree;	/* Start of control area free space */
	int	te_buftop;	/* Top of buffer free area */
	int	te_pscp;
	int	te_pscb;
	scb_t	*te_scb;
	int	te_pisb;
	int	te_pacb;	/* Only CB (chain of 1) */
	cb_t	*te_acb;
	int	te_macparm;	/* MAC init paramter block */
	int	te_txrx;	/* TXRX init block */
	int	te_trcstat;	/* TRC status return block */

	/* Transmit */
	int	te_txfcb_start;	/* First TX FCB in ring */
	int	te_txbusy;	/* Number of busy TX buffers */
	fcb_t	*te_last_txfcb;	/* Last TX FCB used */
	fcb_t	*te_act_txfcb;	/* Oldest active TX FCB */

	/* Receive */
	int	te_rxfcb_start;	/* First RX FCB */
	int	te_rxbdb_start;	/* First RX BDB */
	int	te_rxfcb;	/* Next valid RX FCB */
	int	te_rxbdb;	/* Next valid RX BDB */
	fcb_t	*te_last_rxfcb;	/* Last RX fcb processed (end of list) */
	bdb_t	*te_last_rxbdb;	/* Last RX bdb processed (end of list) */

	/* Interrupt handling */
	u_short	*te_isb;	/* First interrupt word */
	int	te_isbcur;	/* Index to next ISB word to be inspected */
	int	te_resword;	/* resume bits to piggyback on interrupt ACK */

	/* Register addresses/shadows */
	int	te_ctl;		/* Control IO port number */
	int	te_page;	/* Page select I/O port */
	int	te_msr;		/* Shadow of MSR register */
	int	te_laar;	/* LAAR port */
} te_softc_t;

#define	te_if	te_ac.ac_if		/* network-visible interface */
#define	te_addr	te_ac.ac_enaddr		/* hardware Ethernet address */

/* Bits for te_flags */
#define	TE_TRC_HALTED	0x01	/* TRC is being held in reset 
						(TE_HALT || TE_RETRY) */
#define TE_NO_UCODE	0x02	/* TRC needs to be downloaded */
#define TE_ETR		0x04	/* Enable early token release */

/* Configuation */
#define SMCTL_OPER	SMCTL_MSKCBUSY | SMCTL_WCSS
#define TE_CBUSY_WAIT	5000	/* CBUSY timeout, very roughly 5ms */
#define ACB_TIMEOUT	100000	/* sync cmd timeout, usec */
#define TE_WDOG		3	/* Watchdog timeout (sec) */
#define TE_RINGWAIT	30	/* Max time to wait for ring insertion */
#define TE_NTXBUFS	2	/* Number of TX buffers (must be at least 2) */
#define TE_RETRY_CAP	256	/* Max seconds between reinit retries */
#define TE_ISBSIZE	32	/* Max number of pending interrupts */
#define TE_BUFSIZE	2048	/* Size of TX and RX buffers */
#ifdef MAC3DEBUG
#define DBG_MAC3	SMIM_MAC3
#else
#define DBG_MAC3	0
#endif
#define IBITS		(SMIM_MAC2 | SMIM_TX | SMIM_TXQ | SMIM_MACRSRC | \
			SMIM_MACRX | SMIM_FIFO | SMIM_MAC1 | SMIM_TRCI | \
			DBG_MAC3)
#define LAAR_BITS	SMLAAR_L16EN | 1

/* Prototypes */
int teprobe __P((struct device *parent, struct cfdata *cf, void
              *aux));
void teattach __P((struct device *parent, struct device *self, void *aux));
void te_init584 __P((te_softc_t *sc));
int teinit __P((int unit));
void te_trci __P((te_softc_t *sc));
int te_meminit __P((te_softc_t *sc));
int te_memchk __P((te_softc_t *sc, int pat));
int te_fcballoc __P((te_softc_t *sc, int *fcbp, int *bdbp));
int te_ctlalloc __P((te_softc_t *sc, int nbytes));
int te_bufalloc __P((te_softc_t *sc, int nbytes));
void te_shutdown __P((void *arg));
int testart __P((struct ifnet *ifp));
int te_resume_tx __P((te_softc_t *sc));
void proc_txq __P((te_softc_t *sc));
void te_rxint __P((te_softc_t *sc));
int teintr __P((void *arg));
void te_getstat __P((te_softc_t *sc));
void te_inring __P((te_softc_t *sc));
#ifdef MAC3DEBUG
void te_mac3state __P((te_softc_t *sc, int iword));
#endif
void te_retry __P((te_softc_t *sc));
void te_close __P((te_softc_t *sc));
void te_halt __P((te_softc_t *sc));
int tewatchdog __P((int unit));
int te_syncacb __P((te_softc_t *sc));
int teioctl __P((struct ifnet *ifp, int cmd, caddr_t data));
int te_ucode __P((te_softc_t *sc, struct ifreq *ifr));
void te_ucksum __P((te_softc_t *sc));
void teprf __P((te_softc_t *sc, char *fmt, ...));
#ifdef TEDEBUG
int te_shget __P((te_softc_t *sc, struct ifreq *ifr));
#endif

/* Data */
struct cfdriver tecd = {
	NULL, "te", teprobe, teattach, DV_IFNET, sizeof(te_softc_t)
};

#define TE_BADINTR	0xff
/* Map from IRQ number to IR2/IR1/IR0 bits */
static u_char irqmap[16] = {
/* 0 */		TE_BADINTR,
/* 1 */		TE_BADINTR,
/* 2 */		0,
/* 3 */		SMIRR_IR0,
/* 4 */		SMIRR_IR0 | SMIRR_IR1 | SMICR_IR2,
/* 5 */		TE_BADINTR,
/* 6 */		TE_BADINTR,
/* 7 */		SMIRR_IR0 | SMICR_IR2,
/* 8 */		TE_BADINTR,
/* 9 */		0,
/* 10 */	SMICR_IR2,
/* 11 */	SMICR_IR2 | SMIRR_IR0,
/* 12 */	TE_BADINTR,
/* 13 */	TE_BADINTR,
/* 14 */	TE_BADINTR,
/* 15 */	SMICR_IR2 | SMIRR_IR1
};

/*
 * Timer initialization values for 4 and 16 mbit
 */
static u_short timer4[13] = {
	0x00fa,			/* Prescale */
	0x2710,			/* TPT */
	0x2710,			/* TQP */
	0x0a28,			/* TNT */
	0x3e80,			/* TBT */
	0x3a98,			/* TSM */
	0x1b58,			/* TAM */
	0x00c8,			/* TBR */
	0x07d0,			/* TER */
	0x000a,			/* TGT */
	0x1162,			/* THT */
	0x07d0,			/* TRR */
	0x1388			/* TVX */
};

static u_short timer16[13] = {
	0x03e8,			/* Prescale */
	0x9c40,			/* TPT */
	0x9c40,			/* TQP */
	0x0a28,			/* TNT */
	0x3e80,			/* TBT */
	0x3a98,			/* TSM */
	0x1b58,			/* TAM */
	0x00c8,			/* TBR */
	0x07d0,			/* TER */
	0x000a,			/* TGT */
	0x4588,			/* THT */
	0x1f40,			/* TRR */
	0x4e20			/* TVX */
};

/*----------------- Macros and inlines ---------------------*/

/* Store/fetch an int from an sptr_t (char[4]) */
#define ADSET(t, v)	*((int *)(t)) = (v)
#define ADGET(t)	*((int *)(t))

/*
 * Wait for CBUSY to clear w/timeout
 */
extern inline int
te_waitcbusy(sc, msg)
	te_softc_t *sc;
	char *msg;
{
	int timer;

	timer = sc->te_cbusy_maxwait;
	while (inb(sc->te_ctl) & SMCTL_CBUSY) {
		if (!timer--) {
			teprf(sc, msg);
			return (1);
		}
		sc->te_cbusy_wait++;
	}
	return (0);
}

/*
 * Pulse CA
 */
extern inline void
te_hitca(sc)
	te_softc_t *sc;
{

	outb(sc->te_ctl, SMCTL_CA | SMCTL_OPER);
}

/*
 * Currently we leave RAM enabled at all times, te_ramon() and te_ramoff()
 * are (tested) hooks to enable ram only when needed.
 *
 * The ifdefed out code below sets/clears the 16 bit access bit for shared
 * memory (this doesn't work reliably on fast machines). For now this feature
 * is left off, the only reason to shut off shared ram is so that the machine
 * will reboot properly. We do this in an atshutdown instead.
 */

/*
 * Disable shared memory region
 */
extern inline void
te_ramoff(sc)
	te_softc_t *sc;
{

#ifdef RAMSW
	int port = sc->te_laar;
	outb(port, LAAR_BITS);
#endif
}

/*
 * Turn on shared ram, return a token that represents the prior state of
 * shared ram.
 */
extern inline int
te_ramon(sc)
	te_softc_t *sc;
{
#ifdef RAMSW
	int port = sc->te_laar;
	int rc;

	rc = inb(port);
	outb(port, LAAR_BITS | SMLAAR_M16EN);
	return (rc);
#else
	return (0);
#endif
}

/*
 * Switch to prior ram mode
 */
extern inline void
te_ramx(sc, x)
	te_softc_t *sc;
	int x;
{
#ifdef RAMSW
	int port = sc->te_laar;

	outb(port, x);
#endif
}

/*
 * Return virtual address given shared RAM address (assumes page is set)
 */
extern inline void *
te_mptr(sc, saddr)
	te_softc_t *sc;
	int saddr;
{

#ifdef PARANOID
	if (saddr < 0 || saddr >= SM_RSIZE) {
		teprf(sc, "te_mptr: address out of range: 0x%x\n", saddr);
		panic("te_mptr: bad address");
	}
#endif
	return (sc->te_vaddr + (saddr & SMPG_PGMASK));
}

/*
 * Select a RAM page and return pointer (in kernel virtual space) based on
 * shared RAM address.
 */
extern inline void *
te_pgmem(sc, saddr)
	te_softc_t *sc;
	int saddr;
{

	outb(sc->te_page, SMPGBITS(saddr) >> 8);	/* Select page */
	return (te_mptr(sc, saddr));
}

/*
 * Select page 0
 */
extern inline void
te_pg0(sc)
	te_softc_t *sc;
{

	outb(sc->te_page, 0);
}

/*--------- Configuration and low level initialization -------- */

/*
 * Probe routine
 *
 * Determine if card exists by validating the LAN addr/ID checksum
 */
int
teprobe(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	int base = ia->ia_iobase;
	caddr_t maddr = ia->ia_maddr;
	int cksum;
	int i;

	/* Check base address sanity */
	if (base < SM_MINPORT || base > SM_MAXPORT || base & SM_PORTMASK) {
		printf("te%d: Invalid port base: 0x%x\n", cf->cf_unit, base);
		return (0);
	}

	/* See if card exists */
	cksum = 0;
	for (i = 0; i < 7; i++)
		cksum += inb(base + SMIO_LADDR + i);
	cksum = (0xee - cksum & 0xff) & 0xff;
	i = inb(base + SMIO_CKSUM);	/* expected checksum value */
	if (cksum != i)
		return (0);

	/* Check shared ram address */
	if (((int)maddr & SMSHM_MASK) || (maddr < (caddr_t)SMSHM_MIN) ||
	    (maddr > (caddr_t)SMSHM_MAX)) {
		printf("te%d: Invalid shared memory address\n", cf->cf_unit);
		return (0);
	}

	/* Check/allocate an interrupt */
	if (ia->ia_irq == IRQUNK)
		if ((ia->ia_irq = isa_irqalloc(SMIRQ_MASK)) == 0) {
			printf("te%d: no IRQ available\n", cf->cf_unit);
			return (0);
		}
	ia->ia_irq &= SMIRQ_MASK;
	if (ia->ia_irq == IRQNONE) {
		printf("te%d: Invalid IRQ\n", cf->cf_unit);
		return (0);
	}

	/* Fill in additional config paramters */
	ia->ia_msize = SM_WSIZE;
	ia->ia_iosize = SM_NPORTS;

	return (1);
}

/*
 * Fill in the interface stucture and softc, put in HALT state.
 */
void
teattach(parent, self, aux)
	struct device *parent;
	struct device *self;
	void *aux;
{
	te_softc_t *sc = (te_softc_t *)self;
	struct isa_attach_args *ia = (struct isa_attach_args *)aux;
	struct ifnet *ifp = &sc->te_if;
	int i;
	extern int delay_mult;

	/* Print warm fuzzies */
	aprint_naive(": SMC Token Ring");
	printf("\n");

	/*
	 * Save configuration data
	 */
	sc->te_irq = ffs(ia->ia_irq) - 1;
	sc->te_iobase = ia->ia_iobase;
	sc->te_maddr = ia->ia_maddr;
	sc->te_vaddr = (caddr_t)ISA_HOLE_VADDR(ia->ia_maddr);
	sc->te_cbusy_maxwait = (TE_CBUSY_WAIT * delay_mult + 64) / 256;

	/* frequently used ports */
	sc->te_ctl = sc->te_iobase + SMIO_CTL;
	sc->te_page = sc->te_iobase + SMIO_PAGE;
	sc->te_laar = sc->te_iobase + SMIO_LAAR;

	ifp->if_name = tecd.cd_name;
	ifp->if_unit = sc->te_dev.dv_unit;

	/*
	 * Set up the ISA interface chip (in the hopes of catching conflicts
	 * early instead of at ifconfig time). This also fetches the MAC
	 * address and stores it in the interface structure.
	 */
	te_init584(sc);

	sc->te_flags = TE_ETR;
	te_ramon(sc);
	te_ucksum(sc);
	te_ramoff(sc);
	sc->te_state = TE_HALT;
	sc->te_backoff = 1;

	aprint_normal("te%d: address %s\n", ifp->if_unit, 
	    ether_sprintf(sc->te_addr));

	/* Initialize rest of interface structure */
	ifp->if_mtu = ISO88025_MTU;
	ifp->if_init = teinit;			/* Never called by system */
	ifp->if_ioctl = teioctl;
	ifp->if_start = testart;
	ifp->if_watchdog = tewatchdog;
	ifp->if_flags = IFF_BROADCAST | IFF_LINK1;
	token_attach(ifp);

	bpfattach(&ifp->if_bpf, ifp, DLT_IEEE802, sizeof(struct token_header));

	isa_establish(&sc->te_id, &sc->te_dev);
	sc->te_ih.ih_fun = teintr;
	sc->te_ih.ih_arg = (void *)sc;
	intr_establish(ia->ia_irq, &sc->te_ih, DV_NET);

	/* Arrange for callback to unmap memory at shutdown */
	sc->te_ats.func = te_shutdown;
	sc->te_ats.arg = (void *)sc;
	atshutdown(&sc->te_ats, ATSH_ADD);

#ifdef PARANOID
	teprf(sc, "paranoid bounds checks are enabled\n");
#endif
}

/*
 * Set up the ISA interface chip, leaves RAM in 8 bit mode (ie: unuasable)
 */
void
te_init584(sc)
	te_softc_t *sc;
{
	struct ifnet *ifp = &sc->te_if;
	int base = sc->te_iobase;
	u_char *cp;
	int i;
	int ibits;

	/* Disable interrupts during setup */
	outb(base + SMIO_CTL, SMCTL_MSKCBUSY | SMCTL_MSKTINT);

	/* Enable shared ram and keep controller in reset */
	sc->te_msr = ( (int)sc->te_maddr >> 13 & SMMSR_ADDRMASK) | SMMSR_MENB;
	outb(base + SMIO_MSR, sc->te_msr | SMMSR_RST);
	sc->te_flags |= TE_TRC_HALTED;

	ibits = irqmap[sc->te_irq];
	if (ibits == TE_BADINTR) {
		teprf(sc, "Internal IRQ error: irq=%d\n", sc->te_irq);
		panic("te: bad irq");
	}

	outb(base + SMIO_ICR, ibits & SMICR_IR2);
	outb(base + SMIO_BIO, SMBIO_RS_OFF);	/* Disable any BIOS */
	ibits &= SMIRR_IR0|SMIRR_IR1;
	if (ifp->if_flags & IFF_LINK1) {
		/* LINK1 == 16mbit */
		ibits |= SMIRR_16M;
	}
	if (ifp->if_flags & IFF_LINK2) {
		/* LINK2 == UTP */
		ibits |= SMIRR_UTP;
	}
	ibits |= SMIRR_IEN;
	outb(base + SMIO_IRR, ibits);
#ifdef RAMSW
	te_ramoff(sc);				/* Leave ram off */
#else
	outb(base + SMIO_LAAR, LAAR_BITS | SMLAAR_M16EN);
#endif
	outb(base + SMIO_PAGE, 0);

	/* Read MAC address */
	cp = sc->te_addr;
	for (i = 0; i < 6; i++)
		*cp++ = inb(base + SMIO_LADDR + i);

	/* Enable TRC interrupt and shared RAM */
	outb(sc->te_ctl, SMCTL_CLRCBUSY | SMCTL_CLRTINT | SMCTL_OPER);
}

/*
 * Start TRC initialization sequence
 */
int
teinit(unit)
	int unit;
{
	te_softc_t *sc = tecd.cd_devs[unit];
	struct ifnet *ifp = &sc->te_if;
	volatile scb_t *scb;
	int s;

 	if (ifp->if_addrlist == (struct ifaddr *)0)
		goto errout;

	if (sc->te_flags & TE_NO_UCODE) {
		teprf(sc, "microcode not loaded\n");
		ifp->if_flags &= ~IFF_UP;
		goto errout;
	}

	if (ifp->if_flags & IFF_RUNNING)
		return (0);

	s = splimp();

	sc->te_oflags = ifp->if_flags;

	te_init584(sc);		/* Init ISA interface, put TRC into reset */
	te_ramon(sc);
	if (te_meminit(sc)) 	/* Init shared memory strucures */
		goto errout;
	outb(sc->te_iobase + SMIO_MSR, sc->te_msr);	/* release RESET */
	if (te_waitcbusy(sc, "teinit: TRC reset timeout\n"))
		goto errout;
	scb = sc->te_scb;

	/* Mask out all undesirable interrupts */
	scb->ctl = SMCTL_SV|SMCTL_SETIM;
	scb->imctl = ~IBITS;
	te_hitca(sc);
	if (te_waitcbusy(sc,"te_setimask: TRC hung after SETIM\n"))
		goto errout;

	/* Check for TRC fault */
	if (inb(sc->te_ctl) & SMCTL_FAULT) {
		teprf(sc, "TRC fault\n");
		goto errout;
	}

	/* Enable interesting interrupts */
	scb->ctl = SMCTL_SV | SMCTL_CLRIM;
	scb->imctl = IBITS;
	te_hitca(sc);
	if (te_waitcbusy(sc,"te_setimask: TRC hung after CLRIM\n"))
		goto errout;

	sc->te_state = TE_WINIT;
	ifp->if_timer = TE_WDOG;

	splx(s);
	return (0);

errout:
	te_halt(sc);		/* Move back to halted state for safety */
	ifp->if_flags &= ~IFF_UP;
	splx(s);
	return (1);
}

/*
 * TRC initialization complete interrupt
 */
void
te_trci(sc)
	te_softc_t *sc;
{
	mac_conf_t *mp = te_mptr(sc, sc->te_macparm);
	struct ifnet *ifp = &sc->te_if;
	volatile cb_t *acb = sc->te_acb;
	volatile scb_t *scb = sc->te_scb;
	u_short *tp;
	u_short *dmac;
	int i;

	TR("te_trci", sc, sc->te_state);
	ifp->if_timer = 0;
#ifdef DIAGNOSTIC
	if (sc->te_state != TE_WINIT) {
		teprf(sc, "trci invalid state: %d\n", sc->te_state);
		panic("te_trci: invalid state");
	}
#endif
	sc->te_flags &= ~TE_TRC_HALTED;

	/*
	 * Initialize the MAC
	 */
	bzero(mp, sizeof(*mp));
	mp->conf0 = SMIAC0_ONEQUE|SMIAC0_USETPT|SMIAC0_THDREN|SMIAC0_TRIG;
	if (ifp->if_flags & IFF_LINK1) {
		/* 16 Mbit */
		mp->conf0 |= SMIAC0_MRCLK;
		tp = timer16;
		mp->conf1 = SMIAC1_TIF_16;
	} else {
		/* 4 Mbit */
		tp = timer4;
		mp->conf1 = SMIAC1_TIF_4;
	}
	/* Early token release */
	if (!(sc->te_flags & TE_ETR))
		mp->conf0 |= SMIAC0_ETREN;
	if (ifp->if_flags & IFF_PROMISC)
		mp->conf0 |= SMIAC0_PROMIS;
	for (i = 0; i < 13; i++)
		mp->tblock[i] = tp[i];
	dmac = (u_short *)mp->imac;
	tp = (u_short *)sc->te_addr;
	dmac[0] = ntohs(tp[0]);
	dmac[1] = ntohs(tp[1]);
	dmac[2] = ntohs(tp[2]);

	/* Individual Group address (set to group 0) */
	dmac = (u_short *)mp->iga;
	dmac[0] = ntohs(0xc000);
	dmac[1] = ntohs(0x8000);

	/* Functional address (set to unused value) */
	dmac = (u_short *)mp->fa;
	dmac[0] = ntohs(0xc000);

	ADSET(acb->eap, sc->te_macparm);
	acb->pcmd = SMCB_INIT_MAC;
	if (te_syncacb(sc))
		goto again;

	/*
	 * Initialize the RX and TX queues
	 */
	ADSET(acb->eap, sc->te_txrx);
	acb->pcmd = SMCB_INIT_TXRX;
	if (te_syncacb(sc))
		goto again;

	/*
	 * Start the receive queues
	 */
	if (te_waitcbusy(sc, "te_trci: TRC hung before RX resume\n"))
		goto again;
	scb->resctl = SMRES_RXMAC_FCB | SMRES_RXMAC_BDB;
	scb->ctl = SMCTL_SV | SMCTL_RV | SMCTL_NOP;
	te_hitca(sc);
	if (te_waitcbusy(sc, "te_trci: TRC hung after RX resume\n"))
		goto again;

	/*
	 * Insert in ring
	 */
	acb->pcmd = SMCB_INSERT;
	if (te_syncacb(sc))
		goto again;

	/*
	 * Wait for MAC standby state interrupt to tell us we're inserted
	 * and running.
	 */
	sc->te_state = TE_WINRING;
	ifp->if_timer = TE_RINGWAIT;

	return;

again:
	te_retry(sc);
	return;
}

/*
 * Set up shared memory
 *
 * Note: We tell the TRC that the last TX and RX FCB's and BDB's are the
 *	start of the chain, the TRC "eats" the first FCB/BDB when the
 *	queue is started (regardless of the setting of the chain end bit).
 */
int
te_meminit(sc)
	te_softc_t *sc;
{
	scp_t *scp;
	fcb_t *lastfcb;
	bdb_t *lastbdb;
	txrx_init_t *rtp;
	fcb_t *fcb;
	bdb_t *bdb;
	caddr_t	ptr;
	int pfcb;
	int pbdb;
	int i;

	/* Check/clear shared memory */
#ifdef DIAGNOSTIC
	if (te_memchk(sc, 0xaaaaaaaa))
		return (1);
	if (te_memchk(sc, 0x55555555))
		return (1);
	if (te_memchk(sc, 0xffffffff))
		return (1);
#endif
	if (te_memchk(sc, 0x00))
		return (1);


	/* Set up shared memory allocators */
	sc->te_ctlfree = 0;
	sc->te_buftop = SM_ISCPADDR;	/* Top minus space for ISCP */

	/* Allocate basic control blocks */
	sc->te_pscp = te_ctlalloc(sc, sizeof(scp_t));
	sc->te_pscb = te_ctlalloc(sc, sizeof(scb_t));
	sc->te_scb = (scb_t *)te_mptr(sc, sc->te_pscb);
	sc->te_pisb = te_ctlalloc(sc, sizeof(u_short) * TE_ISBSIZE);
	sc->te_isb = (u_short *)te_mptr(sc, sc->te_pisb);
	sc->te_isbcur = 0;
	sc->te_resword = -1;
	sc->te_pacb = te_ctlalloc(sc, sizeof(cb_t));
	sc->te_acb = (cb_t *)te_mptr(sc, sc->te_pacb);

	/* ISCP (stored in network order) */
	ptr = te_pgmem(sc, SM_ISCPADDR);
	ADSET(ptr, htonl(sc->te_pscp));
	te_pg0(sc);

	/* SCP */
	scp = (scp_t *)te_mptr(sc, sc->te_pscp);
	scp->ctl = SCP_IAF | SCP_IDF | SCP_BSTLEN;
	ADSET(scp->scbp, sc->te_pscb);
	ADSET(scp->cbp, sc->te_pacb);
	ADSET(scp->sbp, sc->te_pisb);
	scp->sbsiz = (TE_ISBSIZE - 1) * 2;

	/* CB "chain" (chain of 1) */
	sc->te_acb->ctl = SMCBC_CI|SMCBC_CE;
	ADSET(sc->te_acb->ncbp, sc->te_pacb);	/* Point at self */

	/* MAC parameter block */
	sc->te_macparm = te_ctlalloc(sc, sizeof(mac_conf_t));

	/* TXRX init block */
	sc->te_txrx = te_ctlalloc(sc, sizeof(txrx_init_t));

	/* TRC status block */
	sc->te_trcstat = te_ctlalloc(sc, sizeof(trcstat_t));

	rtp = (txrx_init_t *)te_mptr(sc, sc->te_txrx);

	/*
	 * Set up transmit buffer ring
	 */
	sc->te_txfcb_start = 0;
	for (i = 0; i < TE_NTXBUFS; i++) {
		/* Allocate fcb, bdb, and buffer */
		if (te_fcballoc(sc, &pfcb, &pbdb)) {
			teprf(sc, "no space for TX buffers\n");
			return (1);
		}

		/* Init fcb and link to bdb */
		fcb = (fcb_t *)te_mptr(sc, pfcb);
		if (sc->te_txfcb_start == 0) {
			sc->te_txfcb_start = pfcb;
			fcb->ctl = SMFCT_FI|SMFCT_FE;
		} else {
			fcb->ctl = SMFCT_FI;
			ADSET(lastfcb->nfcbp, pfcb);
		}
		ADSET(fcb->bdbp, pbdb);

		/* Mark end of BDB chain (data buffer already linked in) */
		bdb = (bdb_t *)te_mptr(sc, pbdb);
		bdb->flags = SMBFL_BE;

		/* Short circuit to get to buffer quickly */
		ADSET(fcb->aux, ADGET(bdb->buf));

		lastfcb = fcb;
	}
	/* Point last buffer at first to complete ring */
	ADSET(fcb->nfcbp, sc->te_txfcb_start);
	ADSET(rtp->tx0_fcb, pfcb);

	sc->te_txbusy = 0;
	sc->te_last_txfcb = fcb;
	sc->te_act_txfcb = te_mptr(sc, sc->te_txfcb_start);

	/*
	 * Set up receive buffer rings (use remaining memory). Note: two
	 * chains that are not connected (FCB's and BDB's) are built.
	 */
	sc->te_rxfcb = 0;
	for (;;) {
		/* Allocate fcb, bdb, and buffer */
		if (te_fcballoc(sc, &pfcb, &pbdb)) {
			if (sc->te_rxfcb_start)
				break;		/* Got at least one */
			teprf(sc, "no space for RX buffers\n");
			return (1);
		}

		/* RXTX init block ends up pointing at last fcb/bdb */
		ADSET(rtp->rxmac_fcb, pfcb);
		ADSET(rtp->rxmac_bdb, pbdb);

		fcb = (fcb_t *)te_mptr(sc, pfcb);
		bdb = (bdb_t *)te_mptr(sc, pbdb);

		if (sc->te_rxfcb == 0) {
			/* First one, keep track of start of ring */
			sc->te_rxfcb = sc->te_rxfcb_start = pfcb;
			sc->te_rxbdb = sc->te_rxbdb_start = pbdb;
		} else {
			/* Extend each list */
			ADSET(lastfcb->nfcbp, pfcb);
			ADSET(lastbdb->nbdbp, pbdb);
			lastfcb->ctl &= ~SMFCT_FE;
			lastbdb->flags &= ~SMBFL_BE;
		}

		/* Mark current buffers as ends of their chains */
		fcb->ctl = SMFCT_FI | SMFCT_FE;
		bdb->flags = SMBFL_BE;

		lastfcb = fcb;
		lastbdb = bdb;
	}
	/* Make rings out of both lists */
	ADSET(fcb->nfcbp, sc->te_rxfcb_start);
	ADSET(bdb->nbdbp, sc->te_rxbdb_start);
	sc->te_last_rxfcb = fcb;
	sc->te_last_rxbdb = bdb;
	return (0);
}

/*
 * Pattern memory and then see if it took
 */
int
te_memchk(sc, pat)
	te_softc_t *sc;
	int pat;
{
	int i;
	int pgbase;
	int rc;

	rc = 0;
	for (pgbase = 0; pgbase < SM_RSIZE; pgbase += SM_WSIZE) {
		int *p;

		p = (int *)te_pgmem(sc, pgbase);
		for (i = 0; i < SM_WSIZE/sizeof(int); i++)
			p[i] = pat;
#ifdef DIAGNOSTIC
		for (i = 0; i < SM_WSIZE / sizeof(int); i++)
			if (p[i] != pat) {
				teprf(sc, "shared memory failure page %d\n",
					pgbase);
				rc = -1;
				break;
			}
#endif
	}
	te_pg0(sc);
	return (rc);
}

/*
 * Allocate an FCB/BDB/buffer trio and do some simple initialization.
 */
int
te_fcballoc(sc, fcbp, bdbp)
	te_softc_t *sc;
	int *fcbp;
	int *bdbp;
{
	bdb_t *bdb;
	int dbuf;

	*fcbp = te_ctlalloc(sc, sizeof(fcb_t));
	*bdbp = te_ctlalloc(sc, sizeof(bdb_t));
	if (*fcbp < 0 || *bdbp < 0)
		return (-1);
	dbuf = te_bufalloc(sc, TE_BUFSIZE);
	if (dbuf < 0)
		return (-1);
	bdb = te_mptr(sc, *bdbp);
	ADSET(bdb->buf, dbuf);
	bdb->bflen = TE_BUFSIZE;
	return (0);
}

/*
 * Allocate shared control memory (page 0 only)
 */
int
te_ctlalloc(sc, nbytes)
	te_softc_t *sc;
	int nbytes;
{
	int base = sc->te_ctlfree;

	nbytes = nbytes + 1 & ~1;		/* Round up to short boundary */
	if (base + nbytes > SM_WSIZE) 
		return (-1);
	sc->te_ctlfree += nbytes;
	return (base);
}

/*
 * Allocate shared buffer memory (pages 3,2,1,0). Will not allocate
 * across a page boundary.
 */
int
te_bufalloc(sc, nbytes)
	te_softc_t *sc;
	int nbytes;
{
	int try;

	nbytes = nbytes + 1 & ~1;
	try = sc->te_buftop - nbytes;
	if (try <= sc->te_ctlfree) 
		return (-1);	/* Out of space */
	if (SMPGBITS(try) != SMPGBITS(sc->te_buftop)) {
		sc->te_buftop  = sc->te_buftop & ~SMPG_PGMASK;
		try = sc->te_buftop - nbytes;
		if (try <= sc->te_ctlfree)
			return (-1);
	}
	sc->te_buftop = try;
	return (try);
}

/*
 * Called right before shutdown, put card into reset state, turn off
 * 16 bit memory access (which may interfere with rebooting), and turn
 * off shared memory.
 */
void
te_shutdown(arg)
	void *arg;
{
	te_softc_t *sc = (te_softc_t *)arg;
	int base = sc->te_iobase;

	/* Hold TRC in reset and shut off shared ram */
	outb(base + SMIO_MSR, (sc->te_msr | SMMSR_RST) & ~SMMSR_MENB);

	/* Put memory interface into 8 bit mode */
	outb(base + SMIO_LAAR, LAAR_BITS);
}


/* ------------ Packet output ------------- */

/*
 * Setup output on interface, called only at splimp or interrupt level.
 */
int
testart(ifp)
	struct ifnet *ifp;
{
	te_softc_t *sc = tecd.cd_devs[ifp->if_unit];
	fcb_t *fp;
	caddr_t	dp;
	int len;
	struct mbuf *m;
	struct mbuf *mp;
	int x = te_ramon(sc);

	TR("enter trstart (txbusy)", sc->te_txbusy, 0);
	while (sc->te_txbusy < TE_NTXBUFS) {
		IF_DEQUEUE(&ifp->if_snd, m);
		TR("first m", m, 0);
		if (m == (struct mbuf *) 0)
			break;

		if (ifp->if_bpf)
			bpf_mtap(ifp->if_bpf, m);

		fp = te_mptr(sc, ADGET(sc->te_last_txfcb->nfcbp));
		TR("got fp", fp, 0);

		/*
		 * Fill in the transmit buffer. We rely on MTU to keep
		 * from overflowing the TX buffers.
		 */
		len = 0;
		dp = te_pgmem(sc, ADGET(fp->aux));
		for (mp = m; mp != 0; mp = mp->m_next) {
			int cnt = mp->m_len;

			TR("chunk (m/cnt)", mp, cnt);
			bcopy(mtod(mp, caddr_t), dp, cnt);
			dp += cnt;
			len += cnt;
		}
		te_pg0(sc);
		fp->flen = len;
		TR("total len", len, fp->flen);
		m_freem(m);

		/*
		 * Move end bit to this FCB and resume the queue if idle
		 */
		fp->ctl = SMFCT_FI | SMFCT_FE;
		fp->frst = 0;
		sc->te_last_txfcb->ctl = SMFCT_FI;
		sc->te_last_txfcb = fp;
		if (!sc->te_txbusy++) {
			TR("resuming", 0, 0);
			if (te_resume_tx(sc)) {
				teprf(sc, "testart: TRC not responding\n");
				te_retry(sc);
			}
			ifp->if_flags |= IFF_OACTIVE;
		}
		ifp->if_opackets++;
	}
	te_ramx(sc,x);
	TR("exit trstart (busy/lastlen)", x, sc->te_last_txfcb->flen);
	return (0);
}

/*
 * Resume the transmit queue.
 */
int
te_resume_tx(sc)
	te_softc_t *sc;
{
	scb_t *scb = sc->te_scb;

	if (sc->te_resword != -1) {
		/*
		 * We are in the interrupt handler, it will issue a CA to 
		 * ACK the interrupt, this piggybacks the TX queue resume 
		 * and prevents that ACK from waiting for CBUSY to clear.
		 */
		sc->te_resword = SMRES_TX0_FCB;
		return (0);
	}
	if (te_waitcbusy(sc,"te_resume_tx: TRC not responding\n"))
		return (1);
	scb->resctl = SMRES_TX0_FCB;
	scb->ctl = SMCTL_SV | SMCTL_RV | SMCTL_NOP;
	te_hitca(sc);
	return (0);
}

/*
 * Collect status on transmit buffers and logically free them.
 */
void
proc_txq(sc)
	te_softc_t *sc;
{

	while (sc->te_txbusy) {
		fcb_t *fp = sc->te_act_txfcb;
		int stat = fp->frst;

		if ((stat & SMTXST_FD) == 0)
			return;
		if ((stat & SMTXST_ED) != 0)
			sc->te_if.if_oerrors++;
		sc->te_txbusy--;
		sc->te_act_txfcb = te_mptr(sc, ADGET(fp->nfcbp));
	}
	sc->te_if.if_flags &= ~IFF_OACTIVE;
	return;
}

/* ------------ Packet input ------------- */

/*
 * Process a single FCB on the receive Q
 *
 * Receive Q FCB's are chained together in a loop, the FE bit must always
 * be set on the last available buffer to catch overflow conditions.
 */
void
te_rxint(sc)
	te_softc_t *sc;
{
	fcb_t *fp;
	struct ifnet *ifp = &sc->te_if;
	int pkt_resid;
	struct mbuf *m;
	int m_resid;
	caddr_t m_ptr;
	struct mbuf *mfirst;
	struct mbuf **mlink;
	bdb_t *bp;
	int b_resid;
	int b_ptr;

	fp = te_mptr(sc, sc->te_rxfcb);
#ifdef DIAGNOSTIC
	if (fp->frst & SMRXST_ERR) {
		/* Shouldn't happen, we didn't ask for these */
		teprf(sc, "packet in error received: stat=0x%x\n", fp->frst);
		goto discard;
	}
#endif

nextpkt:
	if ((fp->frst & SMRXST_FD) == 0)
		return;

	pkt_resid = fp->flen;

	/* Set up for first buffer */
	bp = (bdb_t *)te_mptr(sc, ADGET(fp->bdbp));  /* First BDB on this FCB */
	b_resid = min(TE_BUFSIZE, pkt_resid);
	b_ptr = ADGET(bp->buf);

	/* Set up first (and hopefully only) mbuf */
	MGETHDR(m, M_DONTWAIT, MT_DATA);
	if (m == NULL) 
		goto discard;
	m->m_pkthdr.len = pkt_resid;
	m->m_pkthdr.rcvif = ifp;
	m_resid = MHLEN;
	if (pkt_resid > MINCLSIZE) {
		MCLGET(m, M_DONTWAIT);
		if (m->m_flags & M_EXT)
			m_resid = MCLBYTES;
	}
	m->m_len = min(m_resid, pkt_resid);
	m_ptr = mtod(m, caddr_t);
	mfirst = m;
	mlink = &m->m_next;

	for (;;) {
		int this_len = min(m_resid, b_resid);

		bcopy(te_pgmem(sc, b_ptr), m_ptr, this_len);
		te_pg0(sc);			/* Back to control page */

		if ((pkt_resid -= this_len) == 0)
			break;

		if ((b_resid -= this_len) == 0) {
			bp->flags = SMBFL_BE;
			sc->te_last_rxbdb->flags = 0;
			sc->te_last_rxbdb = bp;
			bp = (bdb_t *)te_mptr(sc, ADGET(bp->nbdbp));
			b_resid = min(TE_BUFSIZE, pkt_resid);
			b_ptr = ADGET(bp->buf);
		} else
			b_ptr += this_len;

		if ((m_resid -= this_len) == 0) {
			MGET(m, M_DONTWAIT, MT_DATA);
			if (m == NULL)
				goto mdiscard;
			*mlink = m;
			mlink = &m->m_next;
			m_resid = MHLEN;
			if (pkt_resid > MINCLSIZE) {
				MCLGET(m, M_DONTWAIT);
				if (m->m_flags & M_EXT)
					m_resid = MCLBYTES; 
			}
			m->m_len = min(m_resid, pkt_resid);
			m_ptr = mtod(m, caddr_t);
		} else
			m_ptr += this_len;
#ifdef DIAGNOSTIC
		if ((m_resid < 0) || (b_resid < 0)) {
			teprf(sc, "invalid residual: m_resid=%d b_resid=%d\n",
			    m_resid, b_resid);
			goto mdiscard;
		}
#endif
	}

	/* Advance end of BDB list */
	bp->flags = SMBFL_BE;
	sc->te_last_rxbdb->flags = 0;
	sc->te_last_rxbdb = bp;

	ifp->if_ipackets++;

	if (ifp->if_bpf) {
		bpf_mtap(ifp->if_bpf, mfirst);
		if ((fp->frst & SMRXST_DMATCH) == 0) {
			m_freem(mfirst);
			goto done;
		}
	}

	token_input(ifp, mfirst);
	goto done;

	/*
	 * Release the receive resources on error
	 */
mdiscard:
	m_freem(mfirst);
discard:
	pkt_resid = fp->flen;
	bp = (bdb_t *)te_mptr(sc, ADGET(fp->bdbp)); /* restart at first one */
	for (;;) {
		pkt_resid -= bp->bflen;
		if (pkt_resid <= 0)
			break;
		bp = (bdb_t *)te_mptr(sc, ADGET(bp->nbdbp));
	}
	bp->flags = SMBFL_BE;
	sc->te_last_rxbdb->flags = 0;
	sc->te_last_rxbdb = bp;
	ifp->if_ierrors++;

done:
	/* free this FCB, move to next one */
	fp->ctl = SMFCT_FI | SMFCT_FE;
	fp->frst = 0;
	sc->te_last_rxfcb->ctl = SMFCT_FI;
	sc->te_last_rxfcb = fp;
	fp = te_mptr(sc, sc->te_rxfcb = ADGET(fp->nfcbp));
	goto nextpkt;
}

/* ------------  Interrupt handling/misc ------------- */

/*
 * Controller interrupt.
 */
int
teintr(arg)
	void *arg;
{
	te_softc_t *sc = arg;
	struct ifnet *ifp = &sc->te_if;
	scb_t *scb = sc->te_scb;

	te_ramon(sc);

	/*
	 * Scan forward in ISB interrupt chain
	 */
	sc->te_resword = 0;
	for (;;) {
		u_short iword = sc->te_isb[sc->te_isbcur];

		ITR("teintr: iword", iword, sc->te_isbcur);
		if (iword == 0)
			break;

		switch (ISB_TYPE(iword)) {
		case SMIT_MRX:
			te_rxint(sc);
			break;

		case SMIT_TX:
			proc_txq(sc);
			testart(&sc->te_if);
			break;

		case SMIT_ETX:
			proc_txq(sc);
			/* Restart TX chain if we lost the race */
			if (sc->te_txbusy)
				te_resume_tx(sc);
			testart(&sc->te_if);
			break;

		case SMIT_TRCI:
			te_trci(sc);
			break;

		case SMIT_MAC1:
			teprf(sc, "mac1: removed from ring: 0x%x\n",
			    ISB_SUBTYPE(iword));
			te_retry(sc);
			break;

		case SMIT_MAC2:
			te_getstat(sc);
			break;

#ifdef MAC3DEBUG
		case SMIT_MAC3:
			te_mac3state(sc, iword);
			break;
#endif

		case SMIT_RRI:
		case SMIT_MRRI:
		case SMIT_FIFO:
			teprf(sc, "resource/overrun error 0x%x\n", iword);
			te_retry(sc);
			break;

		default:
			teprf(sc, "Unexpected ISB 0x%x\n", iword);
			break;
		}

		/* If someone moved to HALT or RETRY exit, TRC is stopped */
		if (sc->te_flags & TE_TRC_HALTED) 
			return (1);

		sc->te_isb[sc->te_isbcur] = 0;
		scb->iack = (sc->te_isbcur++) << 1;
		if (sc->te_isbcur == TE_ISBSIZE)
			sc->te_isbcur = 0;
	}

	/*
	 * End of interrupt processing:
	 *   In TRC:
	 *	ACK processed interrupts via iack field (updated above)
	 *	Resume any transmit queues needing to be resumed
	 *	Clear interrupt mask
	 *	Clear interrupt
	 *   In 584:
	 *	Clear latched TRC interrupt bit
	 *	Issue a CA to the TRC to execute the above ops
	 */
	ITR("intr done", sc->te_resword, scb->iack);
	if (te_waitcbusy(sc,"teintr: TRC not responding"))
		te_retry(sc);
	else {
		scb->resctl = sc->te_resword;
		scb->imctl = IBITS;
		sc->te_resword = -1;
		scb->ctl = SMCTL_SV | SMCTL_IV | SMCTL_RV | SMCTL_CLRIM;
		ITR("intr CA ctl/resctl", scb->ctl, scb->resctl);
		outb(sc->te_ctl, SMCTL_CLRTINT | SMCTL_CA | SMCTL_OPER);
	}
	ITR("intr exit", 0, 0);
	te_ramoff(sc);
	return (1);
}

/*
 * Request TRC status, must be run with ACB free.
 */
void
te_getstat(sc)
	te_softc_t *sc;
{
	volatile cb_t *acb = sc->te_acb;
	trcstat_t *ts = te_mptr(sc, sc->te_trcstat);
	struct ifnet *ifp = &sc->te_if;
	int ringstat;
	int why;

	ADSET(acb->eap, sc->te_trcstat);
	acb->pcmd = SMCB_RTRC;
	if (te_syncacb(sc)) {
		if (ifp->if_flags & IFF_UP) {
			te_retry(sc);
		} else {
			te_halt(sc);
		}
		return;
	}

	ringstat = ts->ringstat;
	why = ts->status;

	if ((why & SMTS_RRQINIT) != 0) {
		/* 
		 * MAC wants us to send init request (which we don't
		 * support). This is a good indication that we are
		 * up and running on the ring, so we use this to
		 * transition to the in ring state.
		 */
		te_inring(sc);
	}

	if ((ringstat & SMRS_RINGREC) != 0)
		teprf(sc, "receiving or transmitting contention frames\n");
	if ((ringstat & SMRS_ONLY) != 0)
		teprf(sc, "only station on ring\n");
	if ((ringstat & SMRS_REMOVE) != 0)
		teprf(sc, "remove frame received\n");
	if ((ringstat & SMRS_AUTORMERR) != 0)
		teprf(sc, "lobe test failed\n");
	if ((ringstat & SMRS_LOBEFLT) != 0)
		teprf(sc, "lobe wiring fault\n");
	if ((ringstat & SMRS_TXBEACON) != 0)
		teprf(sc, "transmitting beacon\n");
	if ((ringstat & SMRS_HARDERR) != 0)
		teprf(sc, "receiving or transmitting beacon frames\n");
	if ((ringstat & SMRS_SIGLOSS) != 0)
		teprf(sc, "ring signal lost\n");

	if ((ringstat & (SMRS_REMOVE | SMRS_AUTORMERR)) != 0) {
		struct ifnet *ifp = &sc->te_if;

		teprf(sc, "removed from ring\n");

		/* If bumped due to lobe failure retry, otherwise shut down. */
		if (ifp->if_flags & IFF_UP && (ringstat & SMRS_REMOVE) == 0)
			te_retry(sc);
		else {
			te_halt(sc);
			ifp->if_flags &= ~IFF_UP;
		}
	}
}

/*
 * Transition to TE_INRING state
 */
void
te_inring(sc)
	te_softc_t *sc;
{
	struct ifnet *ifp = &sc->te_if;

	sc->te_backoff = 1;
	sc->te_state = TE_INRING;
	ifp->if_timer = 0;
	ifp->if_flags |= IFF_RUNNING;
	if (((ifp->if_flags ^ sc->te_oflags) &
	    (IFF_UP | IFF_PROMISC | IFF_LINK1 | IFF_LINK2)) != 0)
		/* Something changed, get off ring and maybe try again */
		te_close(sc);
	else {
		teprf(sc, "inserted in ring\n");
		/* XXX should really do an arprequest() here to detect 
		       dup addrs */
		testart(ifp);
	}
}

#ifdef MAC3DEBUG
void
te_mac3state(sc, iword)
	te_softc_t *sc;
	int iword;
{
	char *st;

	switch (ISB_SUBTYPE(iword)) {
	case 0: st = "BYPASS"; break;
	case 1: st = "INSERTED"; break;
	case 2: st = "INITIALIZE"; break;
	case 3: st = "TX_CL_TK"; break;
	case 4: st = "STANDBY"; break;
	case 5: st = "TX_BEACON"; break;
	case 6: st = "ACTIVE"; break;
	case 7: st = "TX_PURGE"; break;
	}
	teprf(sc, "Mac state change to %s\n", st);
}
#endif

/* 
 * Transition to TE_RETRY state (soft error)
 */
void
te_retry(sc)
	te_softc_t *sc;
{
	struct ifnet *ifp = &sc->te_if;

	TR("te_retry", sc->te_state, sc->te_backoff);
	te_init584(sc);			/* Halts the TRC */
	sc->te_state = TE_RETRY;
	ifp->if_flags &= ~(IFF_RUNNING | IFF_OACTIVE);
	ifp->if_timer = sc->te_backoff;	/* Schedule retry */
	teprf(sc, "retrying initialization in %d seconds\n", sc->te_backoff);
}

/*
 * Close the interface, reopen/halt/retry based on interface flags.
 */
void
te_close(sc)
	te_softc_t *sc;
{
	volatile cb_t *acb = sc->te_acb;
	struct ifnet *ifp = &sc->te_if;
	int rc;

	/* Attempt to remove from ring nicely */
	acb->pcmd = SMCB_REMOVE;
	rc = te_syncacb(sc);

	te_halt(sc);
	if (ifp->if_flags & IFF_UP)
		/* Reopen or retry based on status of REMOVE command */
		if (rc)
			te_retry(sc);
		else
			teinit(ifp->if_unit);
	return;
}

/*
 * Stop TRC and reset internal state, this is the transition to the TE_HALT
 * state.
 */
void
te_halt(sc)
	te_softc_t *sc;
{
	struct ifnet *ifp = &sc->te_if;

	TR("te_halt", sc->te_state, 0);
	te_init584(sc);
	sc->te_backoff = 1;
	sc->te_state = TE_HALT;
	ifp->if_timer = 0;
	ifp->if_flags &= ~(IFF_RUNNING | IFF_OACTIVE);
	if_qflush(&ifp->if_snd);
}

/*
 * Watchdog timer expiration
 */
int
tewatchdog(unit)
	int unit;
{
	te_softc_t *sc = tecd.cd_devs[unit];
	struct ifnet *ifp = &sc->te_if;

	TR("watchdog", sc->te_state, 0);

	switch (sc->te_state) {
	case TE_WINIT:
	case TE_WINRING:
		teprf(sc, "TRC timeout, state=%d\n", sc->te_state);
		if (ifp->if_flags & IFF_UP)
			te_retry(sc);
		else
			te_halt(sc);
		break;

	case TE_RETRY:
		/*
		 * Start a new retry
		 */
		if ((ifp->if_flags & IFF_UP) == 0) {
			te_halt(sc);
			return;
		}
		sc->te_backoff <<= 1;
		if (sc->te_backoff > TE_RETRY_CAP)
			sc->te_backoff = TE_RETRY_CAP;
		teinit(sc->te_dev.dv_unit);
		break;

	default:
		teprf(sc, "invalid watchdog state %d\n", sc->te_state);
		panic("tewatchdog: invalid state");
	}
}

/*
 * Run an ACB synchronously (w/timeout of course). Assumes CB is filled
 * in from cmd field down, returns 0 (w/o interrupt indication) if command
 * succeeds, status is left in CB.
 */
int
te_syncacb(sc)
	te_softc_t *sc;
{
	volatile cb_t *acb = sc->te_acb;
	scb_t *scb = sc->te_scb;
	int timer;

	acb->stat = 0;

	TR("syncacb: cmd/eap", acb->pcmd, ADGET(acb->eap));

	/* Issue the command */
	if (te_waitcbusy(sc, "te_syncacb: TRC hung\n"))
		return (1);
	scb->resctl = SMRES_CB;
	scb->ctl = SMCTL_SV | SMCTL_RV | SMCTL_NOP;
	te_hitca(sc);

	/* Wait for completion */
	timer = ACB_TIMEOUT;
	while (timer--) {
		if (acb->stat & SMST_DONE)
			break;
		DELAY(1);
	}
	if ((acb->stat & SMST_DONE) == 0) {
		teprf(sc, "sync cmd timeout, cmd=0x%x\n", acb->pcmd);
		return (1);
	}
	if ((acb->stat & SMST_MASK) != 0) {
		teprf(sc,"sync cmd error, cmd=0x%x stat=0x%x\n",
		    acb->pcmd, acb->stat);
	}
	return (acb->stat & SMST_MASK);
}

/*
 * Process an ioctl request.
 */
int
teioctl(ifp, cmd, data)
	struct ifnet *ifp;
	int cmd;
	caddr_t data;
{
	struct ifaddr *ifa = (struct ifaddr *)data;
	te_softc_t *sc = tecd.cd_devs[ifp->if_unit];
	struct proc *p = curproc;		/* XXX */
	int s;
	int rc;

	rc = 0;
	s = splimp();
	switch (cmd) {
	case SIOCSIFADDR:
		ifp->if_flags |= IFF_UP;
		switch (ifa->ifa_addr->sa_family) {
#ifdef INET
		case AF_INET:
			teinit(ifp->if_unit);
			arp_ifinit((struct arpcom *)ifp, ifa);
			break;
#endif

		default:
			teinit(ifp->if_unit);
			break;
		}
		break;

	case SIOCSIFFLAGS:
		if (ifp->if_flags & IFF_RUNNING) {
			if (((ifp->if_flags ^ sc->te_oflags) &
			    (IFF_UP|IFF_PROMISC|IFF_LINK1|IFF_LINK2)) != 0)
				te_close(sc);
			sc->te_oflags = ifp->if_flags;
		} else {
			if (ifp->if_flags & IFF_UP)
				teinit(ifp->if_unit);
		}
		break;

	case TEIOCUCODE:
		if (rc = suser(p->p_ucred, &p->p_acflag))
			break;
		splx(s);
		return (te_ucode(sc, (struct ifreq *)data));

	case TEIOCETRON:
	case TEIOCETROFF: {
		int oflags = sc->te_flags;

		if (rc = suser(p->p_ucred, &p->p_acflag))
			break;

		if (cmd == TEIOCETRON)
			sc->te_flags |= TE_ETR;
		else
			sc->te_flags &= ~TE_ETR;
		if ((sc->te_flags != oflags) && (ifp->if_flags & IFF_RUNNING))
			te_close(sc);
		break;
	}

#ifdef TEDEBUG
	case TEIOCSHGET:
		splx(s);
		if (rc = suser(p->p_ucred, &p->p_acflag))
			break;
		return (te_shget(sc, (struct ifreq *)data));
		break;
#endif

	default:
		rc = EINVAL;
		break;
	}
	splx(s);
	return (rc);
}

/*
 * Load/verify board microcode from user space buffer, this is only
 * possible when the board is in the TE_HALT state.
 *
 * Success clears the TE_NO_UCODE flag and allows the interface to
 * initialize.
 */
int
te_ucode(sc, ifr)
	te_softc_t *sc;
	struct ifreq *ifr;
{
	int rc;
	int s;
	caddr_t ucbuf;

	if (sc->te_state != TE_HALT)
		return (EBUSY);

	MALLOC(ucbuf, caddr_t, SM_UCODE_SIZE, M_TEMP, M_WAITOK);
	rc = copyin(ifr->ifr_data, ucbuf, SM_UCODE_SIZE);
	s = splimp();
	if (rc == 0) {
		/* Make control store visible in the shared memory region */
		te_pg0(sc);
		te_ramon(sc);
		outb(sc->te_iobase + SMIO_MSR, sc->te_msr | SMMSR_RST);
		outb(sc->te_ctl, 0);			/* clear WCSS */
		bcopy(ucbuf, sc->te_vaddr, SM_UCODE_SIZE);
		if (bcmp(ucbuf, sc->te_vaddr, SM_UCODE_SIZE))
			rc = EIO;
		outb(sc->te_ctl, SMCTL_CLRTINT | SMCTL_CLRCBUSY | SMCTL_OPER);
	}
	FREE(ucbuf, M_TEMP);
	te_ucksum(sc);
	te_ramoff(sc);
	splx(s);
	return (rc);
}

/*
 * Checksum microcode, set/clear the TE_NO_UCODE flag based on results.
 *
 * Warning: this leaves the TRC in reset.
 */
void
te_ucksum(sc)
	te_softc_t *sc;
{
	int s = splimp();
	u_short *pe = (u_short *)(sc->te_vaddr + SM_UCODE_SIZE);
	u_short csum;
	u_short *p;

	csum = 0;
	p = (u_short *)sc->te_vaddr;
	te_pg0(sc);
	outb(sc->te_iobase + SMIO_MSR, sc->te_msr | SMMSR_RST);
	outb(sc->te_ctl, 0);			/* clear WCSS */
	while (p < pe)
		csum += *p++;
	outb(sc->te_ctl, SMCTL_CLRTINT | SMCTL_CLRCBUSY | SMCTL_OPER);
	splx(s);

	if (csum)
		sc->te_flags |= TE_NO_UCODE;
	else
		sc->te_flags &= ~TE_NO_UCODE;
}

/*
 * Print a message preceeded by the interface name
 */
void
#if __STDC__
teprf(te_softc_t *sc, char *fmt, ...)
#else
teprf(sc, fmt, va_alist)
	te_softc_t *sc;
	char *fmt;
	va_dcl;
#endif
{
	va_list ap;
#if __STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	printf("%s%d: %r", sc->te_if.if_name, sc->te_if.if_unit, fmt, ap);
	va_end(ap);
}

#ifdef TEDEBUG
/*
 * Copy shared memory to user buffer
 */
int
te_shget(sc, ifr)
	te_softc_t *sc;
	struct ifreq *ifr;
{
	caddr_t p;
	int pgbase;
	int rc;
	caddr_t	ubuf;

	rc = 0;
	ubuf = ifr->ifr_data;
	for (pgbase = 0; pgbase < SM_RSIZE; pgbase += SM_WSIZE) {
		p = te_pgmem(sc, pgbase);
		if ((rc = copyout(p, ubuf, SM_WSIZE)) != 0)
			break;
		ubuf += SM_WSIZE;
	}
	te_pg0(sc);
	return (rc);
}
#endif
