/*-
 * Copyright (c) 1993, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: if_rh.c,v 2.4 1995/12/12 22:19:54 cp Exp $
 *
 * Modifyed by Aleksei Rudnev, Relcom.EU.net <alex@kiae.su>
 * I add chain_buffer for receive, reset in case of overflow,
 * reset status bits in TX interrupt, and some other things (DTR, modem control
 * is turned on by LINK2 flag).
 * Driver tested on BSDI/based router with 4 HDLC lines, and 1 Ethernet,
 * under heavy load.
 * It is good idea to use chain_buffers on write, but I think it is not
 * as critical as on read. But I hope i'll do it later. Just as error accounting
 * (as we have in if_rn driver).
 *
 * There is one little bug yet - I check if array of receive descriptors
 * are on 64K bit boundary, but don't realloc this array. But I think nobody
 * would see this message, though it is not difficult to add code for this
 * reallocation.
 */

/*
 * SDL Communications RISCom/H2 dual sync/async serial port driver
 *
 * Currently the external clock source and 16 bit HDLC checksum
 * are hardwired; i.e. there is no need to configure the driver
 * to support different line speeds.
 */

/*
 * TODO:
 * - add support for internal clock source and
 *   alternative HDLC checksum
 * - do something about carrier detect etc
 */

#undef  RHDEBUG
#undef  RHDEBUG_X

/* #define RH_PAD  32      * Pad too short packets to 32 bytes */
#define RH_MAXINTR      100

#include <sys/param.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/syslog.h>
#include <sys/device.h>

#if INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>		/* XXX for slcompress/if_p2p.h */
#endif

#include <net/if.h>
#include <net/netisr.h>
#include <net/if_types.h>
#include <net/if_p2p.h>
#include <net/if_c_hdlc.h>

#include <i386/isa/icu.h>
#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/isa/ic/hd64570.h>

#include "if_rhreg.h"

#define	RH_DEFAULT_RATE	56000   /* 56Kbps */
#define	RH_TIMER	10
#define	RH_SLOWTIMER	30
#define	RH_MAXTIMER	600
#define	NRXBUF		8
#define	RXBUFSIZE	(1024 * 4)

#undef dprintf
#ifdef RHDEBUG
#define dprintf(x)      printf x
#ifdef RHDEBUG_X
int _rinb(b, r, s)
{
	int a = RH_SCA(r) + b + 0x8000;
	int v = inb(a);
	printf(" {%x<-%s %x}", v, s, a);
	return(v);
}

void _routb(b, r, v, s)
{
	int a = RH_SCA(r) + b + 0x8000;
	printf(" {%s %x<-%x}", s, a, v & 0xff);
	outb(a, v);
}

#undef rinb
#undef routb
#define rinb(b, r) _rinb(b, r, #r)
#define routb(b, r, v) _routb(b, r, v, #r)
#endif
#else
#define dprintf(x)
#endif

/*
 * Tuneable parameters
 */
#define NTXFRAG 16              /* the max number of TX fragments */

struct rh_softc {
	struct	device sc_dev;		/* base device */
	struct	isadev sc_id;		/* ISA device */
	struct 	intrhand sc_ih;		/* interrupt vectoring */
	struct  p2pcom sc_p2pcom;       /* point-to-point common stuff */
#define	sc_if	sc_p2pcom.p2p_if	/* network-visible interface */	

	struct rh_softc *sc_nextchan;   /* next channel's rh_softc */

	u_short sc_addr;                /* base i/o port */
	u_short sc_msci;                /* channel-adjusted MSCI address */
	u_short sc_dma;                 /* channel-adjusted DMAC address */
	u_short sc_drq;                 /* AT DMA channel */
	int     sc_rate;                /* baud rate on channel */

	int     sc_txphys;              /* physical address of sc_txd */
	int     sc_txbphys;             /* physical address of sc_txbuf */
	struct  mbuf *sc_txtop;         /* the header of the transmitted buffer */
	struct  hd_cbd sc_txd[NTXFRAG]; /* TX buffer descriptors */
	struct  hd_cbd sc_rxd[NRXBUF];  /* RX buffer descriptor */
	char    sc_txbuf[(HDLCMTU + 8) * 2];  /* TX DMA bounce buffer */

	short   sc_nrxc, sc_nrxe;       /* Current rx buf, end rx buf */
	int     sc_rxphys;              /* physical address of sc_rxd */
	int     sc_rxbphys;             /* physical address of sc_rxbuf */
	char    sc_rxbuf[RXBUFSIZE * NRXBUF];  /* RX buffer space */

	int sc_slowtimer;		/* Timer */
};
#define RXD_A(sc, n) ((sc)->sc_rxphys + (n) * (sizeof(struct hd_cbd)))

int     rhprobe __P((struct device *, struct cfdata *, void *));
void    rhforceintr __P((void *));
int     rhwatchdog __P((int));
void    rhattach __P((struct device *, struct device *, void *));
int     rhioctl __P((struct ifnet *, int, caddr_t));
int     rhintr __P((struct rh_softc *));
int     rhstart __P((struct ifnet *));
int     rhmdmctl __P((struct p2pcom *, int));

static void rhrinit __P((struct rh_softc *));
static void rhtinit __P((struct rh_softc *));
static void rheanble __P((struct rh_softc *));
static void rhdisable __P((struct rh_softc *));
static void rhread __P((struct rh_softc *, char *, int));

struct cfdriver rhcd =
	{ NULL, "rh", rhprobe, rhattach, DV_IFNET, sizeof(struct rh_softc) };
 
/*
 * Probe routine
 */
rhprobe(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	register struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	int i, v1, v2;

	/*
	 * Check configuration parameters
	 */
	if (!RH_IOBASEVALID(ia->ia_iobase)) {
		printf("rh%d: illegal base address %x\n", cf->cf_unit,
		    ia->ia_iobase);
		return (0);
	}
	if (!RH_DRQVALID(ia->ia_drq)) {
		printf("rh%d: illegal drq %d\n", cf->cf_unit, ia->ia_drq);
		return (0);
	}
	if (ia->ia_irq != IRQUNK) {
		i = ffs(ia->ia_irq) - 1;
		if (!RH_IRQVALID(i)) {
			printf("rh%d: illegal IRQ %d\n", cf->cf_unit, i);
			return (0);
		}
	}

	/*
	 * Check if card is present
	 */
	/* Reset SCA */
	outb(ia->ia_iobase, 0);
	DELAY(10);
	outb(ia->ia_iobase, RHC_RUN);
	DELAY(5);
	if (inb(ia->ia_iobase) != RHC_RUN)
		return (0);
	routb(ia->ia_iobase, TEPR, 0xff);
	if (rinb(ia->ia_iobase, TEPR) != 0x7)
		return (0);
	routb(ia->ia_iobase, TEPR, 0);
	if (rinb(ia->ia_iobase, TEPR) != 0)
		return (0);

	/*
	 * Automagic irq detection
	 */
	if (ia->ia_irq == IRQUNK)
		ia->ia_irq = isa_discoverintr(rhforceintr, (void *) ia);

	/* Reset SCA */
	outb(ia->ia_iobase, 0);

	if (ffs(ia->ia_irq) - 1 == 0)
		return (0);
	ia->ia_iosize = RH_NPORT;
	return (1);
}

static void
rhforceintr(aux)
	void *aux;
{
	register struct isa_attach_args *ia = (struct isa_attach_args *) aux;

	/* Enable interrupt on tx buffer empty (it sure is) */
	outb(ia->ia_iobase, RHC_RUN | RHC_DMAEN);
	routb(ia->ia_iobase, IE0, ST0_TXRDY);
	routb(ia->ia_iobase, IER0, ISR0_TXRDY0);
	routb(ia->ia_iobase, TRC0, 32);
	routb(ia->ia_iobase, CMD, CMD_TXENB);
}

/*
 * Interface exists: make available by filling in network interface
 * records.  System will initialize the interface when it is ready
 * to accept packets.
 */
void
rhattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct rh_softc *sc = (struct rh_softc *) self;
	struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	struct ifnet *ifp;
	struct rh_softc *csc;
	int i;

	/*
	 * Allow SCA operation
	 */
	DELAY(5);
	outb(ia->ia_iobase, RHC_RUN);
 
	/*
	 * Load the wait controller registers
	 */
	routb(ia->ia_iobase, WCRL, 0);
	routb(ia->ia_iobase, WCRM, 0);
	routb(ia->ia_iobase, WCRH, 0);

	/*
	 * Initialize DMA
	 */
	at_dma_cascade(ia->ia_drq);
	routb(ia->ia_iobase, PCR, PCR_BRC | PCR_ROT);
	routb(ia->ia_iobase, DMER, DMER_DME);
	outb(ia->ia_iobase, RHC_RUN | RHC_DMAEN);

	/*
	 * Initialize interrupts
	 */
	routb(ia->ia_iobase, IER0, ISR0_TXINT0 | ISR0_TXINT1);
	routb(ia->ia_iobase, IER1, ISR1_DMIA0R | ISR1_DMIB0R |
				   ISR1_DMIA1R | ISR1_DMIB1R);

	/*
	 * Initialize interface data
	 */

	aprint_naive(": RISCom/H2");
	printf("\n");
	for (i = 0, csc = sc; i < RH_NCHANS; i++) {
		if (i) {
			csc = (struct rh_softc *) malloc(
				sizeof(struct rh_softc), M_DEVBUF, M_NOWAIT);
			if (csc == 0) {
				printf("rh%d: no memory for the channel 1 data",
					sc->sc_dev.dv_unit);
				return;
			}
			bzero((caddr_t) csc, sizeof(struct rh_softc));
			sc->sc_nextchan = csc;
		}

		csc->sc_nextchan = 0;
		csc->sc_addr = ia->ia_iobase;
		csc->sc_msci = ia->ia_iobase + RH_SCA(MCHAN * i);
		csc->sc_dma = ia->ia_iobase + RH_SCA(CHAN * i);
		csc->sc_drq = ia->ia_drq;
		csc->sc_rate = RH_DEFAULT_RATE;

		/* Calculate addresses */
		csc->sc_txphys = kvtop((caddr_t) sc->sc_txd);
		csc->sc_txbphys = kvtop((caddr_t) sc->sc_txbuf);
		csc->sc_rxphys = kvtop((caddr_t) &sc->sc_rxd);
		csc->sc_rxbphys = kvtop((caddr_t) sc->sc_rxbuf);
		if (((sc->sc_rxphys) >> 16) != (kvtop((caddr_t) &sc->sc_rxd +
		    NRXBUF * sizeof sc->sc_rxd) >> 16)) {
		       printf("ERROR, receive RH chain crosses 64K boundary.n");
		       return;
		}

		ifp = &csc->sc_if;
		ifp->if_unit = (sc->sc_dev.dv_unit * RH_NCHANS) + i;
		ifp->if_name = "rh";
		ifp->if_mtu = HDLCMTU;
		ifp->if_flags = IFF_POINTOPOINT | IFF_MULTICAST;
		ifp->if_type = IFT_NONE;
		ifp->if_ioctl = p2p_ioctl;
		/* ifp->if_output = 0;	crash and burn on early access */
		ifp->if_watchdog = rhwatchdog;
		ifp->if_start = rhstart;
		csc->sc_p2pcom.p2p_mdmctl = rhmdmctl;
		rhrinit(sc);
		rhtinit(sc);
		if_attach(ifp);
		p2p_attach(&csc->sc_p2pcom);
	}

	/* Register the interrupt handler and isa device */
        isa_establish(&sc->sc_id, &sc->sc_dev);
	sc->sc_ih.ih_fun = rhintr;
        sc->sc_ih.ih_arg = (void *)sc;
        intr_establish(ia->ia_irq, &sc->sc_ih, DV_NET);
}


/*
 * Initialize the channel transmit structures
 */
static void
rhtinit(sc)
	register struct rh_softc *sc;
{
	register dma  = sc->sc_dma;
	struct hd_cbd *d;
	int i, j;

	j = sc->sc_txphys + sizeof(struct hd_cbd);
	for (d = sc->sc_txd, i = 0; i < NTXFRAG ; i++, d++) {
		d->cbd_next = j;
		d->cbd_stat = ST2_EOT;
		j += sizeof(struct hd_cbd);
	}
}

/*
 * Initialize the channel receive structures
 */
static void
rhrinit(sc)
	register struct rh_softc *sc;
{
	register dma  = sc->sc_dma;
	struct hd_cbd *d;
	int i, j;

	/* Stop RX DMA, and init internal registers  */
	routb(dma, DSR, 0);
	routb(dma, DCR, DCR_ABORT);
	DELAY(1);
	/* Clear DMA status */
	routb(dma, DSR, rinb(dma,DSR));

	/* Clear DMA channels, and set-up chain mode */
	routb(dma, DMR, 0);
	routb(dma, DMR, DMR_TMOD|DMR_NF);

	/* Build RX buffer chain (ring) */
	j = sc->sc_rxphys;
	for (d = sc->sc_rxd, i = 0; i < NRXBUF; i++, d++) {
		d->cbd_next = RXD_A(sc, ((i + 1) % NRXBUF));
		d->cbd_stat = 0;
		d->cbd_len = 0;
		d->cbd_bp0 = (sc->sc_rxbphys + i * RXBUFSIZE);
		d->cbd_bp1 = (sc->sc_rxbphys + i * RXBUFSIZE) >> 16;
		dprintf(("i %d %x %x\n",i,RXD_A(sc,((i+1)%NRXBUF)),(sc->sc_rxbphys + i*RXBUFSIZE)));
	}
	sc->sc_nrxc = 0;
	sc->sc_nrxe = NRXBUF - 1;
	routb(dma, BFLL, RXBUFSIZE);
	routb(dma, BFLH, (RXBUFSIZE) >> 8);
	routb(dma, EDAL, (RXD_A(sc,sc->sc_nrxe))     );
	routb(dma, EDAH, (RXD_A(sc,sc->sc_nrxe)) >> 8);
	routb(dma, CDAL, (RXD_A(sc,sc->sc_nrxc))     );
	routb(dma, CDAH, (RXD_A(sc,sc->sc_nrxc)) >> 8);
	routb(dma, CPB,  (RXD_A(sc,sc->sc_nrxc)) >> 16);
	dprintf(("CDA=%x EDA=%x\n",RXD_A(sc,sc->sc_nrxc),RXD_A(sc,sc->sc_nrxe)));
}

/*
 * Set transmission parameters and enable the channel
 */
static void
rhenable(sc)
	register struct rh_softc *sc;
{
	register msci = sc->sc_msci;
	register dma  = sc->sc_dma;
	int br, div;
	int chan =  sc->sc_if.if_unit % RH_NCHANS;
#ifdef notyet
	int loop = 0, intsync = 0;
#endif

	dprintf(("enable unit %d:", sc->sc_if.if_unit));
	if (!(sc->sc_if.if_flags & (IFF_UP|IFF_RUNNING)))
	       return;
	/* We have to ENABLE any unit, even in _down_ state - crazy chip! alex */
	routb(msci, CMD, CMD_TXRST);
	routb(msci, CMD, CMD_ABORT);
	/* Reset the channel */
	routb(msci, CMD, CMD_CHRST);
	DELAY(5);
	/* Idle fill pattern -- flag character */
	routb(msci, IDL, 0x7e);

	routb(dma, DMR+TX, 0);
	routb(dma, DCR+TX, DCR_ABORT);

	/* Initialize channel DMA mode registers */
	routb(dma, DMR+TX, DMR_TMOD | DMR_NF);

	/* HDLC, standard CRC */
	routb(msci, MD0, MD0_HDLC | MD0_CRCENB | MD0_CRCINIT | MD0_CRCCCITT |
	    ((sc->sc_if.if_flags & IFF_LINK1) ? MD0_AUTO : 0));

	/* No address field checks */
	routb(msci, MD1, MD1_NOADR);

	/* NRZ, full duplex */
#ifdef notyet
	routb(msci, MD2, MD2_NRZ | (loop ? MD2_LOOPBACK : MD2_DUPLEX));
#else
	routb(msci, MD2, MD2_NRZ | MD2_DUPLEX);
#endif

	/* Reset RX to validate new modes */
	routb(msci, CMD, CMD_RXRST);

	/* Set FIFO depths */
	routb(msci, RRC, 0);    /* start dma immediately */
#ifdef old
	routb(msci, TRC0, 32);  /* lowat */
	routb(msci, TRC1, 63);  /* hiwat */
#else
	routb(msci, TRC0, 7);   /* lowat */
	routb(msci, TRC1, 15);  /* hiwat */
#endif

	/* Load the baud rate generator registers */
	div = RH_OSC_FREQ / sc->sc_rate;
	if (div == 0)
		div = 1;
	br = 0;
	if (div > 10 || !(div & 01)) {  /* avoid br=0 -- it yields duty ratio != 0.5 */
		do {
			br++;
			div /= 2;
		} while (div > 127);
	}
	dprintf((" baud rate=%d -> div=%d br=%d\n", sc->sc_rate, div, br));
	routb(msci, TMC, div);

	/* External clock (shouldn't be hardwired...) */
#ifdef notyet
	if (!intsync) {
#endif
	       routb(msci, RXS, RXS_LINE | br);
	       routb(msci, TXS, TXS_LINE | br);
#ifdef notyet
	} else {
	       routb(msci, RXS, RXS_BRG | br);
	       routb(msci, TXS, TXS_BRG | br);
       }
#endif

	rhtinit(sc);   /* NO dma tx registers setup here */
	rhrinit(sc);   /* But receive dma registeres are loaded here */

	/* Enable SCA transmitter and receiver */
	routb(msci, CMD, CMD_TXENB);
	routb(msci, CMD, CMD_RXENB);

	/* Start RX DMA, and clean DSR bits */
	routb(dma, DSR, DSR_DE | DSR_EOT | DSR_EOM | DSR_BOF );

	/* Enable RX interrupts from the channel */
	routb(dma, DIR, DSR_EOT | DSR_BOF | DSR_EOM);
	routb(msci, IE0, ST0_TXINT);

	/* Enable on-board trasmitter and raise DTR */
	br = inb(sc->sc_addr);
	if (sc->sc_if.if_unit % RH_NCHANS) {
		br &= ~(RHC_DTR1 | RHC_EOUT1);
#ifdef notyet
		br |= RHC_TE1 | (intsync ? RHC_EOUT1 : 0);
#else
		br |= RHC_TE1;
#endif
	} else {
		br &= ~(RHC_DTR0 | RHC_EOUT0);
#ifdef notyet
		br |= RHC_TE0 | (intsync ? RHC_EOUT0 : 0);
#else
		br |= RHC_TE0;
#endif
	}
	outb(sc->sc_addr, br);

	/* Raise RTS and start transmission of the idle pattern */
	routb(msci, CTL, CTL_IDLC);

	/* Kickstart output */
	sc->sc_txtop = 0;
	sc->sc_if.if_flags &= ~IFF_OACTIVE;
	sc->sc_if.if_timer = 0;
	(void) rhstart(&sc->sc_if);
}

int
rh_start(unit)
	int unit;
{
	struct rh_softc *sc0 = rhcd.cd_devs[unit / RH_NCHANS];
	struct rh_softc *sc = (unit % RH_NCHANS) ?  sc0->sc_nextchan : sc0;

	dprintf(("rh%d restarted\n",unit));
	(void) rhstart(&sc->sc_if);
}

/*
 * Start output
 */
int
rhstart(ifp)
	register struct ifnet *ifp;
{
	struct rh_softc *sc0 = rhcd.cd_devs[ifp->if_unit / RH_NCHANS];
	struct rh_softc *sc = (ifp->if_unit % RH_NCHANS)?
					sc0->sc_nextchan : sc0;
	int cnt, phys, bflag;
	int bounce, totlen;
	register struct mbuf *m;
	register struct hd_cbd *d;

	/* Is line alive? If not - don't start, wait 1 second */
	if (!(rinb(sc->sc_msci, ST1) & ST1_FLGD)) {
	       ifp->if_watchdog = rh_start;
	       ifp->if_timer = 1;
	       return;
	};
	/*
	 * Check if we have an available outgoing frame header and
	 * there is a sufficient space in TX output area.
	 */
	IF_DEQUEUE(&sc->sc_p2pcom.p2p_isnd, m);
	if (m == 0)
		IF_DEQUEUE(&sc->sc_if.if_snd, m);
	if (m == 0) {
		dprintf((" EMPTY TX Q\n"));
		return (0);
	}
	ifp->if_flags |= IFF_OACTIVE;
	ifp->if_watchdog = rhwatchdog;
	ifp->if_timer = RH_TIMER + sc->sc_slowtimer;
	dprintf(("TX%d: buf=%x ", ifp->if_unit, m));

	/*
	 * Fill in the descriptors
	 */
	sc->sc_txtop = m;
	bounce = 0;
	bflag = 0;
	totlen = 0;
	cnt = 0;
	d = sc->sc_txd;
scan:   for (; m != 0; cnt++, m = m->m_next) {
		if (m->m_len == 0)
			continue;
		totlen += m->m_len;

		phys = ~0;      /* all 1s */
		if (cnt < NTXFRAG-1)
			phys = kvtop(m->m_data);

		/*
		 * Can't do DMA directly from mbuf?
		 */
		if (phys >= ISA_MAXADDR) {
			dprintf((" %x(%d)->", phys, m->m_len));
			phys = sc->sc_txbphys + bounce;
			if (bounce + m->m_len > sizeof sc->sc_txbuf)
				break;  /* shouldn't happen */
			bcopy(mtod(m, caddr_t), &sc->sc_txbuf[bounce], m->m_len);
			bounce += m->m_len;
			if (bflag) {
				d[-1].cbd_len += m->m_len;
				continue;
			}
			bflag = 2;
		}
		bflag >>= 1;

		/*
		 * Set up the TX buffer descriptor
		 */
		dprintf((" [%d:%d@%x]", cnt, m->m_len, phys));
		d->cbd_bp0 = phys;
		d->cbd_bp1 = phys >> 16;
		d->cbd_len = m->m_len;
		d->cbd_stat = 0;
		d++;
	}
#ifdef RH_PAD
	if (totlen < RH_PAD) {
		static struct mbuf pad;

		pad.m_data = pad.m_dat;
		pad.m_len = RH_PAD - totlen;
		m = &pad;       /* pad.m_next == 0 */
		goto scan;
	}
#endif /* RH_PAD */

	d[-1].cbd_stat = ST2_EOM;       /* End of frame */
	dprintf(("\n"));

	/*
	 * Start output
	 */
	phys = sc->sc_txphys + (char *)d - (char *)(sc->sc_txd);
	routb(sc->sc_dma, EDAL+TX, phys);
	routb(sc->sc_dma, EDAH+TX, phys >> 8);
	routb(sc->sc_dma, CDAL+TX, sc->sc_txphys);
	routb(sc->sc_dma, CDAH+TX, sc->sc_txphys >> 8);
	routb(sc->sc_dma, CPB+TX, sc->sc_txphys >> 16);
	routb(sc->sc_dma, DSR+TX, DSR_DE);

	/*
	 * Enable interrupt on entering the TX idle state
	 * The SCA manual says we should be able to start the
	 * new transmission immediately on completion of
	 * the previous DMA. In practice, it does not allow
	 * enough time for the final flag recognition
	 * by stupid (for example cisco's) HDLC controllers --
	 * they think the first packet was corrupted.
	 *
	 * Apparently we always need to send two flags between
	 * HDLC frames.
	 */
	DELAY(1); /* To prevent IDL interrupt before DMA started */
	routb(sc->sc_msci, IE1, ST1_IDL);
	return (0);
}

/*
 * Interrupt routine
 */
rhintr(sc0)
	register struct rh_softc *sc0;
{
	register struct rh_softc *sc;
	u_char st;
	int len, count;
	int nn, nrcv;

	(void) inb(sc0->sc_addr + 4);   /* For REV A cards */
	dprintf(("I "));
	count = RH_MAXINTR;
loop:
	/*
	 * Packet received?
	 */
	if (st = rinb(sc0->sc_addr, ISR1)) {
		sc = sc0;
		if ((st & (ISR1_DMIA0R | ISR1_DMIB0R)) == 0)
			sc = sc->sc_nextchan;

		/*  Analyze DMA status */
		st = rinb(sc->sc_dma, DSR);
		dprintf(("RX%d %x", sc->sc_if.if_unit,st));
		st &= DSR_EOT | DSR_EOM | DSR_BOF | DSR_COF;
		if (st & (DSR_BOF | DSR_COF)) {
			sc->sc_if.if_ierrors++;
			dprintf(("-dmaerr %x", st));
			sc->sc_if.if_collisions += 1;
			printf("rh%d rcv ovfl, stat=%x\n",sc->sc_if.if_unit,st);
			rhrinit(sc);
			st = 0;
			goto new_rcv;
		}
#if I_WANT_TO_IMITATE_OVERFLOW
		if (sc->sc_if.if_flags & IFF_LINK2 && sc->sc_if.if_opackets % 20 == 0 || sc->sc_if.if_opackets % 20 == 2) {
		       dprintf(("DELAY started..."));
		       for(nrcv = 100;nrcv>0;nrcv--) DELAY(50000);
		       dprintf(("...\n"));
	       }
#endif
		nrcv = 0;
		for (nn = sc->sc_nrxc; nn != sc->sc_nrxe;
		    nn = (++nn % NRXBUF)) {
			u_char c_l,c_h;

			c_l = rinb(sc->sc_dma,CDAL);
			c_h = rinb(sc->sc_dma,CDAH);
			dprintf(("nn=%d stat=%x len=%d ",nn,sc->sc_rxd[nn].cbd_stat,sc->sc_rxd[nn].cbd_len));
/*                        dprintf((" c_l=%x c_h=%x (%x %x)\n",c_l,c_h,(u_char)(RXD_A(sc,nn)&0377),(u_char)((RXD_A(sc,nn)>>8)&0377)));      */
			if (c_l == (u_char)(RXD_A(sc,nn)) &&
			     c_h == (u_char)(RXD_A(sc,nn) >> 8))
			       break;  /* No more completed buffers */

			if (!sc->sc_rxd[nn].cbd_stat) {
			       dprintf((" Filled buffer with status == 0!\n"));
			       goto next;
			}
			nrcv++;
			/* Analyse frame status */
			if ((sc->sc_rxd[nn].cbd_stat & (ST2_EOM)) == 0) {
				dprintf(("-frame buffer overflow"));
				sc->sc_if.if_collisions += 1;
				sc->sc_if.if_ierrors++;
				printf("rh%d buffer ovfl, stat=%x\n",
				    sc->sc_if.if_unit, sc->sc_rxd[nn].cbd_stat);
				rhrinit(sc);
				st = 0;
				goto new_rcv;
			} else if (sc->sc_rxd[nn].cbd_stat &
			   (ST2_SHRT|ST2_ABT|ST2_FRME|ST2_OVRN|ST2_CRCE)) {
				dprintf(("-frmerr %x", st));
				sc->sc_if.if_ierrors++;
			} else {
				len = sc->sc_rxd[nn].cbd_len;
				dprintf(("-len=%d\n", len));
#ifdef RHDEBUG_X
			       {
				       char *p;
				       int i;

				       i = len;
				       if (i >= (79/3))
					       i = 79/3;
				       p = sc->sc_rxbuf+nn*RXBUFSIZE;
				       while (i--)
					       printf("%02x ", *p++ & 0xff);
				       printf("\n");
			       }
#endif /* RHDEBUG_X */
			       rhread(sc, sc->sc_rxbuf + nn * RXBUFSIZE, len);
			       sc->sc_if.if_ipackets++;
		       }
next:
		       sc->sc_nrxc = (nn + 1) % NRXBUF;
		       sc->sc_rxd[nn].cbd_len  = 0;
		       sc->sc_rxd[nn].cbd_stat = 0;
	     };
new_rcv:
	     /* Read ALL packets, now'll update counters */
	     sc->sc_nrxe = (sc->sc_nrxc + NRXBUF - 1) % NRXBUF;
	     routb(sc->sc_dma, EDAL, RXD_A(sc, sc->sc_nrxe));
	     routb(sc->sc_dma, EDAH, RXD_A(sc, sc->sc_nrxe) >> 8);
	     /* Clean status and start DMA */
	     routb(sc->sc_dma, DSR, st|DSR_DWD);
	     routb(sc->sc_dma, DSR, DSR_DE );
	     routb(sc->sc_msci, CMD, CMD_RXENB);
	     dprintf(("R %d pkts %d\n",nrcv,sc->sc_if.if_ierrors));
	} else

	/*
	 * Packet transmitted?
	 */
	if (st = rinb(sc0->sc_addr, ISR0)) {
		sc = sc0;
		if ((st & ISR0_TXINT0) == 0)
			sc = sc->sc_nextchan;

		dprintf((" TX%d", sc->sc_if.if_unit));

		/* Clear DMA status */
		st = rinb(sc->sc_dma, DSR+TX);
		routb(sc->sc_dma,DSR+TX,st); /* Clean status */
#ifdef RHDEBUG
		if (!(st & DSR_EOT | DSR_EOM | DSR_BOF | DSR_COF))
		       dprintf((" dsr %x",st));
#endif
		st &= DSR_EOT | DSR_EOM | DSR_BOF | DSR_COF;
		if (st == 0) {
			dprintf(("-false"));
			if (count-- > 0)
				goto loop;
			return (1);
		}
		routb(sc->sc_dma, DSR+TX, st | DSR_DWD);
		sc->sc_if.if_opackets++;
		sc->sc_slowtimer = 0;

		/* Disable TX IDLE intr */
		routb(sc->sc_msci, IE1, 0);

		/* Release buffer */
		if (sc->sc_txtop)
			m_freem(sc->sc_txtop);
		sc->sc_txtop = 0;

		/* Restart transmission */
		sc->sc_if.if_flags &= ~IFF_OACTIVE;
		sc->sc_if.if_timer = 0;
		(void) rhstart(&sc->sc_if);
	} else {

		/* All sources served */
		dprintf(("\n"));
		return (1);
	}
	goto loop;      /* rescan status registers */
}

/*
 * Modem control
 * flag = 0 -- disable line; flag = 1 -- enable line
 */
int
rhmdmctl(pp, flag)
	struct p2pcom *pp;
	int flag;
{
	struct rh_softc *sc0 = rhcd.cd_devs[pp->p2p_if.if_unit / RH_NCHANS];
	struct rh_softc *sc = (pp->p2p_if.if_unit % RH_NCHANS) ?
				sc0->sc_nextchan : sc0;
	int br;
	int s;

	s = splimp();
	if (flag)
		rhenable(sc);
	else {
		dprintf(("disable unit %d:", sc->sc_if.if_unit));

		/* disable TX & RX interrupts */
		routb(sc->sc_dma, DIR, 0);
		routb(sc->sc_msci, IE1, 0);

		/* Halt dma in progress */
		routb(sc->sc_dma, DSR, 0);
		routb(sc->sc_dma, DSR+TX, 0);
		routb(sc->sc_dma, DCR, DCR_ABORT);
		routb(sc->sc_dma, DCR+TX, DCR_ABORT);

		/* disable transmitter and receiver */
		routb(sc->sc_msci, CMD, CMD_RXDIS);
		routb(sc->sc_msci, CMD, CMD_TXDIS);
		routb(sc->sc_msci, CTL, 0);	       /* drop RTS */
		br = inb(sc->sc_addr);	/* drop DTR and disable transmitter */
		if (sc->sc_if.if_unit % RH_NCHANS) {
			br |= RHC_DTR1;
			br &= ~RHC_TE1;
		} else {
			br |= RHC_DTR0;
			br &= ~RHC_TE0;
		}
		outb(sc->sc_addr, br);

		/* Release tx buffers */
		dprintf(("release txtop %x --", sc->sc_txtop));
		if (sc->sc_txtop)
			m_freem(sc->sc_txtop);
		sc->sc_txtop = 0;
		sc->sc_if.if_timer = 0;
		dprintf(("done\n"));
	}

	/* XXX Report "carrier change" -- should be something more intelligent */
	if (sc->sc_p2pcom.p2p_modem)
		(*sc->sc_p2pcom.p2p_modem)(pp, flag);
	splx(s);
}

/*
 * Pass the accepted IP packet to upper levels
 */
static void
rhread(sc, cp, totlen)
	register struct rh_softc *sc;
	register caddr_t cp;
	int totlen;
{
	if (totlen == 0)
		return;
	
	if ((*sc->sc_p2pcom.p2p_hdrinput)(&sc->sc_p2pcom, (u_char *)cp,
	    totlen) >= 0)
		(*sc->sc_p2pcom.p2p_input)(&sc->sc_p2pcom, 0);
}

int
rhwatchdog(unit)
	int unit;
{
	struct rh_softc *sc0 = rhcd.cd_devs[unit / RH_NCHANS];
	struct rh_softc *sc = (unit % RH_NCHANS) ? sc0->sc_nextchan : sc0;
	int st;

	if (sc0 == sc)
		sc0 = sc->sc_nextchan;

	dprintf(("Status = %x\n", (st = rinb(sc->sc_msci, ST1))));
	printf("rh%d timeout",unit);

	/*  if we receive FLAG, and it is not first reser - restart all board */
	if ((st & ST1_FLGD) && sc->sc_slowtimer != 0) {
		printf(", reset\n");
		rhreset(sc0);
		rhreset(sc);
		rhenable(sc0);
		rhenable(sc);
	} else {
		rhreset(sc);
		rhenable(sc);
		printf("\n");
	}
	sc->sc_if.if_oerrors++;
	sc->sc_slowtimer += (sc->sc_slowtimer * 2 + RH_SLOWTIMER);
	if (sc->sc_slowtimer > RH_MAXTIMER)
		sc->sc_slowtimer = RH_MAXTIMER;
}

rhreset(sc)
	struct rh_softc *sc;
{
	int br;

	/* disable TX & RX interrupts */
	routb(sc->sc_dma, DIR, 0);
	routb(sc->sc_msci, IE1, 0);

	/* Halt dma in progress */
	routb(sc->sc_dma, DSR, 0);
	routb(sc->sc_dma, DSR+TX, 0);
	routb(sc->sc_dma, DCR, DCR_ABORT);
	routb(sc->sc_dma, DCR+TX, DCR_ABORT);

	/* disable transmitter and receiver */
	routb(sc->sc_msci, CMD, CMD_RXDIS);
	routb(sc->sc_msci, CMD, CMD_TXDIS);
	routb(sc->sc_msci, CTL, 0);              /* drop RTS */
	br = inb(sc->sc_addr);		/* drop DTR and disable transmitter */
	if (sc->sc_if.if_unit % RH_NCHANS) {
		br |= RHC_DTR1;
		br &= ~RHC_TE1;
	} else {
		br |= RHC_DTR0;
		br &= ~RHC_TE0;
	}
	outb(sc->sc_addr, br);

	/* Release tx buffers */
	dprintf(("release txtop %x --", sc->sc_txtop));
	if (sc->sc_txtop)
		m_freem(sc->sc_txtop);
	sc->sc_txtop = 0;
}
