/*-
 * Copyright (c) 1994, 1995 Matt Thomas (matt@lkg.dec.com)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software withough specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *	BSDI $Id: if_di.c,v 2.5 1995/12/13 04:18:29 ewv Exp $
 *
 * Id: if_di.c,v 1.3 1995/04/24 23:45:53 thomas Exp
 *
 * Log: if_di.c,v
 * Revision 1.3  1995/04/24  23:45:53  thomas
 * Add EISA autoconiguration for FreeBSD.
 * Add workarounds for some of the strangeness around DEPCAs
 * (ie. NICSR being inb/outb instead of inw/outw)
 *
 * Revision 1.2  1995/04/24  19:03:43  thomas
 * Fix EISA DE422 support.  Correct Copyright message.
 * Remove unused fields.
 *
 * Revision 1.1  1995/04/23  22:07:13  thomas
 * Initial revision
 *
 */

/*
 * DEC EtherWORKS 2 Ethernet Controllers
 * DEC EtherWORKS 3 Ethernet Controllers
 *
 * Written by Matt Thomas
 * BPF support code stolen directly from if_ec.c
 *
 *   This driver supports the DEPCA, DE100, DE101, DE200, DE201,
 *   DE2002, DE203, DE204, DE205, and DE422 cards.
 */

#ifndef __bsdi__
#include "di.h"
#endif

#if NDI > 0 || defined(__bsdi__)

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/mbuf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/malloc.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/if_dl.h>
#include <net/route.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>
#endif

#include <i386/isa/isa.h>
#include <i386/isa/icu.h>

#if defined(__FreeBSD__)
#include <sys/devconf.h>
#include <machine/cpufunc.h>
#include <i386/isa/isa_device.h>
#endif

#if defined(__bsdi__)
#include <sys/device.h>
#include <machine/inline.h>
#include <i386/isa/dma.h>
#include <i386/isa/isavar.h>
#include <i386/eisa/eisa.h>
#endif

#include <i386/isa/ic/am7990.h>
#include <i386/isa/lance.h>
#include <i386/isa/decether.h>

#include <vm/vm.h>

#include "bpfilter.h"
#if NBPFILTER > 0
#include <net/bpf.h>
#endif


/*
 *  These definitions are specific to the DEC "DEPCA-style" NICs.
 *	(DEPCA, DE10x, DE20[012], DE422)
 *
 */
#define	DEPCA_IOBASE_PRIMARY	0x300
#define	DEPCA_IOBASE_SECONDARY	0x200

#define	DEPCA_REG_NICSR		0		/* (RW;16) NI Control / Status */
#define	DEPCA_REG_RDP		4		/* (RW:16) LANCE RDP (data) register */
#define	DEPCA_REG_RAP		6		/* (RW:16) LANCE RAP (address) register */
#define	DEPCA_REG_ADDRROM	12		/* (R : 8) DEPCA Ethernet Address ROM */
#define	DEPCA_IOSIZE		16		/* DEPCAs use 16 bytes of IO space */

#define	DEPCA_NICSR_LED		0x0001		/* Light the LED on the back of the DEPCA */
#define	DEPCA_NICSR_ENABINTR	0x0002		/* Enable Interrupts */
#define	DEPCA_NICSR_MASKINTR	0x0004		/* Mask Interrupts */
#define	DEPCA_NICSR_AAC		0x0008		/* Address Counter Clear */
#define	DEPCA_NICSR_REMOTEBOOT	0x0010		/* Remote Boot Enabled (ignored) */
#define	DEPCA_NICSR_32KRAM	0x0020		/* DEPCA LANCE RAM size 64K (C) / 32K (S) */
#define	DEPCA_NICSR_LOW32K	0x0040		/* Bank Select (A15 = !This Bit) */
#define	DEPCA_NICSR_SHE		0x0080		/* Shared RAM Enabled (ie hide ROM) */
#define	DEPCA_NICSR_BOOTTMO	0x0100		/* Remote Boot Timeout (ignored) */

#define	DEPCA_RDNICSR(iobase)		inb(iobase + DEPCA_REG_NICSR)
#define	DEPCA_WRNICSR(iobase, val)	outb(iobase + DEPCA_REG_NICSR, val)

#define	DEPCA_IDSTR_OFFSET	0xC006		/* ID String Offset */

#define DEPCA_REG_EISAID	0xC80
#define	DEPCA_EISAID_MASK	0xf0ffffff
#define	DEPCA_EISAID_DE422	0x2042A310

#define	DEPCA_REG_EISACFG	0xC0C		/* 0b0000000xxIIIIBBB */
#define	DEPCA_REG_EISACTLSTS	0xC00		/* 0b0000000r00rrB000 */

#if NEISA > 0
static const char *depca_eisa_ids[] = {
    "DEC4220",
};
#endif


typedef enum {
    DEPCA_NOTPRESENT,
    DEPCA_CLASSIC,
    DEPCA_DE100, DEPCA_DE101,
    DEPCA_EE100,
    DEPCA_DE200, DEPCA_DE201, DEPCA_DE202,
    DEPCA_DE422,
    DEPCA_UNKNOWN
} depca_t;

static const char *depca_signatures[] = {
    "DE???",
    "DEPCA",
    "DE100", "DE101",
    "EE100",
    "DE200", "DE201", "DE202",
    "DE422",
    NULL
};

typedef struct {
    unsigned short dds_ds_para;		/* data segment for ROMs */
    unsigned char dds_romerror;		/* error code from roms */
    unsigned char dds_lanceirq;		/* LANCE IRQ */
    unsigned short dds_iooffset;	/* I/O port offset from 0x200 */
    unsigned char dds_mouseirq;		/* mouse IRQ (0=no mouse) */
    unsigned char dds_romerrorcmpl;	/* 1's complement  of rom error */
} depca_diag_status_t;

/*
 * Ethernet status, per interface.
 */
typedef struct {
#if defined(__bsdi__)
    struct device depca_dev;
    struct isadev depca_id;
    struct intrhand depca_ih;
    struct atshutdown depca_ats;
#endif
    lance_softc_t depca_lance;		/* Generic LANCE structure */
    unsigned depca_iobase;		/* Starting I/O base address */
    const char *depca_prodname;		/* product name DE20x-xx */
} depca_softc_t;

#define	depca_if	depca_lance.lance_if
#define	depca_ac	depca_lance.lance_ac

#ifdef __bsdi__
typedef int ifnet_ret_t;
typedef u_short ioport_t;
extern struct cfdriver dicd;
#define	DEPCA_UNIT_TO_SOFTC(unit)	((depca_softc_t *) dicd.cd_devs[(unit)])
#endif

#if defined(__FreeBSD__) || defined(__NetBSD__)
typedef void ifnet_ret_t;
typedef unsigned ioport_t;
static depca_softc_t *depcas[NDI];
#define	DEPCA_UNIT_TO_SOFTC(unit)	(depcas[(unit)])
#define	ISA_HOLE_VADDR(pa)		((caddr_t) ((int) (pa) - 0xa0000 + atdevbase))
#ifndef IRQUNK
#define	IRQUNK	0
#endif
#endif

static void
depca_device_init(
    depca_softc_t *sc)
{
    unsigned nicsr, idx = DEPCA_UNKNOWN, idstr_offset = DEPCA_IDSTR_OFFSET;
    
    /*
     *  Find out how much memory we are dealing with.  Adjust
     *  the ID string offset approriately if we are at
     *  32K.  Make sure the ROM is enabled.
     */
    nicsr = DEPCA_RDNICSR(sc->depca_iobase);
    nicsr &= ~(DEPCA_NICSR_SHE|DEPCA_NICSR_LED|DEPCA_NICSR_ENABINTR);
    
    if (nicsr & DEPCA_NICSR_32KRAM) {
	/*
	 * Make we are going to read the upper
	 * 32K so we do read the ROM.
	 */
	nicsr &= ~DEPCA_NICSR_LOW32K;
	sc->depca_lance.lance_ramoffset = 32 * 1024;
	idstr_offset -= sc->depca_lance.lance_ramsize;
    } else {
	sc->depca_lance.lance_ramoffset = 0;
    }
    DEPCA_WRNICSR(sc->depca_iobase, nicsr);

    if (sc->depca_iobase > DEPCA_IOBASE_PRIMARY) {
	sc->depca_prodname = depca_signatures[DEPCA_DE422];
	idx = DEPCA_DE422;
    } else {
	sc->depca_prodname = depca_signatures[0];
	for (idx = 0; depca_signatures[idx] != NULL; idx++) {
	    if (bcmp(depca_signatures[idx], sc->depca_lance.lance_membase + idstr_offset, 5) == 0) {
		sc->depca_prodname = depca_signatures[idx];
		break;
	    }
	}
    }

    /*
     * Stop the LANCE and try to read the address ROM.
     */
    sc->depca_lance.lance_multicast_filter = decether_multicast_filter;
    sc->depca_lance.lance_input = decether_input;
    sc->depca_lance.lance_rdp = sc->depca_iobase + DEPCA_REG_RDP;
    sc->depca_lance.lance_rap = sc->depca_iobase + DEPCA_REG_RAP;
    sc->depca_lance.lance_csr3 = LN_CSR3_ALE;
    lance_hwreset(&sc->depca_lance);

    if (idx == DEPCA_CLASSIC) {
	DEPCA_WRNICSR(sc->depca_iobase, 0);
	DEPCA_RDNICSR(sc->depca_iobase); /* needed for AAC */
	DEPCA_WRNICSR(sc->depca_iobase, DEPCA_NICSR_AAC);
	DEPCA_RDNICSR(sc->depca_iobase); /* needed for AAC */
    }
    if (decether_read_macaddr(sc->depca_lance.lance_hwaddr, sc->depca_iobase + DEPCA_REG_ADDRROM, 0) < 0)
	return;

    bcopy(sc->depca_lance.lance_hwaddr, sc->depca_lance.lance_ac.ac_enaddr, 6);
    /*
     * Renable shared RAM.
     */
    DEPCA_WRNICSR(sc->depca_iobase, DEPCA_RDNICSR(sc->depca_iobase) | DEPCA_NICSR_SHE);

    if (!lance_init_adapmem(&sc->depca_lance, sizeof(depca_diag_status_t)))
	return;

    DEPCA_WRNICSR(sc->depca_iobase, DEPCA_NICSR_SHE | DEPCA_NICSR_ENABINTR);
    lance_ifreset(&sc->depca_lance);
}

static ifnet_ret_t
depca_ifinit(
    int unit)
{
    lance_ifinit(&DEPCA_UNIT_TO_SOFTC(unit)->depca_lance);
}

static void
depca_ifattach(
    depca_softc_t *sc)
{
    struct ifnet *ifp = &sc->depca_if;

    depca_device_init(sc);

#if !defined(__bsdi__)
    printf("%s%d", ifp->if_name, ifp->if_unit);
    printf(": %s Ethernet address %s\n", 
	   sc->depca_prodname,
	   ether_sprintf(sc->depca_ac.ac_enaddr));
#else
    aprint_naive(": DEC Etherworks III Ethernet");
    aprint_normal(": %s", sc->depca_prodname);
    printf("\n");
    aprint_normal("%s%d: address %s\n", ifp->if_name, ifp->if_unit,
	   ether_sprintf(sc->depca_ac.ac_enaddr));
#endif

    ifp->if_init = depca_ifinit;
    ifp->if_start = lance_ifstart;
#if !defined(__bsdi__)
    ifp->if_output = ether_output;
#endif
    ifp->if_ioctl = decether_ifioctl;

    ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX | IFF_NOTRAILERS;
#if defined(MULTICAST) || defined(__FreeBSD__)
    ifp->if_flags |= IFF_MULTICAST;
#endif /* MULTICAST */

#if !defined(__bsdi__)
    if_attach(ifp);
#else
    ether_attach(ifp);
#endif

#if NBPFILTER > 0
    bpfattach(&ifp->if_bpf, ifp, DLT_EN10MB, sizeof(struct ether_header));
#endif
}

static int
depca_memcheck(
    const unsigned iobase,
    const unsigned maddr,
    unsigned *msize)
{
    const char *dp = (const char *) ISA_HOLE_VADDR(maddr);
    int found = 0, idx, idstr_offset = DEPCA_IDSTR_OFFSET;
    unsigned nicsr;

    /*
     * Make sure the ROM is visible in RAM.
     */
    if (msize)
	*msize = 0x10000;
    nicsr = DEPCA_RDNICSR(iobase);
    nicsr &= ~DEPCA_NICSR_SHE;
    if (nicsr & DEPCA_NICSR_32KRAM) {
	if (*msize)
	    *msize -= 0x8000;
	nicsr &= ~DEPCA_NICSR_LOW32K;
	idstr_offset -= 0x8000;
    }
    DEPCA_WRNICSR(iobase, nicsr);

    /*
     * Go look for the ROM signature.  Once a proper signature
     * has been found, enable the shared RAM and test again.
     * If we having the match I/O port, then the signature will
     * have disappeared.  Unless it's a DEPCA which doesn't have
     * the shared RAM.  Take pity on users of real DEPCAs and 
     * assume they won't put in more than one at a time.
     */
    for (idx = 1; depca_signatures[idx] != NULL; idx++) {
	if (bcmp(depca_signatures[idx], dp + idstr_offset, 5) != 0)
	    continue;
	if (idx == DEPCA_CLASSIC) {
	    if (msize)
		*msize -= 0x4000;
	    return DEPCA_CLASSIC;
	}
	nicsr |= DEPCA_NICSR_SHE;
	DEPCA_WRNICSR(iobase, nicsr);
	if (bcmp(depca_signatures[idx], dp + idstr_offset, 5) != 0)
	    return idx;
	nicsr &= ~DEPCA_NICSR_SHE;
	DEPCA_WRNICSR(iobase, nicsr);
    }
    return DEPCA_UNKNOWN;
}

static int
depca_portcheck(
    unsigned iobase)
{
    unsigned char hwaddr[6];
    unsigned maddr;
#if defined(__bsdi__)
    if (iobase < 0x1000 && !isa_portcheck(iobase, DEPCA_IOSIZE))
	return 0;
#endif

    if (decether_read_macaddr(hwaddr, iobase + DEPCA_REG_ADDRROM, 0) == 0)
	return 1;
    if (iobase > DEPCA_IOBASE_PRIMARY)
	return 0;
    for (maddr = 0xC0000; maddr <= 0xE0000; maddr += 0x10000) {
	if (bcmp(ISA_HOLE_VADDR(maddr + DEPCA_IDSTR_OFFSET), depca_signatures[DEPCA_CLASSIC], 5) == 0) {
	    unsigned nicsr = DEPCA_RDNICSR(iobase);
	    DEPCA_WRNICSR(iobase, nicsr|DEPCA_NICSR_AAC); DEPCA_RDNICSR(iobase);
	    if (decether_read_macaddr(hwaddr, iobase + DEPCA_REG_ADDRROM, 0) == 0)
		return 1;
	    break;
	}
    }
    return 0;
}

static int
depca_isa_probe(
    ioport_t *iobase_p,
    ioport_t *iosize_p,
    unsigned *maddr_p,
    unsigned *msize_p,
    unsigned *irq_p)
{
    unsigned iobase = *iobase_p, iosize = *iosize_p;
    unsigned maddr = *maddr_p, msize = *msize_p, maddr0;
    unsigned irq = *irq_p, irq0 = IRQUNK;
    const depca_diag_status_t *dds;

    /*
     * If we are searching for the DEPCA, then it's either at the
     * primary or secondary address.
     */
    if (iobase) {
	if (iobase != DEPCA_IOBASE_PRIMARY
	        && iobase != DEPCA_IOBASE_SECONDARY
		&& (iobase < 0x1000 || (iobase & 0xFFF) != 0xC00))
	    return 0;
	if (!depca_portcheck(iobase))
	    return 0; 
    } else if (depca_portcheck(DEPCA_IOBASE_PRIMARY)) {
	iobase = DEPCA_IOBASE_PRIMARY;
    } else if (depca_portcheck(DEPCA_IOBASE_SECONDARY)) {
	iobase = DEPCA_IOBASE_SECONDARY;
    } else {
	return 0;
    }
    if (iobase <= DEPCA_IOBASE_PRIMARY)
	iosize = DEPCA_IOSIZE;

    /*
     * Now we need to search the various starting addresses
     * used by the DEPCAs for a diagnostic block written by
     * self-test ROM.
     */
    for (maddr0 = 0xC0000; maddr0 <= 0xE8000; maddr0 += 0x8000) {
	dds = (const depca_diag_status_t *) ISA_HOLE_VADDR(maddr0);
	if (dds->dds_romerror == ((~dds->dds_romerrorcmpl) & 0xFF)) {
	    if (dds->dds_iooffset + DEPCA_IOBASE_SECONDARY == iobase)
		break;
	}
	dds = NULL;
    }

    /*
     * If there was no IRQ supplied and no diagnostic block
     * was found then there's no way to determine the IRQ in use.
     * (actually we force an interrupt but I don't know how to do that)
     */
    if (dds == NULL) {
	/*
	 * No diagnostic block and there's no address configured.
	 * Try to find a ROM signature.
	 */
	if (maddr == 0) {
	    for (maddr0 = 0xC0000; maddr0 <= 0xE0000; maddr0 += 0x10000) {
		if (depca_memcheck(iobase, maddr0, &msize) != DEPCA_UNKNOWN) {
		    maddr = maddr0 + (msize <= 0x8000 ? 0x8000 : 0);
		    break;
		}
	    }
	    if (maddr == 0)
		return 0;
	}
    } else {
	irq0 = (1 << dds->dds_lanceirq);
	if (maddr != 0 && maddr != maddr0)
	    return 0;
	if (depca_memcheck(iobase, maddr0, &msize) == DEPCA_UNKNOWN)
	    return 0;

	maddr = maddr0;
    }

    if (irq == IRQUNK) {
	if (irq0 == IRQUNK)
	    return 0;
	irq = irq0;
    } else if (irq != irq0) {
	return 0;
    }

    *maddr_p = maddr;
    *msize_p = msize;
    *iobase_p = iobase;
    *iosize_p = iosize;
    *irq_p = irq;
    return 1;
}

#if defined(__bsdi__)
static int
depca_probe(
    struct device *parent,
    struct cfdata *cf,
    void *aux)
{
    unsigned nicsr, idx, idstr_offset = DEPCA_IDSTR_OFFSET;
    struct isa_attach_args *ia = (struct isa_attach_args *) aux;
    unsigned irq = IRQUNK, maddr;

    if (ia->ia_bustype == BUS_EISA) {
	int slot;
	if ((slot = eisa_match(cf, ia)) == 0)
	    return 0;
	ia->ia_iobase = (slot << 12) | 0xC00;
	ia->ia_iosize = EISA_NPORT;
	eisa_slotalloc(slot);

	/* EISA bus masters don't use host DMA channels */
	ia->ia_drq = DRQNONE;

    }

    irq = ia->ia_irq;
    maddr = (int) ia->ia_maddr;
    if (!depca_isa_probe(&ia->ia_iobase, &ia->ia_iosize,
			 &maddr, &ia->ia_msize, &irq))
	return 0;
    ia->ia_maddr = (caddr_t) maddr;
    if (ia->ia_iobase > DEPCA_IOBASE_PRIMARY)
	ia->ia_iobase &= 0xF000;

    if (ia->ia_irq != IRQUNK && irq != ia->ia_irq) {
      wrong_irq:
	printf("%s%d: error: desired IRQ of %d does not match device's actual IRQ (%d),\n",
	       cf->cf_driver->cd_name, cf->cf_unit,
	       ffs(ia->ia_irq) - 1, ffs(irq) - 1);
	return 0;
    }
    if (ia->ia_irq == IRQUNK) {
	if ((irq = isa_irqalloc(irq)) == 0)
	    return 0;
	ia->ia_irq = irq;
    }
    return 1;
}

static int
depca_intr(
    void *arg)
{
    depca_softc_t *sc = (depca_softc_t *) arg;
    DEPCA_WRNICSR(sc->depca_iobase,
		  DEPCA_RDNICSR(sc->depca_iobase) ^ DEPCA_NICSR_LED);
    lance_intr(&sc->depca_lance);
    return 1;
}

static void
depca_shutdown(
    void *arg)
{
    depca_softc_t *sc = (depca_softc_t *) arg;
    /*
     * Stop the LANCE (not really needed but why not?)
     */
    lance_hwreset(&sc->depca_lance);
    /*
     * Reenable the ROM so it will go diagnostics on reboot
     */
    DEPCA_WRNICSR(sc->depca_iobase,
		  DEPCA_RDNICSR(sc->depca_iobase) &
		  ~(DEPCA_NICSR_SHE|DEPCA_NICSR_LED));
}

static void
depca_attach(
    struct device *parent,
    struct device *self,
    void *aux)
{
    depca_softc_t *sc = (depca_softc_t *) self;
    struct isa_attach_args *ia = (struct isa_attach_args *) aux;

    sc->depca_if.if_unit = sc->depca_dev.dv_unit;
    sc->depca_if.if_name = sc->depca_dev.dv_cfdata->cf_driver->cd_name;
    sc->depca_iobase = ia->ia_iobase + (ia->ia_iobase > DEPCA_IOBASE_PRIMARY ? 0xC00 : 0);

    sc->depca_lance.lance_ramsize = ia->ia_msize;
    sc->depca_lance.lance_membase = (caddr_t) ISA_HOLE_VADDR(ia->ia_maddr);

    depca_ifattach(sc);

    isa_establish(&sc->depca_id, &sc->depca_dev);

    sc->depca_ih.ih_fun = depca_intr;
    sc->depca_ih.ih_arg = (void *) sc;
    intr_establish(ia->ia_irq, &sc->depca_ih, DV_NET);

    sc->depca_ats.func = depca_shutdown;
    sc->depca_ats.arg = (void *) sc;
    atshutdown(&sc->depca_ats, ATSH_ADD);
}

struct cfdriver dicd = {
    NULL, "di", depca_probe, depca_attach, DV_IFNET, sizeof(depca_softc_t),
#if NEISA > 0
    depca_eisa_ids
#endif
};
#endif /* __bsdi__ */

#if defined(__FreeBSD__)
static int
depca_shutdown(
    struct kern_devconf *kdc,
    int force)
{
    depca_softc_t *sc;
    if (kdc->kdc_unit < NDI && (sc = depcas[kdc->kdc_unit]) != NULL) {
	/*
	 * Stop the LANCE (not really needed but why not?)
	 */
	lance_hwreset(&sc->depca_lance);
	/*
	 * Reenable the ROM so it will go diagnostics on reboot
	 */
	DEPCA_WRNICSR(sc->depca_iobase,
		      DEPCA_RDNICSR(sc->depca_iobase) &
		      ~(DEPCA_NICSR_SHE|DEPCA_NICSR_LED));
    }
    (void) dev_detach(kdc);
    return 0;
}

static struct kern_devconf kdc_depca[NDI] = {
    {	0, 0, 0,		/* filled in by dev_attach */
	"le", 0, { MDDT_ISA, 0, "net" },
	isa_generic_externalize, 0, depca_shutdown, ISA_EXTERNALLEN,
	&kdc_isa0,		/* parent */
	0,			/* parentdata */
	DC_BUSY,		/* network interfaces are always busy */
	"DEC EtherWORKS II Ethernet adapter"
    },
};

static inline void
depca_registerdev(
    struct isa_device *id)
{
    if (id->id_unit > 0)
	kdc_depca[id->id_unit] = kdc_depca[0];
    kdc_depca[id->id_unit].kdc_unit = id->id_unit;
    kdc_depca[id->id_unit].kdc_isa = id;
    dev_attach(&kdc_depca[id->id_unit]);
}

static int depca_eisa_slots = ~1;

static int
depca_probe(
    struct isa_device *id)
{
    ioport_t iobase, iosize;
    unsigned maddr, msize, irq;
    unsigned slot, data;

    if (id->id_unit >= NDI) {
	printf("%s%d: not configured; kernel only built for %d device%s.\n",
	       id->id_driver->name, id->id_unit, NDI, NDI == 1 ? "" : "s");
	return 0;
    }

    iosize = 0;
    iobase = id->id_iobase == -1 ? 0 : id->id_iobase;

    if (iobase == 0) {
	slot = 0x1000 * (ffs(depca_eisa_slots) - 1);
	for (; slot <= 0xF000; slot += 0x1000) {
	    if (depca_eisa_slots & (1 << (slot >> 12))) {
		depca_eisa_slots &= ~(1 << (slot >> 12));
		data = inl(slot + DEPCA_REG_EISAID);
		if ((data & DEPCA_EISAID_MASK) == DEPCA_EISAID_DE422) {
		    iobase = slot | 0xC00;
		    iosize = 0x1000;
		    break;
		}
	    }
	}
    }
    msize = 0;
    maddr = id->id_maddr ? kvtop(id->id_maddr) : 0;
    irq = id->id_irq;

    if (/* !depca_eisa_probe(&iobase, &iosize, &maddr, &msize, &irq)
	    && */ !depca_isa_probe(&iobase, &iosize, &maddr, &msize, &irq))
	return 0;
    if (iobase > DEPCA_IOBASE_PRIMARY)
	iobase &= 0xF000;
    id->id_iobase = iobase;
    id->id_irq = irq;
    id->id_maddr = (void *) ISA_HOLE_VADDR(maddr);
    id->id_msize = msize;
    return iosize;
}

static void
depca_intr(
    int unit)
{
    depca_softc_t *sc = depcas[unit];
    DEPCA_WRNICSR(sc->depca_iobase,
		  DEPCA_RDNICSR(sc->depca_iobase) ^ DEPCA_NICSR_LED);
    lance_intr(&sc->depca_lance);
}

static int
depca_attach(
    struct isa_device *id)
{
    depca_softc_t *sc;

    sc = (depca_softc_t *) malloc(sizeof(*sc), M_DEVBUF, M_NOWAIT);
    if (sc == NULL)
	return 0;
    bzero(sc, sizeof(*sc));

    depcas[id->id_unit] = sc;
    sc->depca_if.if_name = id->id_driver->name;
    sc->depca_if.if_unit = id->id_unit;
    sc->depca_iobase = id->id_iobase + (id->id_iobase > DEPCA_IOBASE_PRIMARY ? 0xC00 : 0);
    sc->depca_lance.lance_ramsize = id->id_msize;
    sc->depca_lance.lance_membase = id->id_maddr;

    id->id_intr = depca_intr;	/* avoids putting it in the config file */

    depca_ifattach(sc);
	       
    depca_registerdev(id);
    return 1;
}
/*
 * This tells the NetBSD/FreeBSD autoconf code how to set us up.
 */
struct isa_driver didriver = {
    depca_probe, depca_attach, "di",
};

#endif /* defined(__FreeBSD__) || defined(__NetBSD__) */
#endif /* NDI > 0 */
