/*-
 * Copyright (c) 1995 Berkeley Software Design, Inc.  All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: aic.c,v 2.10 1995/12/11 16:19:44 cp Exp $
 */

/*
 * Driver for the aic 7{7,8}70 series of controllers
 *
 * Note
 *	Dates refer to when note made, not when any code
 *	was necessarily changed.
 *
 * Termination					July 30, 1995
 *	Need to look through parity code. It may be that
 *	parity should be disabled on a target by target basis.
 */

#include <sys/param.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/device.h>
#include <sys/malloc.h>

#include <vm/vm.h>

#include <dev/scsi/scsi.h>
#include <dev/scsi/scsivar.h>
#include <dev/scsi/disk.h>
#include <dev/scsi/tape.h>

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/eisa/eisa.h>

#include <i386/isa/icu.h>

#include <machine/cpu.h>

#include <i386/pci/pci.h>

#include "aicreg.h"
#include "aic_code.h"		/* actual firmware */

typedef vm_offset_t physaddr_t;
typedef enum {AIC_PCI, AIC_EISA} aic_type_t;

/*
 * information for running an operation
 * the beginninging of this struct must match the
 * aic_scb.def
 */
typedef struct aic_scb {
	physaddr_t	 scb_pa;	/* physical adress of this scb*/
	u_char		 scb_status;	/* scsi status */	
#define  scb_scsiid scb_status
	u_char		 scb_flags;
	u_char		 scb_scsirate;
	u_char		 scb_msgolen;
	u_char		 scb_cdblen;
	u_char		 scb_tag;
	u_char		 scb_target;
	u_char		 scb_filler;
	physaddr_t	 scb_sgentry;	/* current entry in sg array */
	aic_sg_t	 scb_sgactive;
	u_char		 scb_msgout[6];
	u_char		 scb_msgin[6];
	aic_sg_t	 scb_sglist[20];
	struct scsi_cdb  scb_cdb;	/* real scsi cdb from caller */
	struct buf	*scb_bp;
	u_char		 scb_lun;
	u_char		 scb_icmd;	/* operation is for an immediate cmd */
	u_char		 scb_done;	/* icmd is done */
	scintr_fn	 scb_bintr;	/* base interrupt handler */
	struct device	*scb_bintrdev;	/* base device */
	struct aic_scb	*scb_next;
} aic_scb_t;

#define AIC_TARGETS	8
#define AIC_TARGETSCBS	4
/*
 * Informatin needed for a target.
 * The size of this is patched placed in the sram by the driver.
 * The beginning of the struct is shared with the firmware and
 * must match aic_sram.def
 * This struct must match TARGETSIZE defined in aic_code
 */
typedef struct aic_target {		/* target slot */
	physaddr_t	 t_scblist;	/* points at t_scbs */
	physaddr_t	 t_newwork;	/* scb to be started */
	u_char		 t_flags;
	u_char		 t_target;	/* used during reselect */
	u_char		 t_scsirate;
	u_char		 t_filler3;
	vm_offset_t	 t_p_scbs[AIC_TARGETSCBS];
	aic_scb_t	*t_v_scbs[AIC_TARGETSCBS];
} volatile aic_target_t;

typedef struct aic_softc {
	struct hba_softc	sc_hba;
#define sc_flags sc_hba.hba_dev.dv_flags
	struct isadev		sc_isa;
	struct intrhand		sc_ih;
	aic_scb_t	*sc_scb_go;	/* scb currently being started */
	vm_offset_t	 sc_pio;	/* port address for io */
	void 		*sc_mio;	/* where io is mapped in address spade*/
	aic_io_t	*sc_vio;	/* debug only, */
	u_char		 sc_nodisconnect;
	u_char		 sc_nosync;
	u_char		 sc_id;		/* adapter's SCSI ID */
	u_char		 sc_icmd;
	aic_target_t	 sc_targets[AIC_TARGETS];
	u_char		 sc_probetime;
	u_char		 sc_probehung;
	u_char		 sc_watchon;
	u_char		 sc_watchinterval;
	u_int		 sc_ioout;	/* number of outstanding io scbs */
	u_int		 sc_icnt;	/* count of interrupts */
	aic_type_t	 sc_type;
	aic_scb_t	*sc_scb_free;
}aic_softc_t;


/*
 * prototypes needed for cfdriver and hbadriver
 */

static int aicprobe __P((struct device *parent, struct cfdata *cf, void *aux));
static void aicattach __P((struct device *parent, struct device *self,
	void *aux));
static int aicicmd __P((struct hba_softc *hba, int targ, struct scsi_cdb *cdb,
	caddr_t data, int len, int rw));
static int aicdump __P((struct hba_softc *hba, int targ, struct scsi_cdb
	*cdb, caddr_t addr, int len));
static void aicstart __P((struct device *self, struct sq *sq, struct buf
	*bp, scdgo_fn dgo, struct device *dev));
static int aicgo __P((struct device *self, int targ, scintr_fn intr,
	struct device *dev, struct buf *bp, int pad));
static void aicrel __P((struct device *self));
static void aichbareset __P((struct hba_softc *hba, int resetunits));

static char *aic_eisaids[] = {
	"ADP7770",
	"ADP7771",
	0
};

struct cfdriver aiccd = { 
	0, "aic", aicprobe, aicattach, DV_DULL, sizeof (aic_softc_t),
	aic_eisaids
};

static struct hbadriver aichbadriver =
    { aicicmd, aicdump, aicstart, aicgo, aicrel, aichbareset };

/*
 * shifts to pick up values from flags field
 */
#define AIC_F_NOSYNC_S		24	/* units with sync forced off */
#define AIC_F_NODISCONNECT_S	16	/* units with disconnect forced off */
#define AIC_F_IID_S		4	/* initiator id */
#define AIC_F_CLOCK_S	    	0
#define AIC_F_NOPARITY		0x100
#define AIC_F_SLOWSCSI		0x200
#define AIC_F_TERMOFF		0x400
#define AIC_F_ACKNEGOFF		0x800

#define AIC_F_CLOCK_M		0xf
#define AIC_F_IID_M		0xf

#define AIC_F_CLOCK_BITS(x)	(((x) >> AIC_F_CLOCK_S) & AIC_F_CLOCK_M)
#define AIC_F_IID_BITS(x)	(((x) >> AIC_F_IID_S) & AIC_F_IID_M)


#define AIC_VENDOR_ID		0x9004
#define AIC_DEVICE_ID_0		0x7078
#define AIC_DEVICE_ID_1		0x7178

#if 0
#define AIC_SLOWWATCH	hz
#endif
#define AIC_SLOWWATCH	1
#define AIC_FASTWATCH	1

#define PA(addr) pmap_extract(kernel_pmap, (vm_offset_t)(addr))

/*
 * external functions without prototypes
 */
void * mapphys __P((vm_offset_t, int));

/*
 * routines who are know externally
 */
static int aicintr __P((void *sc0));
static int aicmatch __P((pci_devaddr_t *pa));
static void aicwatch __P((void *sc0));

/*
 * routines only called internally
 */
static void start_scb __P((register aic_softc_t *sc, aic_scb_t *scb));
static void done_scb __P((aic_softc_t *sc, aic_scb_t *scb));
static int aicreset_hw __P((aic_softc_t *sc));
static void aic_outb __P((aic_softc_t *sc, u_int offset, u_char value));
static void aic_outl __P((aic_softc_t *sc, u_int offset, u_long value));
static u_char aic_inb __P((aic_softc_t *sc, u_int offset));
static aic_scb_t * create_scb __P((aic_softc_t *sc));
static aic_scb_t * get_scb __P((aic_softc_t *sc));
static void free_scb __P((aic_softc_t *sc, aic_scb_t *scb));
static void aic_cmdcmplt __P((aic_softc_t *sc));
static void start_scb __P((aic_softc_t *sc, aic_scb_t *scb));
static void done_scb __P((aic_softc_t *sc, aic_scb_t *scb));
static void aic_sgmap __P((aic_scb_t *scb));
static int aic_forceintr __P((u_short *iobp));

#ifdef DEBUG
static void aic_trace __P((aic_softc_t *sc, int always));
#endif

#define OUTB_SC(offset, value) \
	aic_outb(sc, (offset), (value))

#define OUTL_SC(offset, value) \
	aic_outl(sc, (offset), (value))


#define INB_SC(offset) \
	aic_inb(sc, (offset))


static int
aic_probepci(struct isa_attach_args *ia)
{
	u_short command;
	pci_devaddr_t *pa;

	if (ia->ia_bustype != BUS_PCI)
		return (0);

	pa = pci_scan(aicmatch);
	if (pa == NULL)
		return (0);

	if (ia->ia_maddr)
		pci_outl(pa, PCI_MEMORY_BA, (u_long)ia->ia_maddr);
	ia->ia_maddr = (caddr_t)pci_inl(pa, PCI_MEMORY_BA);
	ia->ia_msize = 256;
	command = pci_inw(pa, PCI_COMMAND);
	pci_outl(pa, PCI_COMMAND, command & ~1);  /* turn off port access */
	ia->ia_irq = irq_indextomask(pci_inb(pa, PCI_I_LINE));
	if (ffs(ia->ia_irq) - 1 == 0)
		ia->ia_irq = IRQNONE;
	else
		ia->ia_irq |= IRQSHARE;
	ia->ia_iobase = 0;
	return (1);
}

static int
aic_probeeisa(struct cfdata *cf, struct isa_attach_args *ia)
{
	int slot;
	u_short iob;

	if (ia->ia_bustype != BUS_EISA)
		return (0);

	if ((slot = eisa_match(cf, ia)) == 0)
		return (0);

	iob = ia->ia_iobase = slot << 12;
	ia->ia_iosize = EISA_NPORT;
	eisa_slotalloc(slot);
	if (ia->ia_irq != IRQUNK)
		return (1);
	ia->ia_irq = isa_discoverintr(aic_forceintr, (void *) &iob);
	outb(ia->ia_iobase + AIC_HCNTRL, 0);	/* turn of interrupts */
	if (ffs(ia->ia_irq) - 1 == 0)
		ia->ia_irq = IRQNONE;
	return (1);
}


/*
 * standard probe routine
 */
static int
aicprobe(struct device *parent, struct cfdata *cf, void *aux)
{
	struct isa_attach_args *ia = aux;
	int i;

	if (aic_probepci(ia) == 0  && aic_probeeisa(cf, ia) == 0)
		return (0);


#ifdef DEBUG
	i = 0;
	if (AIC_S_SG_OFFSET != (u_int)&((aic_scb_t*)0)->scb_sglist[0]) {
		printf("S_SG_OFFSET wrong should be %x\n",
		    (u_int)&((aic_scb_t*)0)->scb_sglist[0]);
		i = 1;
	}
	if (AIC_S_MSGO_OFFSET != (u_int)&((aic_scb_t*)0)->scb_msgout[0]) {
		printf("S_MSGO_OFFSET wrong should be %x\n",
		    (u_int)&((aic_scb_t*)0)->scb_msgout[0]);
		i = 1;
	}
	if (AIC_S_MSGI_OFFSET != (u_int)&((aic_scb_t*)0)->scb_msgin[0]) {
		printf("S_MSGI_OFFSET wrong should be %x\n",
		    (u_int)&((aic_scb_t*)0)->scb_msgin[0]);
		i = 1;
	}
	if (AIC_S_CDB_OFFSET != (u_int)&((aic_scb_t*)0)->scb_cdb) {
		printf("S_CDB_OFFSET wrong should be %x\n",
		    (u_int)&((aic_scb_t*)0)->scb_cdb);
		i = 1;
	}
	if (AIC_TARGETSIZE != sizeof(aic_target_t)) {
		printf("TARGETSIZE wrong should be %x\n",
		    sizeof(aic_target_t));
		i = 1;
	}
	if (i)
		return 0;
#endif
	return (1);
}

/*
 * standard attach routine.
 */
static void
aicattach(struct device *parent, struct device *self, void *aux)
{
	struct isa_attach_args *ia = aux;
	register aic_softc_t *sc = (aic_softc_t *)self;
	int i;
	aic_scb_t *scb;
	int tmp;


	if (AIC_F_IID_BITS(sc->sc_flags) != 0)
		sc->sc_id = AIC_F_IID_BITS(sc->sc_flags) & 0x7;
	else
		sc->sc_id = 7;

	sc->sc_nodisconnect = sc->sc_flags >> AIC_F_NODISCONNECT_S;
	sc->sc_nosync = sc->sc_flags >> AIC_F_NOSYNC_S;
	if (ia->ia_iobase == 0) {
		sc->sc_mio = 
		    mapphys((vm_offset_t)ia->ia_maddr, ia->ia_msize);
		sc->sc_vio = (aic_io_t*)sc->sc_mio;
		sc->sc_type = AIC_PCI; 
	} else {
		sc->sc_pio = ia->ia_iobase;
		sc->sc_type = AIC_EISA;
	}

	if (aicreset_hw(sc) != 0)
		return;		/* to bad we didn't do this at probe time */

	printf("\n%s: initiator id %d", 
		sc->sc_hba.hba_dev.dv_xname, sc->sc_id); 


	if (sc->sc_flags & AIC_F_NOPARITY)
		printf(", parity disabled");
	else
		printf(", parity enabled");

	printf("\n%s: delaying to permit certain devices to finish self-test",
	    sc->sc_hba.hba_dev.dv_xname);
	DELAY(5000000); 	/* 5 seconds */

	sc->sc_hba.hba_driver = &aichbadriver;
	/*
	 * Link into isa and set interrupt handler.
	 */
	isa_establish(&sc->sc_isa, &sc->sc_hba.hba_dev);
	sc->sc_ih.ih_fun = aicintr;
	sc->sc_ih.ih_arg = sc;
	intr_establish(ia->ia_irq, &sc->sc_ih, DV_DISK);
	printf("\n");

	/*
	 * do everything needed to start firmware
	 */
	scb = create_scb(sc);
	free_scb(sc, scb);
	sc->sc_probetime = 1;
	for (i = 0; i < AIC_TARGETS; i++) {
		aic_target_t *ts = &sc->sc_targets[i];

		ts->t_target = i;
		if (i == sc->sc_id)
			continue;
		ts->t_scblist = PA(ts->t_p_scbs);
		if (scsi_test_unit_ready(&sc->sc_hba, i, 0) < 0)
			continue;
		SCSI_FOUNDTARGET(&sc->sc_hba, i);
		scb = create_scb(sc);
		free_scb(sc, scb);
		ts = &sc->sc_targets[i];
		printf("%s:", sc->sc_hba.hba_targets[i]->t_dev.dv_xname);
		if (sc->sc_nodisconnect & (1 << i))
			printf(" disconnect disabled,");
		else
			printf(" disconnect enabled,");
		if (ts->t_scsirate == 0) {
			printf(" asynchronous\n");
			continue;
		}
		tmp = (((ts->t_scsirate >> 4) & 0xf) + 4) * 25;/* period */
		tmp = 10000 / tmp;				/* MHZ * 10 */
		printf(" synchronous at %d.%dMB/s",
			tmp / 10, tmp % 10);

		printf(", max offset %d\n", ts->t_scsirate & 0xf);

	}
	sc->sc_probetime = 0;
	sc->sc_watchinterval = AIC_SLOWWATCH;
}

/*
 * standard icmd routine
 */
static int
aicicmd(struct hba_softc *hba, int targ, struct scsi_cdb *cdb, caddr_t data,
       int len, int rw)
{
	int status;
	int s;
	aic_scb_t *scb;
	aic_softc_t *sc = (aic_softc_t *)hba;
	struct buf bufhdr;
	int x, reran;
#ifdef DEBUG
	aic_io_t *aic = sc->sc_vio;
#endif

	if (sc->sc_probetime) {
		if (sc->sc_probehung) 
			return (-1);
		x = 0;
		reran = 0;
	}
rerun:
	s = splbio();
	scb = get_scb(sc);
	bufhdr.b_un.b_addr = data;
	bufhdr.b_iocount = bufhdr.b_bcount = len;
	bufhdr.b_flags = rw;
	bufhdr.b_chain = NULL;
	scb->scb_bp = &bufhdr;
	scb->scb_done = 0;

	scb->scb_cdblen = SCSICMDLEN(cdb->cdb_bytes[0]);
	bcopy(cdb, &scb->scb_cdb, scb->scb_cdblen);
	scb->scb_icmd = 1;
	scb->scb_target = targ;
	scb->scb_lun = cdb->cdb_bytes[1] >> 5;
	sc->sc_icmd = 1;
	start_scb(sc, scb);
	while (!scb->scb_done) {
		if (sc->sc_probetime) {
			if (x++ > 3000000) {
				printf("%s: SCSI bus hung\n",
				    sc->sc_hba.hba_dev.dv_xname);
				scb->scb_status = 0x80;
				sc->sc_probehung = 1;
				OUTB_SC(AIC_HCNTRL, AIC_PAUSE);
				while (!(INB_SC(AIC_HCNTRL) & AIC_PAUSE))
					continue;
#ifdef DEBUG
				aic_trace(sc, 1);
#endif
				break;
			}
		}
		aicintr(sc);
	}
	status = scb->scb_status;
	free_scb(sc, scb);
	sc->sc_icmd = 0;
	splx(s);
	if (status & 0x80)
		return (-1);		/* minus one for timeout */
	if (status == 0x08 && sc->sc_probetime && reran++ == 0)
		goto rerun;
	return (status);
}

/*
 * standard dump routine. This routine has to
 * map the physical address into a logical address
 * before it can run the io.
 */
extern char *dumpbufp;

static int
aicdump(struct hba_softc *hba, int targ, struct scsi_cdb *cdb, caddr_t addr,
       int len)
{
	int i;

	for (i = 0; i < len; i += ctob(1)) 
		pmap_enter(kernel_pmap, (vm_offset_t)(dumpbufp + i),
		    (vm_offset_t) addr + i, VM_PROT_READ, TRUE);
	return (aicicmd(hba, targ, cdb, dumpbufp, len, B_WRITE));
}

/*
 * standard start routine.
 */
static void
aicstart(struct device *self, struct sq *sq, struct buf *bp, scdgo_fn dgo,
        struct device *dev)
{
	int s;
	aic_softc_t *sc = (aic_softc_t *)self;
	if (bp) {
		if (sc->sc_scb_go)
			panic ("aic start with go set");
		s = splbio();
		sc->sc_scb_go = get_scb(sc);
		splx(s);
		(*dgo)(dev, &sc->sc_scb_go->scb_cdb);
		return;
	}
	(*dgo)(dev, (struct scsi_cdb *)NULL);
}

/* 
 * standard go routine. Fill in the scb
 * and run the io
 */
static int
aicgo(struct device *self, int targ, scintr_fn intr, struct device *dev,
     struct buf *bp, int pad)
{
	struct aic_softc *sc = (struct aic_softc *)self;
	aic_scb_t *scb;
	scb = sc->sc_scb_go;
	sc->sc_scb_go = NULL;
	scb->scb_bp = bp;
	scb->scb_target = targ;
	scb->scb_bintr = intr;
	scb->scb_bintrdev = dev;
	scb->scb_done = 0;
	scb->scb_icmd = 0;
	scb->scb_cdblen = SCSICMDLEN(scb->scb_cdb.cdb_bytes[0]);
	scb->scb_lun = scb->scb_cdb.cdb_bytes[1] >> 5;
	start_scb(sc, scb);
	return (0);
}

/*
 * standard rel
 */
static void
aicrel(struct device *self)
{
	aic_softc_t *sc = (aic_softc_t *)self;
	struct sq *sq;

	if (sc->sc_scb_go) {
		free_scb(sc, sc->sc_scb_go);
		sc->sc_scb_go = NULL;
	}
	if ((sq = sc->sc_hba.hba_head) != NULL) {
		sc->sc_scb_go = get_scb(sc);
		(*sq->sq_dgo)(sq->sq_dev, &sc->sc_scb_go->scb_cdb);
	}
}

/*
 * standard reset code.
 */
static void
aichbareset(struct hba_softc *hba, int resetunits)
{
	aic_softc_t *sc = (aic_softc_t *)hba;
	aic_target_t *ts;
	int i;

	aicreset_hw(sc);
	printf("%s: delaying 5 seconds after bus reset\n",
	    sc->sc_hba.hba_dev.dv_xname);
	DELAY(5000000); 	/* 5 seconds */

	/* go get all scbs */

	/* start firmware */
	if (resetunits)
		scsi_reset_units(hba);
}


/*
 * standard interrupt handler
 */
static int
aicintr(void *sc0)
{
	aic_softc_t *sc = sc0;
	aic_scb_t *scb;
	int target;
	aic_target_t *ts;
	u_char intstat;
#ifdef DEBUG
	aic_io_t *aic = sc->sc_vio;

	if ((INB_SC(AIC_HCNTRL) & AIC_PAUSE) &&
	    (INB_SC(AIC_SEQCTL) & AIC_STEP))
		aic_trace(sc, 0);
#endif

	intstat = INB_SC(AIC_INTSTAT);

	if ((intstat & AIC_ALLINT) == 0) {
#ifdef DEBUG
		if ((INB_SC(AIC_HCNTRL) & AIC_PAUSE) &&
		    (INB_SC(AIC_SEQCTL) & AIC_STEP))
			OUTB_SC(AIC_HCNTRL, AIC_INTEN);
#endif
		return (0);
	}

	sc->sc_icnt++;

	if (intstat & AIC_CMDCMPLT) {
		if (!sc->sc_icmd || sc->sc_probetime)
			OUTB_SC(AIC_CLRINT, AIC_CMDCMPLT);
		aic_cmdcmplt(sc);
	}

	if (intstat & AIC_SEQINT) {
		printf("sequencer interrupt %x\n", intstat);
		OUTB_SC(AIC_CLRINT, AIC_SEQINT);
	}
#ifdef DEBUG
	if ((INB_SC(AIC_HCNTRL) & AIC_PAUSE) &&
	    (INB_SC(AIC_SEQCTL) & AIC_STEP))
		OUTB_SC(AIC_HCNTRL, AIC_INTEN);
#endif
	return (1);
}

/*
 * routine used by pci_scan to match the device as being
 * an aic7{7,8}70.
 */
static int
aicmatch(pci_devaddr_t *pa)
{
	u_short vendor;
	u_short dev;
	u_char line;

	vendor = pci_inw(pa, PCI_VENDOR_ID);

	if (vendor != AIC_VENDOR_ID)
		return 0;

	dev = pci_inw(pa, PCI_DEVICE_ID);
	if (dev != AIC_DEVICE_ID_0 && dev != AIC_DEVICE_ID_1)
		return 0;

	line = pci_inb(pa, PCI_I_LINE);
	if (line == 0)		/* this device is not enabled in proms */
		return 0;

	return (1);
}

/*
 * routine used to initially create scb's. Only used
 * by aicattach
 */
static aic_scb_t *
create_scb(aic_softc_t *sc)
{
	aic_scb_t *scb;

	scb = malloc(sizeof *scb, M_DEVBUF, M_WAITOK);
	bzero(scb, sizeof *scb);
	scb->scb_pa = PA(scb);
	scb->scb_msgout[1] = MSG_EXT_MESSAGE;
	scb->scb_msgout[2] = 3;			/* length of message  */
	scb->scb_msgout[3] = XMSG_SDTR;		/* synch transfer request */
	if (sc->sc_flags & AIC_F_SLOWSCSI)
		scb->scb_msgout[4] = 200/4;	/* 200 ns 5 MB per second */
	else
		scb->scb_msgout[4] = 100/4;	/* 100 ns 10 MB per second */
	scb->scb_msgout[5] = 8;			/* maximum sync offset */
	return (scb);
}

/*
 * routine to get scb off free list
 */
static aic_scb_t *
get_scb(aic_softc_t *sc)
{
	aic_scb_t *scb;

	if ((scb = sc->sc_scb_free) == NULL) 
		panic ("aic: get_scb no scb");

	sc->sc_scb_free = scb->scb_next;
	scb->scb_flags = 0;
	return (scb);
}

/*
 * routine to return scb to free list
 */
static void
free_scb(aic_softc_t *sc, aic_scb_t *scb)
{
	scb->scb_next = sc->sc_scb_free;
	sc->sc_scb_free = scb;
}

#ifdef DEBUG

static int tracecnt;
int aic_dumpaddr[] = {0x93,0,0,0,0};

static void
aic_trace(aic_softc_t *sc, int always)
{
	int tmp;
	int i;

	if ((tracecnt++ & 0xf) == 0)
		printf("\n");
	tmp = (INB_SC(AIC_SEQADDR1) << 8) + INB_SC(AIC_SEQADDR0) -1;
	printf("%03x ", tmp);
	for (i=0; aic_dumpaddr[i] != 0; i++)
		if (aic_dumpaddr[i] == tmp)
			break;
	if (aic_dumpaddr[i] != 0 || always) {
		for (tmp = 0; tmp < AIC_SRAMSIZE; tmp++) {
			if (!(tmp & 0xf))
				printf("\n");
			printf("%02x ",INB_SC(tmp + AIC_SRAM));
		}
		for (tmp = 0; tmp < AIC_SCBSIZE; tmp++) {
			if (!(tmp & 0xf))
				printf("\n");
			printf("%02x ",INB_SC(tmp + AIC_SCB));
		}
		tracecnt=0;
	}
}
#endif


/*
 * Routine to watch for missed interrupts. Will allow device to operate
 * if it can not interrupt at all.
 */
static void
aicwatch(void *sc0)
{
	struct aic_softc *sc;
	int i, scb, s;
	u_char intstat;
#ifdef DEBUG
	aic_io_t *aic = sc->sc_vio;
#endif

	sc = sc0;
	sc->sc_watchon = 0;

	if (sc->sc_ioout == 0)
		return;

	s = splbio();

	intstat = INB_SC(AIC_INTSTAT);
	if ((intstat & AIC_ALLINT) == 0) {
#ifdef DEBUG
		if ((INB_SC(AIC_HCNTRL) & AIC_PAUSE) &&
		    (INB_SC(AIC_SEQCTL) & AIC_STEP)) {
			aic_trace(sc, 0);
			OUTB_SC(AIC_HCNTRL, AIC_INTEN);
		}
#endif
		splx(s);
		sc->sc_watchon = 1;
		timeout(aicwatch, sc, sc->sc_watchinterval);
		OUTB_SC(AIC_HCNTRL, AIC_INTEN);
		return;
	}
	i = sc->sc_icnt;
	splx(s);
	delay(1);
	s = splbio();
	if (sc->sc_icnt == i) {
		if ( sc->sc_watchinterval != AIC_FASTWATCH)
			printf("%s: missing interrupt(s)\n",
			    sc->sc_hba.hba_dev.dv_xname);
		aicintr(sc);
		sc->sc_watchinterval = AIC_FASTWATCH;
	}
	splx(s);
	if (sc->sc_ioout == 0 || sc->sc_watchon)
		return;
	sc->sc_watchon = 1;
	timeout(aicwatch, sc, sc->sc_watchinterval);
}

static void
aic_cmdcmplt(aic_softc_t *sc)
{
	int i, j;
	aic_target_t *t;
	aic_scb_t *scb;

	for (i = 0; i < AIC_TARGETS; i++) {
		t = &sc->sc_targets[i];
		if (!(t->t_flags & AIC_COMPLETE))
			continue;

		if (!sc->sc_icmd)
			t->t_flags &= ~AIC_COMPLETE;

		for (j = 0; j < AIC_TARGETSCBS; j++) {
			scb = t->t_v_scbs[j];
			if (scb == NULL || !(scb->scb_flags & AIC_S_DONE))
				continue;
			if (sc->sc_icmd && !scb->scb_icmd)
				continue;
			t->t_v_scbs[j] = NULL;
			t->t_p_scbs[j] = NULL;
			done_scb(sc, scb);
		}
	}
}

static void
start_scb(register aic_softc_t *sc, aic_scb_t *scb)
{
	int  i;
	aic_target_t *t;


	t = &sc->sc_targets[scb->scb_target];

	aic_sgmap(scb);
	scb->scb_sgentry = PA(scb->scb_sglist);
	scb->scb_sgactive.sg_flags = 0;

	scb->scb_msgout[0] = scb->scb_lun | MSG_IDENTIFY;

	if (!(sc->sc_nodisconnect & (1 << scb->scb_target)))
		scb->scb_msgout[0] |= MSG_IDENTIFY_DR;

	if (scb->scb_cdb.cdb_bytes[0] != CMD_TEST_UNIT_READY ||
	    scb->scb_lun != 0 || (sc->sc_nosync & (1 << scb->scb_target))) {
		scb->scb_msgolen = 1;
	} else {
		scb->scb_msgolen = 6;
	}

	scb->scb_scsiid = (scb->scb_target << AIC_TID_S) | sc->sc_id;
	scb->scb_scsirate = t->t_scsirate;

	sc->sc_ioout++;
	if (!sc->sc_probetime && !sc->sc_watchon) {
		sc->sc_watchon = 1;
		timeout(aicwatch, sc, sc->sc_watchinterval);
	}
	t->t_p_scbs[0] = PA(scb);
	t->t_v_scbs[0] = scb;
	t->t_newwork = PA(scb) | 1;  /*	Least significant bit flag not addr */

	
	OUTB_SC(AIC_HCNTRL, AIC_PAUSE);
	while (!(INB_SC(AIC_HCNTRL) & AIC_PAUSE))
		continue;
	OUTB_SC(AIC_QINFIFO, 1);
#ifdef DEBUG
	if (INB_SC(AIC_SEQCTL) & AIC_STEP)
		aic_trace(sc, 0);
#endif
	OUTB_SC(AIC_HCNTRL, AIC_INTEN);
	return;
}

static void
done_scb(aic_softc_t *sc, aic_scb_t *scb)
{
	int status;
	int resid = 0;
	aic_sg_t *sg;
	int i;
	aic_target_t *t = &sc->sc_targets[scb->scb_target];

	/*
	 * first see if we we tried to negotiate sync
	 * if we don't get back a SDTR we set the offset
	 * to zero. This makes it look like the target
	 * replied to the negotiation with zero.
	 * Once the target has said no to sync negotiation then it
	 * we never ask again.
	 */

	if (scb->scb_cdb.cdb_bytes[0] == CMD_TEST_UNIT_READY &&
	    scb->scb_lun == 0 && !(sc->sc_nosync & (1 << scb->scb_target))) {
		if (scb->scb_msgin[2] != XMSG_SDTR)
			scb->scb_msgin[4] = 0;
		i = scb->scb_msgin[3] * 4;
		i -= 100;		/* subtract off 100ns minimum setting */
		i /= 25;		/* turn into aic units (25ns) */
		/* 
		 * The transfer rate is slower than we can set to. 
		 * XXXX This is not right.
		 * For now set to our minimum. Should renegotiate
		 * transfer to be off.
		 */
		if (i > 7) 
			i = 7;
		if (scb->scb_msgin[4] == 0) {
			t->t_scsirate = 0;	/* target doesn't want sync */
			sc->sc_nosync |= 1 << scb->scb_target;
		} else
			t->t_scsirate = (i << 4) | scb->scb_msgin[4];
	}
	sc->sc_ioout--;

	/*
	 * icmd wants scb back
	 * it holds its own pointer to it
	 */
	if (scb->scb_icmd) {
		scb->scb_done = 1;
		return;
	}
	status = scb->scb_status & 0x80 ? -1 : scb->scb_status;

	/*
	 * turn physical scatter gather address back into virtual
	 * address
	 */
	sg = scb->scb_sglist;
	sg += (scb->scb_sgentry - PA(scb->scb_sglist)) / sizeof (aic_sg_t);

	/*
	 * loop through any remain scatter gather entries
	 * summing up resid. If Loaded bit is turned of this
	 * means that the firmware completely processed this
	 * scatter gather entry. 
	 */
	if (scb->scb_sgactive.sg_flags & AIC_S_SGF_LOADED) {
		resid += scb->scb_sgactive.sg_count;
		sg++;
	}
	while (sg->sg_flags & AIC_S_SGF_OVERRUN == 0)
		resid += sg->sg_count;

	(*scb->scb_bintr)(scb->scb_bintrdev, status, resid);
	free_scb(sc, scb);
}

/*
 * This code originally lifted from ab_subr.c
 *
 * If the transfer uses a chain (bp->b_chain != NULL), we assume that no
 * <addr,len> pair ever crosses a page boundary.  In any case, at most two
 * pages can be partial (one at the start and one at the end).
 *
 * (Lots of panics here, from sheer raging paranoia.)
 */
static void
aic_sgmap(aic_scb_t *scb)
{
	int len, i, n, o, pages;
	vm_offset_t v;
	struct buf *bp = scb->scb_bp;
	aic_sg_t *sg = scb->scb_sglist;

	/*
	 * ###: We used to xfer 0 bytes at addr 0 for len < 0, but I
	 * think this is an error.  If not, our caller will have to
	 * check len < 0 and change that to len = 0 anyway.
	 */
	len = bp->b_iocount;
	if (len < 0)
		panic("aic_sgmap");
	if (len == 0) {
		sg->sg_flags = AIC_S_SGF_OVERRUN;
		return;
	}
	v = (vm_offset_t)bp->b_un.b_addr;
	o = v & PGOFSET;
	pages = i386_btop(i386_round_page(len + o));

	/*
	 * Transfer may also be chained, in which case we must step
	 * to the next buffer each time bp->b_bcount runs out (note
	 * that len is the head buffer's bp->b_iocount).
	 */

	if (pages == 1) {
		sg->sg_addr = pmap_extract(kernel_pmap, v);
		sg->sg_count = bp->b_bcount;
		sg->sg_flags = 0;
		(++sg)->sg_flags = AIC_S_SGF_OVERRUN;
		return;
	}

	/* First page may start at some offset, and hence be short. */
	n = bp->b_bcount;
	i = NBPG - o;
	sg->sg_addr = pmap_extract(kernel_pmap, v);
	sg->sg_count = i;
	sg->sg_flags = 0;
	sg++;
	v += i;
	n -= i;

	/* Pages 1 through (pages-1), if pages > 2, are full size. */
	for (i = 2; i < pages; i++) {
		if (n <= 0) {
			if (n < 0 || bp->b_chain == NULL)
				panic("aic_sgmap mid chain");
			bp = bp->b_chain;
			n = bp->b_bcount;
			v = (vm_offset_t)bp->b_un.b_addr;
			if (v & PGOFSET)
				panic("aic_sgmap mid addr");
		}
		sg->sg_addr = pmap_extract(kernel_pmap, v);
		sg->sg_count = NBPG;
		sg->sg_flags = 0;
		sg++;
		v += NBPG;
		n -= NBPG;
	}

	/* Last page is n remaining bytes. */
	if (n <= 0) {
		if (n < 0 || bp->b_chain == NULL)
			panic("aic_sgmap last chain");
		bp = bp->b_chain;
		n = bp->b_bcount;
		v = (vm_offset_t)bp->b_un.b_addr;
		if (v & PGOFSET)
			panic("aic_sgmap last addr");
	}
	if (n > NBPG)
		panic("aic_sgmap lastpg %d", n);
	/*
	 * ab_subr looked wrong. It seems like it will plant one extra
	 * address with a length of zero 
	 */
	if (n != 0) {
		sg->sg_addr = pmap_extract(kernel_pmap, v);
		sg->sg_count = n;
		sg->sg_flags = 0;
		sg++;
	}
	(sg)->sg_flags = AIC_S_SGF_OVERRUN;
	return;
}


/*
 * reset the hardware including doing a bus reset. If we allow
 * multiple masters then the bus reset should be conditional.
 */
static int
aicreset_hw(aic_softc_t *sc)
{
	int i;
	u_char* code  = (u_char*)AIC_code;
#ifdef DEBUG
aic_io_t *aic = sc->sc_vio;
#endif

	/*
	 * reset the chip
	 */
	OUTB_SC(AIC_HCNTRL, AIC_CHIPRST | AIC_PAUSE);
	i=10000;
	while (INB_SC(AIC_HCNTRL & AIC_CHIPRST)) {
		if (i-- == 0) {
			printf("aic failed to reset");
			break;
		}
		delay(100);
	}

	OUTB_SC(AIC_HCNTRL, AIC_PAUSE);
	OUTB_SC(AIC_SBLKCTL, AIC_DIAGLEDEN);	/* also selects bus a */

	/*
	 * reset the bus
	 */
	if (sc->sc_type == AIC_EISA) {
		OUTB_SC(AIC_BCTL, AIC_ENABLE);
		OUTB_SC(AIC_BRKADDR1, AIC_BRKDIS);	/* disable breakpoint */
	}

	OUTB_SC(AIC_SCSISEQ, AIC_SCSIRSTO);
	delay(300000);			/* XXX  check were this came from */
	OUTB_SC(AIC_SCSISEQ, 0);

	OUTB_SC(AIC_BUSSPD, 0xc0);



	/*
	 * load firmware
	 */
	OUTB_SC(AIC_SEQCTL, AIC_LOADRAM | AIC_PERRORDIS);
	OUTB_SC(AIC_SEQADDR0, 0);
	OUTB_SC(AIC_SEQADDR1, 0);
	for (i = 0; i < sizeof AIC_code; i++)
		OUTB_SC(AIC_SEQRAM, code[i]);

	/*
	 * verify firmware
	 */
	OUTB_SC(AIC_SEQCTL, 0);
	OUTB_SC(AIC_SEQCTL, AIC_LOADRAM | AIC_PERRORDIS);
	OUTB_SC(AIC_SEQADDR0, 0);
	OUTB_SC(AIC_SEQADDR1, 0);
	for (i = 0; i < sizeof AIC_code; i++) {
		u_char tmp;
		tmp = INB_SC(AIC_SEQRAM);
		if (tmp != code[i]) {
			printf("aic error load seq ram should be %x, was %x\n",
			    code[i], tmp);
			return -1;
		}
	}

	OUTB_SC(AIC_SEQCTL, 0);
	OUTB_SC(AIC_SEQCTL, AIC_SEQRESET);	/* set sequencer pc to zero */

	OUTB_SC(AIC_CLRINT, AIC_SEQINT);
	if (INB_SC(AIC_INTSTAT) & AIC_SEQINT) {
		printf("unable to clear SEQINT\n");
		return -1;
	}

	OUTB_SC(AIC_HCNTRL, 0);			/* let sequencer run */
	for (i = 0; i < 100; i++) {
		if (INB_SC(AIC_INTSTAT) & AIC_SEQINT)
			break;
		delay (1);
	}
	/*
	 * setup sequencer memory 
	 */
	OUTB_SC(AIC_CLRINT, AIC_SEQINT);	/* clear everything */

	for (i = AIC_SCB; i < AIC_SCB + AIC_SCBSIZE; i++)
		OUTB_SC(i, 0);

	for (i = AIC_SRAM; i < AIC_SRAM + AIC_SRAMSIZE; i++)
		OUTB_SC(i, 0);

	OUTL_SC(AIC_targethead, PA(sc->sc_targets));
	OUTB_SC(AIC_cio_targetsize, AIC_t_copysize);
	OUTB_SC(AIC_cio_scbsize, AIC_scb_end);

	OUTB_SC(AIC_SIMODE0, 0);
	OUTB_SC(AIC_SIMODE1, 0);

	OUTB_SC(AIC_CLRSINT0,
	    AIC_SELDO | AIC_SELDI | AIC_SELINGO | AIC_SWRAP | AIC_SPIORDY);

	OUTB_SC(AIC_CLRSINT1, 0xff);	/* clear everything */


	bzero((void*)sc->sc_targets, sizeof sc->sc_targets);
	sc->sc_targets[AIC_TARGETS - 1].t_flags = AIC_LISTEND;

	OUTB_SC(AIC_SXFRCTL0, AIC_SPIOEN | AIC_DFON | AIC_DFPEXP);

	i = AIC_ENSTIMER;
	if (!(sc->sc_flags & AIC_F_TERMOFF))
		i |= AIC_STPWEN; 
	if (!(sc->sc_flags & AIC_F_ACKNEGOFF))
		i |= AIC_ACTNEGEN;
	OUTB_SC(AIC_SXFRCTL1, i);

#if 0
	OUTB_SC(AIC_BUSSPD, 0xc0);
#endif

	OUTB_SC(AIC_CLRINT, 0xff);

	OUTB_SC(AIC_HCNTRL, AIC_INTEN);

	for (i = 0; i < 1000; i++)
	    aicintr(sc);

	return 0;
	
}

static void
aic_outb(aic_softc_t *sc, u_int offset, u_char value)
{
	if (sc->sc_mio) {
		volatile u_char *pnt = sc->sc_mio;

		pnt += offset;
		*pnt = value;
	} else {
		int pnt = sc->sc_pio;

		pnt += offset;
		outb(pnt, value);
	}
}

static void
aic_outl(aic_softc_t *sc, u_int offset, u_long value)
{
	int i;

	if (sc->sc_mio) {
		volatile u_char *pnt = sc->sc_mio;

		pnt += offset;
		for (i = 0; i < 4; i++) 
			*pnt++ = value >> (i * 8);
	} else {
		int pnt = sc->sc_pio;

		pnt += offset;
		for (i = 0; i < 4; i++) 
			outb(pnt++, value >> (i * 8));
	}
}

static u_char
aic_inb(aic_softc_t *sc, u_int offset)
{
	volatile u_char *pnt;
	u_char ret;


	if (sc->sc_mio) {
		volatile u_char *pnt = sc->sc_mio;

		pnt += offset;
		ret = *pnt;
	} else {
		int pnt = sc->sc_pio;

		pnt += offset;
		ret = inb(pnt);
	}
	return ret;
}

static int
aic_forceintr(u_short *iobp)
{
	u_short iob;
	iob = *iobp;
	
	outb(iob + AIC_BCTL, AIC_ENABLE);

	outb(iob + AIC_HCNTRL, 0);
	outb(iob + AIC_CLRINT, 0xff);
	outb(iob + AIC_HCNTRL, AIC_INTEN);
	outb(iob + AIC_HCNTRL, AIC_INTEN | AIC_SWINT);

}
