/*-
 * Copyright (c) 1991, 1992, 1993, 1994 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: aha.c,v 2.3 1995/11/13 16:16:08 cp Exp $
 */

/*
 * Adaptec AHA-1540[BC]/1542[BC] SCSI host adapter driver
 *
 * TODO:
 *	Compute scatter/gather maps when buffers are queued, not when used
 *	Compute CCB's when buffers are queued
 *	Add support for multiple outstanding commands on the same target
 *	Add hacks for AHA-1540A cards
 */

#include <sys/param.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/device.h>
#include <sys/malloc.h>

#include <vm/vm.h>

#include <dev/scsi/scsi.h>
#include <dev/scsi/scsivar.h>
#include <dev/scsi/disk.h>
#include <dev/scsi/tape.h>

#include <machine/cpu.h>

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/isa/icu.h>
#include <i386/isa/ahareg.h>
#include <i386/isa/abvar.h>

extern int maxmem;

/*
 * Having more than one or two mboxes is useful mainly for debugging.
 * These sizes are calculated to fit inside 2 KB.
 */
#define	NMBOX	8
#define	NSCCB	12
#define	MAXSG	17

/*
 * See abvar.h for common variables.  Note, this MUST begin with the
 * ab_softc.
 */
struct aha_softc {
	struct	ab_softc sc_ab;
#define	sc_hba		sc_ab.ab_hba

	/* in and out mailbox info */
#define	mbobase(sc)	((struct mbox24 *)(sc)->sc_ab.ab_mbobase)
#define	mbolim(sc)	((struct mbox24 *)(sc)->sc_ab.ab_mbolim)
#define	mbibase(sc)	((struct mbox24 *)(sc)->sc_ab.ab_mbibase)
#define	mbilim(sc)	((struct mbox24 *)(sc)->sc_ab.ab_mbilim)
	struct	mbox24 *sc_outbox;	/* next available out mbox */
	struct	mbox24 *sc_inbox;	/* next expected in mbox */

	/* command control block information */
#define	sccbbase(sc)	((struct soft_ccb *)(sc)->sc_ab.ab_sccbv)
#define	sccblim(sc)	((struct soft_ccb *)(sc)->sc_ab.ab_sccblim)
	struct	soft_ccb *sc_nextsccb;	/* find free SCCBs starting here */
	struct	soft_ccb *sc_gosccb;	/* send sccb from ahastart to ahago */

	/* bounce buffers, per extant target */
	struct	aha_bounce *sc_bo[8];	/* for faking DMA above 16 MB */
};

/*
 * Macros for incrementing circular pointers.
 */
#define	INC_CIR(v, base, lim) { \
	if (++(v) >= (lim)) \
		(v) = (base); \
}
#define	INC_MBI(mb, sc)	INC_CIR(mb, mbibase(sc), mbilim(sc))
#define	INC_MBO(mb, sc)	INC_CIR(mb, mbobase(sc), mbolim(sc))
#define	INC_SCCB(s, sc)	INC_CIR(s, sccbbase(sc), sccblim(sc))

/*
 * Software CCBs for 24-bit AHA-only code.  Need bounce buffer info
 * as well as CCB and status.
 */
struct soft_ccb {
	struct	ccb24 sccb_ccb;		/* hardware CCB (MUST BE FIRST) */
	scintr_fn sccb_intr;		/* base interrupt handler */
	struct	device *sccb_intrdev;	/* link back to associated unit */
	time_t	sccb_stamp;		/* timestamp (for watchdog) */
	int	sccb_bounces;		/* number of bounce-ins needed */
	struct	sg24 sccb_sg[MAXSG];	/* scatter/gather map */
};

/*
 * DMA bounce support.
 */
struct aha_bounce {
	u_int	bo_p;		/* physical address of bounce page */
	u_int	bo_v;		/* virtual address of bounce page */
	u_int	bo_dst;		/* virtual address of destination */
	u_int	bo_len;		/* amount of data in the page */
};

/*
 * Given the physical address of a CCB (in, e.g., an AHA mbox), convert
 * to the corresponding virtual address for that CCB.
 */
#define	AHA_CCB_TO_SCCB(sc, p) \
	((struct soft_ccb *)((int)(sc)->sc_ab.ab_sccbv + \
	    (AHA_TO_HOST(p) - (sc)->sc_ab.ab_sccb_phys)))

void	ahaattach __P((struct device *parent, struct device *dev, void *args));
static	int ahaccb __P((struct aha_softc *sc, struct soft_ccb *sccb,
	    int synch, int mbcmd));
int	ahadump __P((struct hba_softc *hba, int targ, struct scsi_cdb *cdb,
	    caddr_t buf, int len));
int	ahago __P((struct device *self, int targ, scintr_fn intr,
	    struct device *dev, struct buf *bp, int pad));
void	ahahbareset __P((struct hba_softc *hba, int resetunits));
int	ahaicmd __P((struct hba_softc *hba, int targ, struct scsi_cdb *cdb,
	    caddr_t buf, int len, int rw));
static	struct mbox24 *ahainbox __P((struct aha_softc *sc, struct mbox24 *mb));
int	ahaintr __P((void *sc0));
int	ahaprobe __P((struct device *parent, struct cfdata *cf, void *aux));
void	ahastart __P((struct device *self, struct sq *sq, struct buf *bp,
	    scdgo_fn dgo, struct device *dev));
void	aharel __P((struct device *self));
static	void ahareset __P((struct aha_softc *sc, int hard));
void	ahawatch __P((void *sc0));
static	void bounce_alloc __P((struct aha_softc *sc, int targ));
static	void bounce_in __P((struct aha_softc *sc, struct soft_ccb *sccb));
static	u_int bounce_out __P((struct aha_softc *sc,
    struct soft_ccb *sccb, int indx, vm_offset_t v, vm_size_t len, int rw));
static	struct soft_ccb *sccballoc __P((struct aha_softc *sc,
	    struct scsi_cdb *cdb));
static	void sccbinit __P((struct aha_softc *sc, struct soft_ccb *sccb,
	    int targ, scintr_fn intr, struct device *dev, struct buf *bp));
static void sginit __P((struct aha_softc *sc, struct soft_ccb *sccb,
	    struct buf *bp));

struct cfdriver ahacd =
    { 0, "aha", ahaprobe, ahaattach, DV_DULL, sizeof(struct aha_softc) };
static struct hbadriver ahahbadriver =
    { ahaicmd, ahadump, ahastart, ahago, aharel, ahahbareset };

/*
 * The Adaptec uses 3-byte big-endian physical addresses internally (argh).
 */
#define HOST_TO_AHA(n, cp) \
	((cp)[0] = (n) >> 16, (cp)[1] = (n) >> 8, (cp)[2] = (n))
#define	AHA_TO_HOST(cp) \
	(((cp)[0] << 16) | ((cp)[1] << 8) | (cp)[2])

static void
host_to_aha(n, cp)
	register u_long n;
	register u_char *cp;
{

	HOST_TO_AHA(n, cp);
}

static u_long
aha_to_host(cp)
	register u_char *cp;
{

	return (AHA_TO_HOST(cp));
}

/*
 * Look for an empty outbox.
 * If we don't find one, return NULL.
 */
static __inline struct mbox24 *
scanmbo(sc, start)
	register struct aha_softc *sc;
	struct mbox24 *start;
{
	struct mbox24 *mbo;
	int i;

	for (mbo = start, i = 0; i < sc->sc_ab.ab_mbox_count; ++i) {
		if (mbo->mb_status == MBOX_O_FREE)
			return (mbo);
		INC_MBO(mbo, sc);
	}
	return (NULL);
}

/*
 * Look for a filled inbox.
 * If we don't find one, return start.
 */
static struct mbox24 *
scanmbi(sc, start)
	register struct aha_softc *sc;
	struct mbox24 *start;
{
	struct mbox24 *mbi;
	int i;

	for (mbi = start, i = 0; i < sc->sc_ab.ab_mbox_count; ++i) {
		if (mbi->mb_status != MBOX_I_FREE)
			return (mbi);
		INC_MBI(mbi, sc);
	}
	return (start);
}

/*
 * Probe for an Adaptec or Buslogic.  If we find it we will use it,
 * in 24-bit mode only (hence if you want bha.c to override, it must
 * get probed first).
 */
int
ahaprobe(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{

	return (ab_probe((struct isa_attach_args *)aux, 0));
}

/*
 * Make sure that we don't get stuck.
 * We check for stale mailboxes.
 */
u_long aha_missed_mbif;
u_long aha_aborted_command;
u_long aha_dobusreset = AB_DOBUSRESET; 
u_long aha_wedgetime = AB_WEDGETIME; 


void
ahawatch(sc0)
	void *sc0;
{
	struct aha_softc *sc;
	struct soft_ccb *sccb;
	struct mbox24 *mb;
	int i, op, s;
	time_t now;
	int fasttoolong = 0;
	int slowtoolong = 0;

	sc = sc0;
	sc->sc_ab.ab_watchon = 0;
	if (sc->sc_ab.ab_ioout == 0)
		return;

	/* did we miss an MBIF interrupt? */
	mb = sc->sc_inbox;
	while ((mb = scanmbi(sc, mb))->mb_status != MBOX_I_FREE) {
		s = splbio();	/* avoid races */
		if (mb->mb_status != MBOX_I_FREE) {
			++aha_missed_mbif;
			if (!sc->sc_ab.ab_modifyintdone)
				ab_modifyint(&sc->sc_ab);
			mb = ahainbox(sc, mb);
		}
		splx(s);
		if (sc->sc_ab.ab_ioout == 0)
			return;
	}

	if (!sc->sc_ab.ab_watchon) {
		sc->sc_ab.ab_watchon = 1;
		timeout(ahawatch, sc, sc->sc_ab.ab_watchtime);
	}
	if (--sc->sc_ab.ab_wedgecounter != 0)
		return;
	sc->sc_ab.ab_wedgecounter = sc->sc_ab.ab_wedgetime;

	/*
	 * Scan sccb list, look for stale entries.
	 * If a read or write entry hasn't been serviced in a reasonable
	 * amount of time, complain (should abort the transaction?).
	 */
	now = time.tv_sec;
	sccb = sc->sc_nextsccb;
	for (i = 0; i < sc->sc_ab.ab_sccb_count; ++i) {
		op = sccb->sccb_ccb.ccb_cdb6.cdb_cmd;
		if (sccb->sccb_ccb.ccb_opcode != CCB_FREE &&
		    sccb->sccb_stamp < (now - aha_wedgetime)) {
		    	s = splbio();
		    	if (sccb->sccb_ccb.ccb_opcode != CCB_FREE &&
			    sccb->sccb_stamp < (now - aha_wedgetime)) {
				if (op == CMD_READ10 || op == CMD_WRITE10 ||
				     op == CMD_READ || op == CMD_WRITE) {
					fasttoolong = 1;
					printf("%s: operation time exceeds %d seconds\n",
					    sc->sc_hba.hba_dev.dv_xname,
					    now - sccb->sccb_stamp);
				} else
					slowtoolong = 1;
			}
			splx(s);
		}
		INC_SCCB(sccb, sc);
	}
	if (fasttoolong && !slowtoolong && aha_dobusreset) {
		scintr_fn save_intr[NSCCB];	
		struct	device *save_intrdev[NSCCB];

		s = splbio();
		sccb = sc->sc_nextsccb;
		for (i = 0; i < sc->sc_ab.ab_sccb_count; ++i) {
			if (sccb->sccb_ccb.ccb_opcode != CCB_FREE) {
				save_intr[i] = sccb->sccb_intr;
				save_intrdev[i] = sccb->sccb_intrdev;
			} else 
				save_intr[i] = NULL;
			INC_SCCB(sccb, sc);
		}

		printf("%s: attempting bus reset",
		    sc->sc_hba.hba_dev.dv_xname);

		sc->sc_ab.ab_ioout = 0;
		bzero(sc->sc_ab.ab_mboxv,
		    2 * NMBOX * sizeof(struct mbox24) + 
		    NSCCB * sizeof(struct soft_ccb));

		ahareset(sc, AB_HARDRESET);
		bzero(sc->sc_ab.ab_mboxv,
		    2 * NMBOX * sizeof(struct mbox24) + 
		    NSCCB * sizeof(struct soft_ccb));

		for (i = 0; i < sc->sc_ab.ab_sccb_count; ++i) {
			sccb->sccb_ccb.ccb_opcode = CCB_FREE;
			INC_SCCB(sccb, sc);
		}

		DELAY(5000000);
		printf("\n");
		for (i = 0; i < sc->sc_ab.ab_sccb_count; ++i) {
			struct sq *sq;

			if (save_intr[i] == NULL)
				continue;

			sc->sc_ab.ab_hba.hba_intr = save_intr[i];
			sc->sc_ab.ab_hba.hba_intrdev = save_intrdev[i];

			(*sc->sc_hba.hba_intr)(sc->sc_hba.hba_intrdev, 
			    HBAINTR_BUSRESET, 0);

			if ((sq = sc->sc_hba.hba_head) != NULL) {
				sc->sc_hba.hba_head = sq->sq_forw;
				sc->sc_gosccb =
				    sccballoc(sc, (struct scsi_cdb *)NULL);
				(*sq->sq_dgo)(sq->sq_dev, (struct scsi_cdb *)
				    &sc->sc_gosccb->sccb_ccb.ccb_cdb);
			}
		}
		sc->sc_ab.ab_wedgecounter = sc->sc_ab.ab_wedgetime;
		splx(s);
	}
}

/*
 * Allocate the bounce map for the given target.
 */
static int
aha_bouncealloc(ab, targ)
	struct ab_softc *ab;
	int targ;
{
	struct aha_bounce *bo0, *bo;
	register vm_offset_t p;
	caddr_t buf0;
	vm_offset_t buf;
	int n;

	bo0 = malloc(MAXSG * sizeof *bo, M_DEVBUF, M_WAITOK);
	buf0 = malloc(MAXSG * NBPG, M_DEVBUF, M_WAITOK);
	buf = (vm_offset_t)buf0;
	n = MAXSG;
	for (bo = bo0; --n >= 0; bo++, buf += NBPG) {
		p = pmap_extract(kernel_pmap, buf);
		if (p >= ISA_MAXADDR) {
			printf("%s: can't create bounce map, tg%d unusable\n",
			    ab->ab_hba.hba_dev.dv_xname, targ);
			/* panic("bounce_alloc"); */
			free(bo0, M_DEVBUF);
			free(buf0, M_DEVBUF);
			return (ENOMEM);
		}
		bo->bo_p = p;
		bo->bo_v = buf;
		bo->bo_dst = 0;
		bo->bo_len = 0;
	}
	((struct aha_softc *)ab)->sc_bo[targ] = bo0;
	return (0);
}

void
ahaattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct aha_softc *sc = (struct aha_softc *)self;
	struct soft_ccb *sccb;
	int i;

	/*
	 * Most of the work is in common code, but we need to do some
	 * AHA-specific setup in between, so the job is split.
	 *
	 * XXX dynamically configure sc_mbox_count and sc_sccb_count?
	 */
	sc->sc_ab.ab_ih.ih_fun = ahaintr;
	ab_attach(&sc->sc_ab, (struct isa_attach_args *)aux, 0,
	    NMBOX, sizeof(struct mbox24),
	    NSCCB, sizeof(struct soft_ccb));
	sc->sc_outbox = mbobase(sc);
	sc->sc_inbox = mbibase(sc);

	sccb = sc->sc_ab.ab_sccbv;
	sc->sc_nextsccb = sccb;
	sc->sc_gosccb = NULL;
	for (i = 0; i < sc->sc_ab.ab_sccb_count; ++sccb, ++i)
		sccb->sccb_ccb.ccb_opcode = CCB_FREE;

	sc->sc_hba.hba_driver = &ahahbadriver;
	ahareset(sc, AB_SOFTRESET);

	ab_attach2(&sc->sc_ab,
	    maxmem * NBPG >= ISA_MAXADDR ? aha_bouncealloc : NULL);
}

static void
ahareset(sc, hard)
	struct aha_softc *sc;
	int hard;
{
	u_char out[4];

	out[0] = sc->sc_ab.ab_mbox_count;
	host_to_aha(sc->sc_ab.ab_mbox_phys, &out[1]);
	ab_reset(sc->sc_ab.ab_port, AHA_MBOX_INIT, out, hard);
	if (hard == AB_HARDRESET)
		scsi_reset_units(&sc->sc_hba);
}

void
ahahbareset(hba, resetunits)
	struct hba_softc *hba;
	int resetunits;
{
	struct aha_softc *sc = (struct aha_softc *)hba;

	if (resetunits)
		ahareset(sc, AB_HARDRESET);
	else
		ahareset(sc, AB_SOFTRESET);
}

/*
 * Allocate an outbound bounce page, copying data if necessary.
 */
static u_int
bounce_out(sc, sccb, indx, v, len, rw)
	struct aha_softc *sc;
	struct soft_ccb *sccb;
	int indx;
	vm_offset_t v;
	vm_size_t len;
	int rw;
{
	struct aha_bounce *bo;
	vm_size_t o;

	bo = &sc->sc_bo[CCB_C_TARGET(sccb->sccb_ccb.ccb_control)][indx];
	o = v & PGOFSET;
	if (rw == B_READ) {
		bo->bo_dst = v;
		bo->bo_len = len;
		++sccb->sccb_bounces;
	} else
		bcopy((void *)v, (void *)(bo->bo_v + o), len);
	return (bo->bo_p + o);
}

/*
 * Finish up bouncing in on a read.
 */
static void
bounce_in(sc, sccb)
	struct aha_softc *sc;
	struct soft_ccb *sccb;
{
	vm_offset_t dst;
	struct aha_bounce *bo;

	for (bo = &sc->sc_bo[CCB_C_TARGET(sccb->sccb_ccb.ccb_control)][0];
	    sccb->sccb_bounces > 0; ++bo)
		if ((dst = bo->bo_dst) != 0) {
			bcopy((void *)(bo->bo_v + (dst & PGOFSET)),
			    (void *)dst, bo->bo_len);
			bo->bo_dst = 0;
			--sccb->sccb_bounces;
		}
}

/*
 * Given a buffer describing a transfer, set up a scatter/gather map
 * in a ccb to map that SCSI transfer.
 * Initialize the CCB opcode for the correct transfer type.
 */
static void
sginit(sc, sccb, bp)
	struct aha_softc *sc;
	struct soft_ccb *sccb;
	struct buf *bp;
{
	register int i, len, n, rw, sgpages;
	register u_int phys;
	register vm_offset_t v;
	register struct ccb24 *ccb;
	register struct sg24 *sg;
	u_int aphys;
	struct sg32 sg32[MAXSG];	/* for ab_sgmap */

	ccb = &sccb->sccb_ccb;
	sgpages = ab_sgmap(bp, &sg32[0], MAXSG, &aphys);
	rw = bp->b_flags & B_READ;
	if (sgpages != 0) {
		/*
		 * Translate from sg32 to AHA-style scatter/gather,
		 * using bounce buffers if needed.
		 */
		sg = &sccb->sccb_sg[0];
		v = (vm_offset_t)bp->b_un.b_addr;
		n = bp->b_bcount;
		for (i = 0; i < sgpages; sg++, i++) {
			if (n <= 0) {
				bp = bp->b_chain;
				v = (vm_offset_t)bp->b_un.b_addr;
				n = bp->b_bcount;
			}
			phys = sg32[i].sg_addr;
			len = sg32[i].sg_len;
			if (phys >= ISA_MAXADDR)
				phys = bounce_out(sc, sccb, i, v,
				    (vm_size_t)len, rw);
			HOST_TO_AHA((u_long)phys, sg->sg_addr);
			HOST_TO_AHA((u_long)len, sg->sg_len);
			v += len;
			n -= len;
		}
		/*
		 * Point the CCB at the scatter/gather map.
		 */
		phys = ((int)&sccb->sccb_sg[0] - (int)sc->sc_ab.ab_sccbv) +
		    sc->sc_ab.ab_sccb_phys;
		len = sgpages * sizeof *sg;
		ccb->ccb_opcode = CCB_CMD_SG_RDL;
	} else {
		/*
		 * Point the CCB directly at the transfer, or at a
		 * bounce buffer.
		 */
		len = bp->b_iocount;
		if ((phys = aphys) >= ISA_MAXADDR)
			phys = bounce_out(sc, sccb, 0,
			    (vm_offset_t)bp->b_un.b_addr, (vm_size_t)len, rw);
		ccb->ccb_opcode = CCB_CMD_RDL;
	}
	HOST_TO_AHA((u_long)phys, ccb->ccb_data);
	HOST_TO_AHA((u_long)len, ccb->ccb_datalen);
}

/*
 * Allocate and initialize a software CCB.
 * Must be called at splbio().
 * XXX we wouldn't need splbio() if we didn't call this from ahaintr()!
 */
static struct soft_ccb *
sccballoc(sc, cdb)
	struct aha_softc *sc;
	struct scsi_cdb *cdb;
{
	struct soft_ccb *sccb, *n;
	int i;

	sccb = sc->sc_nextsccb;
	for (i = 0; i < sc->sc_ab.ab_sccb_count; ++i) {
		if (sccb->sccb_ccb.ccb_opcode == CCB_FREE)
			break;
		INC_SCCB(sccb, sc);
	}
	if (sccb->sccb_ccb.ccb_opcode != CCB_FREE)
		panic("sccballoc");
	sccb->sccb_stamp = time.tv_sec;
	sccb->sccb_ccb.ccb_opcode = CCB_CMD_SG_RDL;
	n = sccb;
	INC_SCCB(n, sc);
	sc->sc_nextsccb = n;

	if (cdb)
		bcopy(cdb, sccb->sccb_ccb.ccb_cdbbytes,
		    SCSICMDLEN(cdb->cdb_bytes[0]));

	return (sccb);
}

/*
 * Initialize (most of) an SCCB.
 * We assume that the CDB has already been set up.
 */
static void
sccbinit(sc, sccb, targ, intr, dev, bp)
	struct aha_softc *sc;
	struct soft_ccb *sccb;
	int targ;
	scintr_fn intr;
	struct device *dev;
	struct buf *bp;
{
	struct ccb24 *ccb = &sccb->sccb_ccb;

	/* XXX do we need direction bits if we always check residual count? */
	ccb->ccb_control = CCB_CONTROL(targ, 0, 0, ccb->ccb_cdbbytes[1] >> 5);
	ccb->ccb_cdblen = SCSICMDLEN(ccb->ccb_cdbbytes[0]);

	sginit(sc, sccb, bp);

	/* XXX when we permit multiple commands per target, must change this */
	ccb->ccb_rqslen = 1;	/* disable automatic request sense */
	ccb->ccb_rsv1 = 0;	/* XXX necessary? */
	ccb->ccb_rsv2 = 0;	/* XXX necessary? */

	sccb->sccb_intr = intr;
	sccb->sccb_intrdev = dev;
}

/*
 * Fire off a CCB.
 * If synch is set, poll for completion.
 * Must be called at splbio().
 */
static int
ahaccb(sc, sccb, synch, mbcmd)
	struct aha_softc *sc;
	struct soft_ccb *sccb;
	int synch;
	int mbcmd;
{
	struct ccb24 *ccb;
	u_long ccb_phys;
	struct mbox24 *mb, *n;
	int aha, status;

	if ((mb = scanmbo(sc, sc->sc_outbox)) == NULL)
		/* if we run out of mailboxes here, it's big trouble */
		panic("ahaccb scanmbo");
	n = mb;
	INC_MBO(n, sc);
	sc->sc_outbox = n;
	ccb = &sccb->sccb_ccb;
	ccb_phys = sc->sc_ab.ab_sccb_phys +
	    ((u_long)ccb - (u_long)sc->sc_ab.ab_sccbv);
	HOST_TO_AHA(ccb_phys, mb->mb_ccb);
	mb->mb_cmd = mbcmd;

	/* start the adapter's outbox scan */
	aha = sc->sc_ab.ab_port;
	sc->sc_ab.ab_istat = inb(AHA_INTR(aha)); /* XXX should do more */
	outb(AHA_STAT(aha), AHA_C_IRST);
	while (inb(AHA_STAT(aha)) & AHA_S_CDF)
		continue;
	outb(AHA_DATA(aha), AHA_START_SCSI_CMD);

	if (!synch) {
		sc->sc_ab.ab_ioout++;
		if (!sc->sc_ab.ab_watchon) {
			sc->sc_ab.ab_watchon = 1;
			sc->sc_ab.ab_wedgecounter = sc->sc_ab.ab_wedgetime;
			timeout(ahawatch, sc, sc->sc_ab.ab_watchtime);
		}
		return (0);
	}

	/* wait for notification */
	while ((inb(AHA_INTR(aha)) & AHA_I_ANY) == 0)
		continue;
	sc->sc_ab.ab_istat = inb(AHA_INTR(aha));
	outb(AHA_STAT(aha), AHA_C_IRST);
	if ((sc->sc_ab.ab_istat & AHA_I_MBIF) == 0)
		panic("ahaccb missing mbif");

	/* scan for the inbox */
	/* XXX what do we do if the command hangs? */
	mb = sc->sc_inbox;
	for (;;) {
		mb = scanmbi(sc, mb);
		if (mb->mb_status == MBOX_I_FREE)
			continue;
		if (aha_to_host(mb->mb_ccb) == ccb_phys)
			break;
		INC_MBI(mb, sc);
	}
	sc->sc_ab.ab_resid = aha_to_host(ccb->ccb_datalen);
	if ((sc->sc_ab.ab_stat = ccb->ccb_hastat) == CCB_H_NORMAL)
		status = ccb->ccb_tarstat;
	else if (ccb->ccb_cdbbytes[0] == CMD_TEST_UNIT_READY)
		/* probing, don't make noise */
		status = -1;
	else {
		ab_error(&sc->sc_ab, ccb->ccb_hastat);
		status = -1;
	}
	/*
	 * We may have skipped over other mboxes above, so do not change
	 * sc_inbox.
	 */
	mb->mb_status = MBOX_I_FREE;
	if (sc->sc_gosccb == sccb)
		sc->sc_gosccb = 0;
	if (sccb->sccb_bounces > 0)
		bounce_in(sc, sccb);
	ccb->ccb_opcode = CCB_FREE;

	return (status);
}


int
ahaicmd(hba, targ, cdb, buf, len, rw)
	struct hba_softc *hba;
	int targ;
	struct scsi_cdb *cdb;
	caddr_t buf;
	int len, rw;
{
	struct aha_softc *sc = (struct aha_softc *)hba;
	struct soft_ccb *sccb;
	int s, error;
	struct buf bufhdr;

	s = splbio();
	sccb = sccballoc(sc, cdb);
	splx(s);
	bufhdr.b_un.b_addr = buf;
	bufhdr.b_iocount = bufhdr.b_bcount = len;
	bufhdr.b_flags = rw;
	bufhdr.b_chain = NULL;
	sccbinit(sc, sccb, targ, NULL, NULL, &bufhdr);
	s = splbio();
	error = ahaccb(sc, sccb, 1, MBOX_O_START);
	splx(s);
	return (error);
}

int
ahadump(hba, targ, cdb, buf, len)
	struct hba_softc *hba;
	int targ;
	struct scsi_cdb *cdb;
	caddr_t buf;
	int len;
{
	struct aha_softc *sc = (struct aha_softc *)hba;
	struct soft_ccb *sccb;
	struct ccb24 *ccb;
	u_long from, to, lastfrom;

	/*
	 * Several assumptions:
	 * +	The upper-level dump code always calls us on aligned
	 *	2^n chunks which never cross 64 KB physical memory boundaries
	 * +	We're running in virtual mode with interrupts blocked
	 */

	sccb = sccballoc(sc, cdb);
	ccb = &sccb->sccb_ccb;

	ccb->ccb_opcode = CCB_CMD_RDL;
	ccb->ccb_control = CCB_CONTROL(targ, 0, 0, ccb->ccb_cdbbytes[1] >> 5);
	ccb->ccb_cdblen = SCSICMDLEN(ccb->ccb_cdbbytes[0]);
	ccb->ccb_rqslen = 1;	/* disable automatic request sense */
	if ((vm_offset_t)buf >= ISA_MAXADDR) {
		/* assumes len <= MAXSG * NBPG and contiguous bounce pages */
		from = btoc((u_long)buf);
		to = btoc(sc->sc_bo[targ][0].bo_p);
		lastfrom = btoc((vm_offset_t)&buf[len]);
		while (from < lastfrom)
			physcopyseg(from++, to++);
		buf = (caddr_t)sc->sc_bo[targ][0].bo_p;
	}
	host_to_aha((u_long)buf, ccb->ccb_data);
	host_to_aha((u_long)len, ccb->ccb_datalen);

	sccb->sccb_intr = NULL;
	sccb->sccb_intrdev = 0;

	return (ahaccb(sc, sccb, 1, MBOX_O_START));
}

/*
 * Start a transfer.
 *
 * Since the AHA-154x handles transactions in parallel (with disconnect /
 * reconnect), we don't have to queue requests for the host adapter.
 *
 * This code is NOT re-entrant: (*dgo)() must call ahago() without calling
 * ahastart() first (because of the sc_gosccb kluge).
 */
void
ahastart(self, sq, bp, dgo, dev)
	struct device *self;
	struct sq *sq;
	struct buf *bp;
	scdgo_fn dgo;
	struct device *dev;
{
	struct aha_softc *sc = (struct aha_softc *)self;

	if (bp) {
		/* asynch transaction */
		sc->sc_gosccb = sccballoc(sc, (struct scsi_cdb *)NULL);
		(*dgo)(dev,
		    (struct scsi_cdb *)&sc->sc_gosccb->sccb_ccb.ccb_cdb);
		return;
	}
	/* let ahaicmd() allocate its own sccb */
	(*dgo)(dev, (struct scsi_cdb *)NULL);
}

/*
 * Get the host adapter going on a command.
 *
 * XXX should we allocate the DMA channel here, rather than dedicate one?
 * XXX must be called at splbio() since interrupts can call it
 * XXX but it'd be much better for the work to get done at low ipl!
 */
int
ahago(self, targ, intr, dev, bp, pad)
	struct device *self;
	int targ;
	scintr_fn intr;
	struct device *dev;
	struct buf *bp;
	int pad;
{
	struct aha_softc *sc = (struct aha_softc *)self;

	sccbinit(sc, sc->sc_gosccb, targ, intr, dev, bp);
	return (ahaccb(sc, sc->sc_gosccb, 0, MBOX_O_START));
}

/*
 * Handle a filled in-mailbox; return the next inbox.
 * Leaves sc_inbox pointing at the next inbox as a side effect.
 */
static struct mbox24 *
ahainbox(sc, mbi)
	struct aha_softc *sc;
	struct mbox24 *mbi;
{
	struct soft_ccb *sccb = AHA_CCB_TO_SCCB(sc, mbi->mb_ccb);
	struct ccb24 *ccb = &sccb->sccb_ccb;
	struct sq *sq;
	int status = -1;

	sc->sc_hba.hba_intr = sccb->sccb_intr;
	sc->sc_hba.hba_intrdev = sccb->sccb_intrdev;
	sc->sc_ab.ab_resid = AHA_TO_HOST(ccb->ccb_datalen);
	sc->sc_ab.ab_targ = CCB_C_TARGET(ccb->ccb_control);
	sc->sc_ab.ab_lun = CCB_C_LUN(ccb->ccb_control);
	sc->sc_ab.ab_stat = ccb->ccb_hastat;
	sc->sc_ab.ab_ioout--;
	sc->sc_ab.ab_iototal++;

	switch (mbi->mb_status) {

	case MBOX_I_ERROR:
		if (ccb->ccb_hastat == CCB_H_NORMAL) {
			status = ccb->ccb_tarstat;
			if ((status & STS_MASK) != STS_GOOD)
				break;
		}
		/* print vaguely informative messages */
		ab_prmbox(&sc->sc_ab, mbi->mb_status,
		    ccb->ccb_opcode, ccb->ccb_hastat);
		break;

	case MBOX_I_COMPLETED:
		status = ccb->ccb_tarstat;
		break;

	case MBOX_I_ABORTED:
		break;

	case MBOX_I_ABORT_FAILED:
		if (ccb->ccb_opcode == CCB_FREE) {
			mbi->mb_status = MBOX_I_FREE;
			INC_MBI(mbi, sc);
			return (sc->sc_inbox = mbi);
		}
		/* FALLTHROUGH */

	default:
		ab_prmbox(&sc->sc_ab, mbi->mb_status,
		    ccb->ccb_opcode, ccb->ccb_hastat);
		break;
	}

	/* free up resources */
	mbi->mb_status = MBOX_I_FREE;
	if (sc->sc_gosccb == sccb)
		sc->sc_gosccb = NULL;
	if (sccb->sccb_bounces > 0)
		bounce_in(sc, sccb);
	ccb->ccb_opcode = CCB_FREE;

	if (sc->sc_hba.hba_intr && sc->sc_hba.hba_intrdev) {
		/*
		 * For non-immediate commands,
		 * pass status back to higher levels and
		 * start the next transfer.
		 */
		(*sc->sc_hba.hba_intr)(sc->sc_hba.hba_intrdev, status,
		    sc->sc_ab.ab_resid);
		if ((sq = sc->sc_hba.hba_head) != NULL) {
			sc->sc_hba.hba_head = sq->sq_forw;
			sc->sc_gosccb = sccballoc(sc, (struct scsi_cdb *)NULL);
			(*sq->sq_dgo)(sq->sq_dev, (struct scsi_cdb *)
			    &sc->sc_gosccb->sccb_ccb.ccb_cdb);
		}
	}

	/* increment to the next mailbox */
	INC_MBI(mbi, sc);
	return (sc->sc_inbox = mbi);
}

int
ahaintr(sc0)
	void *sc0;
{
	register struct aha_softc *sc = sc0;
	register struct mbox24 *mbi;

	if (!ab_intr(&sc->sc_ab))
		return (0);
	mbi = sc->sc_inbox;
	/*
	 * Usual case is that the next inbox is full, but immediate commands
	 * screw this up, so we may have to scan.
	 */
	if ((sc->sc_ab.ab_istat & AHA_I_MBIF) && mbi->mb_status == MBOX_I_FREE)
		mbi = scanmbi(sc, mbi);
	while (mbi->mb_status != MBOX_I_FREE)
		mbi = ahainbox(sc, mbi);
	return (1);
}

void
aharel(self)
	struct device *self;
{
	struct aha_softc *sc = (struct aha_softc *)self;
	struct soft_ccb *sccb;
	struct sq *sq;

	if ((sccb = sc->sc_gosccb) != NULL) {
		if (sc->sc_gosccb == sccb)
			sc->sc_gosccb = NULL;
		sccb->sccb_ccb.ccb_opcode = CCB_FREE;
	}
	if ((sq = sc->sc_hba.hba_head) != NULL) {
		sc->sc_gosccb = sccballoc(sc, (struct scsi_cdb *)NULL);
		(*sq->sq_dgo)(sq->sq_dev,
		    (struct scsi_cdb *)&sc->sc_gosccb->sccb_ccb.ccb_cdb);
	}
}
