/*-
 * Copyright (c) 1992, 1993, 1994 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: disk_subr.c,v 2.3 1995/12/07 19:44:37 ewv Exp $
 */

/*
 * Copyright (c) 1982, 1986, 1988, 1993
 *	The Regents of the University of California.  All rights reserved.
 * (c) UNIX System Laboratories, Inc.
 * All or some portions of this file are derived from material licensed
 * to the University of California by American Telephone and Telegraph
 * Co. or Unix System Laboratories, Inc. and are reproduced herein with
 * the permission of UNIX System Laboratories, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)ufs_disksubr.c	8.5 (Berkeley) 1/21/94
 */

/*
 * Support routines for disk drivers;
 * should become the generic disk driver.
 */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/disk.h>
#include <sys/stat.h>
#include <sys/syslog.h>

/*
 * We maintain a linked list of the disks' dkdevice structures.
 * The list is used to find the disk statistics for getkerninfo.
 * The driver is responsible for maintaining the statistics.
 * gatherstats() follows this list to increment dk_time.
 *
 * Disk drivers must call disk_attach() when the drive is configured.
 */

struct dkdevice *diskhead;
char no_disk_label[] = "no disk label";

/*
 * Attach a disk to the stat gathering functions.
 */
void
disk_attach(dk)
	register struct dkdevice *dk;
{
	static struct dkdevice **dkp = &diskhead;

	*dkp = dk;		/* add to end of chain */
	dkp = &dk->dk_next;
}

/*
 * Seek sort for disks.  We depend on the driver which calls us using b_cylin
 * as the current cylinder number.
 *
 * The argument ap structure holds a b_actf activity chain pointer on which we
 * keep two queues, sorted in ascending cylinder order.  The first queue holds
 * those requests which are positioned after the current cylinder (in the first
 * request); the second holds requests which came in after their cylinder number
 * was passed.  Thus we implement a one way scan, retracting after reaching the
 * end of the drive to the first request on the second queue, at which time it
 * becomes the first queue.
 *
 * A one-way scan is natural because of the way UNIX read-ahead blocks are
 * allocated.
 */

void
disksort(ap, bp)
	register struct buf *ap, *bp;
{
	register struct buf *bq;

	/* If the queue is empty, then it's easy. */
	if (ap->b_actf == NULL) {
		bp->b_actf = NULL;
		ap->b_actf = bp;
		return;
	}

	/*
	 * If we lie after the first (currently active) request, then we
	 * must locate the second request list and add ourselves to it.
	 */
	bq = ap->b_actf;
	if (bp->b_cylin < bq->b_cylin) {
		while (bq->b_actf) {
			/*
			 * Check for an ``inversion'' in the normally ascending
			 * cylinder numbers, indicating the start of the second
			 * request list.
			 */
			if (bq->b_actf->b_cylin < bq->b_cylin) {
				/*
				 * Search the second request list for the first
				 * request at a larger cylinder number.  We go
				 * before that; if there is no such request, we
				 * go at end.
				 */
				do {
					if (bp->b_cylin <
					    bq->b_actf->b_cylin)
						goto insert;
					if (bp->b_cylin ==
					    bq->b_actf->b_cylin &&
					    bp->b_blkno < bq->b_actf->b_blkno)
						goto insert;
					bq = bq->b_actf;
				} while (bq->b_actf);
				goto insert;		/* after last */
			}
			bq = bq->b_actf;
		}
		/*
		 * No inversions... we will go after the last, and
		 * be the first request in the second request list.
		 */
		goto insert;
	}
	/*
	 * Request is at/after the current request...
	 * sort in the first request list.
	 */
	while (bq->b_actf) {
		/*
		 * We want to go after the current request if there is an
		 * inversion after it (i.e. it is the end of the first
		 * request list), or if the next request is a larger cylinder
		 * than our request.
		 */
		if (bq->b_actf->b_cylin < bq->b_cylin ||
		    bp->b_cylin < bq->b_actf->b_cylin ||
		    (bp->b_cylin == bq->b_actf->b_cylin &&
		    bp->b_blkno < bq->b_actf->b_blkno))
			goto insert;
		bq = bq->b_actf;
	}
	/*
	 * Neither a second list nor a larger request... we go at the end of
	 * the first list, which is the same as the end of the whole schebang.
	 */
insert:	bp->b_actf = bq->b_actf;
	bq->b_actf = bp;
}

/*
 * Final portion of disk open routine, handles partition open/close state.
 * Should really be part of a generic disk open routine.
 */
dkopenpart(dk, dev, fmt)
	struct dkdevice *dk;
	dev_t dev;
	int fmt;
{
	int part = dk_part(dev), mask = 1 << part;
	struct disklabel *lp = &dk->dk_label;
	struct partition *pp;

	if (part >= lp->d_npartitions)
		return (ENXIO);
	/*
	 * Warn if a partion is opened
	 * that overlaps another partition which is open
	 * unless one is the "raw" partition (whole disk).
	 */
	if ((dk->dk_openmask & mask) == 0 && part != DK_RAWPART) {
		int start, end;

		pp = &dk->dk_label.d_partitions[part];
		start = pp->p_offset;
		end = pp->p_offset + pp->p_size;
		for (pp = lp->d_partitions;
		     pp < &lp->d_partitions[lp->d_npartitions]; pp++) {
			if (pp->p_offset + pp->p_size <= start ||
			    pp->p_offset >= end)
				continue;
			if (pp - lp->d_partitions == DK_RAWPART)
				continue;
			if (dk->dk_openmask & (1 << (pp - lp->d_partitions)))
				log(LOG_WARNING,
				    "%s%c: overlaps open partition (%c)\n",
				    dk->dk_dev.dv_xname, part + 'a',
				    pp - lp->d_partitions + 'a');
		}
	}
	dk->dk_openmask |= mask;
	switch (fmt) {
	case S_IFCHR:
		dk->dk_copenmask |= mask;
		break;
	case S_IFBLK:
		dk->dk_bopenmask |= mask;
		break;
	}
	return (0);
}

/*
 * Close a disk partition on the specified disk.
 */
/* ARGSUSED */
dkclose(dk, dev, flags, fmt, p)
	struct dkdevice *dk;
	dev_t dev;
	int flags, fmt;
	struct proc *p;
{
	int part = dk_part(dev), mask = 1 << part;

	switch (fmt) {
	case S_IFCHR:
		dk->dk_copenmask &= ~mask;
		break;
	case S_IFBLK:
		dk->dk_bopenmask &= ~mask;
		break;
	}
	dk->dk_openmask = dk->dk_copenmask | dk->dk_bopenmask;

	/*
	 * ??? Should wait for i/o to complete on this partition
	 * even if others are open, but wait for work on blkflush().
	 */
#ifdef notyet
	if (dk->dk_openmask == 0) {
		int s = splhigh();

		while (dk->dk_tab.b_actf)
			sleep((caddr_t)dk, PRIBIO);
		splx(s);
		dk->dk_state = CLOSED;
	}
#endif
	return (0);
}

/*
 * Called when a transfer begins at and/or spans the end of a
 * partition.  Return new transfer count (in partition units, i.e.,
 * sectors), or 0 to indicate that we already did the biodone().
 */
int
dktrim(bp, p, bn, sz)
	struct buf *bp;
	struct partition *p;
	daddr_t bn;
	int sz;
{

	/*
	 * Chained transfers happen only from the file system.
	 * They should never cross a partition boundary.
	 * (change this when we recode physio?)
	 */
	if (bp->b_chain) {
		bp->b_error = EINVAL;
		bp->b_flags |= B_ERROR;
		biodone(bp);	/* biodone propagates the error */
		return (0);
	}

	/*
	 * If exactly at end of disk, return EOF on read, error on write;
	 * if transfer begins past end of partition, return error.
	 * Otherwise trim transfer to just reach end of partition.
	 */
	if ((unsigned)bn == p->p_size && bp->b_flags & B_READ)
		bp->b_resid = bp->b_iocount;
	else if ((unsigned)bn >= p->p_size) {
		bp->b_error = EINVAL;
		bp->b_flags |= B_ERROR;
	} else {
		sz = p->p_size - bn;
		bp->b_bcount = bp->b_iocount = dbtob(sz);
		/*
		 * Set NOCACHE to avoid truncated buffers in cache, these
		 * cause a stray size panic in getblk() (via specfs).
		 */
		bp->b_flags |= B_NOCACHE;
		return (sz);
	}
	biodone(bp);
	return (0);
}

/*
 * Search the given sector for a label.
 */
static char *
getdisklabel(bp, lp)
	register struct buf *bp;
	struct disklabel *lp;
{
	register caddr_t e, p;
	char *msg;

	msg = no_disk_label;
	e = bp->b_un.b_addr + bp->b_bcount - sizeof(*lp);
	for (p = bp->b_un.b_addr; p <= e; p += sizeof(long)) {
#define	dlp ((struct disklabel *)p)
		if (dlp->d_magic != DISKMAGIC || dlp->d_magic2 != DISKMAGIC)
			continue;
		if (dlp->d_npartitions > MAXPARTITIONS || dkcksum(dlp) != 0)
			msg = "disk label corrupted";
		else {
			*lp = *dlp;
			return (NULL);
		}
#undef dlp
	}
	return (msg);
}

/*
 * Attempt to read a disk label from a device using the indicated stategy
 * routine.  The label must be partly set up before this: secpercyl (or
 * nsectors and ntracks) and anything required in the strategy routine
 * (e.g., sector size) must be filled in before calling us.  Returns NULL
 * on success and an error string on failure.
 */
char *
readdisklabel(dev, strat, lp, sn)
	dev_t dev;
	void (*strat)();
	register struct disklabel *lp;
	daddr_t sn;
{
	register struct buf *bp;
	u_long p0size, p0off;
	char *msg;

	if (lp->d_secpercyl == 0)
		lp->d_secpercyl = lp->d_nsectors * lp->d_ntracks;
	if (lp->d_secperunit == 0)
		lp->d_secperunit = lp->d_secpercyl * lp->d_ncylinders;
	if (lp->d_npartitions < 1)
		lp->d_npartitions = 1;
	p0size = lp->d_partitions[0].p_size;
	p0off = lp->d_partitions[0].p_offset;
	if (lp->d_partitions[0].p_size == 0)
		lp->d_partitions[0].p_size = 0x1fffffff;
	lp->d_partitions[0].p_offset = 0;

	bp = geteblk((int)lp->d_secsize);
	bp->b_dev = dev;
	bp->b_blkno = sn;
	bp->b_iocount = bp->b_bcount = lp->d_secsize;
	bp->b_flags = B_BUSY | B_READ;
	bp->b_cylin = sn / lp->d_secpercyl;
	(*strat)(bp);
	if (biowait(bp))
		msg = "I/O error";
	else {
		msg = getdisklabel(bp, lp);
#ifdef COMPAT_SUNOS
		if (msg) {
			/* no BSD label; try for a SunOS label */
			bp->b_blkno = 0;
			bp->b_iocount = bp->b_bcount = lp->d_secsize;
			bp->b_flags = B_BUSY | B_READ;
			bp->b_cylin = 0;
			(*strat)(bp);
			if (!biowait(bp) && sun_disklabel(bp->b_un.b_addr, lp))
				msg = NULL;
		}
#endif
	}
	bp->b_flags = B_INVAL | B_AGE;
	brelse(bp);
	if (msg) {
		/* restore the software label (it's better than nothing) */
		lp->d_partitions[0].p_size = p0size;
		lp->d_partitions[0].p_offset = p0off;
	}
	return (msg);
}

/*
 * Check new disk label for sensibility before setting it.
 */
int
setdisklabel(olp, nlp, openmask)
	register struct disklabel *olp, *nlp;
	u_long openmask;
{
	register i;
	register struct partition *opp, *npp;

	if (nlp->d_magic != DISKMAGIC || nlp->d_magic2 != DISKMAGIC ||
	    dkcksum(nlp) != 0)
		return (EINVAL);
	while ((i = ffs((long)openmask)) != 0) {
		i--;
		openmask &= ~(1 << i);
		if (nlp->d_npartitions <= i)
			return (EBUSY);
		opp = &olp->d_partitions[i];
		npp = &nlp->d_partitions[i];
		if (npp->p_offset != opp->p_offset || npp->p_size < opp->p_size)
			return (EBUSY);
		/*
		 * Copy internally-set partition information
		 * if new label doesn't include it.		XXX
		 */
		if (npp->p_fstype == FS_UNUSED && opp->p_fstype != FS_UNUSED) {
			npp->p_fstype = opp->p_fstype;
			npp->p_fsize = opp->p_fsize;
			npp->p_frag = opp->p_frag;
			npp->p_cpg = opp->p_cpg;
		}
	}
 	nlp->d_checksum = 0;
 	nlp->d_checksum = dkcksum(nlp);
	*olp = *nlp;
	return (0);
}

/*
 * Write disk label back to device after modification.
 */
int
writedisklabel(dev, strat, lp, sn)
	dev_t dev;
	void (*strat)();
	register struct disklabel *lp;
	daddr_t sn;
{
	struct buf *bp;
	struct disklabel *dlp;
	int labelpart;
	int error = 0;
	struct partition *pp;

	labelpart = dk_part(dev);
	pp = &lp->d_partitions[labelpart];
	if (pp->p_offset != 0 || sn >= pp->p_size) {
		for (pp = lp->d_partitions, labelpart = 0;
		    labelpart < lp->d_npartitions; pp++, labelpart++)
			if (pp->p_offset == 0 && pp->p_size > sn)
				break;
		if (labelpart >= lp->d_npartitions)
			return (EXDEV);			/* not quite right */
	}
	bp = geteblk((int)lp->d_secsize);
	bp->b_dev = dv_makedev(major(dev), dk_unit(dev), labelpart);
	bp->b_blkno = sn;
	bp->b_iocount = bp->b_bcount = lp->d_secsize;
	bp->b_flags = B_READ | B_BUSY;
	(*strat)(bp);
	if (error = biowait(bp))
		goto done;
	for (dlp = (struct disklabel *)bp->b_data;
	    dlp <= (struct disklabel *)
	      ((char *)bp->b_data + lp->d_secsize - sizeof(*dlp));
	    dlp = (struct disklabel *)((char *)dlp + sizeof(long))) {
		if (dlp->d_magic == DISKMAGIC && dlp->d_magic2 == DISKMAGIC &&
		    dkcksum(dlp) == 0) {
			*dlp = *lp;
			bp->b_flags = B_WRITE | B_BUSY;
			(*strat)(bp);
			error = biowait(bp);
			goto done;
		}
	}
	error = ESRCH;
done:
	brelse(bp);
	return (error);
}

/*
 * Compute checksum for disk label.
 */
dkcksum(lp)
	register struct disklabel *lp;
{
	register u_short *start, *end;
	register u_short sum = 0;

	start = (u_short *)lp;
	end = (u_short *)&lp->d_partitions[lp->d_npartitions];
	while (start < end)
		sum ^= *start++;
	return (sum);
}

/*
 * Disk error is the preface to plaintive error messages
 * about failing disk transfers.  It prints messages of the form

hp0g: hard error reading fsbn 12345 of 12344-12347 (hp0 bn %d cn %d tn %d sn %d)

 * if the offset of the error in the transfer and a disk label
 * are both available.  blkdone should be -1 if the position of the error
 * is unknown; the disklabel pointer may be null from drivers that have not
 * been converted to use them.  The message is printed with printf
 * if pri is LOG_PRINTF, otherwise it uses log at the specified priority.
 * The message should be completed (with at least a newline) with printf
 * or addlog, respectively.  There is no trailing space.
 */
void
diskerr(bp, dname, what, pri, blkdone, lp)
	register struct buf *bp;
	char *dname, *what;
	int pri, blkdone;
	register struct disklabel *lp;
{
	int unit = dk_unit(bp->b_dev), part = dk_part(bp->b_dev);
	register void (*pr) __P((const char *, ...));
	char partname = 'a' + part;
	int sn;

	if (pri != LOG_PRINTF) {
		log(pri, "");
		pr = addlog;
	} else
		pr = printf;
	(*pr)("%s%d%c: %s %sing fsbn ", dname, unit, partname, what,
	    bp->b_flags & B_READ ? "read" : "writ");
	sn = bp->b_blkno;
	if (bp->b_iocount <= DEV_BSIZE)
		(*pr)("%d", sn);
	else {
		if (blkdone >= 0) {
			sn += blkdone;
			(*pr)("%d of ", sn);
		}
		(*pr)("%d-%d", bp->b_blkno,
		    bp->b_blkno + (bp->b_iocount - 1) / DEV_BSIZE);
	}
	if (lp && (blkdone >= 0 || bp->b_iocount <= lp->d_secsize)) {
#ifdef tahoe
		sn *= DEV_BSIZE / lp->d_secsize;		/* XXX */
#endif
		sn += lp->d_partitions[part].p_offset;
		(*pr)(" (%s%d bn %d; cn %d", dname, unit, sn,
		    sn / lp->d_secpercyl);
		sn %= lp->d_secpercyl;
		(*pr)(" tn %d sn %d)", sn / lp->d_nsectors, sn % lp->d_nsectors);
	}
}
