/*	/master/contrib/hylafax/faxd/ModemServer.c++,v 1.1.1.1 1995/11/30 03:32:24 polk Exp */
/*
 * Copyright (c) 1990-1995 Sam Leffler
 * Copyright (c) 1991-1995 Silicon Graphics, Inc.
 * HylaFAX is a trademark of Silicon Graphics
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Sam Leffler and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Sam Leffler and Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 * IN NO EVENT SHALL SAM LEFFLER OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF 
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 */
#include <osfcn.h>
#include <ctype.h>
#include <termios.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/file.h>

#include "Sys.h"

#include "Dispatcher.h"
#include "FaxTrace.h"
#include "FaxMachineLog.h"
#include "ModemServer.h"
#include "UUCPLock.h"
#include "Class0.h"

#include "config.h"

#ifndef O_NOCTTY
#define	O_NOCTTY	0		// no POSIX O_NOCTTY support
#endif

/*
 * HylaFAX Modem Server.
 */

// map ClassModem::BaudRate to numeric value
static const u_int baudRates[] = {
    0,		// BR0
    300,	// BR300
    1200,	// BR1200
    2400,	// BR2400
    4800,	// BR4800
    9600,	// BR9600
    19200,	// BR19200
    38400,	// BR38400
    57600,	// BR57600
    76800,	// BR76800
    115200,	// BR115200
};

ModemServer::ModemServer(const fxStr& devName, const fxStr& devID)
    : modemDevice(devName)
    , modemDevID(devID)
    , configFile(fxStr(FAX_CONFIG) | "." | devID)
{
    state = BASE;
    statusFile = NULL;
    abortCall = FALSE;
    deduceComplain = TRUE;		// first failure causes complaint
    changePriority = TRUE;
    delayConfig = FALSE;

    modemFd = -1;
    modem = NULL;
    curRate = ClassModem::BR0;		// unspecified baud rate

    rcvCC = rcvNext = 0;
    timeout = FALSE;
    log = NULL;
}

ModemServer::~ModemServer()
{
    delete log;
    delete modem;
    if (statusFile)
	fclose(statusFile);
}

#include "faxApp.h"
/*
 * Initialize the server from command line arguments.
 */
void
ModemServer::initialize(int argc, char** argv)
{
    for (GetoptIter iter(argc, argv, faxApp::getOpts()); iter.notDone(); iter++)
	switch (iter.option()) {
	case 'p':
	    changePriority = FALSE;
	    break;
	case 'x':
	    tracingMask &= ~(FAXTRACE_MODEMIO|FAXTRACE_TIMEOUTS);
	    break;
	}
    TIFFSetErrorHandler(NULL);
    TIFFSetWarningHandler(NULL);
    // setup server's status file
    statusFile = Sys::fopen(FAX_STATUSDIR "/" | modemDevID, "w");
    if (statusFile != NULL) {
#if HAS_FCHMOD
	::fchmod(fileno(statusFile), 0644);
#else
	Sys::chmod(FAX_STATUSDIR "/" | modemDevID, 0644);
#endif
	setServerStatus("Initializing server");
    }
    ::umask(077);			// keep all temp files private

    updateConfig(configFile);		// read config file
}

/*
 * Startup the server for the first time.
 */
void
ModemServer::open()
{
    if (lockModem()) {
	fxBool modemReady = setupModem();
	unlockModem();
	if (!modemReady)
	    changeState(MODEMWAIT, pollModemWait);
	else
	    changeState(RUNNING, 0);
    } else {
	traceServer("%s: Can not lock device.", (char*) modemDevice);
	changeState(LOCKWAIT, pollLockWait);
    }
}

/*
 * Close down the server.
 */
void
ModemServer::close()
{
    if (lockModem()) {
	if (modem)
	    modem->hangup();
	discardModem(TRUE);
	unlockModem();
    }
}

const char* ModemServer::stateNames[8] = {
    "BASE",
    "RUNNING",
    "MODEMWAIT",
    "LOCKWAIT",
    "GETTYWAIT",
    "SENDING",
    "ANSWERING",
    "RECEIVING"
};
const char* ModemServer::stateStatus[8] = {
    "Initializing server and modem",		// BASE
    "Running and idle",				// RUNNING
    "Waiting for modem to come ready",		// MODEMWAIT
    "Waiting for modem to come free",		// LOCKWAIT
    "Waiting for login session to terminate",	// GETTYWAIT
    "Sending facsimile",			// SENDING
    "Answering the phone",			// ANSWERING
    "Receiving facsimile",			// RECEIVING
};

/*
 * Change the server's state and, optionally,
 * start a timer running for timeout seconds.
 */
void
ModemServer::changeState(ModemServerState s, long timeout)
{
    if (s != state) {
	if (timeout)
	    traceStatus(FAXTRACE_STATETRANS,
		"STATE CHANGE: %s -> %s (timeout %ld)",
		stateNames[state], stateNames[s], timeout);
	else
	    traceStatus(FAXTRACE_STATETRANS, "STATE CHANGE: %s -> %s",
		stateNames[state], stateNames[s]);
	state = s;
	if (changePriority)
	    setProcessPriority(state);
	setServerStatus(stateStatus[state]);
	if (s == RUNNING)
	    notifyModemReady();			// notify surrogate
    }
    if (timeout)
	Dispatcher::instance().startTimer(timeout, 0, this);
}

#if HAS_SCHEDCTL
#include <sys/schedctl.h>
/*
 * When low latency is required, use a nondegrading process
 * priority; otherwise just remove any nondegrading priority.
 * Note that we assign a high nondegrading priority when sending,
 * answering the telephone, or receiving.  We assume that if the
 * incoming call spawns a getty process that the priority will
 * be reset in the child before the getty is exec'd.
 */
static const int schedCtlParams[8][2] = {
    { NDPRI, 0 },		// BASE
    { NDPRI, 0 },		// RUNNING
    { NDPRI, 0 },		// MODEMWAIT
    { NDPRI, 0 },		// LOCKWAIT
    { NDPRI, 0 },		// GETTYWAIT
    { NDPRI, NDPHIMIN },	// SENDING
    { NDPRI, NDPHIMIN },	// ANSWERING
    { NDPRI, NDPHIMIN },	// RECEIVING
};
#elif HAS_PRIOCNTL
extern "C" {
#include <sys/priocntl.h>
#include <sys/rtpriocntl.h>
#include <sys/tspriocntl.h>
}
static struct SchedInfo {
    const char*	clname;		// scheduling class name
    int		params[3];	// scheduling class parameters
} schedInfo[8] = {
    { "TS", { TS_NOCHANGE, TS_NOCHANGE } },		// BASE
    { "TS", { TS_NOCHANGE, TS_NOCHANGE } },		// RUNNING
    { "TS", { TS_NOCHANGE, TS_NOCHANGE } },		// MODEMWAIT
    { "TS", { TS_NOCHANGE, TS_NOCHANGE } },		// LOCKWAIT
    { "TS", { TS_NOCHANGE, TS_NOCHANGE } },		// GETTYWAIT
    { "RT", { RT_NOCHANGE, RT_NOCHANGE, RT_NOCHANGE } },// SENDING
    { "RT", { RT_NOCHANGE, RT_NOCHANGE, RT_NOCHANGE } },// ANSWERING
    { "RT", { RT_NOCHANGE, RT_NOCHANGE, RT_NOCHANGE } },// RECEIVING
};
#endif

void
ModemServer::setProcessPriority(ModemServerState s)
{
#if HAS_SCHEDCTL
    uid_t euid = ::geteuid();
    if (::seteuid(0) >= 0) {		// must be done as root
	if (::schedctl(schedCtlParams[s][0], 0, schedCtlParams[s][1]) < 0)
	    traceServer("schedctl: %m");
	if (::seteuid(euid) < 0)	// restore previous effective uid
	    traceServer("seteuid(%d): %m", euid);
    } else
	traceServer("seteuid(root): %m");
#elif HAS_PRIOCNTL
    uid_t euid = ::geteuid();
    if (::seteuid(0) >= 0) {		// must be done as root
	const SchedInfo& si = schedInfo[s];
	pcinfo_t pcinfo;
	::strcpy(pcinfo.pc_clname, si.clname);
	if (::priocntl((idtype_t)0, 0, PC_GETCID, (caddr_t)&pcinfo) >= 0) {
	    pcparms_t pcparms;
	    pcparms.pc_cid = pcinfo.pc_cid;
	    if (streq(si.clname, "RT")) {
		rtparms_t* rtp = (rtparms_t*) pcparms.pc_clparms;
		rtp->rt_pri	= si.params[0];
		rtp->rt_tqsecs	= (ulong) si.params[1];
		rtp->rt_tqnsecs	= si.params[2];
	    } else {
		tsparms_t* tsp = (tsparms_t*) pcparms.pc_clparms;
		tsp->ts_uprilim	= si.params[0];
		tsp->ts_upri	= si.params[1];
	    }
	    if (::priocntl(P_PID, P_MYID, PC_SETPARMS, (caddr_t)&pcparms) < 0)
		traceServer("Unable to set %s scheduling parameters: %m",
		    si.clname);
	} else
	    traceServer("priocntl(%s): %m", si.clname);
	if (::seteuid(euid) < 0)	// restore previous effective uid
	    traceServer("setreuid(%d): %m", euid);
    } else
	traceServer("setreuid(root): %m");
#endif
}

/*
 * Record the server status in the status file.
 */
void
ModemServer::setServerStatus(const char* fmt, ...)
{
    if (statusFile == NULL)
	return;
    ::flock(fileno(statusFile), LOCK_EX);
    ::rewind(statusFile);
    va_list ap;
    va_start(ap, fmt);
    ::vfprintf(statusFile, fmt, ap);
    ::fprintf(statusFile, "\n");
    ::fflush(statusFile);
    ::ftruncate(fileno(statusFile), ::ftell(statusFile));
    ::flock(fileno(statusFile), LOCK_UN);
}

void
ModemServer::resetConfig()
{
    if (modem)
	discardModem(TRUE);
    ServerConfig::resetConfig();
}

const fxStr& ModemServer::getConfigFile() const { return configFile; }

/*
 * Setup the modem; if needed.
 */
fxBool
ModemServer::setupModem()
{
    if (!modem) {
	const char* dev = modemDevice;
	if (!openDevice(dev))
	    return (FALSE);
	/*
	 * Deduce modem type and setup configuration info.
	 * The deduceComplain cruft is just to reduce the
	 * noise in the log file when probing for a modem.
	 */
	modem = deduceModem();
	if (!modem) {
	    discardModem(TRUE);
	    if (deduceComplain) {
		traceServer("%s: Can not deduce modem type.", dev);
		deduceComplain = FALSE;
	    }
	    return (FALSE);
	} else {
	    deduceComplain = TRUE;
	    traceServer("MODEM "
		| modem->getManufacturer() | " "
		| modem->getModel() | "/"
		| modem->getRevision());
	}
    } else
	/*
	 * Reset the modem in case some other program
	 * went in and messed with the configuration.
	 */
	modem->reset();
    /*
     * Most modem-related parameters are dealt with
     * in the modem driver.  The speaker volume is
     * kept in the fax server because it often gets
     * changed on the fly.  The phone number has no
     * business in the modem class.
     */
    modem->setSpeakerVolume(speakerVolume);
    return (TRUE);
}

/*
 * Deduce the type of modem supplied to the server
 * and return an instance of the appropriate modem
 * driver class.
 */
ClassModem*
ModemServer::deduceModem()
{
    ClassModem* modem = new Class0Modem(*this, *this);
    if (modem) {
	if (modem->setupModem())
	    return modem;
	delete modem;
    }
    return (NULL);
}

fxStr
ModemServer::getModemCapabilities() const
{
    return modem->getCapabilities();
}

/*
 * Open the tty device associated with the modem
 * and change the device file to be owned by us
 * and with a protected file mode.
 */
fxBool
ModemServer::openDevice(const char* dev)
{
    /*
     * Temporarily become root to open the device.
     * Routines that call setupModem *must* first
     * lock the device with the usual effective uid.
     */
    uid_t euid = ::geteuid();
    if (::seteuid(0) < 0) {
	 traceServer("%s: seteuid root failed (%m)", dev);
	 return (FALSE);
    }
#ifdef O_NDELAY
    /*
     * Open device w/ O_NDELAY to bypass modem
     * control signals, then turn off the flag bit.
     */
    modemFd = Sys::open(dev, O_RDWR|O_NDELAY|O_NOCTTY);
    if (modemFd < 0) {
	::seteuid(euid);
	traceServer("%s: Can not open modem (%m)", dev);
	return (FALSE);
    }
    int flags = ::fcntl(modemFd, F_GETFL, 0);
    if (::fcntl(modemFd, F_SETFL, flags &~ O_NDELAY) < 0) {
	 traceServer("%s: fcntl: %m", dev);
	 ::close(modemFd), modemFd = -1;
	 return (FALSE);
    }
#else
    startTimeout(3*1000);
    modemFd = Sys::open(dev, O_RDWR);
    stopTimeout("opening modem");
    if (modemFd < 0) {
	::seteuid(euid);
	traceServer((timeout ?
		"%s: Can not open modem (timed out)." :
		"%s: Can not open modem (%m)."),
	    dev, errno);
	return (FALSE);
    }
#endif
    /*
     * NB: we stat and use the gid because passing -1
     *     through the gid_t parameter in the prototype
     *	   causes it to get truncated to 65535.
     */
    struct stat sb;
    (void) Sys::fstat(modemFd, sb);
#if HAS_FCHOWN
    if (::fchown(modemFd, UUCPLock::getUUCPUid(), sb.st_gid) < 0)
#else
    if (Sys::chown(dev, UUCPLock::getUUCPUid(), sb.st_gid) < 0)
#endif
	traceServer("%s: chown: %m", dev);
#if HAS_FCHMOD
    if (::fchmod(modemFd, deviceMode) < 0)
#else
    if (Sys::chmod(dev, deviceMode) < 0)
#endif
	traceServer("%s: chmod: %m", dev);
    ::seteuid(euid);
    return (TRUE);
}

fxBool
ModemServer::reopenDevice()
{
    if (modemFd >= 0)
	::close(modemFd), modemFd = -1;
    return openDevice(modemDevice);
}

/*
 * Discard any handle on the modem.
 */
void
ModemServer::discardModem(fxBool dropDTR)
{
    if (modemFd >= 0) {
	if (dropDTR)
	    (void) setDTR(FALSE);			// force hangup
	::close(modemFd), modemFd = -1;			// discard open file
    }
    delete modem, modem = NULL;
}

/*
 * Start a session: a period of time during which
 * carrier is raised and a peer is engaged.
 */
void
ModemServer::beginSession(const fxStr& number)
{
    log = new FaxMachineLog(canonicalizePhoneNumber(number), logMode);
}

/*
 * Terminate a session.
 */
void
ModemServer::endSession()
{
    delete log, log = NULL;
}

/*
 * Return true if a request has been made to abort
 * the current session.  This is true if a previous
 * abort request was made or if an external abort
 * message is dispatched during our processing.
 */
fxBool
ModemServer::abortRequested()
{
#ifndef SERVERABORTBUG
    if (!abortCall) {
	// poll for input so abort commands get processed
	long sec = 0;
	long usec = 0;
	while (Dispatcher::instance().dispatch(sec,usec) && !abortCall)
	    ;
    }
#endif
    return (abortCall);
}

/*
 * Request that a current session be aborted.
 */
void
ModemServer::abortSession()
{
    abortCall = TRUE;
    traceServer("ABORT: job abort requested");
}

/*
 * Dispatcher timer expired routine.  Perform the action
 * associated with the server's state and, possible, transition
 * to a new state.
 */
void
ModemServer::timerExpired(long, long)
{
    switch (state) {
    case MODEMWAIT:
    case LOCKWAIT:
	/*
	 * Waiting for modem to start working.  Retry setup
	 * and either change state or restart the timer.
	 * Note that we unlock the modem before we change
	 * our state to RUNNING after a modem setup so that
	 * any callback doesn't find the modem locked (and
	 * so cause jobs to be requeued).
	 */
	if (lockModem()) {
	    fxBool modemReady = setupModem();
	    unlockModem();
	    if (modemReady)
		changeState(RUNNING, 0);
	    else
		changeState(MODEMWAIT, pollModemWait);
	} else
	    changeState(LOCKWAIT, pollLockWait);
	break;
    }
}

/*
 * Modem support interfaces.  Note that the values
 * returned when we don't have a handle on the modem
 * are selected so that any imaged facsimile should
 * still be sendable.
 */
fxBool ModemServer::modemReady() const
    { return modem != NULL; }

fxBool ModemServer::serverBusy() const
    { return state != RUNNING; }

fxBool ModemServer::modemWaitForRings(u_int rings, CallType& type, CallerID& cid)
    { return modem->waitForRings(rings, type, cid); }
CallType ModemServer::modemAnswerCall(AnswerType atype, fxStr& emsg)
    { return modem->answerCall(atype, emsg); }
void ModemServer::modemHangup()			{ modem->hangup(); }

BaudRate ModemServer::getModemRate() const	{ return baudRates[curRate]; }

/*
 * Server configuration support.
 */

/*
 * Read a configuration file.  Note that we suppress
 * dial string rules setup while reading so that the
 * order of related parameters is not important.  We
 * also setup the local identifier from the fax number
 * if nothing is specified in the config file (for
 * backwards compatibility).
 */
void
ModemServer::readConfig(const fxStr& filename)
{
    dialRulesFile = "";
    delayConfig = TRUE;
    ServerConfig::readConfig(filename);
    delayConfig = FALSE;
    if (dialRulesFile != "")
	setDialRules(dialRulesFile);
    if (localIdentifier == "")
	setLocalIdentifier(canonicalizePhoneNumber(FAXNumber));
}

void
ModemServer::configError(const char* fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
    vtraceStatus(FAXTRACE_SERVER, fmt, ap);
    va_end(ap);
}

void
ModemServer::configTrace(const char* fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
    vtraceStatus(FAXTRACE_CONFIG, fmt, ap);
    va_end(ap);
}

const fxStr& ModemServer::getModemDevice() const	{ return modemDevice; }
const fxStr& ModemServer::getModemDeviceID() const	{ return modemDevID; }
const fxStr& ModemServer::getModemNumber() const	{ return FAXNumber; }

/*
 * Setup the dial string rules.  Note that if we're
 * reading the configuration file (as opposed to
 * reconfiguring based on a FIFO message), then we
 * suppress the actual setup so that other parameters
 * such as the area code can be specified out of
 * order in the configuration file.
 */
void
ModemServer::setDialRules(const char* name)
{
    if (delayConfig)				// delay during config setup
	dialRulesFile = name;
    else
	ServerConfig::setDialRules(name);
}

int ModemServer::getServerTracing() const		{ return (tracingLevel); }
int ModemServer::getSessionTracing() const	{ return (logTracingLevel); }

/*
 * Set the modem speaker volume and if a modem
 * is setup, pass it into the modem driver to
 * pass to the modem.
 */
void
ModemServer::setModemSpeakerVolume(SpeakerVolume level)
{
    ServerConfig::setModemSpeakerVolume(level);
    if (modem)
	modem->setSpeakerVolume(level);
}

/*
 * Tracing support.
 */

void
ModemServer::traceServer(const char* fmt ...)
{
    va_list ap;
    va_start(ap, fmt);
    vtraceStatus(FAXTRACE_SERVER, fmt, ap);
    va_end(ap);
}

void
ModemServer::traceProtocol(const char* fmt ...)
{
    va_list ap;
    va_start(ap, fmt);
    vtraceStatus(FAXTRACE_PROTOCOL, fmt, ap);
    va_end(ap);
}

void
ModemServer::traceStatus(int kind, const char* fmt ...)
{
    va_list ap;
    va_start(ap, fmt);
    vtraceStatus(kind, fmt, ap);
    va_end(ap);
}

extern void vlogInfo(const char* fmt, va_list ap);

void
ModemServer::vtraceStatus(int kind, const char* fmt, va_list ap)
{
    if (log) {
	if (kind == FAXTRACE_SERVER)	// always log server stuff
	    vlogInfo(fmt, ap);
	if (logTracingLevel & kind)
	    log->vlog(fmt, ap);
    } else if (tracingLevel & kind)
	vlogInfo(fmt, ap);
}

#include "StackBuffer.h"

void
ModemServer::traceModemIO(const char* dir, const u_char* data, u_int cc)
{
    if (log) {
	if ((logTracingLevel& FAXTRACE_MODEMIO) == 0)
	    return;
    } else if ((tracingLevel & FAXTRACE_MODEMIO) == 0)
	return;

    const char* hexdigits = "0123456789ABCDEF";
    fxStackBuffer buf;
    for (u_int i = 0; i < cc; i++) {
	u_char b = data[i];
	if (i > 0)
	    buf.put(' ');
	buf.put(hexdigits[b>>4]);
	buf.put(hexdigits[b&0xf]);
    }
    traceStatus(FAXTRACE_MODEMIO, "%s <%u:%.*s>",
	dir, cc, buf.getLength(), (char*) buf);
}

/*
 * Device manipulation.
 */

#ifndef B38400
#define	B38400	B19200
#endif
#ifndef B57600
#define	B57600	B38400
#endif
#ifndef B76800
#define	B76800	B57600
#endif
#ifndef B115200
#define	B115200	B76800
#endif
static speed_t termioBaud[] = {
    B0,		// BR0
    B300,	// BR300
    B1200,	// BR1200
    B2400,	// BR2400
    B4800,	// BR4800
    B9600,	// BR9600
    B19200,	// BR19200
    B38400,	// BR38400
    B57600,	// BR57600
    B76800,	// BR76800
    B115200,	// BR115200
};
#define	NBAUDS	(sizeof (termioBaud) / sizeof (termioBaud[0]))
static const char* flowNames[] = { "NONE", "XON/XOFF", "RTS/CTS", };
static const char* parityNames[] = {
    "8 bits, no parity",	// NONE
    "7 bits, even parity",	// EVEN
    "7 bits, odd parity",	// ODD
};

#if defined(CCTS_OFLOW) && defined(CRTS_IFLOW) && !defined(__NetBSD__)
#undef CRTSCTS				/* BSDi */
#define	CRTSCTS	(CCTS_OFLOW|CRTS_IFLOW)
#endif
#if defined(CTSFLOW) && defined(RTSFLOW)
#define	CRTSCTS	(CTSFLOW|RTSFLOW)	/* SCO */
#endif
#if defined(CNEW_RTSCTS)
#define	CRTSCTS	CNEW_RTSCTS		/* IRIX 5.x */
#endif
#ifndef CRTSCTS
#define	CRTSCTS	0
#endif

static void
setFlow(termios& term, FlowControl iflow, FlowControl oflow)
{
    switch (iflow) {
    case ClassModem::FLOW_NONE:
	term.c_iflag &= ~IXON;
	term.c_cflag &= ~CRTSCTS;
	break;
    case ClassModem::FLOW_XONXOFF:
	term.c_iflag |= IXON;
	term.c_cflag &= ~CRTSCTS;
	break;
    case ClassModem::FLOW_RTSCTS:
	term.c_iflag &= ~IXON;
	term.c_cflag |= CRTSCTS;
	break;
    }
    switch (oflow) {
    case ClassModem::FLOW_NONE:
	term.c_iflag &= ~IXOFF;
	term.c_cflag &= ~CRTSCTS;
	break;
    case ClassModem::FLOW_XONXOFF:
	term.c_iflag |= IXOFF;
	term.c_cflag &= ~CRTSCTS;
	break;
    case ClassModem::FLOW_RTSCTS:
	term.c_iflag &= ~IXOFF;
	term.c_cflag |= CRTSCTS;
	break;
    }
}

fxBool
ModemServer::tcsetattr(int op, struct termios& term)
{
    if (clocalAsRoot) {
	/*
	 * Gag, IRIX 5.2 and beyond permit only the super-user to
	 * change the CLOCAL bit on a tty device with modem control.
	 * However, since there appear to be strange side effects
	 * when doing this with some modems, for now we enable this
	 * from the configuration file so that IRIX users can still
	 * disable it.
	 */
	uid_t euid = ::geteuid();
	(void) ::seteuid(0);
	fxBool ok = (::tcsetattr(modemFd, op, &term) == 0);
	::seteuid(euid);
	return ok;
    } else
	return (::tcsetattr(modemFd, op, &term) == 0);
}

/*
 * Set tty port baud rate and flow control.
 */
fxBool
ModemServer::setBaudRate(BaudRate rate, FlowControl iFlow, FlowControl oFlow)
{
    struct termios term;
    if (::tcgetattr(modemFd, &term) == 0) {
	if (rate >= NBAUDS)
	    rate = NBAUDS-1;
	curRate = rate;				// NB: for use elsewhere
	term.c_oflag = 0;
	term.c_lflag = 0;
	term.c_iflag &= IXON|IXOFF;		// keep these bits
	term.c_cflag &= CRTSCTS;		// and these bits
	setFlow(term, iFlow, oFlow);
	term.c_cflag |= CLOCAL | CS8 | CREAD;
	::cfsetospeed(&term, termioBaud[rate]);
	::cfsetispeed(&term, termioBaud[rate]);
	term.c_cc[VMIN] = 127;			// buffer input by default
	term.c_cc[VTIME] = 1;
	traceStatus(FAXTRACE_MODEMOPS,
	    "MODEM set baud rate: %d baud, input flow %s, output flow %s",
	    baudRates[rate], flowNames[iFlow], flowNames[oFlow]);
	flushModemInput();
#if HAS_TXCD
	/* 
	 * From: Steve Williams <steve@geniers.cuug.ab.ca>
	 *
  	 * Under AIX there is no easy way to determine if hardware
	 * handshaking is already on the terminal Control Discipline
	 * stack.  This is necessary to properly condition the tty
	 * device for RTS/CTS flow control.  Rather than determine
	 * the state of the device we just add/remove the Control
	 * Discipline and ignore any errors.  Note that this is the
	 * only place this must be done because it is the only
	 * method through which the modem drivers enable/disable
	 * RTS/CTS flow control.
	 */
	if (iFlow == FaxModem::FLOW_RTSCTS) {
	    traceStatus(FAXTRACE_MODEMOPS,
		"MODEM add rts control discipline");
	    (void) ::ioctl(modemFd, TXADDCD, "rts");
	} else {
	    traceStatus(FAXTRACE_MODEMOPS,
		"MODEM remove rts control discipline");
	    (void) ::ioctl(modemFd, TXDELCD, "rts");
	}
#endif
	return (tcsetattr(TCSAFLUSH, term));
    } else
	return (FALSE);
}

/*
 * Set tty port baud rate and leave flow control state unchanged.
 */
fxBool
ModemServer::setBaudRate(BaudRate rate)
{
    struct termios term;
    if (::tcgetattr(modemFd, &term) == 0) {
	if (rate >= NBAUDS)
	    rate = NBAUDS-1;
	curRate = rate;				// NB: for use elsewhere
	term.c_oflag = 0;
	term.c_lflag = 0;
	term.c_iflag &= IXON|IXOFF;		// keep these bits
	term.c_cflag &= CRTSCTS;		// and these bits
	term.c_cflag |= CLOCAL | CS8 | CREAD;
	::cfsetospeed(&term, termioBaud[rate]);
	::cfsetispeed(&term, termioBaud[rate]);
	term.c_cc[VMIN] = 127;			// buffer input by default
	term.c_cc[VTIME] = 1;
	traceStatus(FAXTRACE_MODEMOPS,
	    "MODEM set baud rate: %d baud (flow control unchanged)",
	    baudRates[rate]);
	flushModemInput();
	return (tcsetattr(TCSAFLUSH, term));
    } else
	return (FALSE);
}

/*
 * Set tty port parity and number of data bits
 */
fxBool
ModemServer::setParity(Parity parity)
{
    struct termios term;
    if (::tcgetattr(modemFd, &term) == 0) {
	switch(parity) {
	case NONE: 
            term.c_cflag &= ~(CSIZE | PARENB);
            term.c_cflag |= CS8;
	    term.c_iflag &= ~(IGNPAR | ISTRIP);
            break;
        case EVEN:
            term.c_cflag &= ~(CSIZE | PARODD);
            term.c_cflag |= CS7 | PARENB;
            term.c_iflag |= IGNPAR | ISTRIP;
            break;
        case ODD:
            term.c_cflag &= ~CSIZE;
            term.c_cflag |= CS7 | PARENB | PARODD;
            term.c_iflag |= IGNPAR | ISTRIP;
            break;
        }
        traceStatus(FAXTRACE_MODEMOPS, "MODEM set parity: %s",
	    parityNames[parity]);
        flushModemInput();
        return (tcsetattr(TCSANOW, term));
    } else
        return (FALSE);
}

/*
 * Manipulate DTR on tty port.
 *
 * On systems that support explicit DTR control this is done
 * with an ioctl.  Otherwise we assume that setting the baud
 * rate to zero causes DTR to be dropped (asserting DTR is
 * assumed to be implicit in setting a non-zero baud rate).
 *
 * NB: we use the explicit DTR manipulation ioctls because
 *     setting the baud rate to zero on some systems can cause
 *     strange side effects.
 */
fxBool
ModemServer::setDTR(fxBool onoff)
{
    traceStatus(FAXTRACE_MODEMOPS, "MODEM set DTR %s", onoff ? "ON" : "OFF");
#ifdef TIOCMBIS
    int mctl = TIOCM_DTR;
#if defined(sun) || defined(linux)
    /*
     * Happy days! SVR4 passes the arg by value, while
     * SunOS 4.x passes it by reference; is this progress?
     */
    if (::ioctl(modemFd, onoff ? TIOCMBIS : TIOCMBIC, (char *)&mctl) == 0)
#else
    if (::ioctl(modemFd, onoff ? TIOCMBIS : TIOCMBIC, (char *)mctl) == 0)
#endif
	return (TRUE);
    /*
     * Sigh, Sun seems to support this ioctl only on *some*
     * devices (e.g. on-board duarts, but not the ALM-2 card);
     * so if the ioctl that should work fails, we fallback
     * on the usual way of doing things...
     */
#endif
    return (onoff ? TRUE : setBaudRate(ClassModem::BR0));
}

static const char* actNames[] = { "NOW", "DRAIN", "FLUSH" };
static u_int actCode[] = { TCSANOW, TCSADRAIN, TCSAFLUSH };

/*
 * Set tty modes so that the specified handling
 * is done on data being sent and received.  When
 * transmitting binary data, oFlow is FLOW_NONE to
 * disable the transmission of XON/XOFF by the host
 * to the modem.  When receiving binary data, iFlow
 * is FLOW_NONE to cause XON/XOFF from the modem
 * to not be interpreted.  In each case the opposite
 * XON/XOFF handling should be enabled so that any
 * XON/XOFF from/to the modem will be interpreted.
 */
fxBool
ModemServer::setXONXOFF(FlowControl iFlow, FlowControl oFlow, SetAction act)
{
    struct termios term;
    if (::tcgetattr(modemFd, &term) == 0) {
	setFlow(term, iFlow, oFlow);
	traceStatus(FAXTRACE_MODEMOPS,
	    "MODEM set XON/XOFF/%s: input %s, output %s",
	    actNames[act],
	    iFlow == ClassModem::FLOW_NONE ? "ignored" : "interpreted",
	    oFlow == ClassModem::FLOW_NONE ? "disabled" : "generated"
	);
	if (act == ClassModem::ACT_FLUSH)
	    flushModemInput();
	return (tcsetattr(actCode[act], term));
    }
    return (FALSE);
}

#ifdef sgi
#include <sys/stropts.h>
#include <sys/z8530.h>
#endif
#ifdef sun
#include <sys/stropts.h>
#endif

/*
 * Setup process state either for minimum latency (no buffering)
 * or reduced latency (input may be buffered).  We fiddle with
 * the termio structure and, if required, the streams timer
 * that delays the delivery of input data from the UART module
 * upstream to the tty module.
 */
fxBool
ModemServer::setInputBuffering(fxBool on)
{
    traceStatus(FAXTRACE_MODEMOPS, "MODEM input buffering %s",
	on ? "enabled" : "disabled");
#ifdef SIOC_ITIMER
    /*
     * Silicon Graphics systems have a settable timer
     * that causes the UART driver to delay passing
     * data upstream to the tty module.  This can cause
     * anywhere from 20-30ms delay between input characters.
     * We set it to zero when input latency is critical.
     */
    strioctl str;
    str.ic_cmd = SIOC_ITIMER;
    str.ic_timout = (on ? 2 : 0);	// 2 ticks = 20ms (usually)
    str.ic_len = 4;
    int arg = 0;
    str.ic_dp = (char*)&arg;
    if (::ioctl(modemFd, I_STR, &str) < 0)
	traceStatus(FAXTRACE_MODEMOPS, "MODEM ioctl(SIOC_ITIMER): %m");
#endif
#ifdef sun
    /*
     * SunOS has a timer similar to the SIOC_ITIMER described
     * above for input on the on-board serial ports, but it is
     * not generally accessible because it is controlled by a
     * stream control message (M_CTL w/ either MC_SERVICEDEF or
     * MC_SERVICEIMM) and you can not do a putmsg directly to
     * the UART module and the tty driver does not provide an
     * interface.  Also, the ALM-2 driver apparently also has
     * a timer, but does not provide the M_CTL interface that's
     * provided for the on-board ports.  All in all this means
     * that the only way to defeat the timer for the on-board
     * serial ports (and thereby provide enough control for the
     * fax server to work with Class 1 modems) is to implement
     * a streams module in the kernel that provides an interface
     * to the timer--which is what has been done.  In the case of
     * the ALM-2, however, you are just plain out of luck unless
     * you have source code.
     */
    static fxBool zsunbuf_push_tried = FALSE;
    static fxBool zsunbuf_push_ok = FALSE;
    if (on) {			// pop zsunbuf if present to turn on buffering
	char topmodule[FMNAMESZ+1];
        if (zsunbuf_push_ok && ::ioctl(modemFd, I_LOOK, topmodule) >= 0 &&
	  streq(topmodule, "zsunbuf")) {
	    if (::ioctl(modemFd, I_POP, 0) < 0)
		traceStatus(FAXTRACE_MODEMOPS, "MODEM pop zsunbuf failed %m");
	}
    } else {			// push zsunbuf to turn off buffering
        if (!zsunbuf_push_tried) {
            zsunbuf_push_ok = (::ioctl(modemFd, I_PUSH, "zsunbuf") >= 0);
            traceStatus(FAXTRACE_MODEMOPS, "MODEM initial push zsunbuf %s",
                zsunbuf_push_ok ? "succeeded" : "failed");
            zsunbuf_push_tried = TRUE;
        } else if (zsunbuf_push_ok) {
            if (::ioctl(modemFd, I_PUSH, "zsunbuf") < 0)
                traceStatus(FAXTRACE_MODEMOPS, "MODEM push zsunbuf failed %m");
        }
    }
#endif
    struct termios term;
    (void) ::tcgetattr(modemFd, &term);
    if (on) {
	term.c_cc[VMIN] = 127;
	term.c_cc[VTIME] = 1;
    } else {
	term.c_cc[VMIN] = 1;
	term.c_cc[VTIME] = 0;
    }
    return (tcsetattr(TCSANOW, term));
}

fxBool
ModemServer::sendBreak(fxBool pause)
{
    traceStatus(FAXTRACE_MODEMOPS, "MODEM send break");
    flushModemInput();
    if (pause) {
	/*
	 * NB: TCSBRK is supposed to wait for output to drain,
	 * but modems appear to lose data if we don't do this.
	 */
	(void) ::tcdrain(modemFd);
    }
    return (::tcsendbreak(modemFd, 0) == 0);
}

void
ModemServer::startTimeout(long ms)
{
    timer.startTimeout(ms);
    timeout = FALSE;
}

void
ModemServer::stopTimeout(const char* whichdir)
{
    timer.stopTimeout();
    if (timeout = timer.wasTimeout())
	traceStatus(FAXTRACE_MODEMOPS, "TIMEOUT: %s", whichdir);
}

int
ModemServer::getModemLine(char rbuf[], u_int bufSize, long ms)
{
    int c;
    int cc = 0;
    if (ms) startTimeout(ms);
    do {
	while ((c = getModemChar(0)) != EOF && c != '\n')
	    if (c != '\0' && c != '\r' && cc < bufSize)
		rbuf[cc++] = c;
    } while (cc == 0 && c != EOF);
    rbuf[cc] = '\0';
    if (ms) stopTimeout("reading line from modem");
    if (!timeout)
	traceStatus(FAXTRACE_MODEMCOM, "--> [%d:%s]", cc, rbuf);
    return (cc);
}

int
ModemServer::getModemChar(long ms)
{
    if (rcvNext >= rcvCC) {
	int n = 0;
	if (ms) startTimeout(ms);
	do
	    rcvCC = Sys::read(modemFd, (char*) rcvBuf, sizeof (rcvBuf));
	while (n++ < 5 && rcvCC == 0);
	if (ms) stopTimeout("reading from modem");
	if (rcvCC <= 0) {
	    if (rcvCC < 0) {
		if (errno != EINTR)
		    traceStatus(FAXTRACE_MODEMCOM,
			"MODEM READ ERROR: errno %u", errno);
	    }
	    return (EOF);
	} else
	    traceModemIO("-->", rcvBuf, rcvCC);
	rcvNext = 0;
    }
    return (rcvBuf[rcvNext++]);
}

void
ModemServer::modemFlushInput()
{
    flushModemInput();
    (void) ::tcflush(modemFd, TCIFLUSH);
    traceStatus(FAXTRACE_MODEMOPS, "MODEM flush i/o");
}

fxBool
ModemServer::modemStopOutput()
{
    return (::tcflow(modemFd, TCOOFF) == 0);
}

void
ModemServer::flushModemInput()
{
    rcvCC = rcvNext = 0;
}

fxBool
ModemServer::putModem(const void* data, int n, long ms)
{
    traceStatus(FAXTRACE_MODEMCOM, "<-- data [%d]", n);
    return (putModem1(data, n, ms));
}

fxBool
ModemServer::putModem1(const void* data, int n, long ms)
{
    if (ms)
	startTimeout(ms);
    else
	timeout = FALSE;
    int cc = Sys::write(modemFd, (const char*) data, n);
    if (ms)
	stopTimeout("writing to modem");
    if (cc > 0) {
	traceModemIO("<--", (const u_char*) data, cc);
	n -= cc;
    }
    if (cc == -1) {
	if (errno != EINTR)
	    traceStatus(FAXTRACE_MODEMCOM, "MODEM WRITE ERROR: errno %u",
		errno);
    } else if (n != 0)
	traceStatus(FAXTRACE_MODEMCOM, "MODEM WRITE SHORT: sent %u, wrote %u",
	    cc+n, cc);
    return (!timeout && n == 0);
}
