/*
 * types.c : Converters for SearchType and SortType, a converter for
 *	fonts that allows per-application defaults, and an improved
 *	converter for Widget that allows "NULL" to be specified.
 *
 * George Ferguson, ferguson@cs.rochester.edu, 12 Sep 1991.
 * Version 2.0: 23 Apr 1993: For pre-R5 systems, we include the string-
 *		             to-font-struct converter.
 * 30 Jun 1993: Removed extra quotes in stringToSearchType().
 */

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Cardinals.h>
#include "types.h"
#include "xarchie.h"
#include "appres.h"
#include "stringdefs.h"
extern void XmuCvtStringToWidget();	/* original old-style converter */

/*
 * In R5, more of the default converters are available externally.
 * For R4 however, the string-to-font converters are static, so we
 * include the appropriate code below.
 */
#if XtSpecificationRelease<5
static Boolean CvtStringToFontStruct();		/* original static converter */
static int CompareISOLatin1();
#else
extern Boolean XtCvtStringToFontStruct();	/* original converter */
#endif

/*
 * Functions defined here:
 */
void initConverters();
SearchType stringToSearchType();
SortType stringToSortType();
char *searchTypeToString(),*sortTypeToString();

static void strcpyLowered();
static Boolean cvtStringToSearchType(),cvtStringToSortType();
static Boolean cvtStringToFontStruct();
static void cvtStringToWidget(); /* old-style converter */

/* Straight from the Xt manual... */
#define done(TYPE,VALUE)\
{							\
    if (toVal->addr != NULL) {				\
	if (toVal->size < sizeof(TYPE)) {		\
	    toVal->size = sizeof(TYPE);			\
	    return(False);				\
	}						\
	*(TYPE *)(toVal->addr) = (VALUE);		\
    } else {						\
	static TYPE static_val;				\
	static_val = (VALUE);				\
	toVal->addr = (XtPointer)&static_val;		\
    }							\
    toVal->size = sizeof(TYPE);				\
    return(True);					\
}
							
void
initConverters(appContext)
XtAppContext appContext;
{
    static XtConvertArgRec parentCvtArgs[] = {
        {XtBaseOffset, (caddr_t)XtOffset(Widget, core.parent), sizeof(Widget)}
    };
    static XtConvertArgRec displayCvtArgs[] = {
	{XtAddress, (XtPointer)&display, sizeof(Display*)},
    };

    /* Specify XtCacheNone for some reason, broke otherwise */
    XtSetTypeConverter(XtRString,GfRSearchType,cvtStringToSearchType,
		       NULL,ZERO,XtCacheNone,NULL);
    XtSetTypeConverter(XtRString,GfRSortType,cvtStringToSortType,
		       NULL,ZERO,XtCacheNone,NULL);
    /* Override default here */
    XtSetTypeConverter(XtRString,XtRFontStruct,cvtStringToFontStruct,
		       displayCvtArgs,XtNumber(displayCvtArgs),
		       XtCacheByDisplay,NULL);
    /* Have to initialize Form class first or our converter will be	*/
    /* overidden by the class initialization function.			*/
    /* Use the old style here on purpose since that what the default is.*/
    XtInitializeWidgetClass(formWidgetClass);
    XtAppAddConverter(appContext,XtRString,XtRWidget,cvtStringToWidget,
					parentCvtArgs,XtNumber(parentCvtArgs));
}

/*	-	-	-	-	-	-	-	-	*/
/* Xt converters */
static void
strcpyLowered(dst,src)
char *dst,*src;
{
    do {
	if (*src >= 'A' && *src <= 'Z')
	    *dst++ = *src - 'A' + 'a';
	else
	    *dst++ = *src;
    } while (*src++);
}

/* ARGSUSED */
static Boolean
cvtStringToSearchType(display,args,num_args,fromVal,toVal,data)
    Display *display;
    XrmValuePtr args;		/* unused */
    Cardinal    *num_args;      /* unused */
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *data;		/* unused */
{
    char message[256];
    SearchType type;

    strcpyLowered(message,(char*)(fromVal->addr));
    if ((type=stringToSearchType(message)) != GfError)
	done(SearchType,type);
    XtDisplayStringConversionWarning(display,fromVal->addr,GfRSearchType);
    toVal->addr = NULL;
    toVal->size = 0;
    return(False);
}

/* ARGSUSED */
static Boolean
cvtStringToSortType(display,args,num_args,fromVal,toVal,data)
    Display *display;
    XrmValuePtr args;		/* unused */
    Cardinal    *num_args;      /* unused */
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *data;		/* unused */
{
    char message[256];
    SortType type;

    strcpyLowered(message,(char*)(fromVal->addr));
    if ((type=stringToSortType(message)) != GfError)
	done(SortType,type);
    XtDisplayStringConversionWarning(display,fromVal->addr,GfRSortType);
    toVal->addr = NULL;
    toVal->size = 0;
    return(False);
}

/*	-	-	-	-	-	-	-	-	*/
/*
 * cvtStringToFontStruct() : This just checks for the two special names
 *	"xarchieFont" and "xarchieBoldFont" and returns the appropriate
 *	application resource. Otherwise invokes the default converter.
 */

/* ARGSUSED */
static Boolean
cvtStringToFontStruct(display,args,num_args,fromVal,toVal,data)
    Display *display;
    XrmValuePtr args;
    Cardinal    *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *data;
{
    char message[256];

    strcpyLowered(message,(char*)(fromVal->addr));
    if (strcmp(message,"xarchiefont") == 0)
	done(XFontStruct*,appResources.xarchieFont);
    if (strcmp(message,"xarchieboldfont") == 0)
	done(XFontStruct*,appResources.xarchieBoldFont);
#if XtSpecificationRelease<5
    return(CvtStringToFontStruct(display,args,num_args,fromVal,toVal,data));
#else
    return(XtCvtStringToFontStruct(display,args,num_args,fromVal,toVal,data));
#endif
}

/*	-	-	-	-	-	-	-	-	*/
/*
 * cvtStringToWidget() : Allows us to specify "NULL" as a widget name in
 *	a resource file to override compiled-in defaults for composite
 *	widget layouts. Simply calls the regular converter if the string
 *	is not "NULL". Note that this must be registered *after* the
 *	Form class is initialized.
 */
static void
cvtStringToWidget(args,num_args,fromVal,toVal)
XrmValuePtr args;
Cardinal *num_args;
XrmValuePtr fromVal;
XrmValuePtr toVal;
{
    if (strcasecmp(fromVal->addr,"null") == 0) {
	toVal->addr = NULL;
	toVal->size = 0;
    } else {
	XmuCvtStringToWidget(args,num_args,fromVal,toVal);
    }
}

/*	-	-	-	-	-	-	-	-	*/
/* Useful generic converters */

SearchType
stringToSearchType(s)
char *s;
{
    if (strcmp(s,GfNExact) == 0)
	return(GfExact);
    if (strcmp(s,GfNSubstr) == 0)
	return(GfSubstr);
    if (strcmp(s,GfNSubcase) == 0)
	return(GfSubcase);
    if (strcmp(s,GfNRegexp) == 0)
	return(GfRegexp);
    if (strcmp(s,GfNExactSubstr) == 0)
	return(GfExactSubstr);
    if (strcmp(s,GfNExactSubcase) == 0)
	return(GfExactSubcase);
    if (strcmp(s,GfNExactRegexp) == 0)
	return(GfExactRegexp);
    return(GfError);
}

char *
searchTypeToString(type)
SearchType type;
{
    switch (type) {
	case GfExact :		return(GfNExact);
	case GfSubstr : 	return(GfNSubstr);
	case GfSubcase :	return(GfNSubcase);
	case GfRegexp :  	return(GfNRegexp);
	case GfExactSubstr :	return(GfNExactSubstr);
	case GfExactSubcase :	return(GfNExactSubcase);
	case GfExactRegexp :	return(GfNExactRegexp);
	default: return("UNKNOWN");
    }
}

SortType
stringToSortType(s)
char *s;
{
    if (strcmp(s,GfNName) == 0)
	return(GfName);
    if (strcmp(s,GfNDate) == 0)
	return(GfDate);
    if (strcmp(s,GfNWeight) == 0)
	return(GfWeight);
    return(GfError);
}

char *
sortTypeToString(type)
SortType type;
{
    switch (type) {
	case GfName : return(GfNName);
	case GfDate : return(GfNDate);
	case GfWeight : return(GfNWeight);
	default: return("UNKNOWN");
    }
}

/*	-	-	-	-	-	-	-	-	*/
/* The string-to-font-struct converter was static in R4 (and earlier?)
 * so we include it here if necessary. This code is taken from the
 * file lib/Xt/Converters.c in the R4 distribution.
 */

#if XtSpecificationRelease<5
/***********************************************************
Copyright 1987, 1988 by Digital Equipment Corporation, Maynard, Massachusetts,
and the Massachusetts Institute of Technology, Cambridge, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of Digital or MIT not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

******************************************************************/

#include <X11/keysym.h>
#include <X11/Quarks.h>

/*ARGSUSED*/
static Boolean
CvtStringToFontStruct(dpy, args, num_args, fromVal, toVal, closure_ret)
    Display*	dpy;
    XrmValuePtr args;
    Cardinal    *num_args;
    XrmValuePtr	fromVal;
    XrmValuePtr	toVal;
    XtPointer	*closure_ret;
{
    XFontStruct	    *f;
    Display*	display;

    if (*num_args != 1)
     XtAppErrorMsg(XtDisplayToApplicationContext(dpy),
	     "wrongParameters","cvtStringToFontStruct","XtToolkitError",
             "String to font conversion needs display argument",
              (String *) NULL, (Cardinal *)NULL);

    display = *(Display**)args[0].addr;

    if (CompareISOLatin1((String)fromVal->addr, XtDefaultFont) != 0) {
	f = XLoadQueryFont(display, (char *)fromVal->addr);
	if (f != NULL) {
  Done:	    done( XFontStruct*, f);
	}

	XtDisplayStringConversionWarning( dpy, (char*)fromVal->addr,
					  "FontStruct" );
    }

    /* try and get the default font */

    {
	XrmName xrm_name[2];
	XrmClass xrm_class[2];
	XrmRepresentation rep_type;
	XrmValue value;

	xrm_name[0] = XrmStringToName ("xtDefaultFont");
	xrm_name[1] = NULL;
	xrm_class[0] = XrmStringToClass ("XtDefaultFont");
	xrm_class[1] = NULL;
	if (XrmQGetResource(XtDatabase(dpy), xrm_name, xrm_class, 
			    &rep_type, &value)) {
	    if (rep_type == XtQString) {
		f = XLoadQueryFont(display, (char*)value.addr);
		if (f != NULL)
		    goto Done;
		else {
		    XtDisplayStringConversionWarning( dpy, (char*)value.addr,
						      "FontStruct" );
		}
	    } else if (rep_type == XtQFont) {
		f = XQueryFont(dpy, *(Font*)value.addr );
		if (f != NULL) goto Done;
	    } else if (rep_type == XtQFontStruct) {
		f = (XFontStruct*)value.addr;
		goto Done;
	    }
	}
    }
    /* Should really do XListFonts, but most servers support this */
    f = XLoadQueryFont(dpy,"-*-*-*-R-*-*-*-120-*-*-*-*-ISO8859-1");
    if (f != NULL)
	goto Done;

    XtAppErrorMsg(XtDisplayToApplicationContext(dpy),
	     "noFont","cvtStringToFontStruct","XtToolkitError",
             "Unable to load any useable ISO8859-1 font",
              (String *) NULL, (Cardinal *)NULL);
    
    return False;
}

static int CompareISOLatin1 (first, second)
    char *first, *second;
{
    register unsigned char *ap, *bp;

    for (ap = (unsigned char *) first, bp = (unsigned char *) second;
	 *ap && *bp; ap++, bp++) {
	register unsigned char a, b;

	if ((a = *ap) != (b = *bp)) {
	    /* try lowercasing and try again */

	    if ((a >= XK_A) && (a <= XK_Z))
	      a += (XK_a - XK_A);
	    else if ((a >= XK_Agrave) && (a <= XK_Odiaeresis))
	      a += (XK_agrave - XK_Agrave);
	    else if ((a >= XK_Ooblique) && (a <= XK_Thorn))
	      a += (XK_oslash - XK_Ooblique);

	    if ((b >= XK_A) && (b <= XK_Z))
	      b += (XK_a - XK_A);
	    else if ((b >= XK_Agrave) && (b <= XK_Odiaeresis))
	      b += (XK_agrave - XK_Agrave);
	    else if ((b >= XK_Ooblique) && (b <= XK_Thorn))
	      b += (XK_oslash - XK_Ooblique);

	    if (a != b) break;
	}
    }
    return (((int) *bp) - ((int) *ap));
}
#endif /* <R5 */
