// viewchanger.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "viewchanger.h"
#include "controller.h"
#include "editor.h"
#include "dataview.h"
#include "data.h"
#include "smpte.h"


ViewChanger::ViewChanger(DataView* view) : myView(view) {
	myView->ref();
}

ViewChanger::~ViewChanger() { Resource::unref(myView); }

Data *
ViewChanger::viewData() { return myView->dataptr; }

int
ViewChanger::apply() {
	return doApply(myView);
}

//********

const QueryInfo *
ScaleOptionsSetter::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Set Horizontal Scale Options:" }, { nil }
	};
	static QueryInfo info[] = { { labels }, { nil } };
	return info;
}

void
ScaleOptionsSetter::configureRequest(Request& request) {
	request.appendChoice("SMPTE Frame Format:", SMPTE::formatNames(),
		linearEnumToPowerOfTwo(SMPTE::currentFormat()), true);
}

boolean
ScaleOptionsSetter::setValues(Request& request) {
	QueryChoice format;
	request.retrieveChoices(format);
	int fmt = format;
	SMPTE::setFormat(SMPTE::Format(powerOfTwoToLinearEnum(format)));
	return true;
}

int
ScaleOptionsSetter::doApply(DataView* view) {
	view->resetHorizontalScale();
	return true;
}

//********

const QueryInfo *
VisibleFrameRange::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Set Endpoints for Frame Display:" }, { nil }
	};
	static QueryInfo info[] = { { labels }, { nil } };
	return info;
}

void
VisibleFrameRange::configureRequest(Request& request) {
	frameRange = myView->getHorizScaleRange();
	if(myView->frameScaleInFrames()) {
		request.appendValue("Minimum Frame: ", frameRange.intMin(), 
			CharCheck::posIntsOnly);
		request.appendValue("Maximum Frame: ", frameRange.intMax(), 
			CharCheck::posIntsOnly);
	}
	else {
		request.appendValue("Minimum Time: ", frameRange.min(),
			CharCheck::posNumsOnly);
		request.appendValue("Maximum Time: ", frameRange.max(),
			CharCheck::posNumsOnly);
	}
}

boolean
VisibleFrameRange::setValues(Request& request) {
	const int nvals = 2;
	QueryValue vr[nvals];
	request.retrieveValues(vr, nvals);
	frameRange.set(vr[0], vr[1]);
	if(!myView->frameScaleInFrames())
		frameRange *= viewData()->frameRate();
	return true;
}

int
VisibleFrameRange::doApply(DataView* view) {
	view->setVisibleFrameRange(frameRange);
	return true;
}

//********

const QueryInfo *
InsertPoint::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Set Insert Point: " }, { nil }
	};
	static QueryInfo info[] = { { labels }, { nil } };
	return info;
}

void
InsertPoint::configureRequest(Request& request) {
	channelRange = myView->getChannelRange();
	boolean inFrames = myView->frameScaleInFrames();
	request.appendValue(inFrames ? "Frame: " : "Time: ",
		"0", CharCheck::posNumsOnly);
	request.appendValue("First Channel: ", channelRange.intMin(),
		CharCheck::posIntsOnly);
	request.appendValue("Last Channel: ", channelRange.intMax(),
		CharCheck::posIntsOnly);
}

boolean
InsertPoint::setValues(Request& request) {
	const int nvals = 3;
	QueryValue vr[nvals];
	request.retrieveValues(vr, nvals);
	insertPoint = myView->frameScaleInFrames() ? int(vr[0]) :
		round(double(vr[0]) * viewData()->frameRate());
	channelRange.set(vr[1], vr[2]);
	return true;
}

boolean
InsertPoint::confirmValues(Controller* c) {
	int status = true;
	if(insertPoint > viewData()->length()) {
		status = false;
		if(Application::confirm(
			"This insert point is beyond the end of the file.",
				"Do you wish to extend the file length?")) {
			c->editor()->changeLength();
			status = (insertPoint <= viewData()->length());
			if(status == false)
				Application::alert("File length is still too short.",
					"Cancelling command.");
		}
	}
	return status;
}

int
InsertPoint::doApply(DataView* view) {
	view->setInsertPoint(insertPoint, channelRange);
	return true;
}

//********

const QueryInfo *
EditRegion::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Set Edit Region:" }, { nil }
	};
	static QueryInfo info[] = { { labels }, { nil } };
	return info;
}

void
EditRegion::configureRequest(Request& request) {
	boolean inFrames = myView->frameScaleInFrames();
	double framerate = inFrames ? 1.0 : viewData()->frameRate();
	frameRange = myView->getVisibleFrameRange();
	channelRange = myView->getChannelRange();
	request.appendValue(inFrames ? "Start Frame: " : "Start Time: ",
		inFrames ? frameRange.intMin() : frameRange.min()/framerate,
		CharCheck::posNumsOnly);
	request.appendValue(inFrames ? "End Frame: " : "End Time: ",
		inFrames ? frameRange.intMax() : (frameRange.max() + 1)/framerate,
		CharCheck::posNumsOnly);
	request.appendValue("Start Channel: ", channelRange.intMin(),
		CharCheck::posIntsOnly);
	request.appendValue("End Channel: ", channelRange.intMax(),
		CharCheck::posIntsOnly);
}

boolean
EditRegion::setValues(Request& request) {
	const int nvals = 4;
	QueryValue vr[nvals];
	request.retrieveValues(vr, nvals);
	boolean inFrames = myView->frameScaleInFrames();
	double framerate = inFrames ? 1.0 : viewData()->frameRate();
	int start = inFrames? int(vr[0]) : round(double(vr[0]) * framerate);
	int end = inFrames? int(vr[1]) : round(double(vr[1]) * framerate) - 1;
	channelRange.set(vr[2], vr[3]),
	frameRange.set(start, end);
	return true;
}

boolean
EditRegion::confirmValues(Controller* c) {
	int status = true;
	int len = viewData()->length();
	if(len <= frameRange) {
		status = false;
		if(Application::confirm(
			"This edit region extends beyond the end of the file.",
				"Do you wish to extend the file length?")) {
			c->editor()->changeLength();
			status = (viewData()->length() > frameRange);
			if(status == false)
				Application::alert("File length is still too short.",
					"Cancelling command.");
		}
	}
	return status;
}

int
EditRegion::doApply(DataView* view) {
	view->setEditRegion(frameRange, channelRange);
	return true;
}

//********

int
ChannelRange::doApply(DataView* view) {
	view->setChannelRange(channelRange);
	return true;
}

//********

const QueryInfo *
ChannelChannelRange::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Set Channel Display Range:" }, { nil }
	};
	static QueryInfo info[] = { { labels }, { nil } };
	return info;
}

void
ChannelChannelRange::configureRequest(Request& request) {
	channelRange = myView->getChannelRange();
	request.appendValue("First Channel: ", channelRange.intMin(),
		CharCheck::posIntsOnly);
	request.appendValue("Last Channel: ", channelRange.intMax(),
		CharCheck::posIntsOnly);
}

boolean
ChannelChannelRange::setValues(Request& request) {
	const int nvals = 2;
	QueryValue vr[nvals];
	request.retrieveValues(vr, nvals);
	channelRange.set(vr[0], vr[1]);
	return true;
}

//********

const QueryInfo *
FrameChannelRange::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Set Horizontal Display Range:" }, { nil }
	};
	static QueryInfo info[] = { { labels }, { nil } };
	return info;
}

void
FrameChannelRange::configureRequest(Request& request) {
	boolean inIntegers = myView->horizScaleInFrames();
	channelRange = myView->getDisplayedHorizRange();
	if(inIntegers) {
		request.appendValue("Minimum: ", channelRange.intMin(),
			CharCheck::posIntsOnly);
		request.appendValue("Maximum: ", channelRange.intMax(),
			CharCheck::posIntsOnly);
	}
	else {
		request.appendValue("Minimum: ", channelRange.min(),
			CharCheck::posNumsOnly);
		request.appendValue("Maximum: ", channelRange.max(),
			CharCheck::posNumsOnly);
	}
}

boolean
FrameChannelRange::setValues(Request& request) {
	const int nvals = 2;
	QueryValue vr[nvals];
	request.retrieveValues(vr, nvals);
	boolean inIntegers = myView->horizScaleInFrames();
	int totalFrames = viewData()->frameLength();
	double min = inIntegers ? int(vr[0])
		: round(totalFrames * double(vr[0])/channelRange.max());
	double max = inIntegers ? int(vr[1])
		: round(totalFrames * double(vr[1])/channelRange.max());
	channelRange.set(min, max);
	return true;
}
