// sun_dac.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <sys/stat.h>
#include <fcntl.h>
#include <stropts.h>
#include "application.h"
#include "typeconvert.h"
#include "sun_audio.h"
#include "sun_dac.h"
#include "statusaction.h"

#ifndef AUDIO_LINE_IN
#define AUDIO_LINE_IN (0x02)
#endif
#ifndef AUDIO_LINE_OUT
#define AUDIO_LINE_OUT (0x04)
#endif

#define PRINT(x, y) fprintf(stderr, x, y)

extern "C" int poll(struct pollfd*, unsigned long, int);

int SunConverter::sparc_version = 1;
struct pollfd SunConverter::pollStruct;

SunConverter::SunConverter() : ConverterDevice("/dev/audio") {
	BUG("SunConverter::SunConverter()");
	initialize();
	failIf(!confirmDeviceType());
}

SunConverter::~SunConverter() {
	BUG("SunConverter::~SunConverter()");
	stop();
}

boolean
SunConverter::isPlayableFormat(DataType type) {
	if(isSparc10())
		return (type == ShortData || type == MuLawData);
	else
		return (type == MuLawData);
}

DataType
SunConverter::bestPlayableType() {
	return isSparc10() ? ShortData : MuLawData;
}

unsigned
SunConverter::sampleEncoding() {
	unsigned encoding;
	DataType type = dataType();
	switch (type) {
	case MuLawData:
		encoding = AUDIO_ENCODING_ULAW; break;
//	case ALawData:
//		encoding = AUDIO_ENCODING_ALAW; break;
	case CharData:
	case ShortData:
		encoding = AUDIO_ENCODING_LINEAR; break;
	case FloatData:
		encoding = AUDIO_ENCODING_FLOAT; break;
	default:
		encoding = 0; break;
	};
	return encoding;
}

int
SunConverter::pause() {
	BUG("SunConverter::pause()");
	int ret = true;
	if(!paused()) {
		ret = willPlay() ? audio_pause_play(fdesc())
			: willRecord() ? audio_pause_record(fdesc()) : -1;
		ret = (ret == AUDIO_SUCCESS && Super::pause());
		if(!ret)
			error("Can't pause audio device.");
	}
	return ret;
}

int
SunConverter::resume() {
	BUG("SunConverter::resume()");
	int ret = true;
	if(paused()) {
		ret = willPlay() ? audio_resume_play(fdesc())
			: willRecord() ? audio_resume_record(fdesc()) : -1;
		ret = (ret == AUDIO_SUCCESS && Super::resume());
		if(!ret) error("Can't unpause audio device.");
	}
	return ret;
}

int
SunConverter::stop() {
	BUG("SunConverter::stop()");
	int status = true;
	if(running()) {
		if(audio_flush(fdesc()) != AUDIO_SUCCESS) {
			error("Can't flush audio device.");
			status = false;
		}
		status = Super::stop() && status;
	}
	return status;
}

int
SunConverter::doConfigure() {
	BUG("SunConverter::doConfigure()");
	Audio_hdr	Dev_hdr;		// audio header for device
	Audio_hdr	Sound_hdr;		// audio header for sound
	
	// fill in values from sound to be played/recorded
	Sound_hdr.sample_rate = sampleRate();
	Sound_hdr.samples_per_unit = 1;
	Sound_hdr.bytes_per_unit = type_to_sampsize(dataType());
	Sound_hdr.channels = channels();
	Sound_hdr.encoding = sampleEncoding();
	Sound_hdr.data_size = dataSize();
	
	boolean status = true;
	// device is opened with correct mode here, not closed again until stop()
	status = reOpen() && pause();
	status = status && getConfiguration(&Dev_hdr);
	if(status) {
		// reconfigure device if device settings do not match sound
		if(audio_cmp_hdr(&Dev_hdr, &Sound_hdr) != 0)
			status = reconfigure(&Dev_hdr, &Sound_hdr);
	}
	int afd = fdesc();
    if(status && willRecord()) {
		// flush input queue
		if(audio_flush_record(afd) != AUDIO_SUCCESS) {
			error("Converter::configure:  Unable to flush input queue.");
			status = false;
		}	// Set the device up for non-blocking reads
		else if(fcntl(afd, F_SETFL, fcntl(afd, F_GETFL, 0) | O_NDELAY) < 0) {
			error("Converter::configure: fcntl() failed.");
			status = false;
		}
	}
	pollStruct.fd = afd;	// initialize polling struct
	pollStruct.events = willRecord() ? POLLIN : POLLOUT;
#if 0
	if(status && willRecord()) {
		double gain = 0;
		audio_get_record_gain(fdesc(), &gain);
		PRINT("At end of configuration: device rec level:  %f\n\n", gain);
	}
#endif
	return status && resume();		// device unpaused at last instant
}

boolean
SunConverter::getConfiguration(Audio_hdr *devhdr) {
	// Get the device output encoding configuration
	boolean status = true;
	if(willPlay()) {
		if(audio_get_play_config(fdesc(), devhdr) != AUDIO_SUCCESS) {
			error("The device is not an audio device.");
			status = false;
		}
	}
	else if(willRecord()) {
		if(audio_get_record_config(fdesc(), devhdr) != AUDIO_SUCCESS) {
			error("The device is not an audio device.");
			status = false;
		}
	}
	return status;
}

boolean
SunConverter::reconfigure(Audio_hdr *devhdr, Audio_hdr *soundhdr) {
	BUG("SunConverter::reconfigure()");
	char msg[AUDIO_MAX_ENCODE_INFO+64];
	char encoding[AUDIO_MAX_ENCODE_INFO];
	char* action = willRecord() ? "record" : "play";
	// wait for old output to drain, if any
	if(willPlay() && audio_drain(fdesc(), 0) != AUDIO_SUCCESS)
		return false;
	*devhdr = *soundhdr;
	int retval = willPlay() ?
		audio_set_play_config(fdesc(), devhdr)
		: audio_set_record_config(fdesc(), devhdr);
	static const double SampRateThreshold = 0.05;
	switch (retval) {
	case AUDIO_SUCCESS:
		break;
	case AUDIO_ERR_NOEFFECT:
		// Couldn't change the device.
		// Check to see if we're nearly compatible.
		// audio_cmp_hdr() returns >0 if only sample rate difference.
		if(audio_cmp_hdr(devhdr, soundhdr) == 1) {
			double ratio = double(abs(
				int(devhdr->sample_rate - soundhdr->sample_rate))) /
				double(soundhdr->sample_rate);
			if (ratio <= SampRateThreshold) {
				sprintf(msg, "Samp rate was %d, %sing at %d",
					soundhdr->sample_rate, action, devhdr->sample_rate);
				Application::inform(msg, true);
			}
			else {
				sprintf(msg, "Cannot %s at sample rate %d.", action,
					soundhdr->sample_rate);
				Application::alert(msg);
				return false;
			}
		}
		else {
			(void) audio_enc_to_str(soundhdr, encoding);
			sprintf(msg, "Cannot %s sound in format:", action);
			Application::alert(msg, encoding);
			return false;
		}
		break;	/*NOTREACHED*/
	case AUDIO_ERR_ENCODING:
		(void) audio_enc_to_str(soundhdr, encoding);
		sprintf(msg, "Cannot %s sound in format:", action);
		Application::alert(msg, encoding);
		return false;
		break;	/*NOTREACHED*/
	case AUDIO_UNIXERROR:
		if(isSparc10()) {	// Sparc 10 generates this instead of previous
			(void) audio_enc_to_str(soundhdr, encoding);
			sprintf(msg, "Cannot %s sound in format:", action);
			error(msg, encoding);
		}
		else
			error("audio configuration failed");
		return false;
		break;	/*NOTREACHED*/
	default:
		(void) audio_enc_to_str(soundhdr, encoding);
		Application::alert("audio configuration failed for format:", encoding);
		return false;
		break;	/*NOTREACHED*/
	}
	return true;
}

// Write EOF to device, then loop to check for play completion

int
SunConverter::waitForStop(StatusAction* askedToStop) {
	BUG("SunConverter::waitForStop()");
	if(audio_play_eof(fdesc()) != AUDIO_SUCCESS)
		return error("Unable to write EOF to device");
	unsigned EOF_count = 0;
	do {
		usleep(50000);	// sleep .05 secs
		if((*askedToStop)())
			break;
		if(audio_get_play_eof(fdesc(), &EOF_count) != AUDIO_SUCCESS)
			return error("waitForStop: audio_get_play_eof failed");
	} while (EOF_count == 0);
	return true;
}

SunConverter::InputPort
SunConverter::currentInputPort() {
	unsigned audio_port = 0;
	if(audio_get_record_port(fdesc(), &audio_port) != AUDIO_SUCCESS)
		error("Unable to retrieve input port setting.");
	return (audio_port & AUDIO_MICROPHONE) ? Mike : Line;
}

int
SunConverter::setInputPort(InputPort port) {
	unsigned audio_port = (port == Mike) ? AUDIO_MICROPHONE : AUDIO_LINE_IN;
	if(audio_set_record_port(fdesc(), &audio_port) != AUDIO_SUCCESS)
		return error("Can't set audio input port!");
	return true;
}

SunConverter::OutputPort
SunConverter::currentOutputPort() {
	unsigned audio_port = 0;
	if(audio_get_play_port(fdesc(), &audio_port) != AUDIO_SUCCESS)
		error("Unable to retrieve output port setting.");
	return (audio_port & AUDIO_SPEAKER) ? Speaker : Headphone;
}


int
SunConverter::setOutputPort(OutputPort port) {
	unsigned audio_port = (port == Speaker) ? AUDIO_SPEAKER : AUDIO_HEADPHONE;
	// enable line output at all times when hardware permits
	if(isSparc20()) audio_port |= AUDIO_LINE_OUT;
	if(audio_set_play_port(fdesc(), &audio_port) != AUDIO_SUCCESS)
		return error("Can't set audio output port!");
	return true;
}

int
SunConverter::currentRecordLevel() {
	double gain = 0;
	if(audio_get_record_gain(fdesc(), &gain) != AUDIO_SUCCESS)
		error("Unable to retrieve record gain level.");
#ifdef debug
	PRINT("Retrieved rec. level from device:  %f\n\n", gain);
#endif
	return int(gain * 100.0);
}

int
SunConverter::setRecordLevel(int volume) {
	double gain = volume / 100.0;
#ifdef debug
	PRINT("Setting device record level to %f\n", gain);
#endif
	if(audio_set_record_gain(fdesc(), &gain) != AUDIO_SUCCESS)
		return error("Unable to set converter record level.");
#ifdef debug
	audio_get_record_gain(fdesc(), &gain);
	PRINT("Retrieving just-set level for check:  %f\n\n", gain);
#endif
	return true;
}

int
SunConverter::currentPlayLevel() {
	double gain = 0;
	if(audio_get_play_gain(fdesc(), &gain) != AUDIO_SUCCESS)
		error("Unable to retrieve playback gain level.");
#ifdef debug
	PRINT("Retrieved play level from device:  %f\n\n", gain);
#endif
	return round(gain * 100.0);
}

int
SunConverter::setPlayLevel(int volume) {
	double gain = volume / 100.0;
#ifdef debug
	PRINT("Setting device play level to %f\n", gain);
#endif
	if(audio_set_play_gain(fdesc(), &gain) != AUDIO_SUCCESS)
		return error("Unable to set converter play level.");
#ifdef debug
	audio_get_play_gain(fdesc(), &gain);
	PRINT("Retrieving just-set level for check:  %f\n\n", gain);
#endif
	return true;
}

int
SunConverter::confirmDeviceType() {
	struct stat st;
	if(stat(deviceName(), &st) < 0) {
		error("Cannot stat converter device.");
		return false;
	}
	if (!S_ISCHR(st.st_mode)) {
		char msg[128];
		sprintf(msg, "%s %s", deviceName(), "is not an audio device.");
		Application::alert(msg);
		return false;
	}
	return true;
}

#ifndef __GNUG__
extern "C" char* getenv(const char*);
#endif

void
SunConverter::initialize() {
	BUG("SunConverter::initialize()");
	// to avoid multiple binaries, we use environmental vars to distinguish
	const char* enVar = getenv("MXV_SPARC_10");
	sparc_version = 1;	// default
	if(enVar && (!strcmp(enVar, "YES") || !strcmp(enVar, "yes")))
		sparc_version = 10;
	enVar = getenv("MXV_SPARC_20");
	if(enVar && (!strcmp(enVar, "YES") || !strcmp(enVar, "yes")))
		sparc_version = 20;
}

void
SunConverter::waitForDevice() {
	(void) poll(&pollStruct, 1L, -1);
}

// always write 0.1 seconds of sound per buffer

int
SunConverter::writeSize() {
	return int(0.1 * sampleRate() * channels() * type_to_sampsize(dataType()));
}

// later this may be changed

int
SunConverter::readSize() {
	return writeSize();
}
