// formantfilter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "formantfilter.h"
#include "application.h"
#include "controller.h"
#include "datamodifier.h"
#include "lpcdata.h"
#include "query.h"
#include "request.h"

FormantFilter::FormantFilter(Data* input, Data* output,
	LPCData* data, double amp) : LPC_Filter(data, input->length()),
		  SimpleInputOutputFunction(input, output, lpcQueueSize, lpcQueueSize),
		  gain(amp), warpFactor(0) {
	if(LPC_Filter::good())
		initialize();
}

FormantFilter::FormantFilter(Data* input, Data* output, LPCData* data)
		: LPC_Filter(data, input->length()),
		  SimpleInputOutputFunction(input, output, lpcQueueSize, lpcQueueSize),
		  warpFactor(0)
{
	ok(LPC_Filter::good());
}

void
FormantFilter::initialize() {
	if(warpFactor != 0.0) {
		Data* warpData = new Data(FloatData, getLPC()->length() + 1);
		warpData->ref();
		Offset o(warpData, warpFactor);	// set to single value
		o.apply();
		setWarp(warpData);
		Resource::unref(warpData);
	}
	LPC_Filter::initialize();
	restoreState();
	SimpleInputOutputFunction::initialize();
}

void
FormantFilter::restoreState() {
	LPC_Filter::reset();
    SimpleInputOutputFunction::restoreState();
}

int
FormantFilter::doProcessing() {
	BUG("FormantFilter::doProcessing()");
	int status = true;
	while(!inQueueEmpty() && status == true) {
		status = processInput(false);
	}
	return status;
}

double
FormantFilter::doSimpleProcess(double sig) {
	BUG("FormantFilter::doSimpleProcess()");
	float amp = 0.0;
	float dummy1 = 0.0, dummy2 = 0.0;
	float* coeffs = getFrame(&amp, &dummy1, &dummy2);
	return filter(coeffs, sig * amp * gain);
}

const QueryInfo *
FormantFilter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "LPC (Formant) Filter Selected Region:" },
		{ nil }
	};	
	static QueryValueInfo valueInfo[] = {
		{ "Gain Factor:", "1.00", CharCheck::numsOnly },
		{ "Warp Factor:", "0.00", nil, "-1.0|1.0" },
		{ nil }
	};	
	static QueryChoiceInfo choiceInfo[] = {
		{ "Interpolation Mode:", "|Interpolated|Recalculated|", 0x1, true },
		{ nil }
	};
	static QueryInfo info[] = {
		{ labelInfo, "", valueInfo, choiceInfo },
		{ nil }
	};
	return info;
}

boolean
FormantFilter::configure(Controller *c) {
	return SimpleInputOutputFunction::configure(c);
}

boolean
FormantFilter::setValues(Request &request) {
	QueryValue v[2];
	request.retrieveValues(v, 2);
	gain = v[0];
	warpFactor = v[1];
	QueryChoice mode;
	request.retrieveChoices(mode);
	setInterpMode((mode == 0x1) ? Linear : Computed);
	return true;
}

boolean
FormantFilter::confirmValues(Controller *) {
	int status = true;
	char msg[120];
	if(sampRate() != lpcSamplingRate()) {
		sprintf(msg,
			"Warning: LPC data samprate (%d) != selection samprate (%d).",
				lpcSamplingRate(), sampRate());
		status = Application::confirm(msg, "The formant peaks will be skewed.",
			"Continue anyway?", Cancel);
	}
	return status;
}

