// conv_device.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <sys/ioctl.h>

#include "application.h"
#include "conv_device.h"
#include "statusaction.h"

ConverterDevice::ConverterDevice(const char* devName) : device_name(nil) {
	setDeviceName(devName);
}

void
ConverterDevice::setDeviceName(const char* name) {
	delete [] device_name;
	device_name = newstr(name);
}

// to avoid monopolizing the device, it is closed whenever stopped

int
ConverterDevice::stop() {
	BUG("ConverterDevice::stop()");
	int stat = close();		// must close before doing anything else
	if(!stat) Application::error("ConverterDevice::stop:  close error.");
	return RealConverter::stop() && stat;
}

void
ConverterDevice::fail() {
	BUG("ConverterDevice::fail()");
	close();
	RealConverter::fail();
}
	
boolean
ConverterDevice::reOpen() {
	BUG("ConverterDevice::reOpen()");
	close();
	if(!open(deviceName(), (willRecord() ? O_RDONLY : O_WRONLY) | O_NDELAY)) {
		char msg[128];
		sprintf(msg, "ConverterDevice::reOpen: Cannot open %s", deviceName());
		error(msg);
		return false;
	}
	return true;
}

int
ConverterDevice::doConversion(StatusAction* askedToStop) {
	register addr sound = (addr) pointerToData();
	int bytesToGo = dataSize();
	int writeBytes = min(writeSize(), dataSize());
	Application::inform("Playing...");
	int status = true;
	while(bytesToGo > 0) {
		if((*askedToStop)())
			return stop();
		int bytesWritten = write(sound, writeBytes);
		if(bytesWritten < 0) {
			if(errno == EAGAIN || errno == EWOULDBLOCK) {
				waitForDevice();
				continue;
			}
			status = false;
			break;
		}
		sound += bytesWritten;
		bytesToGo -= bytesWritten;
		writeBytes = min(bytesToGo, writeBytes);
#ifdef debug
		fprintf(stderr, "\t%d-byte block written.  %d bytes remaining.\n", bytesWritten, max(0, bytesToGo));
#endif
	}
	if(status == false)
		error("ConverterDevice::doConversion: write error");
	else status = waitForStop(askedToStop);
	return stop() && status;
}

int
ConverterDevice::doRecording(StatusAction* askedToStop) {
	register addr sound = (addr) pointerToData();
	int bytesToGo = dataSize();
	int readBytes = min(readSize(), dataSize());
	int status = 1;
	Application::inform("Recording...");
	while(bytesToGo > 0) {
		if((*askedToStop)())
			break;
		int bytesRead = read(sound, readBytes);
		if(bytesRead < 0) {
			if(errno == EAGAIN || errno == EWOULDBLOCK) {
				waitForDevice();
				continue;
			}
			status = false;
			break;
		}
		sound += bytesRead;
		bytesToGo -= bytesRead;
		readBytes = min(bytesToGo, readBytes);
#ifdef debug
		fprintf(stderr, "\t%d bytes read.  %d bytes remaining.\n", bytesRead, max(0, bytesToGo));
#endif
	}
	if(status == false)
		error("ConverterDevice::doRecording: read error");
	return stop() && status;
}
