/*-
//////////////////////////////////////////////////////////////////////////
//									//
//   Copyright (c) 1995 Migration Associates Corp. All Rights Reserved	//
//									//
// THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF MIGRATION ASSOCIATES	//
//	The copyright notice above does not evidence any   		//
//	actual or intended publication of such source code.		//
//									//
//	Use, duplication, or disclosure by the Government is		//
//	subject to restrictions as set forth in FAR 52.227-19,		//
//	and (for NASA) as supplemented in NASA FAR 18.52.227-19,	//
//	in subparagraph (c) (1) (ii) of Rights in Technical Data	//
//	and Computer Software clause at DFARS 252.227-7013, any		//
//	successor regulations or comparable regulations of other	//
//	Government agencies as appropriate.				//
//									//
//		Migration Associates Corporation			//
//			19935 Hamal Drive				//
//			Monument, CO 80132				//
//									//
//	A license is granted to Berkeley Software Design, Inc. by	//
//	Migration Associates Corporation to redistribute this software	//
//	under the terms and conditions of the software License		//
//	Agreement provided with this distribution.  The Berkeley	//
//	Software Design Inc. software License Agreement specifies the	//
//	terms and conditions for redistribution.			//
//									//
//	UNDER U.S. LAW, THIS SOFTWARE MAY REQUIRE A LICENSE FROM	//
//	THE U.S. COMMERCE DEPARTMENT TO BE EXPORTED.  IT IS YOUR	//
//	REQUIREMENT TO OBTAIN THIS LICENSE PRIOR TO EXPORTATION.	//
//									//
//////////////////////////////////////////////////////////////////////////
*/

#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <stdio.h>
#include <syslog.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/resource.h>

#include "activ.h"
#include "activdb.h"

static	int	disabled_only = 1;
static	int	enabled_only = 1;

static	int	banner = 1;
static	int	long_format = 0;
static	int	reverse_format = 0;
static	int	single_column = 0;

static	int	count = 0;

extern char *__progname;

static	void	print_record(ACTIVDB_Rec *);
static	int	criteria_match(ACTIVDB_Rec *);

extern	int
main(int argc, char **argv)
{
	int	c;
	static	char	*optstr = "bdelr1";

	struct rlimit cds;

	ACTIVDB_Rec activrec;

	(void)signal(SIGQUIT, SIG_IGN);
	(void)signal(SIGINT, SIG_IGN);
	(void)setpriority(PRIO_PROCESS, 0, 0);

	openlog("login", LOG_ODELAY, LOG_AUTH);

	/*
	 * Make sure we never dump core as we might have a
	 * valid user shared-secret in memory.
	 */

	cds.rlim_cur = 0;
	cds.rlim_max = 0;
	if (setrlimit(RLIMIT_CORE, &cds) < 0)
		syslog(LOG_ERR, "couldn't set core dump size to 0: %m");

	if (getuid() != 0) {	/* log it & quietly go away	*/
		syslog(LOG_INFO, "Attempt to use %s by unauthorized user %s",
				__progname, getlogin());
		exit(1);
	}

    	while ((c = getopt(argc, argv, optstr)) != EOF)
		switch(c) {
		case '1':
			single_column++;
			continue;
		case 'b':
			banner = 0;
			continue;
		case 'd':
			disabled_only++;
			enabled_only = 0;
			continue;
		case 'e':
			enabled_only++;
			disabled_only = 0;
			continue;
		case 'l':
			long_format++;
			continue;
		case 'r':
			reverse_format++;
			continue;

		default:
			fprintf(stderr, "Usage is: %s -%s [ username ... ]\n",
					__progname, optstr);
			exit(1);
		}

	argc -= optind;
	argv = &argv[optind];

	/*
	 * If users specified on command line, list only specified users
	 */

	if (argc) {
		if (argc == 1)
			single_column++;

		while (argc--) {
			if (! activdb_getrec(*argv++, &activrec))
				if (criteria_match(&activrec))
					print_record(&activrec);
		}
	}

	/*
	 * No users specified on command line, list all users matching
	 * search criteria.
	 */

	else {
		if (! activdb_firstrec(reverse_format, &activrec)) {
			if (criteria_match(&activrec))
				print_record(&activrec);

			while (! activdb_nextrec(reverse_format, &activrec))
				if (criteria_match(&activrec))
					print_record(&activrec);
		}
	}

	/*
	 * Trailing crlf needed?
	 */

	if (!single_column) {
		if (long_format) {
			if (count & 1)
				printf("\n");
		} else if (count % 4)
			printf("\n");
	}
	exit(0);
}

/*
 * Check a database record for a match against search criteria
 */

static	int
criteria_match(ACTIVDB_Rec *rec)
{
	if (! (rec->flags & ACTIV_ENABLED) && disabled_only)
		return(1);

	if ((rec->flags & ACTIV_ENABLED) && enabled_only)
		return(1);

	return(0);
}

/*
 * Print a database record according to user a specified format
 */

static	void
print_record(ACTIVDB_Rec *rec)
{
	if (long_format) {
		if (single_column) {
			if (!count && banner) {
				printf("User             Status\n");
				printf("---------------- --------\n");
			}
			printf("%-16s %s\n", rec->uname,
			  rec->flags & ACTIV_ENABLED ? " enabled" : "disabled");
		} else {
			if (!count && banner) {
				printf("User             Status    ");
				printf("User             Status  \n");
				printf("---------------- --------  ");
				printf("---------------- --------\n");
			}
			printf("%-16s %s%s", rec->uname,
			   rec->flags & ACTIV_ENABLED ? " enabled" : "disabled",
			   (count & 1) ? "\n" : "  ");
		}
		count++;
	} else {
		if (single_column) {
			if (!count++ && banner) {
				printf("User\n");
				printf("----------------\n");
			}
			printf("%-16s\n", rec->uname);
		} else {
			if (!count++ && banner) {
				printf(
   "User             User             User             User\n"
   "---------------- ---------------- ---------------- ----------------\n");
			}
			printf("%-16s %s", rec->uname,
				(!(count % 4)) ? "\n" : "");
		}
	}
}
