/*	BSDI ppp.c,v 2.11 1995/10/25 18:43:48 prb Exp	*/

/*
 * Establish a PPP connection
 *
 * ppp  - attach dial-in connection if called as a login shell
 * ppp -d [systemid]
 *      - dial-out dodaemon mode (calls on dropped packets)
 * ppp systemid
 *      - call the specified system explicitly
 *
 * Additional parameters:
 *      -s sysfile      - use sysfile instead of /etc/ppp.sys
 *      -x              - turn on debugging
 *      -b              - put self in background after connecting
 *
 * /etc/ppp.sys contains termcap-style records:
 *      at      (str) Auto call unit type.
 *      br      (num) Baud rate.
 *      cm      (str) Map of special characters (as in pppconfig)
 *      cu      (str) Call unit if making a phone call.
 *      di      (bool) Dial-in allowed.
 *      do      (bool) Automatic dial-out allowed.
 *      dt      (num) Dial timeout (in seconds).
 *      du      (bool) This is a dial-up line
 *      dv      (str) Device(s) to open to establish a
 *                    connection.
 *      e0-e9   (str) String to wait for on nth step of logging in
 *                    (after sending s0-s9)
 *      f0-f9   (str) Strings to send if failed to get an expected
 *                    string (e0-e9)
 *      id      (num) Disconnect on idle (n seconds)
 *      if      (str) Space-separated list of ifconfig parameters.
 *      im      (bool) Immediate Connection
 *      in      (num) Interface number.
 *	ku	(bool) Do not mark interface as down when leaving
 *	ld	(str) Script to run when link goes down.
 *	lf	(str) Script to after system logs in but connection fails
 *	li	(str) Script to run system logs in.
 *	lu	(str) Script to run when link comes up.
 *      mc      (num) Max PPP config retries.
 *      mr      (num) PPP MRU.
 *      ms      (str) Stty-style terminal setup string.
 *      mt      (num) Max PPP terminate retries.
 *      pf      (str) Comma-separated list of PPP flags (as in pppconfig)
 *      pn      (str) Telephone number(s) for this host.
 *      sl      (bool) use slip instead of ppp
 *      s0-s9   (str) String to send on nth step of logging in.
 *      t0-t9   (num) Timeout in seconds for e0-e9
 *      tc      (str) Cap. list continuation.
 *      to      (num) PPP retrty timeout (1/10 sec).
 *	wt	(num) Wait timeout for certain operations (such as BOS)
 */

#include "ppp.h"

#include <sys/param.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/signal.h>

#include <net/if.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/pppioctl.h>
#include <net/slip.h>

#include <string.h>

#include <sys/wait.h>
#include <stdarg.h>
#include <setjmp.h>
#include <utmp.h>

void wt_alarm();
void cleanup(int);
void sliphup();
void updateutmp();
int execute(char *, ...);
int startup(char *, char *);

int background = 0;
int debug = 0;
int dodaemon = 0;
char *sysfile = _PATH_SYS;
char *sysname;
jmp_buf wt_env;
int slip_hangup = 0;

char ifname[30];

int Socket;
int amup = 0;
int quiet;
int oldld;

static int busymsg = 0;
static uid_t suid = -1;
static gid_t sgid = -1;

main(ac, av)
	int ac;
	char **av;
{
	extern int optind;
	extern char *optarg;
	extern char *getlogin();
	int dialin = 0;
	char *linkname;
	int i, timo;
	int ld;
	char *reason;
	pid_t pid;
	int rv;
	void (*osig)();

	for (;;) {
		switch (getopt(ac, av, "bdqxs:")) {
		default:
		usage:
			errx(1, "usage: ppp [-s sysfile] [-b] [-d] [-x] [systemname]");
		case 'b':
			background++;
			continue;

		case 'd':
			dodaemon++;
			continue;

		case 'x':
			debug++;
			continue;

		case 'q':
			quiet = 1;
			continue;

		case 's':
			suid = getuid();
			sgid = getgid();
			sysfile = optarg;
			continue;

		case EOF:
			break;
		}
		break;
	}

	if (optind < ac - 1)
		goto usage;

	/* Check permissions */
	if ((dodaemon || debug) && getuid() != 0) {
		if (dodaemon)
			warnx("only superuser can run in dodaemon mode");
		if (debug)
			warnx("only superuser can debug");
		exit(1);
	}

	setbuf(stdout, NULL);

	/* Get the system name */
	if (ac > optind)
		sysname = av[optind];
	else if (dodaemon)
		sysname = 0;
	else {
		if (dodaemon)
			errx(1, "dodaemon mode not allowed for dialin lines");
		sysname = getlogin();
		dialin++;
		if (sysname == 0)
			errx(1, "can't get the login name");
		/*
		 * Standard dial in lines must run their scripts
		 * as root.
		 */
    	    	if (suid == -1)
			suid = geteuid();
	}

	/*
	 * Daemon mode... look out for the dropped packets!
	 */
	if (dodaemon && sysname == 0)
		/*
		 * Scan the system file for all automatic dial-outs
		 * and do forks for every system name.
		 */
		errx(1, "dodaemon w/o sysname IS NOT IMPLEMENTED (yet)");

	getsyscap(sysname, sysfile);

    	if (SL && background)
		errx(1, "slip currently does not support background mode (this is a bug)");

	if (SL && IN != -1)
		errx(1, "slip does not support static interface allocation");

	if (SL && !dialin && (IM == 0 || ID))
		errx(1, "slip requires immediate mode");

	if (dialin && DI == 0)
		errx(1, "dial in is not allowed for %s", sysname);

	if (!dialin && DO == 0)
		errx(1, "dial out is not allowed for %s", sysname);

	if (IN == -1 && IM == 0 && dodaemon)
		errx(1, "%s: interface number (in) required for non-immediate dodaemon mode.", sysname);
	if (IN == -1 && !dialin && IM == 0)
		errx(1, "%s: dynamic interface allocation is not supported\n\ton demand dial connections.", sysname);

	if (SL && LI == 0)
		errx(1, "login script required for SLIP");
	if (SL && LD == 0)
		errx(1, "down script required for SLIP");

    	if (WT == -1)
		WT = WAITTIMEOUT;

	if (quiet)
		tipset_verbose(0);

	tipset_dialtimeout(DT);
	tipset_cmddelay(CD);

	/*
	 * If we in dodaemon mode and we need to wait
	 * for a packet we need to setup the ifname.
	 */
	if (dodaemon && IM == 0)
		snprintf(ifname, sizeof ifname, "%s%d", SL ? "sl" : "ppp", IN);

	/*
	 * Get a socket for ifreqs
	 */
	Socket = socket(AF_INET, SOCK_DGRAM, 0);
	if (Socket < 0)
		ifperror("socket", "AF_INET");

	/* Dial-in? Just set the line discipline and wait */

daemon_loop:
	if (dialin) {
		log("dialin: %s\n", sysname);
		FD = 0;

		/* Set working line parameters */
		set_lparms(FD);
		ioctl(FD, TIOCEXCL, 0);
		/* Let the other end know it is okay to send requests. */
		printf("%s Ready.\n", SL ? "SLIP" : "PPP");
		goto start_protocol;
	}

	/*
	 * Wait for a dropped packet at the interface
	 */
	if (dodaemon && IM == 0) {
		if (startup(ifname, sysname))
			goto timeout;
		waitdrop(ifname);
	}

	/*
	 * Hunt for an available device
	 */
	(void) signal(SIGHUP, cleanup);
	(void) signal(SIGINT, cleanup);
	(void) signal(SIGTERM, cleanup);

	if ((linkname = hunt()) == 0) {
		if (busymsg++ == 0)
			log("%s: all ports busy\n", sysname);
		if (dodaemon) {
			if (debug)
				printf("All ports busy...\n");
			sleep(60);
			goto daemon_loop;
		}
		errx(1, "all ports busy");
	}
    	log("dialing on %s\n", linkname);
	busymsg = 0;
	if (debug)
		printf("Dialing on %s...\n", linkname);

	/*
	 * Dial out
	 */
	reason = Connect(linkname);
	if (reason) {
		close(FD);
		FD = -1;
		unlock();
		log("%s: %s (%s)\n", sysname, reason, linkname);
		if (dodaemon)
			goto daemon_loop;
		errx(1, "%s (%s)", reason, linkname);
	}
	set_lparms(FD);

	/*
	 * Do all the log sequence
	 */
	if (debug)
		printf("\nLog in...\n");
	for (i = 0; i < 10; i++) {
		if (S[i] != 0) {
			if (debug)
				printf("Send: <%s>\n", printable(S[i]));
			write(FD, S[i], strlen(S[i]));
		}
		timo = T[i];
		if (timo == -1) {
			if (E[i] == 0)
				continue;
			timo = 15;
		}
		if (E[i] == 0)
			sleep(timo);
		else if (expect(FD, E[i], timo)) {
			if (F[i] != 0) {
				if (debug)
					printf("Send F: <%s>\n", printable(F[i]));
				write(FD, F[i], strlen(F[i]));
				if (!expect(FD, E[i], timo))
					continue;
			}
			disconnect("login failed");
			close(FD);
			FD = -1;
			unlock();
			log("%s: login failure (%s, expected %s)\n",
				sysname, linkname, printable(E[i]));
			if (dodaemon)
				goto daemon_loop;
			errx(1, "login failed (expected %s)", printable(E[i]));
		}
	}

	/*
	 * Get old line discipline
	 */
	oldld = 0;      /* default... */
	(void) ioctl(FD, TIOCGETD, &oldld);

	if (debug)
		printf("Set %s line disc.\n", SL ? "SLIP" : "PPP");

	/*
	 * Set SLIP or PPP line discipline
	 */
start_protocol:
	if (SL)
		(void) signal(SIGHUP, SIG_IGN);

	ld = SL ? SLIPDISC : PPPDISC;
	if (ioctl(FD, TIOCSETD, &ld) < 0) {
		disconnect("cant set line discipline");
		unlock();
		err(1, "setting line discipline for %s", linkname);
	}

	if (ioctl(FD, SL ? SLIOCGUNIT : PPPIOCSUNIT, &IN) < 0) {
		switch (errno) {
		case ENXIO:
			warnx("no interface");
			break;
		case EBUSY:
			warnx("interface is busy");
			break;
		default:
			warn("%s: assigning %s unit", linkname, SL ? "SLIP"
							   : "PPP");
			break;
		}
		unlock();
		close(FD);
		FD = -1;

		if (!dialin && dodaemon) {
			sleep(60);
			goto daemon_loop;
		}
		exit(1);
	}

	/*
	 * At this point the following should be true:
	 *   1) We know the interface number.
	 *   2) PPP or SLIP line discipline is set.
	 *
	 * If this is a dialup connection now is the time to set the address
	 * on the interface in case the other end wants us to do the
	 * IP address assignment.  So we would like the interface to be
	 * down at this point -- I'm not sure that that will be true.
	 */

	snprintf(ifname, sizeof ifname, "%s%d", SL ? "sl" : "ppp", IN);
	
	if ((!(dodaemon && IM == 0)) && startup(ifname, sysname))
		goto timeout;

	if (dialin)
		updateutmp();

	/*
	 * Wait for the session to come up.
	 * Needs to happen after interface goes up.
	 */
    	if (SL)
		/* SLIP just starts up */;
	else if (setjmp(wt_env) == 0) {
		osig = signal(SIGALRM, wt_alarm);
		alarm(WT);
		i = waittlu(ifname);
		alarm(0);
		signal(SIGALRM, osig);
		if (i == -1) {
			warn("while waiting for beginning of session");
			log("%s: PPPIOCIPWBOS failed on %s\n", sysname, ifname);
			if (LF)
				loginscript(LF);
			goto timeout;
		}
	} else {
		alarm(0);
		signal(SIGALRM, osig);
		warnx("timed out after %d seconds waiting for beginning of session", WT);
		log("%s: PPPIOCIPWBOS timed out on %s\n", sysname, ifname);
		if (LF)
			loginscript(LF);
		goto timeout;
    	}

	log("%s: connected on %s\n", sysname, ifname);

	/*
	 * Run link up script.
	 */
    	amup = 1;
	if (LU && linkscript(LU)) {
		warnx("%s: up script failed", sysname);
		log("%s: up script failed\n", sysname);
	} else {
		/*
		 * End of session
		 */
		if (background) {
			switch (pid = fork()) {
			case -1:
				warnx("failed to fork while attempting to put the session in the background");
				log("%s: couldn't fork\n", sysname);
				break;
			default:
				exit(0);		/* Parent exits. */
			case 0:
				/*
				 * Child continues in background.
				 */
				if (debug)
					printf("Continue in background.\n");
				setsid();
				chdir("/");
				break;
			}
		}
		if (debug)
			printf("Wait for End Of Session...\n");
		(void) ioctl(FD, SL ? SLIOCWEOS : PPPIOCWEOS);
	}

	if (debug)
		printf("Restore & close line\n");
timeout:
	(void) ioctl(FD, TIOCSETD, &oldld);
	close(FD);      				/* hang up! */
	FD = -1;
	unlock();
	log("%s: end of session\n", sysname);

	/*
	 * Run link down script.
	 */
	if (amup && LD && linkscript(LD)) {
		warnx("%s: down script failed", sysname);
		log("%s: down script failed\n", sysname);
	}

	amup = 0;

	if (dodaemon)
		goto daemon_loop;

    	if (SL == 0 && KU == 0)
		ifconfig(ifname, IF, 0);

	exit(0);
}

int
startup(ifname, sysname)
	char *ifname;
	char *sysname;
{
	char dst[SLIP_ADDRLEN];
	struct ifreq ifr;
	int s;
	struct sockaddr_in *sa;
	int r;

	if (LI && loginscript(LI)) {
		warnx("%s: login script failed", sysname);
		log("%s: login script failed\n", sysname);
		if (LF)
			loginscript(LF);
		return(1);
	}

	/*
	 * Load interface and PPP parameters
	 */
	if (SL == 0) {
		ifconfig(ifname, IF, IFF_UP);
		pppconfig(ifname);
	} else {
		dst[0] = 0;

		if ((s = socket(AF_INET, SOCK_DGRAM, 0)) != -1) {
			strncpy(ifr.ifr_name, ifname, sizeof (ifr.ifr_name));
			sa = (struct sockaddr_in *) &ifr.ifr_addr;

			if (ioctl(s, SIOCGIFDSTADDR, (caddr_t) &ifr) == -1)
				warn("failed to get remote IP address for %s",
				    ifname);
			else {
				snprintf(dst, SLIP_ADDRLEN,
				    "Your address is %s\n",
				    inet_ntoa(sa->sin_addr));
				ioctl(FD, SLIOCSENDADDR, dst);
			}
			close(s);
		}
	}
	return(0);
}

static expect_interrupted;

/* Timeout signal handler */
static void
expalarm()
{
	expect_interrupted = 1;
}

/* Timeout signal handler */
static void
sliphup()
{
	slip_hangup = 1;
}


/*
 * Look up for an expected string or until the timeout is expired
 * Returns 0 if successfull.
 */
int
expect(fd, str, timo)
	int fd;
	char *str;
	int timo;
{
	struct sigaction act, oact;
	char buf[128];
	char nchars, expchars;
	char c;

	if (debug)
		printf("Expect <%s> (%d sec)\nGot: <", printable(str), timo);
	nchars = 0;
	expchars = strlen(str);
	expect_interrupted = 0;
	act.sa_handler = expalarm;
	act.sa_mask = 0;
	act.sa_flags = 0;
	sigaction(SIGALRM, &act, &oact);
	alarm(timo);

	do {
		if (read(fd, &c, 1) != 1 || expect_interrupted) {
			alarm(0);
			sigaction(SIGALRM, &oact, 0);
			if (debug)
				printf("> FAILED\n");
			return (1);
		}
		c &= 0177;
		if (debug) {
			if (c < 040)
				printf("^%c", c | 0100);
			else if (c == 0177)
				printf("^?");
			else
				printf("%c", c);
		}
		buf[nchars++] = c;
		if (nchars > expchars)
			bcopy(buf+1, buf, --nchars);
	} while (nchars < expchars || bcmp(buf, str, expchars));
	alarm(0);
	sigaction(SIGALRM, &oact, 0);
	if (debug)
		printf("> OK\n");
	return (0);
}

/*
 * Make string suitable to print out
 */
char *
printable(str)
	char *str;
{
	static char buf[128];
	char *p = buf;
	register c;

	while (c = *str++) {
		c &= 0377;
		if (c < 040) {
			*p++ = '^';
			*p++ = c | 0100;
		} else if (c == 0177) {
			*p++ = '^';
			*p++ = '?';
		} else if (c & 0200) {
			*p++ = '\\';
			*p++ = '0' + (c >> 6);
			*p++ = '0' + ((c >> 3) & 07);
			*p++ = '0' + (c & 07);
		} else
			*p++ = c;
	}
	*p = 0;
	return (buf);
}

void
wt_alarm()
{
    	longjmp(wt_env, 1);
}

/*
 * Cleanup and exit after a signal.
 */
void
cleanup(sig)
	int sig;
{
	char *signame;

	switch (sig) {
	case SIGHUP:
		signame = "SIGHUP"; break;
	case SIGINT:
		signame = "SIGINT"; break;
	case SIGTERM:
		signame = "SIGTERM"; break;
	default:
		signame = "unknown"; break;
	}
	log("%s: end of session, caught %s\n", sysname, signame);

	if (amup && LD != 0)
		linkscript(LD);
	if (SL == 0)
		ifconfig(ifname, IF, 0);
	(void) ioctl(FD, TIOCSETD, &oldld);
	close(FD);

	unlock();
	exit(1);
}

/*
 * Run link up/down script.
 */

linkscript(path)
	char *path;
{
	char dst[24], src[24];
	struct ifreq ifr;
	int s;
	struct sockaddr_in *sa;
    	int pid;
	int r;

    	dst[0] = 0;
    	src[0] = 0;

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) != -1) {
		strncpy(ifr.ifr_name, ifname, sizeof (ifr.ifr_name));
		sa = (struct sockaddr_in *) &ifr.ifr_addr;

		if (ioctl(s, SIOCGIFADDR, (caddr_t) &ifr) == -1)
			warn("failed to get source IP address for %s", ifname);
		else
			strncpy(src, inet_ntoa(sa->sin_addr), sizeof (src));

		if (ioctl(s, SIOCGIFDSTADDR, (caddr_t) &ifr) == -1)
			warn("failed to get remote IP address for %s", ifname);
		else
			strncpy(dst, inet_ntoa(sa->sin_addr), sizeof (dst));
		close(s);
    	}
	src[sizeof(src)-1] = 0;
	dst[sizeof(dst)-1] = 0;

	if (debug)
		printf("Running \"%s %s %s %s %s\" . . . ", 
			path, sysname, ifname, src, dst);
	/*
	 * XXX - there should be a timeout
	 */
	r = execute(path, sysname, ifname, src, dst, 0);
	if (debug)
		printf("done.\n");
	return(r);
}

/*
 * Run link up/down script.
 */

loginscript(path)
	char *path;
{
	int r;

	if (debug)
		printf("Running \"%s %s %s\" . . . ", path, sysname, ifname);
	/*
	 * XXX - there should be a timeout
	 */
	r = execute(path, sysname, ifname, 0);
	if (debug)
		printf("done.\n");
	return(r);
}

void
updateutmp()
{
	struct utmp ut;
	int fd;
	int tty;

	struct ifreq ifr;
	struct sockaddr_in *sa;

	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		log("opening socket: %s\n", strerror(errno));
		return;
	}

	strncpy(ifr.ifr_name, ifname, sizeof (ifr.ifr_name));
	sa = (struct sockaddr_in *) &ifr.ifr_addr;

	if (ioctl(fd, SIOCGIFDSTADDR, (caddr_t) &ifr) == -1) {
		log("getting dst address for utmp: %s\n", strerror(errno));
		close(fd);
		return;
	}
	close(fd);

	if ((tty = ttyslot()) <= 0) {
		log("getting tty slot: %s\n", strerror(errno));
		return;
	}
	if ((fd = open(_PATH_UTMP, O_RDWR)) < 0) {
		log("opening " _PATH_UTMP ": %s\n", strerror(errno));
		return;
	}

	(void)lseek(fd, (off_t)(tty * sizeof(struct utmp)), L_SET);
	if (read(fd, &ut, sizeof(struct utmp)) != sizeof(struct utmp)) {
		log("reading utmp entry: %s\n", strerror(errno));
		return;
	}
	strncpy(ut.ut_host, inet_ntoa(sa->sin_addr), UT_HOSTSIZE);
	(void)lseek(fd, (off_t)(tty * sizeof(struct utmp)), L_SET);
	(void)write(fd, &ut, sizeof(struct utmp));
	(void)close(fd);
}

/*
 * The code below is derived from lib/libc/stdlib/system.c
 *
 * Copyright (c) 1988, 1993
 *	The Regents of the University of California.  All rights reserved.
 */

#define	MAXARGS	64
#define	MAXOPTS	(MAXARGS - 4)

int
execute(char *script, ...)
{
    	va_list ap;
	int pstat;
	pid_t pid;
	int omask;
	sig_t intsave, quitsave;
	char *argv[MAXARGS];
    	char **av = argv;
	static char *script_opts[MAXOPTS] = { 0, };
	char **sov;

	if (SO && script_opts[0] == NULL) {
		sov = script_opts;
		while (sov < &script_opts[MAXOPTS-1] &&
		    (*sov = strsep(&SO, " \t")) != NULL)
			if (**sov != '\0')
				++sov;
		*sov = 0;
		SO = 0;
	}

    	if (*av = strchr(script, '/'))
		++*av;
	else
		*av = script;
	++av;

	sov = script_opts;
    	while (av < &argv[MAXARGS-2] && (*av = *sov++))
		++av;

    	va_start(ap, script);
    	while (av < &argv[MAXARGS-2] && (*av = va_arg(ap, char *)))
		++av;
    	va_end(ap);

	*av = 0;

	omask = sigblock(sigmask(SIGCHLD));

	switch (pid = vfork()) {
	case -1:			/* error */
		(void)sigsetmask(omask);
		warn("%s", script);
		return(-1);
	case 0:				/* child */
    	    	if (sgid != -1)
			setuid(sgid);
    	    	if (suid != -1)
			setuid(suid);
		(void)sigsetmask(omask);
		execv(script, argv);
		_exit(127);
	}

	intsave = signal(SIGINT, SIG_IGN);
	quitsave = signal(SIGQUIT, SIG_IGN);
	pid = waitpid(pid, &pstat, 0);
	(void)sigsetmask(omask);
	(void)signal(SIGINT, intsave);
	(void)signal(SIGQUIT, quitsave);

	return (pid == -1 ? -1 : WIFEXITED(pstat) ? WEXITSTATUS(pstat) : -1);
}
