/*	BSDI kcore.c,v 1.1 1995/11/26 18:00:04 donn Exp	*/

/* Work with kernel crash dumps and live systems through kvm.  This
   module was developed at Lawrence Berkeley Laboratory by Steven
   McCanne (mccanne@ee.lbl.gov).  It is derived from the gdb module core.c.
   The GNU copyright is therefore maintained.

   Copyright 1986, 1987, 1989, 1991 Free Software Foundation, Inc.
   
   This file is part of GDB.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include "defs.h"
#include "frame.h"  /* required by inferior.h */
#include "inferior.h"
#include "symtab.h"
#include "command.h"
#include "bfd.h"
#include "target.h"
#include "gdbcore.h"

#include <kvm.h>
#include <sys/stat.h>

extern int xfer_memory ();
extern void child_attach (), child_create_inferior ();

extern char registers[];

/* KVM library handle */

kvm_t *kd;

/* Forward decl */
extern struct target_ops kernel_core_ops;


/* Discard all vestiges of any previous core file
   and mark data and stack spaces as empty.  */

/* ARGSUSED */
void
kernel_core_close (quitting)
	int quitting;
{
	if (kd != 0) {
		kvm_close(kd);
		kd = 0;
	}
}

/* This routine opens and sets up the core file bfd */

void
kernel_core_open(filename, from_tty)
	char *filename;
	int from_tty;
{
	char *cp;
	struct cleanup *old_chain;
	int ontop;
	char *execfile;
	struct stat stb;
	kvm_t *temp_kd;

	kernel_debugging = 1;

	if (kd != 0) 
		error("kvm already target -- must detach first");

	target_preopen(from_tty);
	if (filename == 0)
		error ("No core file specified.");
	filename = tilde_expand (filename);
	if (filename[0] != '/') {
		cp = concat (current_directory, "/", filename, NULL);
		free(filename);
		filename = cp;
	}
	old_chain = make_cleanup (free, filename);
	execfile = exec_bfd ? bfd_get_filename(exec_bfd) : 0;
	if (execfile == 0) 
		error("No executable image specified");

	/*
	 * If we want the active memory file, just use a null arg for kvm.
	 * The SunOS kvm can't read from the default swap device, unless
	 * /dev/mem is indicated with a null pointer.  This has got to be 
	 * a bug.
	 */
	if (stat(filename, &stb) == 0 && S_ISCHR(stb.st_mode))
		filename = 0;

	temp_kd = kvm_open(execfile, filename, (char *)0,
		      write_files ? O_RDWR : O_RDONLY, "gdb");
	if (temp_kd == 0)
		error("Cannot open kernel core image");

	unpush_target (&kernel_core_ops);
	kd = temp_kd;
	old_chain = make_cleanup(kernel_core_close, (int)kd);
#ifdef notdef
	validate_files ();
#endif
	ontop = !push_target (&kernel_core_ops);
	discard_cleanups (old_chain);
	
	if (ontop) {
		/* Fetch all registers from core file */
		target_fetch_registers (-1);
		
		/* Now, set up the frame cache, and print the top of stack */
		flush_cached_frames();
#if 1
		set_current_frame(create_new_frame(read_register(FP_REGNUM),
						   read_pc()));
#endif
		select_frame(get_current_frame(), 0);
		print_stack_frame(selected_frame, selected_frame_level, 1);
	} else {
		printf (
			"Warning: you won't be able to access this core file until you terminate\n\
your %s; do ``info files''\n", current_target.to_longname);
	}
	/* Machine dependent call to print out panic string etc. */
	kerninfo();
}

void
kernel_core_detach (args, from_tty)
	char *args;
	int from_tty;
{
	if (args)
		error ("Too many arguments");
	unpush_target (&kernel_core_ops);
	if (from_tty)
		printf ("No kernel core file now.\n");
}

static void
kernel_core_files_info (t)
	struct target_ops *t;
{
#ifdef notdef
	struct section_table *p;
	
	printf_filtered ("\t`%s', ", bfd_get_filename(core_bfd));
	wrap_here ("        ");
	printf_filtered ("file type %s.\n", bfd_get_target(core_bfd));
	
	for (p = t->sections; p < t->sections_end; p++) {
		printf_filtered ("\t%s", local_hex_string_custom (p->addr, "08"));
		printf_filtered (" - %s is %s",
				 local_hex_string_custom (p->endaddr, "08"),
				 bfd_section_name (p->bfd, p->sec_ptr));
		if (p->bfd != core_bfd) {
			printf_filtered (" in %s", bfd_get_filename (p->bfd));
		}
		printf_filtered ("\n");
	}
#endif
}

/*
 * Called by the machine dependent module to set a user context.
 * We call kvm_getu() for this desired side effect.
 * BSD kvm doesn't need to do this.
 */
kernel_getu(p)
	u_long *p;
{
	return (0);
}

#ifndef KERNEL_XFER_MEMORY
int
kernel_xfer_memory(addr, cp, len, write, target)
     CORE_ADDR addr;
     char *cp;
     int len;
     int write;
     struct target_ops *target;
{
	if (write)
		return kvm_write(kd, addr, cp, len);
	else
		return kvm_read(kd, addr, cp, len);
}
#endif


static int
ignore (addr, contents)
     CORE_ADDR addr;
     char *contents;
{
  return 0;
}

/* 
 * In target dependent module.
 */
extern void kernel_core_registers();

struct target_ops kernel_core_ops = {
	"kvm",
	"Kernel core file",
	"Use a kernel core file as a target.  Specify the filename of the core file.",
	kernel_core_open,
	kernel_core_close,
	find_default_attach,	/* to_attach */
	kernel_core_detach,
	0,			/* to_resume */
	0,			/* to_wait */
	kernel_core_registers, 
	0,			/* to_store_registers */
	0,			/* to_prepare_to_store */
	kernel_xfer_memory,
	kernel_core_files_info,
	ignore,			/* to_insert_breakpoint */
	ignore,			/* to_remove_breakpoint */
	0,			/* to_terminal_init */
	0,			/* to_terminal_inferior */
	0,			/* to_terminal_ours_for_output */
	0,			/* to_terminal_ours */
	0,			/* to_terminal_info */
	0,			/* to_kill */
	0,			/* to_load */
	0,			/* to_lookup_symbol */
	find_default_create_inferior,
	0,			/* to_mourn_inferior */
	0,			/* to_can_run */
	0,			/* to_notice_signals */
	0,			/* to_thread_alive */
	0,			/* to_stop */
	dummy_stratum,
	0,			/* reserved */
	0,			/* to_has_all_memory */
	1,			/* to_has_memory */
	1,			/* to_has_stack */
	1,			/* to_has_registers */
	0,			/* to_has_execution */
	0,			/* to_sections */
	0,			/* to_sections_end */
	OPS_MAGIC,		/* Always the last thing */
};

void
_initialize_kernel_core()
{
#ifdef notdef
	
	add_com ("kcore", class_files, core_file_command,
		 "Use FILE as core dump for examining memory and registers.\n\
No arg means have no core file.  This command has been superseded by the\n\
`target core' and `detach' commands.");
#endif
	add_target (&kernel_core_ops);
}
